/**
 * Framework7 8.3.3
 * Full featured mobile HTML framework for building iOS & Android apps
 * https://framework7.io/
 *
 * Copyright 2014-2024 Vladimir Kharlampidi
 *
 * Released under the MIT License
 *
 * Released on: March 25, 2024
 */

(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
    typeof define === 'function' && define.amd ? define(factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.Framework7 = factory());
})(this, (function () { 'use strict';

    /**
     * SSR Window 4.0.2
     * Better handling for window object in SSR environment
     * https://github.com/nolimits4web/ssr-window
     *
     * Copyright 2021, Vladimir Kharlampidi
     *
     * Licensed under MIT
     *
     * Released on: December 13, 2021
     */
    /* eslint-disable no-param-reassign */
    function isObject$4(obj) {
      return obj !== null && typeof obj === 'object' && 'constructor' in obj && obj.constructor === Object;
    }
    function extend$4(target, src) {
      if (target === void 0) {
        target = {};
      }
      if (src === void 0) {
        src = {};
      }
      Object.keys(src).forEach(key => {
        if (typeof target[key] === 'undefined') target[key] = src[key];else if (isObject$4(src[key]) && isObject$4(target[key]) && Object.keys(src[key]).length > 0) {
          extend$4(target[key], src[key]);
        }
      });
    }
    const ssrDocument$1 = {
      body: {},
      addEventListener() {},
      removeEventListener() {},
      activeElement: {
        blur() {},
        nodeName: ''
      },
      querySelector() {
        return null;
      },
      querySelectorAll() {
        return [];
      },
      getElementById() {
        return null;
      },
      createEvent() {
        return {
          initEvent() {}
        };
      },
      createElement() {
        return {
          children: [],
          childNodes: [],
          style: {},
          setAttribute() {},
          getElementsByTagName() {
            return [];
          }
        };
      },
      createElementNS() {
        return {};
      },
      importNode() {
        return null;
      },
      location: {
        hash: '',
        host: '',
        hostname: '',
        href: '',
        origin: '',
        pathname: '',
        protocol: '',
        search: ''
      }
    };
    function getDocument$1() {
      const doc = typeof document !== 'undefined' ? document : {};
      extend$4(doc, ssrDocument$1);
      return doc;
    }
    const ssrWindow$1 = {
      document: ssrDocument$1,
      navigator: {
        userAgent: ''
      },
      location: {
        hash: '',
        host: '',
        hostname: '',
        href: '',
        origin: '',
        pathname: '',
        protocol: '',
        search: ''
      },
      history: {
        replaceState() {},
        pushState() {},
        go() {},
        back() {}
      },
      CustomEvent: function CustomEvent() {
        return this;
      },
      addEventListener() {},
      removeEventListener() {},
      getComputedStyle() {
        return {
          getPropertyValue() {
            return '';
          }
        };
      },
      Image() {},
      Date() {},
      screen: {},
      setTimeout() {},
      clearTimeout() {},
      matchMedia() {
        return {};
      },
      requestAnimationFrame(callback) {
        if (typeof setTimeout === 'undefined') {
          callback();
          return null;
        }
        return setTimeout(callback, 0);
      },
      cancelAnimationFrame(id) {
        if (typeof setTimeout === 'undefined') {
          return;
        }
        clearTimeout(id);
      }
    };
    function getWindow$1() {
      const win = typeof window !== 'undefined' ? window : {};
      extend$4(win, ssrWindow$1);
      return win;
    }

    /**
     * Dom7 4.0.6
     * Minimalistic JavaScript library for DOM manipulation, with a jQuery-compatible API
     * https://framework7.io/docs/dom7.html
     *
     * Copyright 2023, Vladimir Kharlampidi
     *
     * Licensed under MIT
     *
     * Released on: February 2, 2023
     */

    /* eslint-disable no-proto */
    function makeReactive(obj) {
      const proto = obj.__proto__;
      Object.defineProperty(obj, '__proto__', {
        get() {
          return proto;
        },
        set(value) {
          proto.__proto__ = value;
        }
      });
    }
    class Dom7 extends Array {
      constructor(items) {
        if (typeof items === 'number') {
          super(items);
        } else {
          super(...(items || []));
          makeReactive(this);
        }
      }
    }
    function arrayFlat(arr) {
      if (arr === void 0) {
        arr = [];
      }
      const res = [];
      arr.forEach(el => {
        if (Array.isArray(el)) {
          res.push(...arrayFlat(el));
        } else {
          res.push(el);
        }
      });
      return res;
    }
    function arrayFilter(arr, callback) {
      return Array.prototype.filter.call(arr, callback);
    }
    function arrayUnique(arr) {
      const uniqueArray = [];
      for (let i = 0; i < arr.length; i += 1) {
        if (uniqueArray.indexOf(arr[i]) === -1) uniqueArray.push(arr[i]);
      }
      return uniqueArray;
    }
    function toCamelCase$1(string) {
      return string.toLowerCase().replace(/-(.)/g, (match, group) => group.toUpperCase());
    }

    // eslint-disable-next-line

    function qsa(selector, context) {
      if (typeof selector !== 'string') {
        return [selector];
      }
      const a = [];
      const res = context.querySelectorAll(selector);
      for (let i = 0; i < res.length; i += 1) {
        a.push(res[i]);
      }
      return a;
    }
    function $$1(selector, context) {
      const window = getWindow$1();
      const document = getDocument$1();
      let arr = [];
      if (!context && selector instanceof Dom7) {
        return selector;
      }
      if (!selector) {
        return new Dom7(arr);
      }
      if (typeof selector === 'string') {
        const html = selector.trim();
        if (html.indexOf('<') >= 0 && html.indexOf('>') >= 0) {
          let toCreate = 'div';
          if (html.indexOf('<li') === 0) toCreate = 'ul';
          if (html.indexOf('<tr') === 0) toCreate = 'tbody';
          if (html.indexOf('<td') === 0 || html.indexOf('<th') === 0) toCreate = 'tr';
          if (html.indexOf('<tbody') === 0) toCreate = 'table';
          if (html.indexOf('<option') === 0) toCreate = 'select';
          const tempParent = document.createElement(toCreate);
          tempParent.innerHTML = html;
          for (let i = 0; i < tempParent.childNodes.length; i += 1) {
            arr.push(tempParent.childNodes[i]);
          }
        } else {
          arr = qsa(selector.trim(), context || document);
        } // arr = qsa(selector, document);
      } else if (selector.nodeType || selector === window || selector === document) {
        arr.push(selector);
      } else if (Array.isArray(selector)) {
        if (selector instanceof Dom7) return selector;
        arr = selector;
      }
      return new Dom7(arrayUnique(arr));
    }
    $$1.fn = Dom7.prototype;

    // eslint-disable-next-line

    function addClass() {
      for (var _len = arguments.length, classes = new Array(_len), _key = 0; _key < _len; _key++) {
        classes[_key] = arguments[_key];
      }
      const classNames = arrayFlat(classes.map(c => c.split(' ')));
      this.forEach(el => {
        el.classList.add(...classNames);
      });
      return this;
    }
    function removeClass() {
      for (var _len2 = arguments.length, classes = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        classes[_key2] = arguments[_key2];
      }
      const classNames = arrayFlat(classes.map(c => c.split(' ')));
      this.forEach(el => {
        el.classList.remove(...classNames);
      });
      return this;
    }
    function toggleClass() {
      for (var _len3 = arguments.length, classes = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
        classes[_key3] = arguments[_key3];
      }
      const classNames = arrayFlat(classes.map(c => c.split(' ')));
      this.forEach(el => {
        classNames.forEach(className => {
          el.classList.toggle(className);
        });
      });
    }
    function hasClass() {
      for (var _len4 = arguments.length, classes = new Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
        classes[_key4] = arguments[_key4];
      }
      const classNames = arrayFlat(classes.map(c => c.split(' ')));
      return arrayFilter(this, el => {
        return classNames.filter(className => el.classList.contains(className)).length > 0;
      }).length > 0;
    }
    function attr(attrs, value) {
      if (arguments.length === 1 && typeof attrs === 'string') {
        // Get attr
        if (this[0]) return this[0].getAttribute(attrs);
        return undefined;
      } // Set attrs

      for (let i = 0; i < this.length; i += 1) {
        if (arguments.length === 2) {
          // String
          this[i].setAttribute(attrs, value);
        } else {
          // Object
          for (const attrName in attrs) {
            this[i][attrName] = attrs[attrName];
            this[i].setAttribute(attrName, attrs[attrName]);
          }
        }
      }
      return this;
    }
    function removeAttr(attr) {
      for (let i = 0; i < this.length; i += 1) {
        this[i].removeAttribute(attr);
      }
      return this;
    }
    function prop(props, value) {
      if (arguments.length === 1 && typeof props === 'string') {
        // Get prop
        if (this[0]) return this[0][props];
      } else {
        // Set props
        for (let i = 0; i < this.length; i += 1) {
          if (arguments.length === 2) {
            // String
            this[i][props] = value;
          } else {
            // Object
            for (const propName in props) {
              this[i][propName] = props[propName];
            }
          }
        }
        return this;
      }
      return this;
    }
    function data(key, value) {
      let el;
      if (typeof value === 'undefined') {
        el = this[0];
        if (!el) return undefined; // Get value

        if (el.dom7ElementDataStorage && key in el.dom7ElementDataStorage) {
          return el.dom7ElementDataStorage[key];
        }
        const dataKey = el.getAttribute(`data-${key}`);
        if (dataKey) {
          return dataKey;
        }
        return undefined;
      } // Set value

      for (let i = 0; i < this.length; i += 1) {
        el = this[i];
        if (!el.dom7ElementDataStorage) el.dom7ElementDataStorage = {};
        el.dom7ElementDataStorage[key] = value;
      }
      return this;
    }
    function removeData(key) {
      for (let i = 0; i < this.length; i += 1) {
        const el = this[i];
        if (el.dom7ElementDataStorage && el.dom7ElementDataStorage[key]) {
          el.dom7ElementDataStorage[key] = null;
          delete el.dom7ElementDataStorage[key];
        }
      }
    }
    function dataset() {
      const el = this[0];
      if (!el) return undefined;
      const dataset = {}; // eslint-disable-line

      if (el.dataset) {
        for (const dataKey in el.dataset) {
          dataset[dataKey] = el.dataset[dataKey];
        }
      } else {
        for (let i = 0; i < el.attributes.length; i += 1) {
          const attr = el.attributes[i];
          if (attr.name.indexOf('data-') >= 0) {
            dataset[toCamelCase$1(attr.name.split('data-')[1])] = attr.value;
          }
        }
      }
      for (const key in dataset) {
        if (dataset[key] === 'false') dataset[key] = false;else if (dataset[key] === 'true') dataset[key] = true;else if (parseFloat(dataset[key]) === dataset[key] * 1) dataset[key] *= 1;
      }
      return dataset;
    }
    function val(value) {
      if (typeof value === 'undefined') {
        // get value
        const el = this[0];
        if (!el) return undefined;
        if (el.multiple && el.nodeName.toLowerCase() === 'select') {
          const values = [];
          for (let i = 0; i < el.selectedOptions.length; i += 1) {
            values.push(el.selectedOptions[i].value);
          }
          return values;
        }
        return el.value;
      } // set value

      for (let i = 0; i < this.length; i += 1) {
        const el = this[i];
        if (Array.isArray(value) && el.multiple && el.nodeName.toLowerCase() === 'select') {
          for (let j = 0; j < el.options.length; j += 1) {
            el.options[j].selected = value.indexOf(el.options[j].value) >= 0;
          }
        } else {
          el.value = value;
        }
      }
      return this;
    }
    function value(value) {
      return this.val(value);
    }
    function transform(transform) {
      for (let i = 0; i < this.length; i += 1) {
        this[i].style.transform = transform;
      }
      return this;
    }
    function transition$1(duration) {
      for (let i = 0; i < this.length; i += 1) {
        this[i].style.transitionDuration = typeof duration !== 'string' ? `${duration}ms` : duration;
      }
      return this;
    }
    function on() {
      for (var _len5 = arguments.length, args = new Array(_len5), _key5 = 0; _key5 < _len5; _key5++) {
        args[_key5] = arguments[_key5];
      }
      let [eventType, targetSelector, listener, capture] = args;
      if (typeof args[1] === 'function') {
        [eventType, listener, capture] = args;
        targetSelector = undefined;
      }
      if (!capture) capture = false;
      function handleLiveEvent(e) {
        const target = e.target;
        if (!target) return;
        const eventData = e.target.dom7EventData || [];
        if (eventData.indexOf(e) < 0) {
          eventData.unshift(e);
        }
        if ($$1(target).is(targetSelector)) listener.apply(target, eventData);else {
          const parents = $$1(target).parents(); // eslint-disable-line

          for (let k = 0; k < parents.length; k += 1) {
            if ($$1(parents[k]).is(targetSelector)) listener.apply(parents[k], eventData);
          }
        }
      }
      function handleEvent(e) {
        const eventData = e && e.target ? e.target.dom7EventData || [] : [];
        if (eventData.indexOf(e) < 0) {
          eventData.unshift(e);
        }
        listener.apply(this, eventData);
      }
      const events = eventType.split(' ');
      let j;
      for (let i = 0; i < this.length; i += 1) {
        const el = this[i];
        if (!targetSelector) {
          for (j = 0; j < events.length; j += 1) {
            const event = events[j];
            if (!el.dom7Listeners) el.dom7Listeners = {};
            if (!el.dom7Listeners[event]) el.dom7Listeners[event] = [];
            el.dom7Listeners[event].push({
              listener,
              proxyListener: handleEvent
            });
            el.addEventListener(event, handleEvent, capture);
          }
        } else {
          // Live events
          for (j = 0; j < events.length; j += 1) {
            const event = events[j];
            if (!el.dom7LiveListeners) el.dom7LiveListeners = {};
            if (!el.dom7LiveListeners[event]) el.dom7LiveListeners[event] = [];
            el.dom7LiveListeners[event].push({
              listener,
              proxyListener: handleLiveEvent
            });
            el.addEventListener(event, handleLiveEvent, capture);
          }
        }
      }
      return this;
    }
    function off() {
      for (var _len6 = arguments.length, args = new Array(_len6), _key6 = 0; _key6 < _len6; _key6++) {
        args[_key6] = arguments[_key6];
      }
      let [eventType, targetSelector, listener, capture] = args;
      if (typeof args[1] === 'function') {
        [eventType, listener, capture] = args;
        targetSelector = undefined;
      }
      if (!capture) capture = false;
      const events = eventType.split(' ');
      for (let i = 0; i < events.length; i += 1) {
        const event = events[i];
        for (let j = 0; j < this.length; j += 1) {
          const el = this[j];
          let handlers;
          if (!targetSelector && el.dom7Listeners) {
            handlers = el.dom7Listeners[event];
          } else if (targetSelector && el.dom7LiveListeners) {
            handlers = el.dom7LiveListeners[event];
          }
          if (handlers && handlers.length) {
            for (let k = handlers.length - 1; k >= 0; k -= 1) {
              const handler = handlers[k];
              if (listener && handler.listener === listener) {
                el.removeEventListener(event, handler.proxyListener, capture);
                handlers.splice(k, 1);
              } else if (listener && handler.listener && handler.listener.dom7proxy && handler.listener.dom7proxy === listener) {
                el.removeEventListener(event, handler.proxyListener, capture);
                handlers.splice(k, 1);
              } else if (!listener) {
                el.removeEventListener(event, handler.proxyListener, capture);
                handlers.splice(k, 1);
              }
            }
          }
        }
      }
      return this;
    }
    function once() {
      const dom = this;
      for (var _len7 = arguments.length, args = new Array(_len7), _key7 = 0; _key7 < _len7; _key7++) {
        args[_key7] = arguments[_key7];
      }
      let [eventName, targetSelector, listener, capture] = args;
      if (typeof args[1] === 'function') {
        [eventName, listener, capture] = args;
        targetSelector = undefined;
      }
      function onceHandler() {
        for (var _len8 = arguments.length, eventArgs = new Array(_len8), _key8 = 0; _key8 < _len8; _key8++) {
          eventArgs[_key8] = arguments[_key8];
        }
        listener.apply(this, eventArgs);
        dom.off(eventName, targetSelector, onceHandler, capture);
        if (onceHandler.dom7proxy) {
          delete onceHandler.dom7proxy;
        }
      }
      onceHandler.dom7proxy = listener;
      return dom.on(eventName, targetSelector, onceHandler, capture);
    }
    function trigger() {
      const window = getWindow$1();
      for (var _len9 = arguments.length, args = new Array(_len9), _key9 = 0; _key9 < _len9; _key9++) {
        args[_key9] = arguments[_key9];
      }
      const events = args[0].split(' ');
      const eventData = args[1];
      for (let i = 0; i < events.length; i += 1) {
        const event = events[i];
        for (let j = 0; j < this.length; j += 1) {
          const el = this[j];
          if (window.CustomEvent) {
            const evt = new window.CustomEvent(event, {
              detail: eventData,
              bubbles: true,
              cancelable: true
            });
            el.dom7EventData = args.filter((data, dataIndex) => dataIndex > 0);
            el.dispatchEvent(evt);
            el.dom7EventData = [];
            delete el.dom7EventData;
          }
        }
      }
      return this;
    }
    function transitionStart$1(callback) {
      const dom = this;
      function fireCallBack(e) {
        if (e.target !== this) return;
        callback.call(this, e);
        dom.off('transitionstart', fireCallBack);
      }
      if (callback) {
        dom.on('transitionstart', fireCallBack);
      }
      return this;
    }
    function transitionEnd$1(callback) {
      const dom = this;
      function fireCallBack(e) {
        if (e.target !== this) return;
        callback.call(this, e);
        dom.off('transitionend', fireCallBack);
      }
      if (callback) {
        dom.on('transitionend', fireCallBack);
      }
      return this;
    }
    function animationEnd(callback) {
      const dom = this;
      function fireCallBack(e) {
        if (e.target !== this) return;
        callback.call(this, e);
        dom.off('animationend', fireCallBack);
      }
      if (callback) {
        dom.on('animationend', fireCallBack);
      }
      return this;
    }
    function width() {
      const window = getWindow$1();
      if (this[0] === window) {
        return window.innerWidth;
      }
      if (this.length > 0) {
        return parseFloat(this.css('width'));
      }
      return null;
    }
    function outerWidth(includeMargins) {
      if (this.length > 0) {
        if (includeMargins) {
          const styles = this.styles();
          return this[0].offsetWidth + parseFloat(styles.getPropertyValue('margin-right')) + parseFloat(styles.getPropertyValue('margin-left'));
        }
        return this[0].offsetWidth;
      }
      return null;
    }
    function height() {
      const window = getWindow$1();
      if (this[0] === window) {
        return window.innerHeight;
      }
      if (this.length > 0) {
        return parseFloat(this.css('height'));
      }
      return null;
    }
    function outerHeight(includeMargins) {
      if (this.length > 0) {
        if (includeMargins) {
          const styles = this.styles();
          return this[0].offsetHeight + parseFloat(styles.getPropertyValue('margin-top')) + parseFloat(styles.getPropertyValue('margin-bottom'));
        }
        return this[0].offsetHeight;
      }
      return null;
    }
    function offset() {
      if (this.length > 0) {
        const window = getWindow$1();
        const document = getDocument$1();
        const el = this[0];
        const box = el.getBoundingClientRect();
        const body = document.body;
        const clientTop = el.clientTop || body.clientTop || 0;
        const clientLeft = el.clientLeft || body.clientLeft || 0;
        const scrollTop = el === window ? window.scrollY : el.scrollTop;
        const scrollLeft = el === window ? window.scrollX : el.scrollLeft;
        return {
          top: box.top + scrollTop - clientTop,
          left: box.left + scrollLeft - clientLeft
        };
      }
      return null;
    }
    function hide() {
      for (let i = 0; i < this.length; i += 1) {
        this[i].style.display = 'none';
      }
      return this;
    }
    function show() {
      const window = getWindow$1();
      for (let i = 0; i < this.length; i += 1) {
        const el = this[i];
        if (el.style.display === 'none') {
          el.style.display = '';
        }
        if (window.getComputedStyle(el, null).getPropertyValue('display') === 'none') {
          // Still not visible
          el.style.display = 'block';
        }
      }
      return this;
    }
    function styles() {
      const window = getWindow$1();
      if (this[0]) return window.getComputedStyle(this[0], null);
      return {};
    }
    function css(props, value) {
      const window = getWindow$1();
      let i;
      if (arguments.length === 1) {
        if (typeof props === 'string') {
          // .css('width')
          if (this[0]) return window.getComputedStyle(this[0], null).getPropertyValue(props);
        } else {
          // .css({ width: '100px' })
          for (i = 0; i < this.length; i += 1) {
            for (const prop in props) {
              this[i].style[prop] = props[prop];
            }
          }
          return this;
        }
      }
      if (arguments.length === 2 && typeof props === 'string') {
        // .css('width', '100px')
        for (i = 0; i < this.length; i += 1) {
          this[i].style[props] = value;
        }
        return this;
      }
      return this;
    }
    function each(callback) {
      if (!callback) return this;
      this.forEach((el, index) => {
        callback.apply(el, [el, index]);
      });
      return this;
    }
    function filter(callback) {
      const result = arrayFilter(this, callback);
      return $$1(result);
    }
    function html(html) {
      if (typeof html === 'undefined') {
        return this[0] ? this[0].innerHTML : null;
      }
      for (let i = 0; i < this.length; i += 1) {
        this[i].innerHTML = html;
      }
      return this;
    }
    function text(text) {
      if (typeof text === 'undefined') {
        return this[0] ? this[0].textContent.trim() : null;
      }
      for (let i = 0; i < this.length; i += 1) {
        this[i].textContent = text;
      }
      return this;
    }
    function is(selector) {
      const window = getWindow$1();
      const document = getDocument$1();
      const el = this[0];
      let compareWith;
      let i;
      if (!el || typeof selector === 'undefined') return false;
      if (typeof selector === 'string') {
        if (el.matches) return el.matches(selector);
        if (el.webkitMatchesSelector) return el.webkitMatchesSelector(selector);
        if (el.msMatchesSelector) return el.msMatchesSelector(selector);
        compareWith = $$1(selector);
        for (i = 0; i < compareWith.length; i += 1) {
          if (compareWith[i] === el) return true;
        }
        return false;
      }
      if (selector === document) {
        return el === document;
      }
      if (selector === window) {
        return el === window;
      }
      if (selector.nodeType || selector instanceof Dom7) {
        compareWith = selector.nodeType ? [selector] : selector;
        for (i = 0; i < compareWith.length; i += 1) {
          if (compareWith[i] === el) return true;
        }
        return false;
      }
      return false;
    }
    function index() {
      let child = this[0];
      let i;
      if (child) {
        i = 0; // eslint-disable-next-line

        while ((child = child.previousSibling) !== null) {
          if (child.nodeType === 1) i += 1;
        }
        return i;
      }
      return undefined;
    }
    function eq(index) {
      if (typeof index === 'undefined') return this;
      const length = this.length;
      if (index > length - 1) {
        return $$1([]);
      }
      if (index < 0) {
        const returnIndex = length + index;
        if (returnIndex < 0) return $$1([]);
        return $$1([this[returnIndex]]);
      }
      return $$1([this[index]]);
    }
    function append() {
      let newChild;
      const document = getDocument$1();
      for (let k = 0; k < arguments.length; k += 1) {
        newChild = k < 0 || arguments.length <= k ? undefined : arguments[k];
        for (let i = 0; i < this.length; i += 1) {
          if (typeof newChild === 'string') {
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newChild;
            while (tempDiv.firstChild) {
              this[i].appendChild(tempDiv.firstChild);
            }
          } else if (newChild instanceof Dom7) {
            for (let j = 0; j < newChild.length; j += 1) {
              this[i].appendChild(newChild[j]);
            }
          } else {
            this[i].appendChild(newChild);
          }
        }
      }
      return this;
    }
    function appendTo(parent) {
      $$1(parent).append(this);
      return this;
    }
    function prepend(newChild) {
      const document = getDocument$1();
      let i;
      let j;
      for (i = 0; i < this.length; i += 1) {
        if (typeof newChild === 'string') {
          const tempDiv = document.createElement('div');
          tempDiv.innerHTML = newChild;
          for (j = tempDiv.childNodes.length - 1; j >= 0; j -= 1) {
            this[i].insertBefore(tempDiv.childNodes[j], this[i].childNodes[0]);
          }
        } else if (newChild instanceof Dom7) {
          for (j = 0; j < newChild.length; j += 1) {
            this[i].insertBefore(newChild[j], this[i].childNodes[0]);
          }
        } else {
          this[i].insertBefore(newChild, this[i].childNodes[0]);
        }
      }
      return this;
    }
    function prependTo(parent) {
      $$1(parent).prepend(this);
      return this;
    }
    function insertBefore$1(selector) {
      const before = $$1(selector);
      for (let i = 0; i < this.length; i += 1) {
        if (before.length === 1) {
          before[0].parentNode.insertBefore(this[i], before[0]);
        } else if (before.length > 1) {
          for (let j = 0; j < before.length; j += 1) {
            before[j].parentNode.insertBefore(this[i].cloneNode(true), before[j]);
          }
        }
      }
    }
    function insertAfter(selector) {
      const after = $$1(selector);
      for (let i = 0; i < this.length; i += 1) {
        if (after.length === 1) {
          after[0].parentNode.insertBefore(this[i], after[0].nextSibling);
        } else if (after.length > 1) {
          for (let j = 0; j < after.length; j += 1) {
            after[j].parentNode.insertBefore(this[i].cloneNode(true), after[j].nextSibling);
          }
        }
      }
    }
    function next(selector) {
      if (this.length > 0) {
        if (selector) {
          if (this[0].nextElementSibling && $$1(this[0].nextElementSibling).is(selector)) {
            return $$1([this[0].nextElementSibling]);
          }
          return $$1([]);
        }
        if (this[0].nextElementSibling) return $$1([this[0].nextElementSibling]);
        return $$1([]);
      }
      return $$1([]);
    }
    function nextAll(selector) {
      const nextEls = [];
      let el = this[0];
      if (!el) return $$1([]);
      while (el.nextElementSibling) {
        const next = el.nextElementSibling; // eslint-disable-line

        if (selector) {
          if ($$1(next).is(selector)) nextEls.push(next);
        } else nextEls.push(next);
        el = next;
      }
      return $$1(nextEls);
    }
    function prev(selector) {
      if (this.length > 0) {
        const el = this[0];
        if (selector) {
          if (el.previousElementSibling && $$1(el.previousElementSibling).is(selector)) {
            return $$1([el.previousElementSibling]);
          }
          return $$1([]);
        }
        if (el.previousElementSibling) return $$1([el.previousElementSibling]);
        return $$1([]);
      }
      return $$1([]);
    }
    function prevAll(selector) {
      const prevEls = [];
      let el = this[0];
      if (!el) return $$1([]);
      while (el.previousElementSibling) {
        const prev = el.previousElementSibling; // eslint-disable-line

        if (selector) {
          if ($$1(prev).is(selector)) prevEls.push(prev);
        } else prevEls.push(prev);
        el = prev;
      }
      return $$1(prevEls);
    }
    function siblings(selector) {
      return this.nextAll(selector).add(this.prevAll(selector));
    }
    function parent(selector) {
      const parents = []; // eslint-disable-line

      for (let i = 0; i < this.length; i += 1) {
        if (this[i].parentNode !== null) {
          if (selector) {
            if ($$1(this[i].parentNode).is(selector)) parents.push(this[i].parentNode);
          } else {
            parents.push(this[i].parentNode);
          }
        }
      }
      return $$1(parents);
    }
    function parents(selector) {
      const parents = []; // eslint-disable-line

      for (let i = 0; i < this.length; i += 1) {
        let parent = this[i].parentNode; // eslint-disable-line

        while (parent) {
          if (selector) {
            if ($$1(parent).is(selector)) parents.push(parent);
          } else {
            parents.push(parent);
          }
          parent = parent.parentNode;
        }
      }
      return $$1(parents);
    }
    function closest(selector) {
      let closest = this; // eslint-disable-line

      if (typeof selector === 'undefined') {
        return $$1([]);
      }
      if (!closest.is(selector)) {
        closest = closest.parents(selector).eq(0);
      }
      return closest;
    }
    function find(selector) {
      const foundElements = [];
      for (let i = 0; i < this.length; i += 1) {
        const found = this[i].querySelectorAll(selector);
        for (let j = 0; j < found.length; j += 1) {
          foundElements.push(found[j]);
        }
      }
      return $$1(foundElements);
    }
    function children(selector) {
      const children = []; // eslint-disable-line

      for (let i = 0; i < this.length; i += 1) {
        const childNodes = this[i].children;
        for (let j = 0; j < childNodes.length; j += 1) {
          if (!selector || $$1(childNodes[j]).is(selector)) {
            children.push(childNodes[j]);
          }
        }
      }
      return $$1(children);
    }
    function remove() {
      for (let i = 0; i < this.length; i += 1) {
        if (this[i].parentNode) this[i].parentNode.removeChild(this[i]);
      }
      return this;
    }
    function detach() {
      return this.remove();
    }
    function add() {
      const dom = this;
      let i;
      let j;
      for (var _len10 = arguments.length, els = new Array(_len10), _key10 = 0; _key10 < _len10; _key10++) {
        els[_key10] = arguments[_key10];
      }
      for (i = 0; i < els.length; i += 1) {
        const toAdd = $$1(els[i]);
        for (j = 0; j < toAdd.length; j += 1) {
          dom.push(toAdd[j]);
        }
      }
      return dom;
    }
    function empty() {
      for (let i = 0; i < this.length; i += 1) {
        const el = this[i];
        if (el.nodeType === 1) {
          for (let j = 0; j < el.childNodes.length; j += 1) {
            if (el.childNodes[j].parentNode) {
              el.childNodes[j].parentNode.removeChild(el.childNodes[j]);
            }
          }
          el.textContent = '';
        }
      }
      return this;
    }

    // eslint-disable-next-line

    function scrollTo() {
      const window = getWindow$1();
      for (var _len11 = arguments.length, args = new Array(_len11), _key11 = 0; _key11 < _len11; _key11++) {
        args[_key11] = arguments[_key11];
      }
      let [left, top, duration, easing, callback] = args;
      if (args.length === 4 && typeof easing === 'function') {
        callback = easing;
        [left, top, duration, callback, easing] = args;
      }
      if (typeof easing === 'undefined') easing = 'swing';
      return this.each(function animate() {
        const el = this;
        let currentTop;
        let currentLeft;
        let maxTop;
        let maxLeft;
        let newTop;
        let newLeft;
        let scrollTop; // eslint-disable-line

        let scrollLeft; // eslint-disable-line

        let animateTop = top > 0 || top === 0;
        let animateLeft = left > 0 || left === 0;
        if (typeof easing === 'undefined') {
          easing = 'swing';
        }
        if (animateTop) {
          currentTop = el.scrollTop;
          if (!duration) {
            el.scrollTop = top;
          }
        }
        if (animateLeft) {
          currentLeft = el.scrollLeft;
          if (!duration) {
            el.scrollLeft = left;
          }
        }
        if (!duration) return;
        if (animateTop) {
          maxTop = el.scrollHeight - el.offsetHeight;
          newTop = Math.max(Math.min(top, maxTop), 0);
        }
        if (animateLeft) {
          maxLeft = el.scrollWidth - el.offsetWidth;
          newLeft = Math.max(Math.min(left, maxLeft), 0);
        }
        let startTime = null;
        if (animateTop && newTop === currentTop) animateTop = false;
        if (animateLeft && newLeft === currentLeft) animateLeft = false;
        function render(time) {
          if (time === void 0) {
            time = new Date().getTime();
          }
          if (startTime === null) {
            startTime = time;
          }
          const progress = Math.max(Math.min((time - startTime) / duration, 1), 0);
          const easeProgress = easing === 'linear' ? progress : 0.5 - Math.cos(progress * Math.PI) / 2;
          let done;
          if (animateTop) scrollTop = currentTop + easeProgress * (newTop - currentTop);
          if (animateLeft) scrollLeft = currentLeft + easeProgress * (newLeft - currentLeft);
          if (animateTop && newTop > currentTop && scrollTop >= newTop) {
            el.scrollTop = newTop;
            done = true;
          }
          if (animateTop && newTop < currentTop && scrollTop <= newTop) {
            el.scrollTop = newTop;
            done = true;
          }
          if (animateLeft && newLeft > currentLeft && scrollLeft >= newLeft) {
            el.scrollLeft = newLeft;
            done = true;
          }
          if (animateLeft && newLeft < currentLeft && scrollLeft <= newLeft) {
            el.scrollLeft = newLeft;
            done = true;
          }
          if (done) {
            if (callback) callback();
            return;
          }
          if (animateTop) el.scrollTop = scrollTop;
          if (animateLeft) el.scrollLeft = scrollLeft;
          window.requestAnimationFrame(render);
        }
        window.requestAnimationFrame(render);
      });
    } // scrollTop(top, duration, easing, callback) {

    function scrollTop() {
      for (var _len12 = arguments.length, args = new Array(_len12), _key12 = 0; _key12 < _len12; _key12++) {
        args[_key12] = arguments[_key12];
      }
      let [top, duration, easing, callback] = args;
      if (args.length === 3 && typeof easing === 'function') {
        [top, duration, callback, easing] = args;
      }
      const dom = this;
      if (typeof top === 'undefined') {
        if (dom.length > 0) return dom[0].scrollTop;
        return null;
      }
      return dom.scrollTo(undefined, top, duration, easing, callback);
    }
    function scrollLeft() {
      for (var _len13 = arguments.length, args = new Array(_len13), _key13 = 0; _key13 < _len13; _key13++) {
        args[_key13] = arguments[_key13];
      }
      let [left, duration, easing, callback] = args;
      if (args.length === 3 && typeof easing === 'function') {
        [left, duration, callback, easing] = args;
      }
      const dom = this;
      if (typeof left === 'undefined') {
        if (dom.length > 0) return dom[0].scrollLeft;
        return null;
      }
      return dom.scrollTo(left, undefined, duration, easing, callback);
    }

    // eslint-disable-next-line

    function animate(initialProps, initialParams) {
      const window = getWindow$1();
      const els = this;
      const a = {
        props: Object.assign({}, initialProps),
        params: Object.assign({
          duration: 300,
          easing: 'swing' // or 'linear'

          /* Callbacks
          begin(elements)
          complete(elements)
          progress(elements, complete, remaining, start, tweenValue)
          */
        }, initialParams),
        elements: els,
        animating: false,
        que: [],
        easingProgress(easing, progress) {
          if (easing === 'swing') {
            return 0.5 - Math.cos(progress * Math.PI) / 2;
          }
          if (typeof easing === 'function') {
            return easing(progress);
          }
          return progress;
        },
        stop() {
          if (a.frameId) {
            window.cancelAnimationFrame(a.frameId);
          }
          a.animating = false;
          a.elements.each(el => {
            const element = el;
            delete element.dom7AnimateInstance;
          });
          a.que = [];
        },
        done(complete) {
          a.animating = false;
          a.elements.each(el => {
            const element = el;
            delete element.dom7AnimateInstance;
          });
          if (complete) complete(els);
          if (a.que.length > 0) {
            const que = a.que.shift();
            a.animate(que[0], que[1]);
          }
        },
        animate(props, params) {
          if (a.animating) {
            a.que.push([props, params]);
            return a;
          }
          const elements = []; // Define & Cache Initials & Units

          a.elements.each((el, index) => {
            let initialFullValue;
            let initialValue;
            let unit;
            let finalValue;
            let finalFullValue;
            if (!el.dom7AnimateInstance) a.elements[index].dom7AnimateInstance = a;
            elements[index] = {
              container: el
            };
            Object.keys(props).forEach(prop => {
              initialFullValue = window.getComputedStyle(el, null).getPropertyValue(prop).replace(',', '.');
              initialValue = parseFloat(initialFullValue);
              unit = initialFullValue.replace(initialValue, '');
              finalValue = parseFloat(props[prop]);
              finalFullValue = props[prop] + unit;
              elements[index][prop] = {
                initialFullValue,
                initialValue,
                unit,
                finalValue,
                finalFullValue,
                currentValue: initialValue
              };
            });
          });
          let startTime = null;
          let time;
          let elementsDone = 0;
          let propsDone = 0;
          let done;
          let began = false;
          a.animating = true;
          function render() {
            time = new Date().getTime();
            let progress;
            let easeProgress; // let el;

            if (!began) {
              began = true;
              if (params.begin) params.begin(els);
            }
            if (startTime === null) {
              startTime = time;
            }
            if (params.progress) {
              // eslint-disable-next-line
              params.progress(els, Math.max(Math.min((time - startTime) / params.duration, 1), 0), startTime + params.duration - time < 0 ? 0 : startTime + params.duration - time, startTime);
            }
            elements.forEach(element => {
              const el = element;
              if (done || el.done) return;
              Object.keys(props).forEach(prop => {
                if (done || el.done) return;
                progress = Math.max(Math.min((time - startTime) / params.duration, 1), 0);
                easeProgress = a.easingProgress(params.easing, progress);
                const {
                  initialValue,
                  finalValue,
                  unit
                } = el[prop];
                el[prop].currentValue = initialValue + easeProgress * (finalValue - initialValue);
                const currentValue = el[prop].currentValue;
                if (finalValue > initialValue && currentValue >= finalValue || finalValue < initialValue && currentValue <= finalValue) {
                  el.container.style[prop] = finalValue + unit;
                  propsDone += 1;
                  if (propsDone === Object.keys(props).length) {
                    el.done = true;
                    elementsDone += 1;
                  }
                  if (elementsDone === elements.length) {
                    done = true;
                  }
                }
                if (done) {
                  a.done(params.complete);
                  return;
                }
                el.container.style[prop] = currentValue + unit;
              });
            });
            if (done) return; // Then call

            a.frameId = window.requestAnimationFrame(render);
          }
          a.frameId = window.requestAnimationFrame(render);
          return a;
        }
      };
      if (a.elements.length === 0) {
        return els;
      }
      let animateInstance;
      for (let i = 0; i < a.elements.length; i += 1) {
        if (a.elements[i].dom7AnimateInstance) {
          animateInstance = a.elements[i].dom7AnimateInstance;
        } else a.elements[i].dom7AnimateInstance = a;
      }
      if (!animateInstance) {
        animateInstance = a;
      }
      if (initialProps === 'stop') {
        animateInstance.stop();
      } else {
        animateInstance.animate(a.props, a.params);
      }
      return els;
    }
    function stop() {
      const els = this;
      for (let i = 0; i < els.length; i += 1) {
        if (els[i].dom7AnimateInstance) {
          els[i].dom7AnimateInstance.stop();
        }
      }
    }
    const noTrigger = 'resize scroll'.split(' ');
    function shortcut(name) {
      function eventHandler() {
        for (var _len14 = arguments.length, args = new Array(_len14), _key14 = 0; _key14 < _len14; _key14++) {
          args[_key14] = arguments[_key14];
        }
        if (typeof args[0] === 'undefined') {
          for (let i = 0; i < this.length; i += 1) {
            if (noTrigger.indexOf(name) < 0) {
              if (name in this[i]) this[i][name]();else {
                $$1(this[i]).trigger(name);
              }
            }
          }
          return this;
        }
        return this.on(name, ...args);
      }
      return eventHandler;
    }
    const click = shortcut('click');
    const blur = shortcut('blur');
    const focus = shortcut('focus');
    const focusin = shortcut('focusin');
    const focusout = shortcut('focusout');
    const keyup = shortcut('keyup');
    const keydown = shortcut('keydown');
    const keypress = shortcut('keypress');
    const submit = shortcut('submit');
    const change = shortcut('change');
    const mousedown = shortcut('mousedown');
    const mousemove = shortcut('mousemove');
    const mouseup = shortcut('mouseup');
    const mouseenter = shortcut('mouseenter');
    const mouseleave = shortcut('mouseleave');
    const mouseout = shortcut('mouseout');
    const mouseover = shortcut('mouseover');
    const touchstart = shortcut('touchstart');
    const touchend = shortcut('touchend');
    const touchmove = shortcut('touchmove');
    const resize = shortcut('resize');
    const scroll = shortcut('scroll');

    var methods = /*#__PURE__*/Object.freeze({
        __proto__: null,
        $: $$1,
        add: add,
        addClass: addClass,
        animate: animate,
        animationEnd: animationEnd,
        append: append,
        appendTo: appendTo,
        attr: attr,
        blur: blur,
        change: change,
        children: children,
        click: click,
        closest: closest,
        css: css,
        data: data,
        dataset: dataset,
        default: $$1,
        detach: detach,
        each: each,
        empty: empty,
        eq: eq,
        filter: filter,
        find: find,
        focus: focus,
        focusin: focusin,
        focusout: focusout,
        hasClass: hasClass,
        height: height,
        hide: hide,
        html: html,
        index: index,
        insertAfter: insertAfter,
        insertBefore: insertBefore$1,
        is: is,
        keydown: keydown,
        keypress: keypress,
        keyup: keyup,
        mousedown: mousedown,
        mouseenter: mouseenter,
        mouseleave: mouseleave,
        mousemove: mousemove,
        mouseout: mouseout,
        mouseover: mouseover,
        mouseup: mouseup,
        next: next,
        nextAll: nextAll,
        off: off,
        offset: offset,
        on: on,
        once: once,
        outerHeight: outerHeight,
        outerWidth: outerWidth,
        parent: parent,
        parents: parents,
        prepend: prepend,
        prependTo: prependTo,
        prev: prev,
        prevAll: prevAll,
        prop: prop,
        remove: remove,
        removeAttr: removeAttr,
        removeClass: removeClass,
        removeData: removeData,
        resize: resize,
        scroll: scroll,
        scrollLeft: scrollLeft,
        scrollTo: scrollTo,
        scrollTop: scrollTop,
        show: show,
        siblings: siblings,
        stop: stop,
        styles: styles,
        submit: submit,
        text: text,
        toggleClass: toggleClass,
        touchend: touchend,
        touchmove: touchmove,
        touchstart: touchstart,
        transform: transform,
        transition: transition$1,
        transitionEnd: transitionEnd$1,
        transitionStart: transitionStart$1,
        trigger: trigger,
        val: val,
        value: value,
        width: width
    });

    // eslint-disable-next-line
    Object.keys(methods).forEach(methodName => {
      if (methodName === '$') return;
      $$1.fn[methodName] = methods[methodName];
    });
    var $ = $$1;

    /* eslint-disable */
    function signum(num) {
      return num < 0 ? -1 : 0 === num ? 0 : 1;
    }
    function lerp(start, stop, amount) {
      return (1 - amount) * start + amount * stop;
    }
    function clampInt(min, max, input) {
      return input < min ? min : input > max ? max : input;
    }
    function clampDouble(min, max, input) {
      return input < min ? min : input > max ? max : input;
    }
    function sanitizeDegreesDouble(degrees) {
      return (degrees %= 360) < 0 && (degrees += 360), degrees;
    }
    function rotationDirection(from, to) {
      return sanitizeDegreesDouble(to - from) <= 180 ? 1 : -1;
    }
    function differenceDegrees(a, b) {
      return 180 - Math.abs(Math.abs(a - b) - 180);
    }
    function matrixMultiply(row, matrix) {
      return [row[0] * matrix[0][0] + row[1] * matrix[0][1] + row[2] * matrix[0][2], row[0] * matrix[1][0] + row[1] * matrix[1][1] + row[2] * matrix[1][2], row[0] * matrix[2][0] + row[1] * matrix[2][1] + row[2] * matrix[2][2]];
    }
    const SRGB_TO_XYZ = [[.41233895, .35762064, .18051042], [.2126, .7152, .0722], [.01932141, .11916382, .95034478]],
      XYZ_TO_SRGB = [[3.2413774792388685, -1.5376652402851851, -.49885366846268053], [-.9691452513005321, 1.8758853451067872, .04156585616912061], [.05562093689691305, -.20395524564742123, 1.0571799111220335]],
      WHITE_POINT_D65 = [95.047, 100, 108.883];
    function argbFromRgb(red, green, blue) {
      return (255 << 24 | (255 & red) << 16 | (255 & green) << 8 | 255 & blue) >>> 0;
    }
    function argbFromLinrgb(linrgb) {
      return argbFromRgb(delinearized(linrgb[0]), delinearized(linrgb[1]), delinearized(linrgb[2]));
    }
    function redFromArgb(argb) {
      return argb >> 16 & 255;
    }
    function greenFromArgb(argb) {
      return argb >> 8 & 255;
    }
    function blueFromArgb(argb) {
      return 255 & argb;
    }
    function argbFromXyz(x, y, z) {
      const matrix = XYZ_TO_SRGB,
        linearR = matrix[0][0] * x + matrix[0][1] * y + matrix[0][2] * z,
        linearG = matrix[1][0] * x + matrix[1][1] * y + matrix[1][2] * z,
        linearB = matrix[2][0] * x + matrix[2][1] * y + matrix[2][2] * z;
      return argbFromRgb(delinearized(linearR), delinearized(linearG), delinearized(linearB));
    }
    function xyzFromArgb(argb) {
      return matrixMultiply([linearized(redFromArgb(argb)), linearized(greenFromArgb(argb)), linearized(blueFromArgb(argb))], SRGB_TO_XYZ);
    }
    function argbFromLstar(lstar) {
      const component = delinearized(yFromLstar(lstar));
      return argbFromRgb(component, component, component);
    }
    function lstarFromArgb(argb) {
      return 116 * labF(xyzFromArgb(argb)[1] / 100) - 16;
    }
    function yFromLstar(lstar) {
      return 100 * labInvf((lstar + 16) / 116);
    }
    function lstarFromY(y) {
      return 116 * labF(y / 100) - 16;
    }
    function linearized(rgbComponent) {
      const normalized = rgbComponent / 255;
      return normalized <= .040449936 ? normalized / 12.92 * 100 : 100 * Math.pow((normalized + .055) / 1.055, 2.4);
    }
    function delinearized(rgbComponent) {
      const normalized = rgbComponent / 100;
      let delinearized = 0;
      return delinearized = normalized <= .0031308 ? 12.92 * normalized : 1.055 * Math.pow(normalized, 1 / 2.4) - .055, clampInt(0, 255, Math.round(255 * delinearized));
    }
    function whitePointD65() {
      return WHITE_POINT_D65;
    }
    function labF(t) {
      return t > 216 / 24389 ? Math.pow(t, 1 / 3) : (903.2962962962963 * t + 16) / 116;
    }
    function labInvf(ft) {
      const ft3 = ft * ft * ft;
      return ft3 > 216 / 24389 ? ft3 : (116 * ft - 16) / 903.2962962962963;
    }
    class ViewingConditions {
      static make(whitePoint, adaptingLuminance, backgroundLstar, surround, discountingIlluminant) {
        if (whitePoint === void 0) {
          whitePoint = whitePointD65();
        }
        if (adaptingLuminance === void 0) {
          adaptingLuminance = 200 / Math.PI * yFromLstar(50) / 100;
        }
        if (backgroundLstar === void 0) {
          backgroundLstar = 50;
        }
        if (surround === void 0) {
          surround = 2;
        }
        if (discountingIlluminant === void 0) {
          discountingIlluminant = !1;
        }
        const xyz = whitePoint,
          rW = .401288 * xyz[0] + .650173 * xyz[1] + -.051461 * xyz[2],
          gW = -.250268 * xyz[0] + 1.204414 * xyz[1] + .045854 * xyz[2],
          bW = -.002079 * xyz[0] + .048952 * xyz[1] + .953127 * xyz[2],
          f = .8 + surround / 10,
          c = f >= .9 ? lerp(.59, .69, 10 * (f - .9)) : lerp(.525, .59, 10 * (f - .8));
        let d = discountingIlluminant ? 1 : f * (1 - 1 / 3.6 * Math.exp((-adaptingLuminance - 42) / 92));
        d = d > 1 ? 1 : d < 0 ? 0 : d;
        const nc = f,
          rgbD = [d * (100 / rW) + 1 - d, d * (100 / gW) + 1 - d, d * (100 / bW) + 1 - d],
          k = 1 / (5 * adaptingLuminance + 1),
          k4 = k * k * k * k,
          k4F = 1 - k4,
          fl = k4 * adaptingLuminance + .1 * k4F * k4F * Math.cbrt(5 * adaptingLuminance),
          n = yFromLstar(backgroundLstar) / whitePoint[1],
          z = 1.48 + Math.sqrt(n),
          nbb = .725 / Math.pow(n, .2),
          ncb = nbb,
          rgbAFactors = [Math.pow(fl * rgbD[0] * rW / 100, .42), Math.pow(fl * rgbD[1] * gW / 100, .42), Math.pow(fl * rgbD[2] * bW / 100, .42)],
          rgbA = [400 * rgbAFactors[0] / (rgbAFactors[0] + 27.13), 400 * rgbAFactors[1] / (rgbAFactors[1] + 27.13), 400 * rgbAFactors[2] / (rgbAFactors[2] + 27.13)];
        return new ViewingConditions(n, (2 * rgbA[0] + rgbA[1] + .05 * rgbA[2]) * nbb, nbb, ncb, c, nc, rgbD, fl, Math.pow(fl, .25), z);
      }
      constructor(n, aw, nbb, ncb, c, nc, rgbD, fl, fLRoot, z) {
        this.n = n, this.aw = aw, this.nbb = nbb, this.ncb = ncb, this.c = c, this.nc = nc, this.rgbD = rgbD, this.fl = fl, this.fLRoot = fLRoot, this.z = z;
      }
    }
    ViewingConditions.DEFAULT = ViewingConditions.make();
    class Cam16 {
      constructor(hue, chroma, j, q, m, s, jstar, astar, bstar) {
        this.hue = hue, this.chroma = chroma, this.j = j, this.q = q, this.m = m, this.s = s, this.jstar = jstar, this.astar = astar, this.bstar = bstar;
      }
      distance(other) {
        const dJ = this.jstar - other.jstar,
          dA = this.astar - other.astar,
          dB = this.bstar - other.bstar,
          dEPrime = Math.sqrt(dJ * dJ + dA * dA + dB * dB);
        return 1.41 * Math.pow(dEPrime, .63);
      }
      static fromInt(argb) {
        return Cam16.fromIntInViewingConditions(argb, ViewingConditions.DEFAULT);
      }
      static fromIntInViewingConditions(argb, viewingConditions) {
        const green = (65280 & argb) >> 8,
          blue = 255 & argb,
          redL = linearized((16711680 & argb) >> 16),
          greenL = linearized(green),
          blueL = linearized(blue),
          x = .41233895 * redL + .35762064 * greenL + .18051042 * blueL,
          y = .2126 * redL + .7152 * greenL + .0722 * blueL,
          z = .01932141 * redL + .11916382 * greenL + .95034478 * blueL,
          rC = .401288 * x + .650173 * y - .051461 * z,
          gC = -.250268 * x + 1.204414 * y + .045854 * z,
          bC = -.002079 * x + .048952 * y + .953127 * z,
          rD = viewingConditions.rgbD[0] * rC,
          gD = viewingConditions.rgbD[1] * gC,
          bD = viewingConditions.rgbD[2] * bC,
          rAF = Math.pow(viewingConditions.fl * Math.abs(rD) / 100, .42),
          gAF = Math.pow(viewingConditions.fl * Math.abs(gD) / 100, .42),
          bAF = Math.pow(viewingConditions.fl * Math.abs(bD) / 100, .42),
          rA = 400 * signum(rD) * rAF / (rAF + 27.13),
          gA = 400 * signum(gD) * gAF / (gAF + 27.13),
          bA = 400 * signum(bD) * bAF / (bAF + 27.13),
          a = (11 * rA + -12 * gA + bA) / 11,
          b = (rA + gA - 2 * bA) / 9,
          u = (20 * rA + 20 * gA + 21 * bA) / 20,
          p2 = (40 * rA + 20 * gA + bA) / 20,
          atanDegrees = 180 * Math.atan2(b, a) / Math.PI,
          hue = atanDegrees < 0 ? atanDegrees + 360 : atanDegrees >= 360 ? atanDegrees - 360 : atanDegrees,
          hueRadians = hue * Math.PI / 180,
          ac = p2 * viewingConditions.nbb,
          j = 100 * Math.pow(ac / viewingConditions.aw, viewingConditions.c * viewingConditions.z),
          q = 4 / viewingConditions.c * Math.sqrt(j / 100) * (viewingConditions.aw + 4) * viewingConditions.fLRoot,
          huePrime = hue < 20.14 ? hue + 360 : hue,
          t = 5e4 / 13 * (.25 * (Math.cos(huePrime * Math.PI / 180 + 2) + 3.8)) * viewingConditions.nc * viewingConditions.ncb * Math.sqrt(a * a + b * b) / (u + .305),
          alpha = Math.pow(t, .9) * Math.pow(1.64 - Math.pow(.29, viewingConditions.n), .73),
          c = alpha * Math.sqrt(j / 100),
          m = c * viewingConditions.fLRoot,
          s = 50 * Math.sqrt(alpha * viewingConditions.c / (viewingConditions.aw + 4)),
          jstar = (1 + 100 * .007) * j / (1 + .007 * j),
          mstar = 1 / .0228 * Math.log(1 + .0228 * m),
          astar = mstar * Math.cos(hueRadians),
          bstar = mstar * Math.sin(hueRadians);
        return new Cam16(hue, c, j, q, m, s, jstar, astar, bstar);
      }
      static fromJch(j, c, h) {
        return Cam16.fromJchInViewingConditions(j, c, h, ViewingConditions.DEFAULT);
      }
      static fromJchInViewingConditions(j, c, h, viewingConditions) {
        const q = 4 / viewingConditions.c * Math.sqrt(j / 100) * (viewingConditions.aw + 4) * viewingConditions.fLRoot,
          m = c * viewingConditions.fLRoot,
          alpha = c / Math.sqrt(j / 100),
          s = 50 * Math.sqrt(alpha * viewingConditions.c / (viewingConditions.aw + 4)),
          hueRadians = h * Math.PI / 180,
          jstar = (1 + 100 * .007) * j / (1 + .007 * j),
          mstar = 1 / .0228 * Math.log(1 + .0228 * m),
          astar = mstar * Math.cos(hueRadians),
          bstar = mstar * Math.sin(hueRadians);
        return new Cam16(h, c, j, q, m, s, jstar, astar, bstar);
      }
      static fromUcs(jstar, astar, bstar) {
        return Cam16.fromUcsInViewingConditions(jstar, astar, bstar, ViewingConditions.DEFAULT);
      }
      static fromUcsInViewingConditions(jstar, astar, bstar, viewingConditions) {
        const a = astar,
          b = bstar,
          m = Math.sqrt(a * a + b * b),
          c = (Math.exp(.0228 * m) - 1) / .0228 / viewingConditions.fLRoot;
        let h = Math.atan2(b, a) * (180 / Math.PI);
        h < 0 && (h += 360);
        const j = jstar / (1 - .007 * (jstar - 100));
        return Cam16.fromJchInViewingConditions(j, c, h, viewingConditions);
      }
      toInt() {
        return this.viewed(ViewingConditions.DEFAULT);
      }
      viewed(viewingConditions) {
        const alpha = 0 === this.chroma || 0 === this.j ? 0 : this.chroma / Math.sqrt(this.j / 100),
          t = Math.pow(alpha / Math.pow(1.64 - Math.pow(.29, viewingConditions.n), .73), 1 / .9),
          hRad = this.hue * Math.PI / 180,
          eHue = .25 * (Math.cos(hRad + 2) + 3.8),
          ac = viewingConditions.aw * Math.pow(this.j / 100, 1 / viewingConditions.c / viewingConditions.z),
          p1 = eHue * (5e4 / 13) * viewingConditions.nc * viewingConditions.ncb,
          p2 = ac / viewingConditions.nbb,
          hSin = Math.sin(hRad),
          hCos = Math.cos(hRad),
          gamma = 23 * (p2 + .305) * t / (23 * p1 + 11 * t * hCos + 108 * t * hSin),
          a = gamma * hCos,
          b = gamma * hSin,
          rA = (460 * p2 + 451 * a + 288 * b) / 1403,
          gA = (460 * p2 - 891 * a - 261 * b) / 1403,
          bA = (460 * p2 - 220 * a - 6300 * b) / 1403,
          rCBase = Math.max(0, 27.13 * Math.abs(rA) / (400 - Math.abs(rA))),
          rC = signum(rA) * (100 / viewingConditions.fl) * Math.pow(rCBase, 1 / .42),
          gCBase = Math.max(0, 27.13 * Math.abs(gA) / (400 - Math.abs(gA))),
          gC = signum(gA) * (100 / viewingConditions.fl) * Math.pow(gCBase, 1 / .42),
          bCBase = Math.max(0, 27.13 * Math.abs(bA) / (400 - Math.abs(bA))),
          bC = signum(bA) * (100 / viewingConditions.fl) * Math.pow(bCBase, 1 / .42),
          rF = rC / viewingConditions.rgbD[0],
          gF = gC / viewingConditions.rgbD[1],
          bF = bC / viewingConditions.rgbD[2];
        return argbFromXyz(1.86206786 * rF - 1.01125463 * gF + .14918677 * bF, .38752654 * rF + .62144744 * gF - .00897398 * bF, -.0158415 * rF - .03412294 * gF + 1.04996444 * bF);
      }
      static fromXyzInViewingConditions(x, y, z, viewingConditions) {
        const rC = .401288 * x + .650173 * y - .051461 * z,
          gC = -.250268 * x + 1.204414 * y + .045854 * z,
          bC = -.002079 * x + .048952 * y + .953127 * z,
          rD = viewingConditions.rgbD[0] * rC,
          gD = viewingConditions.rgbD[1] * gC,
          bD = viewingConditions.rgbD[2] * bC,
          rAF = Math.pow(viewingConditions.fl * Math.abs(rD) / 100, .42),
          gAF = Math.pow(viewingConditions.fl * Math.abs(gD) / 100, .42),
          bAF = Math.pow(viewingConditions.fl * Math.abs(bD) / 100, .42),
          rA = 400 * signum(rD) * rAF / (rAF + 27.13),
          gA = 400 * signum(gD) * gAF / (gAF + 27.13),
          bA = 400 * signum(bD) * bAF / (bAF + 27.13),
          a = (11 * rA + -12 * gA + bA) / 11,
          b = (rA + gA - 2 * bA) / 9,
          u = (20 * rA + 20 * gA + 21 * bA) / 20,
          p2 = (40 * rA + 20 * gA + bA) / 20,
          atanDegrees = 180 * Math.atan2(b, a) / Math.PI,
          hue = atanDegrees < 0 ? atanDegrees + 360 : atanDegrees >= 360 ? atanDegrees - 360 : atanDegrees,
          hueRadians = hue * Math.PI / 180,
          ac = p2 * viewingConditions.nbb,
          J = 100 * Math.pow(ac / viewingConditions.aw, viewingConditions.c * viewingConditions.z),
          Q = 4 / viewingConditions.c * Math.sqrt(J / 100) * (viewingConditions.aw + 4) * viewingConditions.fLRoot,
          huePrime = hue < 20.14 ? hue + 360 : hue,
          t = 5e4 / 13 * (1 / 4 * (Math.cos(huePrime * Math.PI / 180 + 2) + 3.8)) * viewingConditions.nc * viewingConditions.ncb * Math.sqrt(a * a + b * b) / (u + .305),
          alpha = Math.pow(t, .9) * Math.pow(1.64 - Math.pow(.29, viewingConditions.n), .73),
          C = alpha * Math.sqrt(J / 100),
          M = C * viewingConditions.fLRoot,
          s = 50 * Math.sqrt(alpha * viewingConditions.c / (viewingConditions.aw + 4)),
          jstar = (1 + 100 * .007) * J / (1 + .007 * J),
          mstar = Math.log(1 + .0228 * M) / .0228,
          astar = mstar * Math.cos(hueRadians),
          bstar = mstar * Math.sin(hueRadians);
        return new Cam16(hue, C, J, Q, M, s, jstar, astar, bstar);
      }
      xyzInViewingConditions(viewingConditions) {
        const alpha = 0 === this.chroma || 0 === this.j ? 0 : this.chroma / Math.sqrt(this.j / 100),
          t = Math.pow(alpha / Math.pow(1.64 - Math.pow(.29, viewingConditions.n), .73), 1 / .9),
          hRad = this.hue * Math.PI / 180,
          eHue = .25 * (Math.cos(hRad + 2) + 3.8),
          ac = viewingConditions.aw * Math.pow(this.j / 100, 1 / viewingConditions.c / viewingConditions.z),
          p1 = eHue * (5e4 / 13) * viewingConditions.nc * viewingConditions.ncb,
          p2 = ac / viewingConditions.nbb,
          hSin = Math.sin(hRad),
          hCos = Math.cos(hRad),
          gamma = 23 * (p2 + .305) * t / (23 * p1 + 11 * t * hCos + 108 * t * hSin),
          a = gamma * hCos,
          b = gamma * hSin,
          rA = (460 * p2 + 451 * a + 288 * b) / 1403,
          gA = (460 * p2 - 891 * a - 261 * b) / 1403,
          bA = (460 * p2 - 220 * a - 6300 * b) / 1403,
          rCBase = Math.max(0, 27.13 * Math.abs(rA) / (400 - Math.abs(rA))),
          rC = signum(rA) * (100 / viewingConditions.fl) * Math.pow(rCBase, 1 / .42),
          gCBase = Math.max(0, 27.13 * Math.abs(gA) / (400 - Math.abs(gA))),
          gC = signum(gA) * (100 / viewingConditions.fl) * Math.pow(gCBase, 1 / .42),
          bCBase = Math.max(0, 27.13 * Math.abs(bA) / (400 - Math.abs(bA))),
          bC = signum(bA) * (100 / viewingConditions.fl) * Math.pow(bCBase, 1 / .42),
          rF = rC / viewingConditions.rgbD[0],
          gF = gC / viewingConditions.rgbD[1],
          bF = bC / viewingConditions.rgbD[2];
        return [1.86206786 * rF - 1.01125463 * gF + .14918677 * bF, .38752654 * rF + .62144744 * gF - .00897398 * bF, -.0158415 * rF - .03412294 * gF + 1.04996444 * bF];
      }
    }
    class HctSolver {
      static sanitizeRadians(angle) {
        return (angle + 8 * Math.PI) % (2 * Math.PI);
      }
      static trueDelinearized(rgbComponent) {
        const normalized = rgbComponent / 100;
        let delinearized = 0;
        return delinearized = normalized <= .0031308 ? 12.92 * normalized : 1.055 * Math.pow(normalized, 1 / 2.4) - .055, 255 * delinearized;
      }
      static chromaticAdaptation(component) {
        const af = Math.pow(Math.abs(component), .42);
        return 400 * signum(component) * af / (af + 27.13);
      }
      static hueOf(linrgb) {
        const scaledDiscount = matrixMultiply(linrgb, HctSolver.SCALED_DISCOUNT_FROM_LINRGB),
          rA = HctSolver.chromaticAdaptation(scaledDiscount[0]),
          gA = HctSolver.chromaticAdaptation(scaledDiscount[1]),
          bA = HctSolver.chromaticAdaptation(scaledDiscount[2]),
          a = (11 * rA + -12 * gA + bA) / 11,
          b = (rA + gA - 2 * bA) / 9;
        return Math.atan2(b, a);
      }
      static areInCyclicOrder(a, b, c) {
        return HctSolver.sanitizeRadians(b - a) < HctSolver.sanitizeRadians(c - a);
      }
      static intercept(source, mid, target) {
        return (mid - source) / (target - source);
      }
      static lerpPoint(source, t, target) {
        return [source[0] + (target[0] - source[0]) * t, source[1] + (target[1] - source[1]) * t, source[2] + (target[2] - source[2]) * t];
      }
      static setCoordinate(source, coordinate, target, axis) {
        const t = HctSolver.intercept(source[axis], coordinate, target[axis]);
        return HctSolver.lerpPoint(source, t, target);
      }
      static isBounded(x) {
        return 0 <= x && x <= 100;
      }
      static nthVertex(y, n) {
        const kR = HctSolver.Y_FROM_LINRGB[0],
          kG = HctSolver.Y_FROM_LINRGB[1],
          kB = HctSolver.Y_FROM_LINRGB[2],
          coordA = n % 4 <= 1 ? 0 : 100,
          coordB = n % 2 == 0 ? 0 : 100;
        if (n < 4) {
          const g = coordA,
            b = coordB,
            r = (y - g * kG - b * kB) / kR;
          return HctSolver.isBounded(r) ? [r, g, b] : [-1, -1, -1];
        }
        if (n < 8) {
          const b = coordA,
            r = coordB,
            g = (y - r * kR - b * kB) / kG;
          return HctSolver.isBounded(g) ? [r, g, b] : [-1, -1, -1];
        }
        {
          const r = coordA,
            g = coordB,
            b = (y - r * kR - g * kG) / kB;
          return HctSolver.isBounded(b) ? [r, g, b] : [-1, -1, -1];
        }
      }
      static bisectToSegment(y, targetHue) {
        let left = [-1, -1, -1],
          right = left,
          leftHue = 0,
          rightHue = 0,
          initialized = !1,
          uncut = !0;
        for (let n = 0; n < 12; n++) {
          const mid = HctSolver.nthVertex(y, n);
          if (mid[0] < 0) continue;
          const midHue = HctSolver.hueOf(mid);
          initialized ? (uncut || HctSolver.areInCyclicOrder(leftHue, midHue, rightHue)) && (uncut = !1, HctSolver.areInCyclicOrder(leftHue, targetHue, midHue) ? (right = mid, rightHue = midHue) : (left = mid, leftHue = midHue)) : (left = mid, right = mid, leftHue = midHue, rightHue = midHue, initialized = !0);
        }
        return [left, right];
      }
      static midpoint(a, b) {
        return [(a[0] + b[0]) / 2, (a[1] + b[1]) / 2, (a[2] + b[2]) / 2];
      }
      static criticalPlaneBelow(x) {
        return Math.floor(x - .5);
      }
      static criticalPlaneAbove(x) {
        return Math.ceil(x - .5);
      }
      static bisectToLimit(y, targetHue) {
        const segment = HctSolver.bisectToSegment(y, targetHue);
        let left = segment[0],
          leftHue = HctSolver.hueOf(left),
          right = segment[1];
        for (let axis = 0; axis < 3; axis++) if (left[axis] !== right[axis]) {
          let lPlane = -1,
            rPlane = 255;
          left[axis] < right[axis] ? (lPlane = HctSolver.criticalPlaneBelow(HctSolver.trueDelinearized(left[axis])), rPlane = HctSolver.criticalPlaneAbove(HctSolver.trueDelinearized(right[axis]))) : (lPlane = HctSolver.criticalPlaneAbove(HctSolver.trueDelinearized(left[axis])), rPlane = HctSolver.criticalPlaneBelow(HctSolver.trueDelinearized(right[axis])));
          for (let i = 0; i < 8 && !(Math.abs(rPlane - lPlane) <= 1); i++) {
            const mPlane = Math.floor((lPlane + rPlane) / 2),
              midPlaneCoordinate = HctSolver.CRITICAL_PLANES[mPlane],
              mid = HctSolver.setCoordinate(left, midPlaneCoordinate, right, axis),
              midHue = HctSolver.hueOf(mid);
            HctSolver.areInCyclicOrder(leftHue, targetHue, midHue) ? (right = mid, rPlane = mPlane) : (left = mid, leftHue = midHue, lPlane = mPlane);
          }
        }
        return HctSolver.midpoint(left, right);
      }
      static inverseChromaticAdaptation(adapted) {
        const adaptedAbs = Math.abs(adapted),
          base = Math.max(0, 27.13 * adaptedAbs / (400 - adaptedAbs));
        return signum(adapted) * Math.pow(base, 1 / .42);
      }
      static findResultByJ(hueRadians, chroma, y) {
        let j = 11 * Math.sqrt(y);
        const viewingConditions = ViewingConditions.DEFAULT,
          tInnerCoeff = 1 / Math.pow(1.64 - Math.pow(.29, viewingConditions.n), .73),
          p1 = .25 * (Math.cos(hueRadians + 2) + 3.8) * (5e4 / 13) * viewingConditions.nc * viewingConditions.ncb,
          hSin = Math.sin(hueRadians),
          hCos = Math.cos(hueRadians);
        for (let iterationRound = 0; iterationRound < 5; iterationRound++) {
          const jNormalized = j / 100,
            alpha = 0 === chroma || 0 === j ? 0 : chroma / Math.sqrt(jNormalized),
            t = Math.pow(alpha * tInnerCoeff, 1 / .9),
            p2 = viewingConditions.aw * Math.pow(jNormalized, 1 / viewingConditions.c / viewingConditions.z) / viewingConditions.nbb,
            gamma = 23 * (p2 + .305) * t / (23 * p1 + 11 * t * hCos + 108 * t * hSin),
            a = gamma * hCos,
            b = gamma * hSin,
            rA = (460 * p2 + 451 * a + 288 * b) / 1403,
            gA = (460 * p2 - 891 * a - 261 * b) / 1403,
            bA = (460 * p2 - 220 * a - 6300 * b) / 1403,
            linrgb = matrixMultiply([HctSolver.inverseChromaticAdaptation(rA), HctSolver.inverseChromaticAdaptation(gA), HctSolver.inverseChromaticAdaptation(bA)], HctSolver.LINRGB_FROM_SCALED_DISCOUNT);
          if (linrgb[0] < 0 || linrgb[1] < 0 || linrgb[2] < 0) return 0;
          const kR = HctSolver.Y_FROM_LINRGB[0],
            kG = HctSolver.Y_FROM_LINRGB[1],
            kB = HctSolver.Y_FROM_LINRGB[2],
            fnj = kR * linrgb[0] + kG * linrgb[1] + kB * linrgb[2];
          if (fnj <= 0) return 0;
          if (4 === iterationRound || Math.abs(fnj - y) < .002) return linrgb[0] > 100.01 || linrgb[1] > 100.01 || linrgb[2] > 100.01 ? 0 : argbFromLinrgb(linrgb);
          j -= (fnj - y) * j / (2 * fnj);
        }
        return 0;
      }
      static solveToInt(hueDegrees, chroma, lstar) {
        if (chroma < 1e-4 || lstar < 1e-4 || lstar > 99.9999) return argbFromLstar(lstar);
        const hueRadians = (hueDegrees = sanitizeDegreesDouble(hueDegrees)) / 180 * Math.PI,
          y = yFromLstar(lstar),
          exactAnswer = HctSolver.findResultByJ(hueRadians, chroma, y);
        if (0 !== exactAnswer) return exactAnswer;
        return argbFromLinrgb(HctSolver.bisectToLimit(y, hueRadians));
      }
      static solveToCam(hueDegrees, chroma, lstar) {
        return Cam16.fromInt(HctSolver.solveToInt(hueDegrees, chroma, lstar));
      }
    }
    HctSolver.SCALED_DISCOUNT_FROM_LINRGB = [[.001200833568784504, .002389694492170889, .0002795742885861124], [.0005891086651375999, .0029785502573438758, .0003270666104008398], [.00010146692491640572, .0005364214359186694, .0032979401770712076]], HctSolver.LINRGB_FROM_SCALED_DISCOUNT = [[1373.2198709594231, -1100.4251190754821, -7.278681089101213], [-271.815969077903, 559.6580465940733, -32.46047482791194], [1.9622899599665666, -57.173814538844006, 308.7233197812385]], HctSolver.Y_FROM_LINRGB = [.2126, .7152, .0722], HctSolver.CRITICAL_PLANES = [.015176349177441876, .045529047532325624, .07588174588720938, .10623444424209313, .13658714259697685, .16693984095186062, .19729253930674434, .2276452376616281, .2579979360165119, .28835063437139563, .3188300904430532, .350925934958123, .3848314933096426, .42057480301049466, .458183274052838, .4976837250274023, .5391024159806381, .5824650784040898, .6277969426914107, .6751227633498623, .7244668422128921, .775853049866786, .829304845476233, .8848452951698498, .942497089126609, 1.0022825574869039, 1.0642236851973577, 1.1283421258858297, 1.1946592148522128, 1.2631959812511864, 1.3339731595349034, 1.407011200216447, 1.4823302800086415, 1.5599503113873272, 1.6398909516233677, 1.7221716113234105, 1.8068114625156377, 1.8938294463134073, 1.9832442801866852, 2.075074464868551, 2.1693382909216234, 2.2660538449872063, 2.36523901573795, 2.4669114995532007, 2.5710888059345764, 2.6777882626779785, 2.7870270208169257, 2.898822059350997, 3.0131901897720907, 3.1301480604002863, 3.2497121605402226, 3.3718988244681087, 3.4967242352587946, 3.624204428461639, 3.754355295633311, 3.887192587735158, 4.022731918402185, 4.160988767090289, 4.301978482107941, 4.445716283538092, 4.592217266055746, 4.741496401646282, 4.893568542229298, 5.048448422192488, 5.20615066083972, 5.3666897647573375, 5.5300801301023865, 5.696336044816294, 5.865471690767354, 6.037501145825082, 6.212438385869475, 6.390297286737924, 6.571091626112461, 6.7548350853498045, 6.941541251256611, 7.131223617812143, 7.323895587840543, 7.5195704746346665, 7.7182615035334345, 7.919981813454504, 8.124744458384042, 8.332562408825165, 8.543448553206703, 8.757415699253682, 8.974476575321063, 9.194643831691977, 9.417930041841839, 9.644347703669503, 9.873909240696694, 10.106627003236781, 10.342513269534024, 10.58158024687427, 10.8238400726681, 11.069304815507364, 11.317986476196008, 11.569896988756009, 11.825048221409341, 12.083451977536606, 12.345119996613247, 12.610063955123938, 12.878295467455942, 13.149826086772048, 13.42466730586372, 13.702830557985108, 13.984327217668513, 14.269168601521828, 14.55736596900856, 14.848930523210871, 15.143873411576273, 15.44220572664832, 15.743938506781891, 16.04908273684337, 16.35764934889634, 16.66964922287304, 16.985093187232053, 17.30399201960269, 17.62635644741625, 17.95219714852476, 18.281524751807332, 18.614349837764564, 18.95068293910138, 19.290534541298456, 19.633915083172692, 19.98083495742689, 20.331304511189067, 20.685334046541502, 21.042933821039977, 21.404114048223256, 21.76888489811322, 22.137256497705877, 22.50923893145328, 22.884842241736916, 23.264076429332462, 23.6469514538663, 24.033477234264016, 24.42366364919083, 24.817520537484558, 25.21505769858089, 25.61628489293138, 26.021211842414342, 26.429848230738664, 26.842203703840827, 27.258287870275353, 27.678110301598522, 28.10168053274597, 28.529008062403893, 28.96010235337422, 29.39497283293396, 29.83362889318845, 30.276079891419332, 30.722335150426627, 31.172403958865512, 31.62629557157785, 32.08401920991837, 32.54558406207592, 33.010999283389665, 33.4802739966603, 33.953417292456834, 34.430438229418264, 34.911345834551085, 35.39614910352207, 35.88485700094671, 36.37747846067349, 36.87402238606382, 37.37449765026789, 37.87891309649659, 38.38727753828926, 38.89959975977785, 39.41588851594697, 39.93615253289054, 40.460400508064545, 40.98864111053629, 41.520882981230194, 42.05713473317016, 42.597404951718396, 43.141702194811224, 43.6900349931913, 44.24241185063697, 44.798841244188324, 45.35933162437017, 45.92389141541209, 46.49252901546552, 47.065252796817916, 47.64207110610409, 48.22299226451468, 48.808024568002054, 49.3971762874833, 49.9904556690408, 50.587870934119984, 51.189430279724725, 51.79514187861014, 52.40501387947288, 53.0190544071392, 53.637271562750364, 54.259673423945976, 54.88626804504493, 55.517063457223934, 56.15206766869424, 56.79128866487574, 57.43473440856916, 58.08241284012621, 58.734331877617365, 59.39049941699807, 60.05092333227251, 60.715611475655585, 61.38457167773311, 62.057811747619894, 62.7353394731159, 63.417162620860914, 64.10328893648692, 64.79372614476921, 65.48848194977529, 66.18756403501224, 66.89098006357258, 67.59873767827808, 68.31084450182222, 69.02730813691093, 69.74813616640164, 70.47333615344107, 71.20291564160104, 71.93688215501312, 72.67524319850172, 73.41800625771542, 74.16517879925733, 74.9167682708136, 75.67278210128072, 76.43322770089146, 77.1981124613393, 77.96744375590167, 78.74122893956174, 79.51947534912904, 80.30219030335869, 81.08938110306934, 81.88105503125999, 82.67721935322541, 83.4778813166706, 84.28304815182372, 85.09272707154808, 85.90692527145302, 86.72564993000343, 87.54890820862819, 88.3767072518277, 89.2090541872801, 90.04595612594655, 90.88742016217518, 91.73345337380438, 92.58406282226491, 93.43925555268066, 94.29903859396902, 95.16341895893969, 96.03240364439274, 96.9059996312159, 97.78421388448044, 98.6670533535366, 99.55452497210776];
    class Hct {
      static from(hue, chroma, tone) {
        return new Hct(HctSolver.solveToInt(hue, chroma, tone));
      }
      static fromInt(argb) {
        return new Hct(argb);
      }
      toInt() {
        return this.argb;
      }
      get hue() {
        return this.internalHue;
      }
      set hue(newHue) {
        this.setInternalState(HctSolver.solveToInt(newHue, this.internalChroma, this.internalTone));
      }
      get chroma() {
        return this.internalChroma;
      }
      set chroma(newChroma) {
        this.setInternalState(HctSolver.solveToInt(this.internalHue, newChroma, this.internalTone));
      }
      get tone() {
        return this.internalTone;
      }
      set tone(newTone) {
        this.setInternalState(HctSolver.solveToInt(this.internalHue, this.internalChroma, newTone));
      }
      constructor(argb) {
        this.argb = argb;
        const cam = Cam16.fromInt(argb);
        this.internalHue = cam.hue, this.internalChroma = cam.chroma, this.internalTone = lstarFromArgb(argb), this.argb = argb;
      }
      setInternalState(argb) {
        const cam = Cam16.fromInt(argb);
        this.internalHue = cam.hue, this.internalChroma = cam.chroma, this.internalTone = lstarFromArgb(argb), this.argb = argb;
      }
      inViewingConditions(vc) {
        const viewedInVc = Cam16.fromInt(this.toInt()).xyzInViewingConditions(vc),
          recastInVc = Cam16.fromXyzInViewingConditions(viewedInVc[0], viewedInVc[1], viewedInVc[2], ViewingConditions.make());
        return Hct.from(recastInVc.hue, recastInVc.chroma, lstarFromY(viewedInVc[1]));
      }
    }
    class Blend {
      static harmonize(designColor, sourceColor) {
        const fromHct = Hct.fromInt(designColor),
          toHct = Hct.fromInt(sourceColor),
          differenceDegrees$1 = differenceDegrees(fromHct.hue, toHct.hue),
          rotationDegrees = Math.min(.5 * differenceDegrees$1, 15),
          outputHue = sanitizeDegreesDouble(fromHct.hue + rotationDegrees * rotationDirection(fromHct.hue, toHct.hue));
        return Hct.from(outputHue, fromHct.chroma, fromHct.tone).toInt();
      }
      static hctHue(from, to, amount) {
        const ucs = Blend.cam16Ucs(from, to, amount),
          ucsCam = Cam16.fromInt(ucs),
          fromCam = Cam16.fromInt(from);
        return Hct.from(ucsCam.hue, fromCam.chroma, lstarFromArgb(from)).toInt();
      }
      static cam16Ucs(from, to, amount) {
        const fromCam = Cam16.fromInt(from),
          toCam = Cam16.fromInt(to),
          fromJ = fromCam.jstar,
          fromA = fromCam.astar,
          fromB = fromCam.bstar,
          jstar = fromJ + (toCam.jstar - fromJ) * amount,
          astar = fromA + (toCam.astar - fromA) * amount,
          bstar = fromB + (toCam.bstar - fromB) * amount;
        return Cam16.fromUcs(jstar, astar, bstar).toInt();
      }
    }
    class Contrast {
      static ratioOfTones(toneA, toneB) {
        return toneA = clampDouble(0, 100, toneA), toneB = clampDouble(0, 100, toneB), Contrast.ratioOfYs(yFromLstar(toneA), yFromLstar(toneB));
      }
      static ratioOfYs(y1, y2) {
        const lighter = y1 > y2 ? y1 : y2;
        return (lighter + 5) / ((lighter === y2 ? y1 : y2) + 5);
      }
      static lighter(tone, ratio) {
        if (tone < 0 || tone > 100) return -1;
        const darkY = yFromLstar(tone),
          lightY = ratio * (darkY + 5) - 5,
          realContrast = Contrast.ratioOfYs(lightY, darkY),
          delta = Math.abs(realContrast - ratio);
        if (realContrast < ratio && delta > .04) return -1;
        const returnValue = lstarFromY(lightY) + .4;
        return returnValue < 0 || returnValue > 100 ? -1 : returnValue;
      }
      static darker(tone, ratio) {
        if (tone < 0 || tone > 100) return -1;
        const lightY = yFromLstar(tone),
          darkY = (lightY + 5) / ratio - 5,
          realContrast = Contrast.ratioOfYs(lightY, darkY),
          delta = Math.abs(realContrast - ratio);
        if (realContrast < ratio && delta > .04) return -1;
        const returnValue = lstarFromY(darkY) - .4;
        return returnValue < 0 || returnValue > 100 ? -1 : returnValue;
      }
      static lighterUnsafe(tone, ratio) {
        const lighterSafe = Contrast.lighter(tone, ratio);
        return lighterSafe < 0 ? 100 : lighterSafe;
      }
      static darkerUnsafe(tone, ratio) {
        const darkerSafe = Contrast.darker(tone, ratio);
        return darkerSafe < 0 ? 0 : darkerSafe;
      }
    }
    class DislikeAnalyzer {
      static isDisliked(hct) {
        const huePasses = Math.round(hct.hue) >= 90 && Math.round(hct.hue) <= 111,
          chromaPasses = Math.round(hct.chroma) > 16,
          tonePasses = Math.round(hct.tone) < 65;
        return huePasses && chromaPasses && tonePasses;
      }
      static fixIfDisliked(hct) {
        return DislikeAnalyzer.isDisliked(hct) ? Hct.from(hct.hue, hct.chroma, 70) : hct;
      }
    }
    class DynamicColor {
      static fromPalette(args) {
        return new DynamicColor(args.name ?? "", args.palette, args.tone, args.isBackground ?? !1, args.background, args.secondBackground, args.contrastCurve, args.toneDeltaPair);
      }
      constructor(name, palette, tone, isBackground, background, secondBackground, contrastCurve, toneDeltaPair) {
        if (this.name = name, this.palette = palette, this.tone = tone, this.isBackground = isBackground, this.background = background, this.secondBackground = secondBackground, this.contrastCurve = contrastCurve, this.toneDeltaPair = toneDeltaPair, this.hctCache = new Map(), !background && secondBackground) throw new Error(`Color ${name} has secondBackgrounddefined, but background is not defined.`);
        if (!background && contrastCurve) throw new Error(`Color ${name} has contrastCurvedefined, but background is not defined.`);
        if (background && !contrastCurve) throw new Error(`Color ${name} has backgrounddefined, but contrastCurve is not defined.`);
      }
      getArgb(scheme) {
        return this.getHct(scheme).toInt();
      }
      getHct(scheme) {
        const cachedAnswer = this.hctCache.get(scheme);
        if (null != cachedAnswer) return cachedAnswer;
        const tone = this.getTone(scheme),
          answer = this.palette(scheme).getHct(tone);
        return this.hctCache.size > 4 && this.hctCache.clear(), this.hctCache.set(scheme, answer), answer;
      }
      getTone(scheme) {
        const decreasingContrast = scheme.contrastLevel < 0;
        if (this.toneDeltaPair) {
          const toneDeltaPair = this.toneDeltaPair(scheme),
            roleA = toneDeltaPair.roleA,
            roleB = toneDeltaPair.roleB,
            delta = toneDeltaPair.delta,
            polarity = toneDeltaPair.polarity,
            stayTogether = toneDeltaPair.stayTogether,
            bgTone = this.background(scheme).getTone(scheme),
            aIsNearer = "nearer" === polarity || "lighter" === polarity && !scheme.isDark || "darker" === polarity && scheme.isDark,
            nearer = aIsNearer ? roleA : roleB,
            farther = aIsNearer ? roleB : roleA,
            amNearer = this.name === nearer.name,
            expansionDir = scheme.isDark ? 1 : -1,
            nContrast = nearer.contrastCurve.getContrast(scheme.contrastLevel),
            fContrast = farther.contrastCurve.getContrast(scheme.contrastLevel),
            nInitialTone = nearer.tone(scheme);
          let nTone = Contrast.ratioOfTones(bgTone, nInitialTone) >= nContrast ? nInitialTone : DynamicColor.foregroundTone(bgTone, nContrast);
          const fInitialTone = farther.tone(scheme);
          let fTone = Contrast.ratioOfTones(bgTone, fInitialTone) >= fContrast ? fInitialTone : DynamicColor.foregroundTone(bgTone, fContrast);
          return decreasingContrast && (nTone = DynamicColor.foregroundTone(bgTone, nContrast), fTone = DynamicColor.foregroundTone(bgTone, fContrast)), (fTone - nTone) * expansionDir >= delta || (fTone = clampDouble(0, 100, nTone + delta * expansionDir), (fTone - nTone) * expansionDir >= delta || (nTone = clampDouble(0, 100, fTone - delta * expansionDir))), 50 <= nTone && nTone < 60 ? expansionDir > 0 ? (nTone = 60, fTone = Math.max(fTone, nTone + delta * expansionDir)) : (nTone = 49, fTone = Math.min(fTone, nTone + delta * expansionDir)) : 50 <= fTone && fTone < 60 && (stayTogether ? expansionDir > 0 ? (nTone = 60, fTone = Math.max(fTone, nTone + delta * expansionDir)) : (nTone = 49, fTone = Math.min(fTone, nTone + delta * expansionDir)) : fTone = expansionDir > 0 ? 60 : 49), amNearer ? nTone : fTone;
        }
        {
          let answer = this.tone(scheme);
          if (null == this.background) return answer;
          const bgTone = this.background(scheme).getTone(scheme),
            desiredRatio = this.contrastCurve.getContrast(scheme.contrastLevel);
          if (Contrast.ratioOfTones(bgTone, answer) >= desiredRatio || (answer = DynamicColor.foregroundTone(bgTone, desiredRatio)), decreasingContrast && (answer = DynamicColor.foregroundTone(bgTone, desiredRatio)), this.isBackground && 50 <= answer && answer < 60 && (answer = Contrast.ratioOfTones(49, bgTone) >= desiredRatio ? 49 : 60), this.secondBackground) {
            const [bg1, bg2] = [this.background, this.secondBackground],
              [bgTone1, bgTone2] = [bg1(scheme).getTone(scheme), bg2(scheme).getTone(scheme)],
              [upper, lower] = [Math.max(bgTone1, bgTone2), Math.min(bgTone1, bgTone2)];
            if (Contrast.ratioOfTones(upper, answer) >= desiredRatio && Contrast.ratioOfTones(lower, answer) >= desiredRatio) return answer;
            const lightOption = Contrast.lighter(upper, desiredRatio),
              darkOption = Contrast.darker(lower, desiredRatio),
              availables = [];
            -1 !== lightOption && availables.push(lightOption), -1 !== darkOption && availables.push(darkOption);
            return DynamicColor.tonePrefersLightForeground(bgTone1) || DynamicColor.tonePrefersLightForeground(bgTone2) ? lightOption < 0 ? 100 : lightOption : 1 === availables.length ? availables[0] : darkOption < 0 ? 0 : darkOption;
          }
          return answer;
        }
      }
      static foregroundTone(bgTone, ratio) {
        const lighterTone = Contrast.lighterUnsafe(bgTone, ratio),
          darkerTone = Contrast.darkerUnsafe(bgTone, ratio),
          lighterRatio = Contrast.ratioOfTones(lighterTone, bgTone),
          darkerRatio = Contrast.ratioOfTones(darkerTone, bgTone);
        if (DynamicColor.tonePrefersLightForeground(bgTone)) {
          const negligibleDifference = Math.abs(lighterRatio - darkerRatio) < .1 && lighterRatio < ratio && darkerRatio < ratio;
          return lighterRatio >= ratio || lighterRatio >= darkerRatio || negligibleDifference ? lighterTone : darkerTone;
        }
        return darkerRatio >= ratio || darkerRatio >= lighterRatio ? darkerTone : lighterTone;
      }
      static tonePrefersLightForeground(tone) {
        return Math.round(tone) < 60;
      }
      static toneAllowsLightForeground(tone) {
        return Math.round(tone) <= 49;
      }
      static enableLightForeground(tone) {
        return DynamicColor.tonePrefersLightForeground(tone) && !DynamicColor.toneAllowsLightForeground(tone) ? 49 : tone;
      }
    }
    var Variant;
    !function (Variant) {
      Variant[Variant.MONOCHROME = 0] = "MONOCHROME", Variant[Variant.NEUTRAL = 1] = "NEUTRAL", Variant[Variant.TONAL_SPOT = 2] = "TONAL_SPOT", Variant[Variant.VIBRANT = 3] = "VIBRANT", Variant[Variant.EXPRESSIVE = 4] = "EXPRESSIVE", Variant[Variant.FIDELITY = 5] = "FIDELITY", Variant[Variant.CONTENT = 6] = "CONTENT", Variant[Variant.RAINBOW = 7] = "RAINBOW", Variant[Variant.FRUIT_SALAD = 8] = "FRUIT_SALAD";
    }(Variant || (Variant = {}));
    class ContrastCurve {
      constructor(low, normal, medium, high) {
        this.low = low, this.normal = normal, this.medium = medium, this.high = high;
      }
      getContrast(contrastLevel) {
        return contrastLevel <= -1 ? this.low : contrastLevel < 0 ? lerp(this.low, this.normal, (contrastLevel - -1) / 1) : contrastLevel < .5 ? lerp(this.normal, this.medium, (contrastLevel - 0) / .5) : contrastLevel < 1 ? lerp(this.medium, this.high, (contrastLevel - .5) / .5) : this.high;
      }
    }
    class ToneDeltaPair {
      constructor(roleA, roleB, delta, polarity, stayTogether) {
        this.roleA = roleA, this.roleB = roleB, this.delta = delta, this.polarity = polarity, this.stayTogether = stayTogether;
      }
    }
    function isFidelity(scheme) {
      return scheme.variant === Variant.FIDELITY || scheme.variant === Variant.CONTENT;
    }
    function isMonochrome(scheme) {
      return scheme.variant === Variant.MONOCHROME;
    }
    function findDesiredChromaByTone(hue, chroma, tone, byDecreasingTone) {
      let answer = tone,
        closestToChroma = Hct.from(hue, chroma, tone);
      if (closestToChroma.chroma < chroma) {
        let chromaPeak = closestToChroma.chroma;
        for (; closestToChroma.chroma < chroma;) {
          answer += byDecreasingTone ? -1 : 1;
          const potentialSolution = Hct.from(hue, chroma, answer);
          if (chromaPeak > potentialSolution.chroma) break;
          if (Math.abs(potentialSolution.chroma - chroma) < .4) break;
          Math.abs(potentialSolution.chroma - chroma) < Math.abs(closestToChroma.chroma - chroma) && (closestToChroma = potentialSolution), chromaPeak = Math.max(chromaPeak, potentialSolution.chroma);
        }
      }
      return answer;
    }
    function viewingConditionsForAlbers(scheme) {
      return ViewingConditions.make(void 0, void 0, scheme.isDark ? 30 : 80, void 0, void 0);
    }
    function performAlbers(prealbers, scheme) {
      const albersd = prealbers.inViewingConditions(viewingConditionsForAlbers(scheme));
      return DynamicColor.tonePrefersLightForeground(prealbers.tone) && !DynamicColor.toneAllowsLightForeground(albersd.tone) ? DynamicColor.enableLightForeground(prealbers.tone) : DynamicColor.enableLightForeground(albersd.tone);
    }
    class MaterialDynamicColors {
      static highestSurface(s) {
        return s.isDark ? MaterialDynamicColors.surfaceBright : MaterialDynamicColors.surfaceDim;
      }
    }
    MaterialDynamicColors.contentAccentToneDelta = 15, MaterialDynamicColors.primaryPaletteKeyColor = DynamicColor.fromPalette({
      name: "primary_palette_key_color",
      palette: s => s.primaryPalette,
      tone: s => s.primaryPalette.keyColor.tone
    }), MaterialDynamicColors.secondaryPaletteKeyColor = DynamicColor.fromPalette({
      name: "secondary_palette_key_color",
      palette: s => s.secondaryPalette,
      tone: s => s.secondaryPalette.keyColor.tone
    }), MaterialDynamicColors.tertiaryPaletteKeyColor = DynamicColor.fromPalette({
      name: "tertiary_palette_key_color",
      palette: s => s.tertiaryPalette,
      tone: s => s.tertiaryPalette.keyColor.tone
    }), MaterialDynamicColors.neutralPaletteKeyColor = DynamicColor.fromPalette({
      name: "neutral_palette_key_color",
      palette: s => s.neutralPalette,
      tone: s => s.neutralPalette.keyColor.tone
    }), MaterialDynamicColors.neutralVariantPaletteKeyColor = DynamicColor.fromPalette({
      name: "neutral_variant_palette_key_color",
      palette: s => s.neutralVariantPalette,
      tone: s => s.neutralVariantPalette.keyColor.tone
    }), MaterialDynamicColors.background = DynamicColor.fromPalette({
      name: "background",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 6 : 98,
      isBackground: !0
    }), MaterialDynamicColors.onBackground = DynamicColor.fromPalette({
      name: "on_background",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.background,
      contrastCurve: new ContrastCurve(3, 3, 4.5, 7)
    }), MaterialDynamicColors.surface = DynamicColor.fromPalette({
      name: "surface",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 6 : 98,
      isBackground: !0
    }), MaterialDynamicColors.surfaceDim = DynamicColor.fromPalette({
      name: "surface_dim",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 6 : 87,
      isBackground: !0
    }), MaterialDynamicColors.surfaceBright = DynamicColor.fromPalette({
      name: "surface_bright",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 24 : 98,
      isBackground: !0
    }), MaterialDynamicColors.surfaceContainerLowest = DynamicColor.fromPalette({
      name: "surface_container_lowest",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 4 : 100,
      isBackground: !0
    }), MaterialDynamicColors.surfaceContainerLow = DynamicColor.fromPalette({
      name: "surface_container_low",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 10 : 96,
      isBackground: !0
    }), MaterialDynamicColors.surfaceContainer = DynamicColor.fromPalette({
      name: "surface_container",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 12 : 94,
      isBackground: !0
    }), MaterialDynamicColors.surfaceContainerHigh = DynamicColor.fromPalette({
      name: "surface_container_high",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 17 : 92,
      isBackground: !0
    }), MaterialDynamicColors.surfaceContainerHighest = DynamicColor.fromPalette({
      name: "surface_container_highest",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 22 : 90,
      isBackground: !0
    }), MaterialDynamicColors.onSurface = DynamicColor.fromPalette({
      name: "on_surface",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.surfaceVariant = DynamicColor.fromPalette({
      name: "surface_variant",
      palette: s => s.neutralVariantPalette,
      tone: s => s.isDark ? 30 : 90,
      isBackground: !0
    }), MaterialDynamicColors.onSurfaceVariant = DynamicColor.fromPalette({
      name: "on_surface_variant",
      palette: s => s.neutralVariantPalette,
      tone: s => s.isDark ? 80 : 30,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11)
    }), MaterialDynamicColors.inverseSurface = DynamicColor.fromPalette({
      name: "inverse_surface",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 90 : 20
    }), MaterialDynamicColors.inverseOnSurface = DynamicColor.fromPalette({
      name: "inverse_on_surface",
      palette: s => s.neutralPalette,
      tone: s => s.isDark ? 20 : 95,
      background: s => MaterialDynamicColors.inverseSurface,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.outline = DynamicColor.fromPalette({
      name: "outline",
      palette: s => s.neutralVariantPalette,
      tone: s => s.isDark ? 60 : 50,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1.5, 3, 4.5, 7)
    }), MaterialDynamicColors.outlineVariant = DynamicColor.fromPalette({
      name: "outline_variant",
      palette: s => s.neutralVariantPalette,
      tone: s => s.isDark ? 30 : 80,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7)
    }), MaterialDynamicColors.shadow = DynamicColor.fromPalette({
      name: "shadow",
      palette: s => s.neutralPalette,
      tone: s => 0
    }), MaterialDynamicColors.scrim = DynamicColor.fromPalette({
      name: "scrim",
      palette: s => s.neutralPalette,
      tone: s => 0
    }), MaterialDynamicColors.surfaceTint = DynamicColor.fromPalette({
      name: "surface_tint",
      palette: s => s.primaryPalette,
      tone: s => s.isDark ? 80 : 40,
      isBackground: !0
    }), MaterialDynamicColors.primary = DynamicColor.fromPalette({
      name: "primary",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 100 : 0 : s.isDark ? 80 : 40,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.primaryContainer, MaterialDynamicColors.primary, 15, "nearer", !1)
    }), MaterialDynamicColors.onPrimary = DynamicColor.fromPalette({
      name: "on_primary",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 10 : 90 : s.isDark ? 20 : 100,
      background: s => MaterialDynamicColors.primary,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.primaryContainer = DynamicColor.fromPalette({
      name: "primary_container",
      palette: s => s.primaryPalette,
      tone: s => isFidelity(s) ? performAlbers(s.sourceColorHct, s) : isMonochrome(s) ? s.isDark ? 85 : 25 : s.isDark ? 30 : 90,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.primaryContainer, MaterialDynamicColors.primary, 15, "nearer", !1)
    }), MaterialDynamicColors.onPrimaryContainer = DynamicColor.fromPalette({
      name: "on_primary_container",
      palette: s => s.primaryPalette,
      tone: s => isFidelity(s) ? DynamicColor.foregroundTone(MaterialDynamicColors.primaryContainer.tone(s), 4.5) : isMonochrome(s) ? s.isDark ? 0 : 100 : s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.primaryContainer,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.inversePrimary = DynamicColor.fromPalette({
      name: "inverse_primary",
      palette: s => s.primaryPalette,
      tone: s => s.isDark ? 40 : 80,
      background: s => MaterialDynamicColors.inverseSurface,
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11)
    }), MaterialDynamicColors.secondary = DynamicColor.fromPalette({
      name: "secondary",
      palette: s => s.secondaryPalette,
      tone: s => s.isDark ? 80 : 40,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.secondaryContainer, MaterialDynamicColors.secondary, 15, "nearer", !1)
    }), MaterialDynamicColors.onSecondary = DynamicColor.fromPalette({
      name: "on_secondary",
      palette: s => s.secondaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 10 : 100 : s.isDark ? 20 : 100,
      background: s => MaterialDynamicColors.secondary,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.secondaryContainer = DynamicColor.fromPalette({
      name: "secondary_container",
      palette: s => s.secondaryPalette,
      tone: s => {
        const initialTone = s.isDark ? 30 : 90;
        if (isMonochrome(s)) return s.isDark ? 30 : 85;
        if (!isFidelity(s)) return initialTone;
        let answer = findDesiredChromaByTone(s.secondaryPalette.hue, s.secondaryPalette.chroma, initialTone, !s.isDark);
        return answer = performAlbers(s.secondaryPalette.getHct(answer), s), answer;
      },
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.secondaryContainer, MaterialDynamicColors.secondary, 15, "nearer", !1)
    }), MaterialDynamicColors.onSecondaryContainer = DynamicColor.fromPalette({
      name: "on_secondary_container",
      palette: s => s.secondaryPalette,
      tone: s => isFidelity(s) ? DynamicColor.foregroundTone(MaterialDynamicColors.secondaryContainer.tone(s), 4.5) : s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.secondaryContainer,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.tertiary = DynamicColor.fromPalette({
      name: "tertiary",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 90 : 25 : s.isDark ? 80 : 40,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.tertiaryContainer, MaterialDynamicColors.tertiary, 15, "nearer", !1)
    }), MaterialDynamicColors.onTertiary = DynamicColor.fromPalette({
      name: "on_tertiary",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 10 : 90 : s.isDark ? 20 : 100,
      background: s => MaterialDynamicColors.tertiary,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.tertiaryContainer = DynamicColor.fromPalette({
      name: "tertiary_container",
      palette: s => s.tertiaryPalette,
      tone: s => {
        if (isMonochrome(s)) return s.isDark ? 60 : 49;
        if (!isFidelity(s)) return s.isDark ? 30 : 90;
        const albersTone = performAlbers(s.tertiaryPalette.getHct(s.sourceColorHct.tone), s),
          proposedHct = s.tertiaryPalette.getHct(albersTone);
        return DislikeAnalyzer.fixIfDisliked(proposedHct).tone;
      },
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.tertiaryContainer, MaterialDynamicColors.tertiary, 15, "nearer", !1)
    }), MaterialDynamicColors.onTertiaryContainer = DynamicColor.fromPalette({
      name: "on_tertiary_container",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? s.isDark ? 0 : 100 : isFidelity(s) ? DynamicColor.foregroundTone(MaterialDynamicColors.tertiaryContainer.tone(s), 4.5) : s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.tertiaryContainer,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.error = DynamicColor.fromPalette({
      name: "error",
      palette: s => s.errorPalette,
      tone: s => s.isDark ? 80 : 40,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.errorContainer, MaterialDynamicColors.error, 15, "nearer", !1)
    }), MaterialDynamicColors.onError = DynamicColor.fromPalette({
      name: "on_error",
      palette: s => s.errorPalette,
      tone: s => s.isDark ? 20 : 100,
      background: s => MaterialDynamicColors.error,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.errorContainer = DynamicColor.fromPalette({
      name: "error_container",
      palette: s => s.errorPalette,
      tone: s => s.isDark ? 30 : 90,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.errorContainer, MaterialDynamicColors.error, 15, "nearer", !1)
    }), MaterialDynamicColors.onErrorContainer = DynamicColor.fromPalette({
      name: "on_error_container",
      palette: s => s.errorPalette,
      tone: s => s.isDark ? 90 : 10,
      background: s => MaterialDynamicColors.errorContainer,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.primaryFixed = DynamicColor.fromPalette({
      name: "primary_fixed",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? 40 : 90,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.primaryFixed, MaterialDynamicColors.primaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.primaryFixedDim = DynamicColor.fromPalette({
      name: "primary_fixed_dim",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? 30 : 80,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.primaryFixed, MaterialDynamicColors.primaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.onPrimaryFixed = DynamicColor.fromPalette({
      name: "on_primary_fixed",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? 100 : 10,
      background: s => MaterialDynamicColors.primaryFixedDim,
      secondBackground: s => MaterialDynamicColors.primaryFixed,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.onPrimaryFixedVariant = DynamicColor.fromPalette({
      name: "on_primary_fixed_variant",
      palette: s => s.primaryPalette,
      tone: s => isMonochrome(s) ? 90 : 30,
      background: s => MaterialDynamicColors.primaryFixedDim,
      secondBackground: s => MaterialDynamicColors.primaryFixed,
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11)
    }), MaterialDynamicColors.secondaryFixed = DynamicColor.fromPalette({
      name: "secondary_fixed",
      palette: s => s.secondaryPalette,
      tone: s => isMonochrome(s) ? 80 : 90,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.secondaryFixed, MaterialDynamicColors.secondaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.secondaryFixedDim = DynamicColor.fromPalette({
      name: "secondary_fixed_dim",
      palette: s => s.secondaryPalette,
      tone: s => isMonochrome(s) ? 70 : 80,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.secondaryFixed, MaterialDynamicColors.secondaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.onSecondaryFixed = DynamicColor.fromPalette({
      name: "on_secondary_fixed",
      palette: s => s.secondaryPalette,
      tone: s => 10,
      background: s => MaterialDynamicColors.secondaryFixedDim,
      secondBackground: s => MaterialDynamicColors.secondaryFixed,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.onSecondaryFixedVariant = DynamicColor.fromPalette({
      name: "on_secondary_fixed_variant",
      palette: s => s.secondaryPalette,
      tone: s => isMonochrome(s) ? 25 : 30,
      background: s => MaterialDynamicColors.secondaryFixedDim,
      secondBackground: s => MaterialDynamicColors.secondaryFixed,
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11)
    }), MaterialDynamicColors.tertiaryFixed = DynamicColor.fromPalette({
      name: "tertiary_fixed",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? 40 : 90,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.tertiaryFixed, MaterialDynamicColors.tertiaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.tertiaryFixedDim = DynamicColor.fromPalette({
      name: "tertiary_fixed_dim",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? 30 : 80,
      isBackground: !0,
      background: s => MaterialDynamicColors.highestSurface(s),
      contrastCurve: new ContrastCurve(1, 1, 3, 7),
      toneDeltaPair: s => new ToneDeltaPair(MaterialDynamicColors.tertiaryFixed, MaterialDynamicColors.tertiaryFixedDim, 10, "lighter", !0)
    }), MaterialDynamicColors.onTertiaryFixed = DynamicColor.fromPalette({
      name: "on_tertiary_fixed",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? 100 : 10,
      background: s => MaterialDynamicColors.tertiaryFixedDim,
      secondBackground: s => MaterialDynamicColors.tertiaryFixed,
      contrastCurve: new ContrastCurve(4.5, 7, 11, 21)
    }), MaterialDynamicColors.onTertiaryFixedVariant = DynamicColor.fromPalette({
      name: "on_tertiary_fixed_variant",
      palette: s => s.tertiaryPalette,
      tone: s => isMonochrome(s) ? 90 : 30,
      background: s => MaterialDynamicColors.tertiaryFixedDim,
      secondBackground: s => MaterialDynamicColors.tertiaryFixed,
      contrastCurve: new ContrastCurve(3, 4.5, 7, 11)
    });
    class TonalPalette {
      static fromInt(argb) {
        const hct = Hct.fromInt(argb);
        return TonalPalette.fromHct(hct);
      }
      static fromHct(hct) {
        return new TonalPalette(hct.hue, hct.chroma, hct);
      }
      static fromHueAndChroma(hue, chroma) {
        return new TonalPalette(hue, chroma, TonalPalette.createKeyColor(hue, chroma));
      }
      constructor(hue, chroma, keyColor) {
        this.hue = hue, this.chroma = chroma, this.keyColor = keyColor, this.cache = new Map();
      }
      static createKeyColor(hue, chroma) {
        let smallestDeltaHct = Hct.from(hue, chroma, 50),
          smallestDelta = Math.abs(smallestDeltaHct.chroma - chroma);
        for (let delta = 1; delta < 50; delta += 1) {
          if (Math.round(chroma) === Math.round(smallestDeltaHct.chroma)) return smallestDeltaHct;
          const hctAdd = Hct.from(hue, chroma, 50 + delta),
            hctAddDelta = Math.abs(hctAdd.chroma - chroma);
          hctAddDelta < smallestDelta && (smallestDelta = hctAddDelta, smallestDeltaHct = hctAdd);
          const hctSubtract = Hct.from(hue, chroma, 50 - delta),
            hctSubtractDelta = Math.abs(hctSubtract.chroma - chroma);
          hctSubtractDelta < smallestDelta && (smallestDelta = hctSubtractDelta, smallestDeltaHct = hctSubtract);
        }
        return smallestDeltaHct;
      }
      tone(tone) {
        let argb = this.cache.get(tone);
        return void 0 === argb && (argb = Hct.from(this.hue, this.chroma, tone).toInt(), this.cache.set(tone, argb)), argb;
      }
      getHct(tone) {
        return Hct.fromInt(this.tone(tone));
      }
    }
    class CorePalette {
      static of(argb) {
        return new CorePalette(argb, !1);
      }
      static contentOf(argb) {
        return new CorePalette(argb, !0);
      }
      static fromColors(colors) {
        return CorePalette.createPaletteFromColors(!1, colors);
      }
      static contentFromColors(colors) {
        return CorePalette.createPaletteFromColors(!0, colors);
      }
      static createPaletteFromColors(content, colors) {
        const palette = new CorePalette(colors.primary, content);
        if (colors.secondary) {
          const p = new CorePalette(colors.secondary, content);
          palette.a2 = p.a1;
        }
        if (colors.tertiary) {
          const p = new CorePalette(colors.tertiary, content);
          palette.a3 = p.a1;
        }
        if (colors.error) {
          const p = new CorePalette(colors.error, content);
          palette.error = p.a1;
        }
        if (colors.neutral) {
          const p = new CorePalette(colors.neutral, content);
          palette.n1 = p.n1;
        }
        if (colors.neutralVariant) {
          const p = new CorePalette(colors.neutralVariant, content);
          palette.n2 = p.n2;
        }
        return palette;
      }
      constructor(argb, isContent) {
        const hct = Hct.fromInt(argb),
          hue = hct.hue,
          chroma = hct.chroma;
        isContent ? (this.a1 = TonalPalette.fromHueAndChroma(hue, chroma), this.a2 = TonalPalette.fromHueAndChroma(hue, chroma / 3), this.a3 = TonalPalette.fromHueAndChroma(hue + 60, chroma / 2), this.n1 = TonalPalette.fromHueAndChroma(hue, Math.min(chroma / 12, 4)), this.n2 = TonalPalette.fromHueAndChroma(hue, Math.min(chroma / 6, 8))) : (this.a1 = TonalPalette.fromHueAndChroma(hue, Math.max(48, chroma)), this.a2 = TonalPalette.fromHueAndChroma(hue, 16), this.a3 = TonalPalette.fromHueAndChroma(hue + 60, 24), this.n1 = TonalPalette.fromHueAndChroma(hue, 4), this.n2 = TonalPalette.fromHueAndChroma(hue, 8)), this.error = TonalPalette.fromHueAndChroma(25, 84);
      }
    }
    class Scheme {
      get primary() {
        return this.props.primary;
      }
      get onPrimary() {
        return this.props.onPrimary;
      }
      get primaryContainer() {
        return this.props.primaryContainer;
      }
      get onPrimaryContainer() {
        return this.props.onPrimaryContainer;
      }
      get secondary() {
        return this.props.secondary;
      }
      get onSecondary() {
        return this.props.onSecondary;
      }
      get secondaryContainer() {
        return this.props.secondaryContainer;
      }
      get onSecondaryContainer() {
        return this.props.onSecondaryContainer;
      }
      get tertiary() {
        return this.props.tertiary;
      }
      get onTertiary() {
        return this.props.onTertiary;
      }
      get tertiaryContainer() {
        return this.props.tertiaryContainer;
      }
      get onTertiaryContainer() {
        return this.props.onTertiaryContainer;
      }
      get error() {
        return this.props.error;
      }
      get onError() {
        return this.props.onError;
      }
      get errorContainer() {
        return this.props.errorContainer;
      }
      get onErrorContainer() {
        return this.props.onErrorContainer;
      }
      get background() {
        return this.props.background;
      }
      get onBackground() {
        return this.props.onBackground;
      }
      get surface() {
        return this.props.surface;
      }
      get onSurface() {
        return this.props.onSurface;
      }
      get surfaceVariant() {
        return this.props.surfaceVariant;
      }
      get onSurfaceVariant() {
        return this.props.onSurfaceVariant;
      }
      get outline() {
        return this.props.outline;
      }
      get outlineVariant() {
        return this.props.outlineVariant;
      }
      get shadow() {
        return this.props.shadow;
      }
      get scrim() {
        return this.props.scrim;
      }
      get inverseSurface() {
        return this.props.inverseSurface;
      }
      get inverseOnSurface() {
        return this.props.inverseOnSurface;
      }
      get inversePrimary() {
        return this.props.inversePrimary;
      }
      static light(argb) {
        return Scheme.lightFromCorePalette(CorePalette.of(argb));
      }
      static dark(argb) {
        return Scheme.darkFromCorePalette(CorePalette.of(argb));
      }
      static lightContent(argb) {
        return Scheme.lightFromCorePalette(CorePalette.contentOf(argb));
      }
      static darkContent(argb) {
        return Scheme.darkFromCorePalette(CorePalette.contentOf(argb));
      }
      static lightFromCorePalette(core) {
        return new Scheme({
          primary: core.a1.tone(40),
          onPrimary: core.a1.tone(100),
          primaryContainer: core.a1.tone(90),
          onPrimaryContainer: core.a1.tone(10),
          secondary: core.a2.tone(40),
          onSecondary: core.a2.tone(100),
          secondaryContainer: core.a2.tone(90),
          onSecondaryContainer: core.a2.tone(10),
          tertiary: core.a3.tone(40),
          onTertiary: core.a3.tone(100),
          tertiaryContainer: core.a3.tone(90),
          onTertiaryContainer: core.a3.tone(10),
          error: core.error.tone(40),
          onError: core.error.tone(100),
          errorContainer: core.error.tone(90),
          onErrorContainer: core.error.tone(10),
          background: core.n1.tone(99),
          onBackground: core.n1.tone(10),
          surface: core.n1.tone(99),
          onSurface: core.n1.tone(10),
          surfaceVariant: core.n2.tone(90),
          onSurfaceVariant: core.n2.tone(30),
          outline: core.n2.tone(50),
          outlineVariant: core.n2.tone(80),
          shadow: core.n1.tone(0),
          scrim: core.n1.tone(0),
          inverseSurface: core.n1.tone(20),
          inverseOnSurface: core.n1.tone(95),
          inversePrimary: core.a1.tone(80)
        });
      }
      static darkFromCorePalette(core) {
        return new Scheme({
          primary: core.a1.tone(80),
          onPrimary: core.a1.tone(20),
          primaryContainer: core.a1.tone(30),
          onPrimaryContainer: core.a1.tone(90),
          secondary: core.a2.tone(80),
          onSecondary: core.a2.tone(20),
          secondaryContainer: core.a2.tone(30),
          onSecondaryContainer: core.a2.tone(90),
          tertiary: core.a3.tone(80),
          onTertiary: core.a3.tone(20),
          tertiaryContainer: core.a3.tone(30),
          onTertiaryContainer: core.a3.tone(90),
          error: core.error.tone(80),
          onError: core.error.tone(20),
          errorContainer: core.error.tone(30),
          onErrorContainer: core.error.tone(80),
          background: core.n1.tone(10),
          onBackground: core.n1.tone(90),
          surface: core.n1.tone(10),
          onSurface: core.n1.tone(90),
          surfaceVariant: core.n2.tone(30),
          onSurfaceVariant: core.n2.tone(80),
          outline: core.n2.tone(60),
          outlineVariant: core.n2.tone(30),
          shadow: core.n1.tone(0),
          scrim: core.n1.tone(0),
          inverseSurface: core.n1.tone(90),
          inverseOnSurface: core.n1.tone(20),
          inversePrimary: core.a1.tone(40)
        });
      }
      constructor(props) {
        this.props = props;
      }
      toJSON() {
        return {
          ...this.props
        };
      }
    }
    function hexFromArgb(argb) {
      const r = redFromArgb(argb),
        g = greenFromArgb(argb),
        b = blueFromArgb(argb),
        outParts = [r.toString(16), g.toString(16), b.toString(16)];
      for (const [i, part] of outParts.entries()) 1 === part.length && (outParts[i] = "0" + part);
      return "#" + outParts.join("");
    }
    function argbFromHex(hex) {
      const isThree = 3 === (hex = hex.replace("#", "")).length,
        isSix = 6 === hex.length,
        isEight = 8 === hex.length;
      if (!isThree && !isSix && !isEight) throw new Error("unexpected hex " + hex);
      let r = 0,
        g = 0,
        b = 0;
      return isThree ? (r = parseIntHex(hex.slice(0, 1).repeat(2)), g = parseIntHex(hex.slice(1, 2).repeat(2)), b = parseIntHex(hex.slice(2, 3).repeat(2))) : isSix ? (r = parseIntHex(hex.slice(0, 2)), g = parseIntHex(hex.slice(2, 4)), b = parseIntHex(hex.slice(4, 6))) : isEight && (r = parseIntHex(hex.slice(2, 4)), g = parseIntHex(hex.slice(4, 6)), b = parseIntHex(hex.slice(6, 8))), (255 << 24 | (255 & r) << 16 | (255 & g) << 8 | 255 & b) >>> 0;
    }
    function parseIntHex(value) {
      return parseInt(value, 16);
    }
    function themeFromSourceColor(source, customColors) {
      if (customColors === void 0) {
        customColors = [];
      }
      const palette = CorePalette.of(source);
      return {
        source: source,
        schemes: {
          light: Scheme.light(source),
          dark: Scheme.dark(source)
        },
        palettes: {
          primary: palette.a1,
          secondary: palette.a2,
          tertiary: palette.a3,
          neutral: palette.n1,
          neutralVariant: palette.n2,
          error: palette.error
        },
        customColors: customColors.map(c => customColor(source, c))
      };
    }
    function customColor(source, color) {
      let value = color.value;
      const from = value,
        to = source;
      color.blend && (value = Blend.harmonize(from, to));
      const tones = CorePalette.of(value).a1;
      return {
        color: color,
        value: value,
        light: {
          color: tones.tone(40),
          onColor: tones.tone(100),
          colorContainer: tones.tone(90),
          onColorContainer: tones.tone(10)
        },
        dark: {
          color: tones.tone(80),
          onColor: tones.tone(20),
          colorContainer: tones.tone(30),
          onColorContainer: tones.tone(90)
        }
      };
    }

    // eslint-disable-next-line

    /* eslint-disable */
    // prettier-ignore
    function toRGBA(d) {
      const r = Math.round;
      const l = d.length;
      const rgba = {};
      if (d.slice(0, 3).toLowerCase() === 'rgb') {
        d = d.replace(' ', '').split(',');
        rgba[0] = parseInt(d[0].slice(d[3].toLowerCase() === 'a' ? 5 : 4), 10);
        rgba[1] = parseInt(d[1], 10);
        rgba[2] = parseInt(d[2], 10);
        rgba[3] = d[3] ? parseFloat(d[3]) : -1;
      } else {
        if (l < 6) d = parseInt(String(d[1]) + d[1] + d[2] + d[2] + d[3] + d[3] + (l > 4 ? String(d[4]) + d[4] : ''), 16);else d = parseInt(d.slice(1), 16);
        rgba[0] = d >> 16 & 255;
        rgba[1] = d >> 8 & 255;
        rgba[2] = d & 255;
        rgba[3] = l === 9 || l === 5 ? r((d >> 24 & 255) / 255 * 10000) / 10000 : -1;
      }
      return rgba;
    }

    // prettier-ignore
    function blend(from, to, p) {
      if (p === void 0) {
        p = 0.5;
      }
      const r = Math.round;
      from = from.trim();
      to = to.trim();
      const b = p < 0;
      p = b ? p * -1 : p;
      const f = toRGBA(from);
      const t = toRGBA(to);
      if (to[0] === 'r') {
        return 'rgb' + (to[3] === 'a' ? 'a(' : '(') + r((t[0] - f[0]) * p + f[0]) + ',' + r((t[1] - f[1]) * p + f[1]) + ',' + r((t[2] - f[2]) * p + f[2]) + (f[3] < 0 && t[3] < 0 ? '' : ',' + (f[3] > -1 && t[3] > -1 ? r(((t[3] - f[3]) * p + f[3]) * 10000) / 10000 : t[3] < 0 ? f[3] : t[3])) + ')';
      }
      return '#' + (0x100000000 + (f[3] > -1 && t[3] > -1 ? r(((t[3] - f[3]) * p + f[3]) * 255) : t[3] > -1 ? r(t[3] * 255) : f[3] > -1 ? r(f[3] * 255) : 255) * 0x1000000 + r((t[0] - f[0]) * p + f[0]) * 0x10000 + r((t[1] - f[1]) * p + f[1]) * 0x100 + r((t[2] - f[2]) * p + f[2])).toString(16).slice(f[3] > -1 || t[3] > -1 ? 1 : 3);
    }
    /* eslint-enable */

    const materialColors = function (hexColor) {
      if (hexColor === void 0) {
        hexColor = '';
      }
      const theme = themeFromSourceColor(argbFromHex(`#${hexColor.replace('#', '')}`));
      [0.05, 0.08, 0.11, 0.12, 0.14].forEach((amount, index) => {
        theme.schemes.light.props[`surface${index + 1}`] = argbFromHex(blend(hexFromArgb(theme.schemes.light.props.surface), hexFromArgb(theme.schemes.light.props.primary), amount));
        theme.schemes.dark.props[`surface${index + 1}`] = argbFromHex(blend(hexFromArgb(theme.schemes.dark.props.surface), hexFromArgb(theme.schemes.dark.props.primary), amount));
      });
      const name = n => {
        return n.split('').map(char => char.toUpperCase() === char && char !== '-' && char !== '7' ? `-${char.toLowerCase()}` : char).join('');
      };
      const shouldSkip = prop => {
        const skip = ['tertiary', 'shadow', 'scrim', 'error', 'background'];
        return skip.filter(v => prop.toLowerCase().includes(v)).length > 0;
      };
      const light = {};
      const dark = {};
      Object.keys(theme.schemes.light.props).forEach(prop => {
        if (shouldSkip(prop)) return;
        light[name(`--f7-md-${prop}`)] = hexFromArgb(theme.schemes.light.props[prop]);
      });
      Object.keys(theme.schemes.dark.props).forEach(prop => {
        if (shouldSkip(prop)) return;
        dark[name(`--f7-md-${prop}`)] = hexFromArgb(theme.schemes.dark.props[prop]);
      });
      return {
        light,
        dark
      };
    };

    let uniqueNum = 0;
    function uniqueNumber() {
      uniqueNum += 1;
      return uniqueNum;
    }
    function id$1(mask, map) {
      if (mask === void 0) {
        mask = 'xxxxxxxxxx';
      }
      if (map === void 0) {
        map = '0123456789abcdef';
      }
      const length = map.length;
      return mask.replace(/x/g, () => map[Math.floor(Math.random() * length)]);
    }
    const mdPreloaderContent = `
  <span class="preloader-inner">
    <svg viewBox="0 0 36 36">
      <circle cx="18" cy="18" r="16"></circle>
    </svg>
  </span>
`.trim();
    const iosPreloaderContent = `
  <span class="preloader-inner">
    ${[0, 1, 2, 3, 4, 5, 6, 7].map(() => '<span class="preloader-inner-line"></span>').join('')}
  </span>
`.trim();
    function eventNameToColonCase(eventName) {
      let hasColon;
      return eventName.split('').map((char, index) => {
        if (char.match(/[A-Z]/) && index !== 0 && !hasColon) {
          hasColon = true;
          return `:${char.toLowerCase()}`;
        }
        return char.toLowerCase();
      }).join('');
    }
    function deleteProps$1(obj) {
      const object = obj;
      Object.keys(object).forEach(key => {
        try {
          object[key] = null;
        } catch (e) {
          // no setter for object
        }
        try {
          delete object[key];
        } catch (e) {
          // something got wrong
        }
      });
    }
    function requestAnimationFrame$1(callback) {
      const window = getWindow$1();
      return window.requestAnimationFrame(callback);
    }
    function cancelAnimationFrame$1(frameId) {
      const window = getWindow$1();
      return window.cancelAnimationFrame(frameId);
    }
    function nextTick$1(callback, delay) {
      if (delay === void 0) {
        delay = 0;
      }
      return setTimeout(callback, delay);
    }
    function nextFrame$1(callback) {
      return requestAnimationFrame$1(() => {
        requestAnimationFrame$1(callback);
      });
    }
    function now$1() {
      return Date.now();
    }
    function parseUrlQuery(url) {
      const window = getWindow$1();
      const query = {};
      let urlToParse = url || window.location.href;
      let i;
      let params;
      let param;
      let length;
      if (typeof urlToParse === 'string' && urlToParse.length) {
        urlToParse = urlToParse.indexOf('?') > -1 ? urlToParse.replace(/\S*\?/, '') : '';
        params = urlToParse.split('&').filter(paramsPart => paramsPart !== '');
        length = params.length;
        for (i = 0; i < length; i += 1) {
          param = params[i].replace(/#\S+/g, '').split('=');
          query[decodeURIComponent(param[0])] = typeof param[1] === 'undefined' ? undefined : decodeURIComponent(param.slice(1).join('=')) || '';
        }
      }
      return query;
    }
    function getTranslate$1(el, axis) {
      if (axis === void 0) {
        axis = 'x';
      }
      const window = getWindow$1();
      let matrix;
      let curTransform;
      let transformMatrix;
      const curStyle = window.getComputedStyle(el, null);
      if (window.WebKitCSSMatrix) {
        curTransform = curStyle.transform || curStyle.webkitTransform;
        if (curTransform.split(',').length > 6) {
          curTransform = curTransform.split(', ').map(a => a.replace(',', '.')).join(', ');
        }
        // Some old versions of Webkit choke when 'none' is passed; pass
        // empty string instead in this case
        transformMatrix = new window.WebKitCSSMatrix(curTransform === 'none' ? '' : curTransform);
      } else {
        transformMatrix = curStyle.MozTransform || curStyle.OTransform || curStyle.MsTransform || curStyle.msTransform || curStyle.transform || curStyle.getPropertyValue('transform').replace('translate(', 'matrix(1, 0, 0, 1,');
        matrix = transformMatrix.toString().split(',');
      }
      if (axis === 'x') {
        // Latest Chrome and webkits Fix
        if (window.WebKitCSSMatrix) curTransform = transformMatrix.m41;
        // Crazy IE10 Matrix
        else if (matrix.length === 16) curTransform = parseFloat(matrix[12]);
        // Normal Browsers
        else curTransform = parseFloat(matrix[4]);
      }
      if (axis === 'y') {
        // Latest Chrome and webkits Fix
        if (window.WebKitCSSMatrix) curTransform = transformMatrix.m42;
        // Crazy IE10 Matrix
        else if (matrix.length === 16) curTransform = parseFloat(matrix[13]);
        // Normal Browsers
        else curTransform = parseFloat(matrix[5]);
      }
      return curTransform || 0;
    }
    function serializeObject(obj, parents) {
      if (parents === void 0) {
        parents = [];
      }
      if (typeof obj === 'string') return obj;
      const resultArray = [];
      const separator = '&';
      let newParents;
      function varName(name) {
        if (parents.length > 0) {
          let parentParts = '';
          for (let j = 0; j < parents.length; j += 1) {
            if (j === 0) parentParts += parents[j];else parentParts += `[${encodeURIComponent(parents[j])}]`;
          }
          return `${parentParts}[${encodeURIComponent(name)}]`;
        }
        return encodeURIComponent(name);
      }
      function varValue(value) {
        return encodeURIComponent(value);
      }
      Object.keys(obj).forEach(prop => {
        let toPush;
        if (Array.isArray(obj[prop])) {
          toPush = [];
          for (let i = 0; i < obj[prop].length; i += 1) {
            if (!Array.isArray(obj[prop][i]) && typeof obj[prop][i] === 'object') {
              newParents = parents.slice();
              newParents.push(prop);
              newParents.push(String(i));
              toPush.push(serializeObject(obj[prop][i], newParents));
            } else {
              toPush.push(`${varName(prop)}[]=${varValue(obj[prop][i])}`);
            }
          }
          if (toPush.length > 0) resultArray.push(toPush.join(separator));
        } else if (obj[prop] === null || obj[prop] === '') {
          resultArray.push(`${varName(prop)}=`);
        } else if (typeof obj[prop] === 'object') {
          // Object, convert to named array
          newParents = parents.slice();
          newParents.push(prop);
          toPush = serializeObject(obj[prop], newParents);
          if (toPush !== '') resultArray.push(toPush);
        } else if (typeof obj[prop] !== 'undefined' && obj[prop] !== '') {
          // Should be string or plain value
          resultArray.push(`${varName(prop)}=${varValue(obj[prop])}`);
        } else if (obj[prop] === '') resultArray.push(varName(prop)); // eslint-disable-line
      });

      return resultArray.join(separator);
    }
    function isObject$3(o) {
      return typeof o === 'object' && o !== null && o.constructor && o.constructor === Object;
    }
    function merge() {
      for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }
      const to = args[0];
      args.splice(0, 1);
      const from = args;
      for (let i = 0; i < from.length; i += 1) {
        const nextSource = args[i];
        if (nextSource !== undefined && nextSource !== null) {
          const keysArray = Object.keys(Object(nextSource));
          for (let nextIndex = 0, len = keysArray.length; nextIndex < len; nextIndex += 1) {
            const nextKey = keysArray[nextIndex];
            const desc = Object.getOwnPropertyDescriptor(nextSource, nextKey);
            if (desc !== undefined && desc.enumerable) {
              to[nextKey] = nextSource[nextKey];
            }
          }
        }
      }
      return to;
    }
    function extend$3() {
      let deep = true;
      let to;
      let from;
      for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }
      if (typeof args[0] === 'boolean') {
        deep = args[0];
        to = args[1];
        args.splice(0, 2);
        from = args;
      } else {
        to = args[0];
        args.splice(0, 1);
        from = args;
      }
      for (let i = 0; i < from.length; i += 1) {
        const nextSource = args[i];
        if (nextSource !== undefined && nextSource !== null) {
          const keysArray = Object.keys(Object(nextSource));
          for (let nextIndex = 0, len = keysArray.length; nextIndex < len; nextIndex += 1) {
            const nextKey = keysArray[nextIndex];
            const desc = Object.getOwnPropertyDescriptor(nextSource, nextKey);
            if (desc !== undefined && desc.enumerable) {
              if (!deep) {
                to[nextKey] = nextSource[nextKey];
              } else if (isObject$3(to[nextKey]) && isObject$3(nextSource[nextKey])) {
                extend$3(to[nextKey], nextSource[nextKey]);
              } else if (!isObject$3(to[nextKey]) && isObject$3(nextSource[nextKey])) {
                to[nextKey] = {};
                extend$3(to[nextKey], nextSource[nextKey]);
              } else {
                to[nextKey] = nextSource[nextKey];
              }
            }
          }
        }
      }
      return to;
    }
    function colorHexToRgb(hex) {
      const h = hex.replace(/^#?([a-f\d])([a-f\d])([a-f\d])$/i, (m, r, g, b) => r + r + g + g + b + b);
      const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(h);
      return result ? result.slice(1).map(n => parseInt(n, 16)) : null;
    }
    function colorRgbToHex(r, g, b) {
      const result = [r, g, b].map(n => {
        const hex = n.toString(16);
        return hex.length === 1 ? `0${hex}` : hex;
      }).join('');
      return `#${result}`;
    }
    function colorRgbToHsl(r, g, b) {
      r /= 255; // eslint-disable-line
      g /= 255; // eslint-disable-line
      b /= 255; // eslint-disable-line
      const max = Math.max(r, g, b);
      const min = Math.min(r, g, b);
      const d = max - min;
      let h;
      if (d === 0) h = 0;else if (max === r) h = (g - b) / d % 6;else if (max === g) h = (b - r) / d + 2;else if (max === b) h = (r - g) / d + 4;
      const l = (min + max) / 2;
      const s = d === 0 ? 0 : d / (1 - Math.abs(2 * l - 1));
      if (h < 0) h = 360 / 60 + h;
      return [h * 60, s, l];
    }
    function colorHslToRgb(h, s, l) {
      const c = (1 - Math.abs(2 * l - 1)) * s;
      const hp = h / 60;
      const x = c * (1 - Math.abs(hp % 2 - 1));
      let rgb1;
      if (Number.isNaN(h) || typeof h === 'undefined') {
        rgb1 = [0, 0, 0];
      } else if (hp <= 1) rgb1 = [c, x, 0];else if (hp <= 2) rgb1 = [x, c, 0];else if (hp <= 3) rgb1 = [0, c, x];else if (hp <= 4) rgb1 = [0, x, c];else if (hp <= 5) rgb1 = [x, 0, c];else if (hp <= 6) rgb1 = [c, 0, x];
      const m = l - c / 2;
      return rgb1.map(n => Math.max(0, Math.min(255, Math.round(255 * (n + m)))));
    }
    function colorHsbToHsl(h, s, b) {
      const HSL = {
        h,
        s: 0,
        l: 0
      };
      const HSB = {
        h,
        s,
        b
      };
      HSL.l = (2 - HSB.s) * HSB.b / 2;
      HSL.s = HSL.l && HSL.l < 1 ? HSB.s * HSB.b / (HSL.l < 0.5 ? HSL.l * 2 : 2 - HSL.l * 2) : HSL.s;
      return [HSL.h, HSL.s, HSL.l];
    }
    function colorHslToHsb(h, s, l) {
      const HSB = {
        h,
        s: 0,
        b: 0
      };
      const HSL = {
        h,
        s,
        l
      };
      const t = HSL.s * (HSL.l < 0.5 ? HSL.l : 1 - HSL.l);
      HSB.b = HSL.l + t;
      HSB.s = HSL.l > 0 ? 2 * t / HSB.b : HSB.s;
      return [HSB.h, HSB.s, HSB.b];
    }
    const getShadeTintColors = rgb => {
      const hsl = colorRgbToHsl(...rgb);
      const hslShade = [hsl[0], hsl[1], Math.max(0, hsl[2] - 0.08)];
      const hslTint = [hsl[0], hsl[1], Math.max(0, hsl[2] + 0.08)];
      const shade = colorRgbToHex(...colorHslToRgb(...hslShade));
      const tint = colorRgbToHex(...colorHslToRgb(...hslTint));
      return {
        shade,
        tint
      };
    };
    function colorThemeCSSProperties() {
      let hex;
      let rgb;
      for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
        args[_key3] = arguments[_key3];
      }
      if (args.length === 1) {
        hex = args[0];
        rgb = colorHexToRgb(hex);
      } else if (args.length === 3) {
        rgb = args;
        hex = colorRgbToHex(...rgb);
      }
      if (!rgb) return {};
      const {
        light,
        dark
      } = materialColors(hex);
      const shadeTintIos = getShadeTintColors(rgb);
      const shadeTintMdLight = getShadeTintColors(colorHexToRgb(light['--f7-md-primary']));
      const shadeTintMdDark = getShadeTintColors(colorHexToRgb(dark['--f7-md-primary']));
      Object.keys(light).forEach(key => {
        if (key.includes('surface-')) {
          light[`${key}-rgb`] = colorHexToRgb(light[key]);
        }
      });
      Object.keys(dark).forEach(key => {
        if (key.includes('surface-')) {
          dark[`${key}-rgb`] = colorHexToRgb(dark[key]);
        }
      });
      return {
        ios: {
          '--f7-theme-color': 'var(--f7-ios-primary)',
          '--f7-theme-color-rgb': 'var(--f7-ios-primary-rgb)',
          '--f7-theme-color-shade': 'var(--f7-ios-primary-shade)',
          '--f7-theme-color-tint': 'var(--f7-ios-primary-tint)'
        },
        md: {
          '--f7-theme-color': 'var(--f7-md-primary)',
          '--f7-theme-color-rgb': 'var(--f7-md-primary-rgb)',
          '--f7-theme-color-shade': 'var(--f7-md-primary-shade)',
          '--f7-theme-color-tint': 'var(--f7-md-primary-tint)'
        },
        light: {
          '--f7-ios-primary': hex,
          '--f7-ios-primary-shade': shadeTintIos.shade,
          '--f7-ios-primary-tint': shadeTintIos.tint,
          '--f7-ios-primary-rgb': rgb.join(', '),
          '--f7-md-primary-shade': shadeTintMdLight.shade,
          '--f7-md-primary-tint': shadeTintMdLight.tint,
          '--f7-md-primary-rgb': colorHexToRgb(light['--f7-md-primary']).join(', '),
          ...light
        },
        dark: {
          '--f7-md-primary-shade': shadeTintMdDark.shade,
          '--f7-md-primary-tint': shadeTintMdDark.tint,
          '--f7-md-primary-rgb': colorHexToRgb(dark['--f7-md-primary']).join(', '),
          ...dark
        }
      };
    }
    function bindMethods(instance, obj) {
      Object.keys(obj).forEach(key => {
        if (isObject$3(obj[key])) {
          Object.keys(obj[key]).forEach(subKey => {
            if (typeof obj[key][subKey] === 'function') {
              obj[key][subKey] = obj[key][subKey].bind(instance);
            }
          });
        }
        instance[key] = obj[key];
      });
    }
    function flattenArray() {
      const arr = [];
      for (var _len4 = arguments.length, args = new Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
        args[_key4] = arguments[_key4];
      }
      args.forEach(arg => {
        if (Array.isArray(arg)) arr.push(...flattenArray(...arg));else arr.push(arg);
      });
      return arr;
    }
    function colorThemeCSSStyles(colors) {
      if (colors === void 0) {
        colors = {};
      }
      const stringifyObject = obj => {
        let res = '';
        Object.keys(obj).forEach(key => {
          res += `${key}:${obj[key]};`;
        });
        return res;
      };
      const colorVars = colorThemeCSSProperties(colors.primary);
      const primary = [`:root{`, stringifyObject(colorVars.light), `--swiper-theme-color:var(--f7-theme-color);`, ...Object.keys(colors).map(colorName => `--f7-color-${colorName}: ${colors[colorName]};`), `}`, `.dark{`, stringifyObject(colorVars.dark), `}`, `.ios, .ios .dark{`, stringifyObject(colorVars.ios), '}', `.md, .md .dark{`, stringifyObject(colorVars.md), '}'].join('');
      const restVars = {};
      Object.keys(colors).forEach(colorName => {
        const colorValue = colors[colorName];
        restVars[colorName] = colorThemeCSSProperties(colorValue);
      });

      // rest
      let rest = '';
      Object.keys(colors).forEach(colorName => {
        const {
          light,
          dark,
          ios,
          md
        } = restVars[colorName];
        const whiteColorVars = `
    --f7-ios-primary: #ffffff;
    --f7-ios-primary-shade: #ebebeb;
    --f7-ios-primary-tint: #ffffff;
    --f7-ios-primary-rgb: 255, 255, 255;
    --f7-md-primary-shade: #eee;
    --f7-md-primary-tint: #fff;
    --f7-md-primary-rgb: 255, 255, 255;
    --f7-md-primary: #fff;
    --f7-md-on-primary: #000;
    --f7-md-primary-container: #fff;
    --f7-md-on-primary-container: #000;
    --f7-md-secondary: #fff;
    --f7-md-on-secondary: #000;
    --f7-md-secondary-container: #555;
    --f7-md-on-secondary-container: #fff;
    --f7-md-surface: #fff;
    --f7-md-on-surface: #000;
    --f7-md-surface-variant: #333;
    --f7-md-on-surface-variant: #fff;
    --f7-md-outline: #fff;
    --f7-md-outline-variant: #fff;
    --f7-md-inverse-surface: #000;
    --f7-md-inverse-on-surface: #fff;
    --f7-md-inverse-primary: #000;
    --f7-md-surface-1: #f8f8f8;
    --f7-md-surface-2: #f1f1f1;
    --f7-md-surface-3: #e7e7e7;
    --f7-md-surface-4: #e1e1e1;
    --f7-md-surface-5: #d7d7d7;
    --f7-md-surface-variant-rgb: 51, 51, 51;
    --f7-md-on-surface-variant-rgb: 255, 255, 255;
    --f7-md-surface-1-rgb: 248, 248, 248;
    --f7-md-surface-2-rgb: 241, 241, 241;
    --f7-md-surface-3-rgb: 231, 231, 231;
    --f7-md-surface-4-rgb: 225, 225, 225;
    --f7-md-surface-5-rgb: 215, 215, 215;
    `;
        const blackColorVars = `
    --f7-ios-primary: #000;
    --f7-ios-primary-shade: #000;
    --f7-ios-primary-tint: #232323;
    --f7-ios-primary-rgb: 0, 0, 0;
    --f7-md-primary-shade: #000;
    --f7-md-primary-tint: #232323;
    --f7-md-primary-rgb: 0, 0, 0;
    --f7-md-primary: #000;
    --f7-md-on-primary: #fff;
    --f7-md-primary-container: #000;
    --f7-md-on-primary-container: #fff;
    --f7-md-secondary: #000;
    --f7-md-on-secondary: #fff;
    --f7-md-secondary-container: #aaa;
    --f7-md-on-secondary-container: #000;
    --f7-md-surface: #000;
    --f7-md-on-surface: #fff;
    --f7-md-surface-variant: #ccc;
    --f7-md-on-surface-variant: #000;
    --f7-md-outline: #000;
    --f7-md-outline-variant: #000;
    --f7-md-inverse-surface: #fff;
    --f7-md-inverse-on-surface: #000;
    --f7-md-inverse-primary: #fff;
    --f7-md-surface-1: #070707;
    --f7-md-surface-2: #161616;
    --f7-md-surface-3: #232323;
    --f7-md-surface-4: #303030;
    --f7-md-surface-5: #373737;
    --f7-md-surface-variant-rgb: 204, 204, 204;
    --f7-md-on-surface-variant-rgb: 0, 0, 0;
    --f7-md-surface-1-rgb: 7, 7, 7;
    --f7-md-surface-2-rgb: 22, 22, 22;
    --f7-md-surface-3-rgb: 35, 35, 35;
    --f7-md-surface-4-rgb: 48, 48, 48;
    --f7-md-surface-5-rgb: 55, 55, 55;
    `;
        /* eslint-disable */
        const lightString = colorName === 'white' ? whiteColorVars : colorName === 'black' ? blackColorVars : stringifyObject(light);
        const darkString = colorName === 'white' ? whiteColorVars : colorName === 'black' ? blackColorVars : stringifyObject(dark);
        /* eslint-enable */
        rest += [`.color-${colorName} {`, lightString, `--swiper-theme-color: var(--f7-theme-color);`, `}`, `.color-${colorName}.dark, .color-${colorName} .dark, .dark .color-${colorName} {`, darkString, `--swiper-theme-color: var(--f7-theme-color);`, `}`, `.ios .color-${colorName}, .ios.color-${colorName}, .ios .dark .color-${colorName}, .ios .dark.color-${colorName} {`, stringifyObject(ios), `}`, `.md .color-${colorName}, .md.color-${colorName}, .md .dark .color-${colorName}, .md .dark.color-${colorName} {`, stringifyObject(md), `}`,
        // text color
        `.text-color-${colorName} {`, `--f7-theme-color-text-color: ${colors[colorName]};`, `}`,
        // bg color
        `.bg-color-${colorName} {`, `--f7-theme-color-bg-color: ${colors[colorName]};`, `}`,
        // border color
        `.border-color-${colorName} {`, `--f7-theme-color-border-color: ${colors[colorName]};`, `}`,
        // ripple color
        `.ripple-color-${colorName} {`, `--f7-theme-color-ripple-color: rgba(${light['--f7-ios-primary-rgb']}, 0.3);`, `}`].join('');
      });
      return `${primary}${rest}`;
    }

    var utils = /*#__PURE__*/Object.freeze({
        __proto__: null,
        bindMethods: bindMethods,
        cancelAnimationFrame: cancelAnimationFrame$1,
        colorHexToRgb: colorHexToRgb,
        colorHsbToHsl: colorHsbToHsl,
        colorHslToHsb: colorHslToHsb,
        colorHslToRgb: colorHslToRgb,
        colorRgbToHex: colorRgbToHex,
        colorRgbToHsl: colorRgbToHsl,
        colorThemeCSSProperties: colorThemeCSSProperties,
        colorThemeCSSStyles: colorThemeCSSStyles,
        deleteProps: deleteProps$1,
        eventNameToColonCase: eventNameToColonCase,
        extend: extend$3,
        flattenArray: flattenArray,
        getTranslate: getTranslate$1,
        id: id$1,
        iosPreloaderContent: iosPreloaderContent,
        isObject: isObject$3,
        mdPreloaderContent: mdPreloaderContent,
        merge: merge,
        nextFrame: nextFrame$1,
        nextTick: nextTick$1,
        now: now$1,
        parseUrlQuery: parseUrlQuery,
        requestAnimationFrame: requestAnimationFrame$1,
        serializeObject: serializeObject,
        uniqueNumber: uniqueNumber
    });

    let support$1;
    function calcSupport$1() {
      const window = getWindow$1();
      const document = getDocument$1();
      return {
        touch: !!('ontouchstart' in window || window.DocumentTouch && document instanceof window.DocumentTouch),
        pointerEvents: !!window.PointerEvent && 'maxTouchPoints' in window.navigator && window.navigator.maxTouchPoints >= 0,
        passiveListener: function checkPassiveListener() {
          let supportsPassive = false;
          try {
            const opts = Object.defineProperty({}, 'passive', {
              // eslint-disable-next-line
              get() {
                supportsPassive = true;
              }
            });
            window.addEventListener('testPassiveListener', null, opts);
          } catch (e) {
            // No support
          }
          return supportsPassive;
        }(),
        intersectionObserver: function checkObserver() {
          return 'IntersectionObserver' in window;
        }()
      };
    }
    function getSupport$1() {
      if (!support$1) {
        support$1 = calcSupport$1();
      }
      return support$1;
    }

    let deviceCalculated;
    function calcDevice$1(_temp) {
      let {
        userAgent
      } = _temp === void 0 ? {} : _temp;
      const support = getSupport$1();
      const window = getWindow$1();
      const platform = window.navigator.platform;
      const ua = userAgent || window.navigator.userAgent;
      const device = {
        ios: false,
        android: false,
        androidChrome: false,
        desktop: false,
        iphone: false,
        ipod: false,
        ipad: false,
        edge: false,
        ie: false,
        firefox: false,
        macos: false,
        windows: false,
        cordova: !!window.cordova,
        electron: false,
        capacitor: !!window.Capacitor,
        nwjs: false
      };
      const screenWidth = window.screen.width;
      const screenHeight = window.screen.height;
      const android = ua.match(/(Android);?[\s\/]+([\d.]+)?/); // eslint-disable-line
      let ipad = ua.match(/(iPad).*OS\s([\d_]+)/);
      const ipod = ua.match(/(iPod)(.*OS\s([\d_]+))?/);
      const iphone = !ipad && ua.match(/(iPhone\sOS|iOS|iPhone;\sCPU\sOS)\s([\d_]+)/);
      const ie = ua.indexOf('MSIE ') >= 0 || ua.indexOf('Trident/') >= 0;
      const edge = ua.indexOf('Edge/') >= 0;
      const firefox = ua.indexOf('Gecko/') >= 0 && ua.indexOf('Firefox/') >= 0;
      const windows = platform === 'Win32';
      const electron = ua.toLowerCase().indexOf('electron') >= 0;
      const nwjs = typeof nw !== 'undefined' && typeof process !== 'undefined' && typeof process.versions !== 'undefined' && typeof process.versions.nw !== 'undefined';
      let macos = platform === 'MacIntel';

      // iPadOs 13 fix
      const iPadScreens = ['1024x1366', '1366x1024', '834x1194', '1194x834', '834x1112', '1112x834', '768x1024', '1024x768', '820x1180', '1180x820', '810x1080', '1080x810'];
      if (!ipad && macos && support.touch && iPadScreens.indexOf(`${screenWidth}x${screenHeight}`) >= 0) {
        ipad = ua.match(/(Version)\/([\d.]+)/);
        if (!ipad) ipad = [0, 1, '13_0_0'];
        macos = false;
      }
      device.ie = ie;
      device.edge = edge;
      device.firefox = firefox;

      // Android
      if (android) {
        device.os = 'android';
        device.osVersion = android[2];
        device.android = true;
        device.androidChrome = ua.toLowerCase().indexOf('chrome') >= 0;
      }
      if (ipad || iphone || ipod) {
        device.os = 'ios';
        device.ios = true;
      }
      // iOS
      if (iphone && !ipod) {
        device.osVersion = iphone[2].replace(/_/g, '.');
        device.iphone = true;
      }
      if (ipad) {
        device.osVersion = ipad[2].replace(/_/g, '.');
        device.ipad = true;
      }
      if (ipod) {
        device.osVersion = ipod[3] ? ipod[3].replace(/_/g, '.') : null;
        device.ipod = true;
      }
      // iOS 8+ changed UA
      if (device.ios && device.osVersion && ua.indexOf('Version/') >= 0) {
        if (device.osVersion.split('.')[0] === '10') {
          device.osVersion = ua.toLowerCase().split('version/')[1].split(' ')[0];
        }
      }

      // Webview
      device.webView = !!((iphone || ipad || ipod) && (ua.match(/.*AppleWebKit(?!.*Safari)/i) || window.navigator.standalone)) || window.matchMedia && window.matchMedia('(display-mode: standalone)').matches;
      device.webview = device.webView;
      device.standalone = device.webView;

      // Desktop
      device.desktop = !(device.ios || device.android) || electron || nwjs;
      if (device.desktop) {
        device.electron = electron;
        device.nwjs = nwjs;
        device.macos = macos;
        device.windows = windows;
        if (device.macos) {
          device.os = 'macos';
        }
        if (device.windows) {
          device.os = 'windows';
        }
      }

      // Pixel Ratio
      device.pixelRatio = window.devicePixelRatio || 1;

      // Color Scheme
      const DARK = '(prefers-color-scheme: dark)';
      const LIGHT = '(prefers-color-scheme: light)';
      device.prefersColorScheme = function prefersColorTheme() {
        let theme;
        if (window.matchMedia && window.matchMedia(LIGHT).matches) {
          theme = 'light';
        }
        if (window.matchMedia && window.matchMedia(DARK).matches) {
          theme = 'dark';
        }
        return theme;
      };

      // Export object
      return device;
    }
    const IS_BROWSER = (() => {
      const document = getDocument$1();
      try {
        // eslint-disable-next-line no-restricted-globals
        return Boolean(document && document.body && document.body.getBoundingClientRect && document.body.getBoundingClientRect().width > 0);
      } catch (e) {
        return false;
      }
    })();
    function getDevice$1(overrides, reset) {
      if (overrides === void 0) {
        overrides = {};
      }
      if (reset === void 0) {
        reset = IS_BROWSER;
      }
      if (!deviceCalculated || reset) {
        deviceCalculated = calcDevice$1(overrides);
      }
      return deviceCalculated;
    }

    class EventsClass {
      constructor(parents) {
        if (parents === void 0) {
          parents = [];
        }
        const self = this;
        self.eventsParents = parents;
        self.eventsListeners = {};
      }
      on(events, handler, priority) {
        const self = this;
        if (typeof handler !== 'function') return self;
        const method = priority ? 'unshift' : 'push';
        events.split(' ').forEach(event => {
          if (!self.eventsListeners[event]) self.eventsListeners[event] = [];
          self.eventsListeners[event][method](handler);
        });
        return self;
      }
      once(events, handler, priority) {
        const self = this;
        if (typeof handler !== 'function') return self;
        function onceHandler() {
          self.off(events, onceHandler);
          if (onceHandler.f7proxy) {
            delete onceHandler.f7proxy;
          }
          for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
          }
          handler.apply(self, args);
        }
        onceHandler.f7proxy = handler;
        return self.on(events, onceHandler, priority);
      }
      off(events, handler) {
        const self = this;
        if (!self.eventsListeners) return self;
        events.split(' ').forEach(event => {
          if (typeof handler === 'undefined') {
            self.eventsListeners[event] = [];
          } else if (self.eventsListeners[event]) {
            self.eventsListeners[event].forEach((eventHandler, index) => {
              if (eventHandler === handler || eventHandler.f7proxy && eventHandler.f7proxy === handler) {
                self.eventsListeners[event].splice(index, 1);
              }
            });
          }
        });
        return self;
      }
      emit() {
        const self = this;
        if (!self.eventsListeners) return self;
        let events;
        let data;
        let context;
        let eventsParents;
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        if (typeof args[0] === 'string' || Array.isArray(args[0])) {
          events = args[0];
          data = args.slice(1, args.length);
          context = self;
          eventsParents = self.eventsParents;
        } else {
          events = args[0].events;
          data = args[0].data;
          context = args[0].context || self;
          eventsParents = args[0].local ? [] : args[0].parents || self.eventsParents;
        }
        const eventsArray = Array.isArray(events) ? events : events.split(' ');
        const localEvents = eventsArray.map(eventName => eventName.replace('local::', ''));
        const parentEvents = eventsArray.filter(eventName => eventName.indexOf('local::') < 0);
        localEvents.forEach(event => {
          if (self.eventsListeners && self.eventsListeners[event]) {
            const handlers = [];
            self.eventsListeners[event].forEach(eventHandler => {
              handlers.push(eventHandler);
            });
            handlers.forEach(eventHandler => {
              eventHandler.apply(context, data);
            });
          }
        });
        if (eventsParents && eventsParents.length > 0) {
          eventsParents.forEach(eventsParent => {
            eventsParent.emit(parentEvents, ...data);
          });
        }
        return self;
      }
    }

    class Framework7Class extends EventsClass {
      constructor(params, parents) {
        if (params === void 0) {
          params = {};
        }
        if (parents === void 0) {
          parents = [];
        }
        super(parents);
        const self = this;
        self.params = params;
        if (self.params && self.params.on) {
          Object.keys(self.params.on).forEach(eventName => {
            self.on(eventName, self.params.on[eventName]);
          });
        }
      }

      // eslint-disable-next-line
      useModuleParams(module, instanceParams) {
        if (module.params) {
          const originalParams = {};
          Object.keys(module.params).forEach(paramKey => {
            if (typeof instanceParams[paramKey] === 'undefined') return;
            originalParams[paramKey] = extend$3({}, instanceParams[paramKey]);
          });
          extend$3(instanceParams, module.params);
          Object.keys(originalParams).forEach(paramKey => {
            extend$3(instanceParams[paramKey], originalParams[paramKey]);
          });
        }
      }
      useModulesParams(instanceParams) {
        const instance = this;
        if (!instance.modules) return;
        Object.keys(instance.modules).forEach(moduleName => {
          const module = instance.modules[moduleName];
          // Extend params
          if (module.params) {
            extend$3(instanceParams, module.params);
          }
        });
      }
      useModule(moduleName, moduleParams) {
        if (moduleName === void 0) {
          moduleName = '';
        }
        if (moduleParams === void 0) {
          moduleParams = {};
        }
        const instance = this;
        if (!instance.modules) return;
        const module = typeof moduleName === 'string' ? instance.modules[moduleName] : moduleName;
        if (!module) return;

        // Extend instance methods and props
        if (module.instance) {
          Object.keys(module.instance).forEach(modulePropName => {
            const moduleProp = module.instance[modulePropName];
            if (typeof moduleProp === 'function') {
              instance[modulePropName] = moduleProp.bind(instance);
            } else {
              instance[modulePropName] = moduleProp;
            }
          });
        }
        // Add event listeners
        if (module.on && instance.on) {
          Object.keys(module.on).forEach(moduleEventName => {
            instance.on(moduleEventName, module.on[moduleEventName]);
          });
        }
        // Add vnode hooks
        if (module.vnode) {
          if (!instance.vnodeHooks) instance.vnodeHooks = {};
          Object.keys(module.vnode).forEach(vnodeId => {
            Object.keys(module.vnode[vnodeId]).forEach(hookName => {
              const handler = module.vnode[vnodeId][hookName];
              if (!instance.vnodeHooks[hookName]) instance.vnodeHooks[hookName] = {};
              if (!instance.vnodeHooks[hookName][vnodeId]) instance.vnodeHooks[hookName][vnodeId] = [];
              instance.vnodeHooks[hookName][vnodeId].push(handler.bind(instance));
            });
          });
        }
        // Module create callback
        if (module.create) {
          module.create.bind(instance)(moduleParams);
        }
      }
      useModules(modulesParams) {
        if (modulesParams === void 0) {
          modulesParams = {};
        }
        const instance = this;
        if (!instance.modules) return;
        Object.keys(instance.modules).forEach(moduleName => {
          const moduleParams = modulesParams[moduleName] || {};
          instance.useModule(moduleName, moduleParams);
        });
      }
      static set components(components) {
        const Class = this;
        if (!Class.use) return;
        Class.use(components);
      }
      static installModule(module) {
        const Class = this;
        if (!Class.prototype.modules) Class.prototype.modules = {};
        const name = module.name || `${Object.keys(Class.prototype.modules).length}_${now$1()}`;
        Class.prototype.modules[name] = module;
        // Prototype
        if (module.proto) {
          Object.keys(module.proto).forEach(key => {
            Class.prototype[key] = module.proto[key];
          });
        }
        // Class
        if (module.static) {
          Object.keys(module.static).forEach(key => {
            Class[key] = module.static[key];
          });
        }
        // Callback
        if (module.install) {
          for (var _len = arguments.length, params = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
            params[_key - 1] = arguments[_key];
          }
          module.install.apply(Class, params);
        }
        return Class;
      }
      static use(module) {
        const Class = this;
        if (Array.isArray(module)) {
          module.forEach(m => Class.installModule(m));
          return Class;
        }
        for (var _len2 = arguments.length, params = new Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
          params[_key2 - 1] = arguments[_key2];
        }
        return Class.installModule(module, ...params);
      }
    }

    function ConstructorMethods(parameters) {
      if (parameters === void 0) {
        parameters = {};
      }
      const {
        defaultSelector,
        constructor: Constructor,
        domProp,
        app,
        addMethods
      } = parameters;
      const methods = {
        create() {
          for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
          }
          if (app) return new Constructor(app, ...args);
          return new Constructor(...args);
        },
        get(el) {
          if (el === void 0) {
            el = defaultSelector;
          }
          if (el instanceof Constructor) return el;
          const $el = $(el);
          if ($el.length === 0) return undefined;
          return $el[0][domProp];
        },
        destroy(el) {
          const instance = methods.get(el);
          if (instance && instance.destroy) return instance.destroy();
          return undefined;
        }
      };
      if (addMethods && Array.isArray(addMethods)) {
        addMethods.forEach(methodName => {
          methods[methodName] = function (el) {
            if (el === void 0) {
              el = defaultSelector;
            }
            const instance = methods.get(el);
            for (var _len2 = arguments.length, args = new Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
              args[_key2 - 1] = arguments[_key2];
            }
            if (instance && instance[methodName]) return instance[methodName](...args);
            return undefined;
          };
        });
      }
      return methods;
    }

    function ModalMethods(parameters) {
      if (parameters === void 0) {
        parameters = {};
      }
      const {
        defaultSelector,
        constructor: Constructor,
        app
      } = parameters;
      const methods = extend$3(ConstructorMethods({
        defaultSelector,
        constructor: Constructor,
        app,
        domProp: 'f7Modal'
      }), {
        open(el, animate, targetEl) {
          let $el = $(el);
          if ($el.length > 1 && targetEl) {
            // check if same modal in other page
            const $targetPage = $(targetEl).parents('.page');
            if ($targetPage.length) {
              $el.each(modalEl => {
                const $modalEl = $(modalEl);
                if ($modalEl.parents($targetPage)[0] === $targetPage[0]) {
                  $el = $modalEl;
                }
              });
            }
          }
          if ($el.length > 1) {
            $el = $el.eq($el.length - 1);
          }
          if (!$el.length) return undefined;
          let instance = $el[0].f7Modal;
          if (!instance) {
            const params = $el.dataset();
            instance = new Constructor(app, {
              el: $el,
              ...params
            });
          }
          return instance.open(animate);
        },
        close(el, animate, targetEl) {
          if (el === void 0) {
            el = defaultSelector;
          }
          let $el = $(el);
          if (!$el.length) return undefined;
          if ($el.length > 1) {
            // check if close link (targetEl) in this modal
            let $parentEl;
            if (targetEl) {
              const $targetEl = $(targetEl);
              if ($targetEl.length) {
                $parentEl = $targetEl.parents($el);
              }
            }
            if ($parentEl && $parentEl.length > 0) {
              $el = $parentEl;
            } else {
              $el = $el.eq($el.length - 1);
            }
          }
          let instance = $el[0].f7Modal;
          if (!instance) {
            const params = $el.dataset();
            instance = new Constructor(app, {
              el: $el,
              ...params
            });
          }
          return instance.close(animate);
        }
      });
      return methods;
    }

    const fetchedModules = [];
    function loadModule(moduleToLoad) {
      const Framework7 = this;
      const window = getWindow$1();
      const document = getDocument$1();
      return new Promise((resolve, reject) => {
        const app = Framework7.instance;
        let modulePath;
        let moduleObj;
        let moduleFunc;
        if (!moduleToLoad) {
          reject(new Error('Framework7: Lazy module must be specified'));
          return;
        }
        function install(module) {
          Framework7.use(module);
          if (app) {
            app.useModuleParams(module, app.params);
            app.useModule(module);
          }
        }
        if (typeof moduleToLoad === 'string') {
          const matchNamePattern = moduleToLoad.match(/([a-z0-9-]*)/i);
          if (moduleToLoad.indexOf('.') < 0 && matchNamePattern && matchNamePattern[0].length === moduleToLoad.length) {
            if (!app || app && !app.params.lazyModulesPath) {
              reject(new Error('Framework7: "lazyModulesPath" app parameter must be specified to fetch module by name'));
              return;
            }
            modulePath = `${app.params.lazyModulesPath}/${moduleToLoad}/${moduleToLoad}.lazy.js`;
          } else {
            modulePath = moduleToLoad;
          }
        } else if (typeof moduleToLoad === 'function') {
          moduleFunc = moduleToLoad;
        } else {
          // considering F7-Plugin object
          moduleObj = moduleToLoad;
        }
        if (moduleFunc) {
          const module = moduleFunc(Framework7, false);
          if (!module) {
            reject(new Error("Framework7: Can't find Framework7 component in specified component function"));
            return;
          }
          // Check if it was added
          if (Framework7.prototype.modules && Framework7.prototype.modules[module.name]) {
            resolve();
            return;
          }
          // Install It
          install(module);
          resolve();
        }
        if (moduleObj) {
          const module = moduleObj;
          if (!module) {
            reject(new Error("Framework7: Can't find Framework7 component in specified component"));
            return;
          }
          // Check if it was added
          if (Framework7.prototype.modules && Framework7.prototype.modules[module.name]) {
            resolve();
            return;
          }
          // Install It
          install(module);
          resolve();
        }
        if (modulePath) {
          if (fetchedModules.indexOf(modulePath) >= 0) {
            resolve();
            return;
          }
          fetchedModules.push(modulePath);
          const scriptLoad = new Promise((resolveScript, rejectScript) => {
            fetch(modulePath).then(res => res.text()).then(scriptContent => {
              const callbackId = id$1();
              const callbackLoadName = `f7_component_loader_callback_${callbackId}`;
              const scriptEl = document.createElement('script');
              scriptEl.innerHTML = `window.${callbackLoadName} = function (Framework7, Framework7AutoInstallComponent) {return ${scriptContent.trim()}}`;
              $('head').append(scriptEl);
              const componentLoader = window[callbackLoadName];
              delete window[callbackLoadName];
              $(scriptEl).remove();
              const module = componentLoader(Framework7, false);
              if (!module) {
                rejectScript(new Error(`Framework7: Can't find Framework7 component in ${modulePath} file`));
                return;
              }

              // Check if it was added
              if (Framework7.prototype.modules && Framework7.prototype.modules[module.name]) {
                resolveScript();
                return;
              }

              // Install It
              install(module);
              resolveScript();
            }).catch(err => {
              rejectScript(err);
            });
          });
          const styleLoad = new Promise(resolveStyle => {
            fetch(modulePath.replace('.lazy.js', app.rtl ? '.rtl.css' : '.css').replace('.js', app.rtl ? '.rtl.css' : '.css')).then(res => res.text()).then(styleContent => {
              const styleEl = document.createElement('style');
              styleEl.innerHTML = styleContent;
              $('head').append(styleEl);
              resolveStyle();
            }).catch(() => {
              resolveStyle();
            });
          });
          Promise.all([scriptLoad, styleLoad]).then(() => {
            resolve();
          }).catch(err => {
            reject(err);
          });
        }
      });
    }

    /* eslint-disable prefer-rest-params */
    const $jsx$1 = function (tag, props) {
      const attrs = props || {};
      for (var _len = arguments.length, args = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        args[_key - 2] = arguments[_key];
      }
      const children = args || [];
      const attrsString = Object.keys(attrs).map(attr => {
        if (attr[0] === '_') {
          if (attrs[attr]) return attr.replace('_', '');
          return '';
        }
        return `${attr}="${attrs[attr]}"`;
      }).filter(attr => !!attr).join(' ');
      if (['path', 'img', 'circle', 'polygon', 'line', 'input'].indexOf(tag) >= 0) {
        return `<${tag} ${attrsString} />`.trim();
      }
      const childrenContent = children.filter(c => !!c).map(c => Array.isArray(c) ? c.join('') : c).join('');
      return `<${tag} ${attrsString}>${childrenContent}</${tag}>`.trim();
    };

    /* eslint-disable no-underscore-dangle */

    class Framework7 extends Framework7Class {
      constructor(params) {
        if (params === void 0) {
          params = {};
        }
        super(params);
        // eslint-disable-next-line
        if (Framework7.instance && typeof window !== 'undefined') {
          throw new Error("Framework7 is already initialized and can't be initialized more than once");
        }
        const device = getDevice$1({
          userAgent: params.userAgent || undefined
        });
        const support = getSupport$1();
        const passedParams = extend$3({}, params);

        // App Instance
        const app = this;
        app.device = device;
        app.support = support;
        const w = getWindow$1();
        const d = getDocument$1();
        Framework7.instance = app;

        // Default
        const defaults = {
          el: 'body',
          theme: 'auto',
          routes: [],
          name: 'Framework7',
          lazyModulesPath: null,
          initOnDeviceReady: true,
          init: true,
          darkMode: undefined,
          iosTranslucentBars: true,
          iosTranslucentModals: true,
          component: undefined,
          componentUrl: undefined,
          userAgent: null,
          url: null,
          colors: {
            primary: '#007aff',
            red: '#ff3b30',
            green: '#4cd964',
            blue: '#2196f3',
            pink: '#ff2d55',
            yellow: '#ffcc00',
            orange: '#ff9500',
            purple: '#9c27b0',
            deeppurple: '#673ab7',
            lightblue: '#5ac8fa',
            teal: '#009688',
            lime: '#cddc39',
            deeporange: '#ff6b22',
            white: '#ffffff',
            black: '#000000'
          }
        };

        // Extend defaults with modules params
        app.useModulesParams(defaults);

        // Extend defaults with passed params
        app.params = extend$3(defaults, params);
        extend$3(app, {
          // App Name
          name: app.params.name,
          // Routes
          routes: app.params.routes,
          // Theme
          theme: function getTheme() {
            if (app.params.theme === 'auto') {
              if (device.ios) return 'ios';
              return 'md';
            }
            return app.params.theme;
          }(),
          // Initially passed parameters
          passedParams,
          online: w.navigator.onLine,
          colors: app.params.colors,
          darkMode: app.params.darkMode
        });
        if (params.store) app.params.store = params.store;

        // Save Root
        if (app.$el && app.$el[0]) {
          app.$el[0].f7 = app;
        }

        // Install Modules
        app.useModules();

        // Init Store
        app.initStore();

        // Init
        if (app.params.init) {
          if (device.cordova && app.params.initOnDeviceReady) {
            $(d).on('deviceready', () => {
              app.init();
            });
          } else {
            app.init();
          }
        }

        // Return app instance
        return app;
      }
      setColorTheme(color) {
        if (!color) return;
        const app = this;
        app.colors.primary = color;
        app.setColors();
      }
      setColors() {
        const app = this;
        const document = getDocument$1();
        if (!app.colorsStyleEl) {
          app.colorsStyleEl = document.createElement('style');
          document.head.prepend(app.colorsStyleEl);
        }
        app.colorsStyleEl.textContent = app.utils.colorThemeCSSStyles(app.colors);
      }
      mount(rootEl) {
        const app = this;
        const window = getWindow$1();
        const document = getDocument$1();
        const $rootEl = $(rootEl || app.params.el).eq(0);
        app.$el = $rootEl;
        if (app.$el && app.$el[0]) {
          app.el = app.$el[0];
          app.el.f7 = app;
          app.rtl = $rootEl.css('direction') === 'rtl';
        }

        // Auto Dark Mode
        const DARK = '(prefers-color-scheme: dark)';
        const LIGHT = '(prefers-color-scheme: light)';
        app.mq = {};
        if (window.matchMedia) {
          app.mq.dark = window.matchMedia(DARK);
          app.mq.light = window.matchMedia(LIGHT);
        }
        app.colorSchemeListener = function colorSchemeListener(_ref) {
          let {
            matches,
            media
          } = _ref;
          if (!matches) {
            return;
          }
          const html = document.querySelector('html');
          if (media === DARK) {
            html.classList.add('dark');
            app.darkMode = true;
            app.emit('darkModeChange', true);
          } else if (media === LIGHT) {
            html.classList.remove('dark');
            app.darkMode = false;
            app.emit('darkModeChange', false);
          }
        };
        app.emit('mount');
      }
      initStore() {
        const app = this;
        if (typeof app.params.store !== 'undefined' && app.params.store.__store) {
          app.store = app.params.store;
        } else {
          app.store = app.createStore(app.params.store);
        }
      }
      enableAutoDarkMode() {
        const window = getWindow$1();
        const document = getDocument$1();
        if (!window.matchMedia) return;
        const app = this;
        const html = document.querySelector('html');
        if (app.mq.dark && app.mq.light) {
          app.mq.dark.addEventListener('change', app.colorSchemeListener);
          app.mq.light.addEventListener('change', app.colorSchemeListener);
        }
        if (app.mq.dark && app.mq.dark.matches) {
          html.classList.add('dark');
          app.darkMode = true;
          app.emit('darkModeChange', true);
        } else if (app.mq.light && app.mq.light.matches) {
          html.classList.remove('dark');
          app.darkMode = false;
          app.emit('darkModeChange', false);
        }
      }
      disableAutoDarkMode() {
        const window = getWindow$1();
        if (!window.matchMedia) return;
        const app = this;
        if (app.mq.dark) app.mq.dark.removeEventListener('change', app.colorSchemeListener);
        if (app.mq.light) app.mq.light.removeEventListener('change', app.colorSchemeListener);
      }
      setDarkMode(mode) {
        const app = this;
        if (mode === 'auto') {
          app.enableAutoDarkMode();
        } else {
          app.disableAutoDarkMode();
          $('html')[mode ? 'addClass' : 'removeClass']('dark');
          app.darkMode = mode;
        }
      }
      initAppComponent(callback) {
        const app = this;
        app.router.componentLoader(app.params.component, app.params.componentUrl, {
          componentOptions: {
            el: app.$el[0]
          }
        }, el => {
          app.$el = $(el);
          app.$el[0].f7 = app;
          app.$elComponent = el.f7Component;
          app.el = app.$el[0];
          if (callback) callback();
        }, () => {});
      }
      init(rootEl) {
        const app = this;
        app.setColors();
        app.mount(rootEl);
        const init = () => {
          if (app.initialized) return;
          app.$el.addClass('framework7-initializing');

          // RTL attr
          if (app.rtl) {
            $('html').attr('dir', 'rtl');
          }

          // Auto Dark Mode
          if (typeof app.params.darkMode === 'undefined') {
            app.darkMode = $('html').hasClass('dark');
          } else {
            app.setDarkMode(app.params.darkMode);
          }

          // Watch for online/offline state
          const window = getWindow$1();
          window.addEventListener('offline', () => {
            app.online = false;
            app.emit('offline');
            app.emit('connection', false);
          });
          window.addEventListener('online', () => {
            app.online = true;
            app.emit('online');
            app.emit('connection', true);
          });

          // Root class
          app.$el.addClass('framework7-root');

          // Theme class
          $('html').removeClass('ios md').addClass(app.theme);

          // iOS Translucent
          if (app.params.iosTranslucentBars && app.theme === 'ios') {
            $('html').addClass('ios-translucent-bars');
          }
          if (app.params.iosTranslucentModals && app.theme === 'ios') {
            $('html').addClass('ios-translucent-modals');
          }

          // Init class
          nextFrame$1(() => {
            app.$el.removeClass('framework7-initializing');
          });
          // Emit, init other modules
          app.initialized = true;
          app.emit('init');
        };
        if (app.params.component || app.params.componentUrl) {
          app.initAppComponent(() => {
            init();
          });
        } else {
          init();
        }
        return app;
      }

      // eslint-disable-next-line
      loadModule() {
        return Framework7.loadModule(...arguments);
      }

      // eslint-disable-next-line
      loadModules() {
        return Framework7.loadModules(...arguments);
      }
      getVnodeHooks(hook, id) {
        const app = this;
        if (!app.vnodeHooks || !app.vnodeHooks[hook]) return [];
        return app.vnodeHooks[hook][id] || [];
      }

      // eslint-disable-next-line
      get $() {
        return $;
      }
      static get Dom7() {
        return $;
      }
      static get $() {
        return $;
      }
      static get device() {
        return getDevice$1();
      }
      static get support() {
        return getSupport$1();
      }
      static get Class() {
        return Framework7Class;
      }
      static get Events() {
        return EventsClass;
      }
    }
    Framework7.$jsx = $jsx$1;
    Framework7.ModalMethods = ModalMethods;
    Framework7.ConstructorMethods = ConstructorMethods;
    Framework7.loadModule = loadModule;
    Framework7.loadModules = function loadModules(modules) {
      return Promise.all(modules.map(module => Framework7.loadModule(module)));
    };

    var DeviceModule = {
      name: 'device',
      static: {
        getDevice: getDevice$1
      },
      on: {
        init() {
          const document = getDocument$1();
          const device = getDevice$1();
          const classNames = [];
          const html = document.querySelector('html');
          const metaStatusbar = document.querySelector('meta[name="apple-mobile-web-app-status-bar-style"]');
          if (!html) return;
          if (device.standalone && device.ios && metaStatusbar && metaStatusbar.content === 'black-translucent') {
            classNames.push('device-full-viewport');
          }

          // Pixel Ratio
          classNames.push(`device-pixel-ratio-${Math.floor(device.pixelRatio)}`);
          // OS classes
          if (device.os && !device.desktop) {
            classNames.push(`device-${device.os}`);
          } else if (device.desktop) {
            classNames.push('device-desktop');
            if (device.os) {
              classNames.push(`device-${device.os}`);
            }
          }
          if (device.cordova) {
            classNames.push('device-cordova');
          }
          if (device.capacitor) {
            classNames.push('device-capacitor');
          }

          // Add html classes
          classNames.forEach(className => {
            html.classList.add(className);
          });
        }
      }
    };

    var SupportModule = {
      name: 'support',
      static: {
        getSupport: getSupport$1
      }
    };

    var UtilsModule = {
      name: 'utils',
      proto: {
        utils
      },
      static: {
        utils
      }
    };

    var ResizeModule = {
      name: 'resize',
      create() {
        const app = this;
        app.getSize = () => {
          if (!app.el) return {
            width: 0,
            height: 0,
            left: 0,
            top: 0
          };
          const offset = app.$el.offset();
          const [width, height, left, top] = [app.el.offsetWidth, app.el.offsetHeight, offset.left, offset.top];
          app.width = width;
          app.height = height;
          app.left = left;
          app.top = top;
          return {
            width,
            height,
            left,
            top
          };
        };
      },
      on: {
        init() {
          const app = this;
          const window = getWindow$1();

          // Get Size
          app.getSize();

          // Emit resize
          window.addEventListener('resize', () => {
            app.emit('resize');
          }, false);

          // Emit orientationchange
          window.addEventListener('orientationchange', () => {
            app.emit('orientationchange');
          });
        },
        orientationchange() {
          const document = getDocument$1();
          const device = getDevice$1();
          // Fix iPad weird body scroll
          if (device.ipad) {
            document.body.scrollLeft = 0;
            setTimeout(() => {
              document.body.scrollLeft = 0;
            }, 0);
          }
        },
        resize() {
          const app = this;
          app.getSize();
        }
      }
    };

    /* eslint-disable no-nested-ternary */
    function initTouch() {
      const app = this;
      const device = getDevice$1();
      const support = getSupport$1();
      const window = getWindow$1();
      const document = getDocument$1();
      const params = app.params.touch;
      const useRipple = params[`${app.theme}TouchRipple`];
      if (device.ios && device.webView) {
        // Strange hack required for iOS 8 webview to work on inputs
        window.addEventListener('touchstart', () => {});
      }
      let touchStartX;
      let touchStartY;
      let targetElement;
      let isMoved;
      let tapHoldFired;
      let tapHoldTimeout;
      let preventClick;
      let activableElement;
      let activeTimeout;
      let rippleWave;
      let rippleTarget;
      let rippleTimeout;
      function findActivableElement(el) {
        const target = $(el);
        const parents = target.parents(params.activeStateElements);
        if (target.closest('.no-active-state').length) {
          return null;
        }
        let activable;
        if (target.is(params.activeStateElements)) {
          activable = target;
        }
        if (parents.length > 0) {
          activable = activable ? activable.add(parents) : parents;
        }
        if (activable && activable.length > 1) {
          const newActivable = [];
          let preventPropagation;
          for (let i = 0; i < activable.length; i += 1) {
            if (!preventPropagation) {
              newActivable.push(activable[i]);
              if (activable.eq(i).hasClass('prevent-active-state-propagation') || activable.eq(i).hasClass('no-active-state-propagation')) {
                preventPropagation = true;
              }
            }
          }
          activable = $(newActivable);
        }
        return activable || target;
      }
      function isInsideScrollableView(el) {
        const pageContent = el.parents('.page-content');
        return pageContent.length > 0;
      }
      function addActive() {
        if (!activableElement) return;
        activableElement.addClass('active-state');
      }
      function removeActive() {
        if (!activableElement) return;
        activableElement.removeClass('active-state');
        activableElement = null;
      }

      // Ripple handlers
      function findRippleElement(el) {
        const rippleElements = params.touchRippleElements;
        const $el = $(el);
        if ($el.is(rippleElements)) {
          if ($el.hasClass('no-ripple')) {
            return false;
          }
          return $el;
        }
        if ($el.parents(rippleElements).length > 0) {
          const rippleParent = $el.parents(rippleElements).eq(0);
          if (rippleParent.hasClass('no-ripple')) {
            return false;
          }
          return rippleParent;
        }
        return false;
      }
      function createRipple($el, x, y) {
        if (!$el) return;
        rippleWave = app.touchRipple.create(app, $el, x, y);
      }
      function removeRipple() {
        if (!rippleWave) return;
        rippleWave.remove();
        rippleWave = undefined;
        rippleTarget = undefined;
      }
      function rippleTouchStart(el) {
        rippleTarget = findRippleElement(el);
        if (!rippleTarget || rippleTarget.length === 0) {
          rippleTarget = undefined;
          return;
        }
        const inScrollable = isInsideScrollableView(rippleTarget);
        if (!inScrollable) {
          removeRipple();
          createRipple(rippleTarget, touchStartX, touchStartY);
        } else {
          clearTimeout(rippleTimeout);
          rippleTimeout = setTimeout(() => {
            removeRipple();
            createRipple(rippleTarget, touchStartX, touchStartY);
          }, 80);
        }
      }
      function rippleTouchMove() {
        clearTimeout(rippleTimeout);
        removeRipple();
      }
      function rippleTouchEnd() {
        if (!rippleWave && rippleTarget && !isMoved) {
          clearTimeout(rippleTimeout);
          createRipple(rippleTarget, touchStartX, touchStartY);
          setTimeout(removeRipple, 0);
        } else {
          removeRipple();
        }
      }

      // Mouse Handlers
      function handleMouseDown(e) {
        const $activableEl = findActivableElement(e.target);
        if ($activableEl) {
          $activableEl.addClass('active-state');
          if ('which' in e && e.which === 3) {
            setTimeout(() => {
              $('.active-state').removeClass('active-state');
            }, 0);
          }
        }
        if (useRipple) {
          touchStartX = e.pageX;
          touchStartY = e.pageY;
          rippleTouchStart(e.target, e.pageX, e.pageY);
        }
      }
      function handleMouseMove() {
        if (!params.activeStateOnMouseMove) {
          $('.active-state').removeClass('active-state');
        }
        if (useRipple) {
          rippleTouchMove();
        }
      }
      function handleMouseUp() {
        $('.active-state').removeClass('active-state');
        if (useRipple) {
          rippleTouchEnd();
        }
      }
      function handleTouchCancel() {
        targetElement = null;

        // Remove Active State
        clearTimeout(activeTimeout);
        clearTimeout(tapHoldTimeout);
        if (params.activeState) {
          removeActive();
        }

        // Remove Ripple
        if (useRipple) {
          rippleTouchEnd();
        }
      }
      let isScrolling;
      let isSegmentedStrong = false;
      let segmentedStrongEl = null;
      const touchMoveActivableIos = '.dialog-button, .actions-button';
      let isTouchMoveActivable = false;
      let touchmoveActivableEl = null;
      function handleTouchStart(e) {
        if (!e.isTrusted) return true;
        isMoved = false;
        tapHoldFired = false;
        preventClick = false;
        isScrolling = undefined;
        if (e.targetTouches.length > 1) {
          if (activableElement) removeActive();
          return true;
        }
        if (e.touches.length > 1 && activableElement) {
          removeActive();
        }
        if (params.tapHold) {
          if (tapHoldTimeout) clearTimeout(tapHoldTimeout);
          tapHoldTimeout = setTimeout(() => {
            if (e && e.touches && e.touches.length > 1) return;
            tapHoldFired = true;
            e.preventDefault();
            preventClick = true;
            $(e.target).trigger('taphold', e);
            app.emit('taphold', e);
          }, params.tapHoldDelay);
        }
        targetElement = e.target;
        touchStartX = e.targetTouches[0].pageX;
        touchStartY = e.targetTouches[0].pageY;
        isSegmentedStrong = e.target.closest('.segmented-strong .button-active, .segmented-strong .tab-link-active');
        isTouchMoveActivable = app.theme === 'ios' && e.target.closest(touchMoveActivableIos);
        if (isSegmentedStrong) {
          segmentedStrongEl = isSegmentedStrong.closest('.segmented-strong');
        }
        if (params.activeState) {
          activableElement = findActivableElement(targetElement);
          if (activableElement && !isInsideScrollableView(activableElement)) {
            addActive();
          } else if (activableElement) {
            activeTimeout = setTimeout(addActive, 80);
          }
        }
        if (useRipple) {
          rippleTouchStart(targetElement, touchStartX, touchStartY);
        }
        return true;
      }
      function handleTouchMove(e) {
        if (!e.isTrusted) return;
        let touch;
        let distance;
        let shouldRemoveActive = true;
        if (e.type === 'touchmove') {
          touch = e.targetTouches[0];
          distance = params.touchClicksDistanceThreshold;
        }
        const touchCurrentX = e.targetTouches[0].pageX;
        const touchCurrentY = e.targetTouches[0].pageY;
        if (typeof isScrolling === 'undefined') {
          isScrolling = !!(isScrolling || Math.abs(touchCurrentY - touchStartY) > Math.abs(touchCurrentX - touchStartX));
        }
        if (isTouchMoveActivable || !isScrolling && isSegmentedStrong && segmentedStrongEl) {
          if (e.cancelable) e.preventDefault();
        }
        if (!isScrolling && isSegmentedStrong && segmentedStrongEl) {
          const elementFromPoint = document.elementFromPoint(e.targetTouches[0].clientX, e.targetTouches[0].clientY);
          const buttonEl = elementFromPoint.closest('.segmented-strong .button:not(.button-active):not(.tab-link-active)');
          if (buttonEl && segmentedStrongEl.contains(buttonEl)) {
            $(buttonEl).trigger('click', 'f7Segmented');
            targetElement = buttonEl;
          }
        }
        if (distance && touch) {
          const pageX = touch.pageX;
          const pageY = touch.pageY;
          if (Math.abs(pageX - touchStartX) > distance || Math.abs(pageY - touchStartY) > distance) {
            isMoved = true;
          }
        } else {
          isMoved = true;
        }
        if (isMoved) {
          preventClick = true;
          // Keep active state on touchMove (for dialog and actions buttons)
          if (isTouchMoveActivable) {
            const elementFromPoint = document.elementFromPoint(e.targetTouches[0].clientX, e.targetTouches[0].clientY);
            touchmoveActivableEl = elementFromPoint.closest(touchMoveActivableIos);
            if (touchmoveActivableEl && activableElement && activableElement[0] === touchmoveActivableEl) {
              shouldRemoveActive = false;
            } else if (touchmoveActivableEl) {
              setTimeout(() => {
                activableElement = findActivableElement(touchmoveActivableEl);
                addActive();
              });
            }
          }
          if (params.tapHold) {
            clearTimeout(tapHoldTimeout);
          }
          if (params.activeState && shouldRemoveActive) {
            clearTimeout(activeTimeout);
            removeActive();
          }
          if (useRipple) {
            rippleTouchMove();
          }
        }
      }
      function handleTouchEnd(e) {
        if (!e.isTrusted) return true;
        isScrolling = undefined;
        isSegmentedStrong = false;
        segmentedStrongEl = null;
        isTouchMoveActivable = false;
        clearTimeout(activeTimeout);
        clearTimeout(tapHoldTimeout);
        if (touchmoveActivableEl) {
          $(touchmoveActivableEl).trigger('click', 'f7TouchMoveActivable');
          touchmoveActivableEl = null;
        }
        if (document.activeElement === e.target) {
          if (params.activeState) removeActive();
          if (useRipple) {
            rippleTouchEnd();
          }
          return true;
        }
        if (params.activeState) {
          addActive();
          setTimeout(removeActive, 0);
        }
        if (useRipple) {
          rippleTouchEnd();
        }
        if (params.tapHoldPreventClicks && tapHoldFired || preventClick) {
          if (e.cancelable) e.preventDefault();
          preventClick = true;
          return false;
        }
        return true;
      }
      function handleClick(e) {
        const isOverswipe = e && e.detail && e.detail === 'f7Overswipe';
        const isSegmented = e && e.detail && e.detail === 'f7Segmented';
        // eslint-disable-next-line
        const isTouchMoveActivable = e && e.detail && e.detail === 'f7TouchMoveActivable';
        let localPreventClick = preventClick;
        if (targetElement && e.target !== targetElement) {
          if (isOverswipe || isSegmented || isTouchMoveActivable) {
            localPreventClick = false;
          } else {
            localPreventClick = true;
          }
        } else if (isTouchMoveActivable) {
          localPreventClick = false;
        }
        if (params.tapHold && params.tapHoldPreventClicks && tapHoldFired) {
          localPreventClick = true;
        }
        if (localPreventClick) {
          e.stopImmediatePropagation();
          e.stopPropagation();
          e.preventDefault();
        }
        if (params.tapHold) {
          tapHoldTimeout = setTimeout(() => {
            tapHoldFired = false;
          }, device.ios || device.androidChrome ? 100 : 400);
        }
        preventClick = false;
        targetElement = null;
        return !localPreventClick;
      }
      function emitAppTouchEvent(name, e) {
        app.emit({
          events: name,
          data: [e]
        });
      }
      function appClick(e) {
        emitAppTouchEvent('click', e);
      }
      function appTouchStartActive(e) {
        emitAppTouchEvent('touchstart touchstart:active', e);
      }
      function appTouchMoveActive(e) {
        emitAppTouchEvent('touchmove touchmove:active', e);
      }
      function appTouchEndActive(e) {
        emitAppTouchEvent('touchend touchend:active', e);
      }
      function appTouchStartPassive(e) {
        emitAppTouchEvent('touchstart:passive', e);
      }
      function appTouchMovePassive(e) {
        emitAppTouchEvent('touchmove:passive', e);
      }
      function appTouchEndPassive(e) {
        emitAppTouchEvent('touchend:passive', e);
      }
      const passiveListener = support.passiveListener ? {
        passive: true
      } : false;
      const passiveListenerCapture = support.passiveListener ? {
        passive: true,
        capture: true
      } : true;
      const activeListener = support.passiveListener ? {
        passive: false
      } : false;
      const activeListenerCapture = support.passiveListener ? {
        passive: false,
        capture: true
      } : true;
      document.addEventListener('click', appClick, true);
      if (support.passiveListener) {
        document.addEventListener(app.touchEvents.start, appTouchStartActive, activeListenerCapture);
        document.addEventListener(app.touchEvents.move, appTouchMoveActive, activeListener);
        document.addEventListener(app.touchEvents.end, appTouchEndActive, activeListener);
        document.addEventListener(app.touchEvents.start, appTouchStartPassive, passiveListenerCapture);
        document.addEventListener(app.touchEvents.move, appTouchMovePassive, passiveListener);
        document.addEventListener(app.touchEvents.end, appTouchEndPassive, passiveListener);
      } else {
        document.addEventListener(app.touchEvents.start, e => {
          appTouchStartActive(e);
          appTouchStartPassive(e);
        }, true);
        document.addEventListener(app.touchEvents.move, e => {
          appTouchMoveActive(e);
          appTouchMovePassive(e);
        }, false);
        document.addEventListener(app.touchEvents.end, e => {
          appTouchEndActive(e);
          appTouchEndPassive(e);
        }, false);
      }
      if (support.touch) {
        app.on('click', handleClick);
        app.on('touchstart', handleTouchStart);
        app.on('touchmove', handleTouchMove);
        app.on('touchend', handleTouchEnd);
        document.addEventListener('touchcancel', handleTouchCancel, {
          passive: true
        });
      } else if (params.activeState) {
        app.on('touchstart', handleMouseDown);
        app.on('touchmove', handleMouseMove);
        app.on('touchend', handleMouseUp);
        document.addEventListener('pointercancel', handleMouseUp, {
          passive: true
        });
      }
      document.addEventListener('contextmenu', e => {
        if (params.disableContextMenu && (device.ios || device.android || device.cordova || window.Capacitor && window.Capacitor.isNative)) {
          e.preventDefault();
        }
        if (useRipple) {
          if (activableElement) removeActive();
          rippleTouchEnd();
        }
      });
    }
    var TouchModule = {
      name: 'touch',
      params: {
        touch: {
          // Clicks
          touchClicksDistanceThreshold: 5,
          // ContextMenu
          disableContextMenu: false,
          // Tap Hold
          tapHold: false,
          tapHoldDelay: 750,
          tapHoldPreventClicks: true,
          // Active State
          activeState: true,
          activeStateElements: 'a, button, label, span, .actions-button, .stepper-button, .stepper-button-plus, .stepper-button-minus, .card-expandable, .link, .item-link, .accordion-item-toggle',
          activeStateOnMouseMove: false,
          mdTouchRipple: true,
          iosTouchRipple: false,
          touchRippleElements: '.ripple, .link, .item-link, .list label.item-content, .list-button, .links-list a, .button, button, .input-clear-button, .dialog-button, .tab-link, .item-radio, .item-checkbox, .actions-button, .searchbar-disable-button, .fab a, .checkbox, .radio, .data-table .sortable-cell:not(.input-cell), .notification-close-button, .stepper-button, .stepper-button-minus, .stepper-button-plus, .list.accordion-list .accordion-item-toggle',
          touchRippleInsetElements: '.ripple-inset, .icon-only, .searchbar-disable-button, .input-clear-button, .notification-close-button, .md .navbar .link.back'
        }
      },
      create() {
        const app = this;
        const support = getSupport$1();
        extend$3(app, {
          touchEvents: {
            start: support.touch ? 'touchstart' : support.pointerEvents ? 'pointerdown' : 'mousedown',
            move: support.touch ? 'touchmove' : support.pointerEvents ? 'pointermove' : 'mousemove',
            end: support.touch ? 'touchend' : support.pointerEvents ? 'pointerup' : 'mouseup'
          }
        });
      },
      on: {
        init: initTouch
      }
    };

    /**
     * Tokenize input string.
     */
    function lexer(str) {
      var tokens = [];
      var i = 0;
      while (i < str.length) {
        var char = str[i];
        if (char === "*" || char === "+" || char === "?") {
          tokens.push({
            type: "MODIFIER",
            index: i,
            value: str[i++]
          });
          continue;
        }
        if (char === "\\") {
          tokens.push({
            type: "ESCAPED_CHAR",
            index: i++,
            value: str[i++]
          });
          continue;
        }
        if (char === "{") {
          tokens.push({
            type: "OPEN",
            index: i,
            value: str[i++]
          });
          continue;
        }
        if (char === "}") {
          tokens.push({
            type: "CLOSE",
            index: i,
            value: str[i++]
          });
          continue;
        }
        if (char === ":") {
          var name = "";
          var j = i + 1;
          while (j < str.length) {
            var code = str.charCodeAt(j);
            if (
            // `0-9`
            code >= 48 && code <= 57 ||
            // `A-Z`
            code >= 65 && code <= 90 ||
            // `a-z`
            code >= 97 && code <= 122 ||
            // `_`
            code === 95) {
              name += str[j++];
              continue;
            }
            break;
          }
          if (!name) throw new TypeError("Missing parameter name at ".concat(i));
          tokens.push({
            type: "NAME",
            index: i,
            value: name
          });
          i = j;
          continue;
        }
        if (char === "(") {
          var count = 1;
          var pattern = "";
          var j = i + 1;
          if (str[j] === "?") {
            throw new TypeError("Pattern cannot start with \"?\" at ".concat(j));
          }
          while (j < str.length) {
            if (str[j] === "\\") {
              pattern += str[j++] + str[j++];
              continue;
            }
            if (str[j] === ")") {
              count--;
              if (count === 0) {
                j++;
                break;
              }
            } else if (str[j] === "(") {
              count++;
              if (str[j + 1] !== "?") {
                throw new TypeError("Capturing groups are not allowed at ".concat(j));
              }
            }
            pattern += str[j++];
          }
          if (count) throw new TypeError("Unbalanced pattern at ".concat(i));
          if (!pattern) throw new TypeError("Missing pattern at ".concat(i));
          tokens.push({
            type: "PATTERN",
            index: i,
            value: pattern
          });
          i = j;
          continue;
        }
        tokens.push({
          type: "CHAR",
          index: i,
          value: str[i++]
        });
      }
      tokens.push({
        type: "END",
        index: i,
        value: ""
      });
      return tokens;
    }
    /**
     * Parse a string for the raw tokens.
     */
    function parse(str, options) {
      if (options === void 0) {
        options = {};
      }
      var tokens = lexer(str);
      var _a = options.prefixes,
        prefixes = _a === void 0 ? "./" : _a;
      var defaultPattern = "[^".concat(escapeString(options.delimiter || "/#?"), "]+?");
      var result = [];
      var key = 0;
      var i = 0;
      var path = "";
      var tryConsume = function (type) {
        if (i < tokens.length && tokens[i].type === type) return tokens[i++].value;
      };
      var mustConsume = function (type) {
        var value = tryConsume(type);
        if (value !== undefined) return value;
        var _a = tokens[i],
          nextType = _a.type,
          index = _a.index;
        throw new TypeError("Unexpected ".concat(nextType, " at ").concat(index, ", expected ").concat(type));
      };
      var consumeText = function () {
        var result = "";
        var value;
        while (value = tryConsume("CHAR") || tryConsume("ESCAPED_CHAR")) {
          result += value;
        }
        return result;
      };
      while (i < tokens.length) {
        var char = tryConsume("CHAR");
        var name = tryConsume("NAME");
        var pattern = tryConsume("PATTERN");
        if (name || pattern) {
          var prefix = char || "";
          if (prefixes.indexOf(prefix) === -1) {
            path += prefix;
            prefix = "";
          }
          if (path) {
            result.push(path);
            path = "";
          }
          result.push({
            name: name || key++,
            prefix: prefix,
            suffix: "",
            pattern: pattern || defaultPattern,
            modifier: tryConsume("MODIFIER") || ""
          });
          continue;
        }
        var value = char || tryConsume("ESCAPED_CHAR");
        if (value) {
          path += value;
          continue;
        }
        if (path) {
          result.push(path);
          path = "";
        }
        var open = tryConsume("OPEN");
        if (open) {
          var prefix = consumeText();
          var name_1 = tryConsume("NAME") || "";
          var pattern_1 = tryConsume("PATTERN") || "";
          var suffix = consumeText();
          mustConsume("CLOSE");
          result.push({
            name: name_1 || (pattern_1 ? key++ : ""),
            pattern: name_1 && !pattern_1 ? defaultPattern : pattern_1,
            prefix: prefix,
            suffix: suffix,
            modifier: tryConsume("MODIFIER") || ""
          });
          continue;
        }
        mustConsume("END");
      }
      return result;
    }
    /**
     * Compile a string to a template function for the path.
     */
    function compile(str, options) {
      return tokensToFunction(parse(str, options), options);
    }
    /**
     * Expose a method for transforming tokens into the path function.
     */
    function tokensToFunction(tokens, options) {
      if (options === void 0) {
        options = {};
      }
      var reFlags = flags(options);
      var _a = options.encode,
        encode = _a === void 0 ? function (x) {
          return x;
        } : _a,
        _b = options.validate,
        validate = _b === void 0 ? true : _b;
      // Compile all the tokens into regexps.
      var matches = tokens.map(function (token) {
        if (typeof token === "object") {
          return new RegExp("^(?:".concat(token.pattern, ")$"), reFlags);
        }
      });
      return function (data) {
        var path = "";
        for (var i = 0; i < tokens.length; i++) {
          var token = tokens[i];
          if (typeof token === "string") {
            path += token;
            continue;
          }
          var value = data ? data[token.name] : undefined;
          var optional = token.modifier === "?" || token.modifier === "*";
          var repeat = token.modifier === "*" || token.modifier === "+";
          if (Array.isArray(value)) {
            if (!repeat) {
              throw new TypeError("Expected \"".concat(token.name, "\" to not repeat, but got an array"));
            }
            if (value.length === 0) {
              if (optional) continue;
              throw new TypeError("Expected \"".concat(token.name, "\" to not be empty"));
            }
            for (var j = 0; j < value.length; j++) {
              var segment = encode(value[j], token);
              if (validate && !matches[i].test(segment)) {
                throw new TypeError("Expected all \"".concat(token.name, "\" to match \"").concat(token.pattern, "\", but got \"").concat(segment, "\""));
              }
              path += token.prefix + segment + token.suffix;
            }
            continue;
          }
          if (typeof value === "string" || typeof value === "number") {
            var segment = encode(String(value), token);
            if (validate && !matches[i].test(segment)) {
              throw new TypeError("Expected \"".concat(token.name, "\" to match \"").concat(token.pattern, "\", but got \"").concat(segment, "\""));
            }
            path += token.prefix + segment + token.suffix;
            continue;
          }
          if (optional) continue;
          var typeOfMessage = repeat ? "an array" : "a string";
          throw new TypeError("Expected \"".concat(token.name, "\" to be ").concat(typeOfMessage));
        }
        return path;
      };
    }
    /**
     * Create path match function from `path-to-regexp` spec.
     */
    function match(str, options) {
      var keys = [];
      var re = pathToRegexp(str, keys, options);
      return regexpToFunction(re, keys, options);
    }
    /**
     * Create a path match function from `path-to-regexp` output.
     */
    function regexpToFunction(re, keys, options) {
      if (options === void 0) {
        options = {};
      }
      var _a = options.decode,
        decode = _a === void 0 ? function (x) {
          return x;
        } : _a;
      return function (pathname) {
        var m = re.exec(pathname);
        if (!m) return false;
        var path = m[0],
          index = m.index;
        var params = Object.create(null);
        var _loop_1 = function (i) {
          if (m[i] === undefined) return "continue";
          var key = keys[i - 1];
          if (key.modifier === "*" || key.modifier === "+") {
            params[key.name] = m[i].split(key.prefix + key.suffix).map(function (value) {
              return decode(value, key);
            });
          } else {
            params[key.name] = decode(m[i], key);
          }
        };
        for (var i = 1; i < m.length; i++) {
          _loop_1(i);
        }
        return {
          path: path,
          index: index,
          params: params
        };
      };
    }
    /**
     * Escape a regular expression string.
     */
    function escapeString(str) {
      return str.replace(/([.+*?=^!:${}()[\]|/\\])/g, "\\$1");
    }
    /**
     * Get the flags for a regexp from the options.
     */
    function flags(options) {
      return options && options.sensitive ? "" : "i";
    }
    /**
     * Pull out keys from a regexp.
     */
    function regexpToRegexp(path, keys) {
      if (!keys) return path;
      var groupsRegex = /\((?:\?<(.*?)>)?(?!\?)/g;
      var index = 0;
      var execResult = groupsRegex.exec(path.source);
      while (execResult) {
        keys.push({
          // Use parenthesized substring match if available, index otherwise
          name: execResult[1] || index++,
          prefix: "",
          suffix: "",
          modifier: "",
          pattern: ""
        });
        execResult = groupsRegex.exec(path.source);
      }
      return path;
    }
    /**
     * Transform an array into a regexp.
     */
    function arrayToRegexp(paths, keys, options) {
      var parts = paths.map(function (path) {
        return pathToRegexp(path, keys, options).source;
      });
      return new RegExp("(?:".concat(parts.join("|"), ")"), flags(options));
    }
    /**
     * Create a path regexp from string input.
     */
    function stringToRegexp(path, keys, options) {
      return tokensToRegexp(parse(path, options), keys, options);
    }
    /**
     * Expose a function for taking tokens and returning a RegExp.
     */
    function tokensToRegexp(tokens, keys, options) {
      if (options === void 0) {
        options = {};
      }
      var _a = options.strict,
        strict = _a === void 0 ? false : _a,
        _b = options.start,
        start = _b === void 0 ? true : _b,
        _c = options.end,
        end = _c === void 0 ? true : _c,
        _d = options.encode,
        encode = _d === void 0 ? function (x) {
          return x;
        } : _d,
        _e = options.delimiter,
        delimiter = _e === void 0 ? "/#?" : _e,
        _f = options.endsWith,
        endsWith = _f === void 0 ? "" : _f;
      var endsWithRe = "[".concat(escapeString(endsWith), "]|$");
      var delimiterRe = "[".concat(escapeString(delimiter), "]");
      var route = start ? "^" : "";
      // Iterate over the tokens and create our regexp string.
      for (var _i = 0, tokens_1 = tokens; _i < tokens_1.length; _i++) {
        var token = tokens_1[_i];
        if (typeof token === "string") {
          route += escapeString(encode(token));
        } else {
          var prefix = escapeString(encode(token.prefix));
          var suffix = escapeString(encode(token.suffix));
          if (token.pattern) {
            if (keys) keys.push(token);
            if (prefix || suffix) {
              if (token.modifier === "+" || token.modifier === "*") {
                var mod = token.modifier === "*" ? "?" : "";
                route += "(?:".concat(prefix, "((?:").concat(token.pattern, ")(?:").concat(suffix).concat(prefix, "(?:").concat(token.pattern, "))*)").concat(suffix, ")").concat(mod);
              } else {
                route += "(?:".concat(prefix, "(").concat(token.pattern, ")").concat(suffix, ")").concat(token.modifier);
              }
            } else {
              if (token.modifier === "+" || token.modifier === "*") {
                route += "((?:".concat(token.pattern, ")").concat(token.modifier, ")");
              } else {
                route += "(".concat(token.pattern, ")").concat(token.modifier);
              }
            }
          } else {
            route += "(?:".concat(prefix).concat(suffix, ")").concat(token.modifier);
          }
        }
      }
      if (end) {
        if (!strict) route += "".concat(delimiterRe, "?");
        route += !options.endsWith ? "$" : "(?=".concat(endsWithRe, ")");
      } else {
        var endToken = tokens[tokens.length - 1];
        var isEndDelimited = typeof endToken === "string" ? delimiterRe.indexOf(endToken[endToken.length - 1]) > -1 : endToken === undefined;
        if (!strict) {
          route += "(?:".concat(delimiterRe, "(?=").concat(endsWithRe, "))?");
        }
        if (!isEndDelimited) {
          route += "(?=".concat(delimiterRe, "|").concat(endsWithRe, ")");
        }
      }
      return new RegExp(route, flags(options));
    }
    /**
     * Normalize the given path string, returning a regular expression.
     *
     * An empty array can be passed in for the keys, which will hold the
     * placeholder key descriptions. For example, using `/user/:id`, `keys` will
     * contain `[{ name: 'id', delimiter: '/', optional: false, repeat: false }]`.
     */
    function pathToRegexp(path, keys, options) {
      if (path instanceof RegExp) return regexpToRegexp(path, keys);
      if (Array.isArray(path)) return arrayToRegexp(path, keys, options);
      return stringToRegexp(path, keys, options);
    }

    const History$1 = {
      queue: [],
      clearQueue() {
        if (History$1.queue.length === 0) return;
        const currentQueue = History$1.queue.shift();
        currentQueue();
      },
      routerQueue: [],
      clearRouterQueue() {
        if (History$1.routerQueue.length === 0) return;
        const currentQueue = History$1.routerQueue.pop();
        const {
          router,
          stateUrl,
          action
        } = currentQueue;
        let animate = router.params.animate;
        if (router.params.browserHistoryAnimate === false) animate = false;
        if (action === 'back') {
          router.back({
            animate,
            browserHistory: false
          });
        }
        if (action === 'load') {
          router.navigate(stateUrl, {
            animate,
            browserHistory: false
          });
        }
      },
      handle(e) {
        if (History$1.blockPopstate) return;
        const app = this;
        // const mainView = app.views.main;
        let state = e.state;
        History$1.previousState = History$1.state;
        History$1.state = state;
        History$1.allowChange = true;
        History$1.clearQueue();
        state = History$1.state;
        if (!state) state = {};
        app.views.forEach(view => {
          const router = view.router;
          let viewState = state[view.id];
          if (!viewState && view.params.browserHistory) {
            viewState = {
              url: view.router.history[0]
            };
          }
          if (!viewState) return;
          const stateUrl = viewState.url || undefined;
          let animate = router.params.animate;
          if (router.params.browserHistoryAnimate === false) animate = false;
          if (stateUrl !== router.url) {
            if (router.history.indexOf(stateUrl) >= 0) {
              // Go Back
              if (router.allowPageChange) {
                router.back({
                  animate,
                  browserHistory: false
                });
              } else {
                History$1.routerQueue.push({
                  action: 'back',
                  router
                });
              }
            } else if (router.allowPageChange) {
              // Load page
              router.navigate(stateUrl, {
                animate,
                browserHistory: false
              });
            } else {
              History$1.routerQueue.unshift({
                action: 'load',
                stateUrl,
                router
              });
            }
          }
        });
      },
      initViewState(viewId, viewState) {
        const window = getWindow$1();
        const newState = extend$3({}, History$1.state || {}, {
          [viewId]: viewState
        });
        History$1.state = newState;
        window.history.replaceState(newState, '');
      },
      push(viewId, viewState, url) {
        const window = getWindow$1();
        const document = getDocument$1();
        /* eslint-disable no-param-reassign */
        if (url.substr(-3) === '#!/') {
          url = url.replace('#!/', '');
          if (url === '') {
            url = document.location.href;
            if (url.includes('#!/')) {
              url = document.location.href.split('#!/')[0];
            }
          }
        }
        /* eslint-enable no-param-reassign */
        if (!History$1.allowChange) {
          History$1.queue.push(() => {
            History$1.push(viewId, viewState, url);
          });
          return;
        }
        History$1.previousState = History$1.state;
        const newState = extend$3({}, History$1.previousState || {}, {
          [viewId]: viewState
        });
        History$1.state = newState;
        window.history.pushState(newState, '', url);
      },
      replace(viewId, viewState, url) {
        const window = getWindow$1();
        if (url.substr(-3) === '#!/') {
          // eslint-disable-next-line
          url = url.replace('#!/', '');
        }
        if (!History$1.allowChange) {
          History$1.queue.push(() => {
            History$1.replace(viewId, viewState, url);
          });
          return;
        }
        History$1.previousState = History$1.state;
        const newState = extend$3({}, History$1.previousState || {}, {
          [viewId]: viewState
        });
        History$1.state = newState;
        window.history.replaceState(newState, '', url);
      },
      go(index) {
        const window = getWindow$1();
        History$1.allowChange = false;
        window.history.go(index);
      },
      back() {
        const window = getWindow$1();
        History$1.allowChange = false;
        window.history.back();
      },
      allowChange: true,
      previousState: {},
      state: {},
      blockPopstate: true,
      init(app) {
        const window = getWindow$1();
        const document = getDocument$1();
        History$1.state = window.history.state;
        $(window).on('load', () => {
          setTimeout(() => {
            History$1.blockPopstate = false;
          }, 0);
        });
        if (document.readyState && document.readyState === 'complete') {
          History$1.blockPopstate = false;
        }
        $(window).on('popstate', History$1.handle.bind(app));
      }
    };

    function SwipeBack(r) {
      const router = r;
      const {
        $el,
        $navbarsEl,
        app,
        params
      } = router;
      const support = getSupport$1();
      const device = getDevice$1();
      let isTouched = false;
      let isMoved = false;
      const touchesStart = {};
      let isScrolling;
      let $currentPageEl = [];
      let $previousPageEl = [];
      let viewContainerWidth;
      let touchesDiff;
      let allowViewTouchMove = true;
      let touchStartTime;
      let $currentNavbarEl = [];
      let $previousNavbarEl = [];
      let dynamicNavbar;
      let $pageShadowEl;
      let $pageOpacityEl;
      let animatableNavEls;
      const paramsSwipeBackAnimateShadow = params[`${app.theme}SwipeBackAnimateShadow`];
      const paramsSwipeBackAnimateOpacity = params[`${app.theme}SwipeBackAnimateOpacity`];
      const paramsSwipeBackActiveArea = params[`${app.theme}SwipeBackActiveArea`];
      const paramsSwipeBackThreshold = params[`${app.theme}SwipeBackThreshold`];
      const transformOrigin = app.rtl ? 'right center' : 'left center';
      const transformOriginTitleLarge = app.rtl ? 'calc(100% - var(--f7-navbar-large-title-padding-left) - var(--f7-safe-area-left)) center' : 'calc(var(--f7-navbar-large-title-padding-left) + var(--f7-safe-area-left)) center';
      function animatableNavElements() {
        const els = [];
        const inverter = app.rtl ? -1 : 1;
        const currentNavIsTransparent = $currentNavbarEl.hasClass('navbar-transparent') && !$currentNavbarEl.hasClass('navbar-large') && !$currentNavbarEl.hasClass('navbar-transparent-visible');
        const currentNavIsLarge = $currentNavbarEl.hasClass('navbar-large');
        const currentNavIsCollapsed = $currentNavbarEl.hasClass('navbar-large-collapsed');
        const currentNavIsLargeTransparent = $currentNavbarEl.hasClass('navbar-large-transparent') || $currentNavbarEl.hasClass('navbar-large') && $currentNavbarEl.hasClass('navbar-transparent');
        const previousNavIsTransparent = $previousNavbarEl.hasClass('navbar-transparent') && !$previousNavbarEl.hasClass('navbar-large') && !$previousNavbarEl.hasClass('navbar-transparent-visible');
        const previousNavIsLarge = $previousNavbarEl.hasClass('navbar-large');
        const previousNavIsCollapsed = $previousNavbarEl.hasClass('navbar-large-collapsed');
        const previousNavIsLargeTransparent = $previousNavbarEl.hasClass('navbar-large-transparent') || $previousNavbarEl.hasClass('navbar-large') && $previousNavbarEl.hasClass('navbar-transparent');
        const fromLarge = currentNavIsLarge && !currentNavIsCollapsed;
        const toLarge = previousNavIsLarge && !previousNavIsCollapsed;
        const $currentNavElements = $currentNavbarEl.find('.left, .title, .right, .subnavbar, .fading, .title-large, .navbar-bg');
        const $previousNavElements = $previousNavbarEl.find('.left, .title, .right, .subnavbar, .fading, .title-large, .navbar-bg');
        let activeNavBackIconText;
        let previousNavBackIconText;
        if (params.iosAnimateNavbarBackIcon) {
          if ($currentNavbarEl.hasClass('sliding') || $currentNavbarEl.find('.navbar-inner.sliding').length) {
            activeNavBackIconText = $currentNavbarEl.find('.left').find('.back .icon + span').eq(0);
          } else {
            activeNavBackIconText = $currentNavbarEl.find('.left.sliding').find('.back .icon + span').eq(0);
          }
          if ($previousNavbarEl.hasClass('sliding') || $previousNavbarEl.find('.navbar-inner.sliding').length) {
            previousNavBackIconText = $previousNavbarEl.find('.left').find('.back .icon + span').eq(0);
          } else {
            previousNavBackIconText = $previousNavbarEl.find('.left.sliding').find('.back .icon + span').eq(0);
          }
          if (activeNavBackIconText.length) {
            $previousNavElements.each(el => {
              if (!$(el).hasClass('title')) return;
              el.f7NavbarLeftOffset += activeNavBackIconText.prev('.icon')[0].offsetWidth;
            });
          }
        }
        $currentNavElements.each(navEl => {
          const $navEl = $(navEl);
          const isSubnavbar = $navEl.hasClass('subnavbar');
          const isLeft = $navEl.hasClass('left');
          const isTitle = $navEl.hasClass('title');
          const isBg = $navEl.hasClass('navbar-bg');
          if ((isTitle || isBg) && currentNavIsTransparent) return;
          if (!fromLarge && $navEl.hasClass('.title-large')) return;
          const el = {
            el: navEl
          };
          if (fromLarge) {
            if (isTitle) return;
            if ($navEl.hasClass('title-large')) {
              if (els.indexOf(el) < 0) els.push(el);
              el.overflow = 'visible';
              $navEl.find('.title-large-text').each(subNavEl => {
                els.push({
                  el: subNavEl,
                  transform: progress => `translateX(${progress * 100 * inverter}%)`
                });
              });
              return;
            }
          }
          if (toLarge) {
            if (!fromLarge) {
              if ($navEl.hasClass('title-large')) {
                if (els.indexOf(el) < 0) els.push(el);
                el.opacity = 0;
              }
            }
            if (isLeft) {
              if (els.indexOf(el) < 0) els.push(el);
              el.opacity = progress => 1 - progress ** 0.33;
              $navEl.find('.back span').each(subNavEl => {
                els.push({
                  el: subNavEl,
                  'transform-origin': transformOrigin,
                  transform: progress => `translateX(calc(${progress} * (var(--f7-navbarTitleLargeOffset) - var(--f7-navbarLeftTextOffset)))) translateY(calc(${progress} * (var(--f7-navbar-large-title-height) - var(--f7-navbar-large-title-padding-vertical) / 2))) scale(${1 + 1 * progress})`
                });
              });
              return;
            }
          }
          if (isBg) {
            if (els.indexOf(el) < 0) els.push(el);
            if (!fromLarge && !toLarge) {
              if (currentNavIsCollapsed) {
                if (currentNavIsLargeTransparent) {
                  el.className = 'ios-swipeback-navbar-bg-large';
                }
                el.transform = progress => `translateX(${100 * progress * inverter}%) translateY(calc(-1 * var(--f7-navbar-large-title-height)))`;
              } else {
                el.transform = progress => `translateX(${100 * progress * inverter}%)`;
              }
            }
            if (!fromLarge && toLarge) {
              el.className = 'ios-swipeback-navbar-bg-large';
              el.transform = progress => `translateX(${100 * progress * inverter}%) translateY(calc(-1 * ${1 - progress} * var(--f7-navbar-large-title-height)))`;
            }
            if (fromLarge && toLarge) {
              el.transform = progress => `translateX(${100 * progress * inverter}%)`;
            }
            if (fromLarge && !toLarge) {
              el.transform = progress => `translateX(${100 * progress * inverter}%) translateY(calc(-${progress} * var(--f7-navbar-large-title-height)))`;
            }
            return;
          }
          if ($navEl.hasClass('title-large')) return;
          const isSliding = $navEl.hasClass('sliding') || $navEl.parents('.navbar-inner.sliding').length;
          if (els.indexOf(el) < 0) els.push(el);
          if (!isSubnavbar || isSubnavbar && !isSliding) {
            el.opacity = progress => 1 - progress ** 0.33;
          }
          if (isSliding) {
            let transformTarget = el;
            if (isLeft && activeNavBackIconText.length && params.iosAnimateNavbarBackIcon) {
              const textEl = {
                el: activeNavBackIconText[0]
              };
              transformTarget = textEl;
              els.push(textEl);
            }
            transformTarget.transform = progress => {
              let activeNavTranslate = progress * transformTarget.el.f7NavbarRightOffset;
              if (device.pixelRatio === 1) activeNavTranslate = Math.round(activeNavTranslate);
              if (isSubnavbar && currentNavIsLarge) {
                return `translate3d(${activeNavTranslate}px, calc(-1 * var(--f7-navbar-large-collapse-progress) * var(--f7-navbar-large-title-height)), 0)`;
              }
              return `translate3d(${activeNavTranslate}px,0,0)`;
            };
          }
        });
        $previousNavElements.each(navEl => {
          const $navEl = $(navEl);
          const isSubnavbar = $navEl.hasClass('subnavbar');
          const isLeft = $navEl.hasClass('left');
          const isTitle = $navEl.hasClass('title');
          const isBg = $navEl.hasClass('navbar-bg');
          if ((isTitle || isBg) && previousNavIsTransparent) return;
          const el = {
            el: navEl
          };
          if (toLarge) {
            if (isTitle) return;
            if (els.indexOf(el) < 0) els.push(el);
            if ($navEl.hasClass('title-large')) {
              el.opacity = 1;
              el.overflow = 'visible';
              $navEl.find('.title-large-text').each(subNavEl => {
                els.push({
                  el: subNavEl,
                  'transform-origin': transformOriginTitleLarge,
                  opacity: progress => progress ** 3,
                  transform: progress => `translateX(calc(${1 - progress} * (var(--f7-navbarLeftTextOffset) - var(--f7-navbarTitleLargeOffset)))) translateY(calc(${progress - 1} * var(--f7-navbar-large-title-height) + ${1 - progress} * var(--f7-navbar-large-title-padding-vertical))) scale(${0.5 + progress * 0.5})`
                });
              });
              return;
            }
          }
          if (isBg) {
            if (els.indexOf(el) < 0) els.push(el);
            if (!fromLarge && !toLarge) {
              if (previousNavIsCollapsed) {
                if (previousNavIsLargeTransparent) {
                  el.className = 'ios-swipeback-navbar-bg-large';
                }
                el.transform = progress => `translateX(${(-100 + 100 * progress) * inverter}%) translateY(calc(-1 * var(--f7-navbar-large-title-height)))`;
              } else {
                el.transform = progress => `translateX(${(-100 + 100 * progress) * inverter}%)`;
              }
            }
            if (!fromLarge && toLarge) {
              el.transform = progress => `translateX(${(-100 + 100 * progress) * inverter}%) translateY(calc(-1 * ${1 - progress} * var(--f7-navbar-large-title-height)))`;
            }
            if (fromLarge && !toLarge) {
              el.className = 'ios-swipeback-navbar-bg-large';
              el.transform = progress => `translateX(${(-100 + 100 * progress) * inverter}%) translateY(calc(-${progress} * var(--f7-navbar-large-title-height)))`;
            }
            if (fromLarge && toLarge) {
              el.transform = progress => `translateX(${(-100 + 100 * progress) * inverter}%)`;
            }
            return;
          }
          if ($navEl.hasClass('title-large')) return;
          const isSliding = $navEl.hasClass('sliding') || $previousNavbarEl.children('.navbar-inner.sliding').length;
          if (els.indexOf(el) < 0) els.push(el);
          if (!isSubnavbar || isSubnavbar && !isSliding) {
            el.opacity = progress => progress ** 3;
          }
          if (isSliding) {
            let transformTarget = el;
            if (isLeft && previousNavBackIconText.length && params.iosAnimateNavbarBackIcon) {
              const textEl = {
                el: previousNavBackIconText[0]
              };
              transformTarget = textEl;
              els.push(textEl);
            }
            transformTarget.transform = progress => {
              let previousNavTranslate = transformTarget.el.f7NavbarLeftOffset * (1 - progress);
              if (device.pixelRatio === 1) previousNavTranslate = Math.round(previousNavTranslate);
              if (isSubnavbar && previousNavIsLarge) {
                return `translate3d(${previousNavTranslate}px, calc(-1 * var(--f7-navbar-large-collapse-progress) * var(--f7-navbar-large-title-height)), 0)`;
              }
              return `translate3d(${previousNavTranslate}px,0,0)`;
            };
          }
        });
        return els;
      }
      function setAnimatableNavElements(_temp) {
        let {
          progress,
          reset,
          transition,
          reflow
        } = _temp === void 0 ? {} : _temp;
        const styles = ['overflow', 'transform', 'transform-origin', 'opacity'];
        if (transition === true || transition === false) {
          for (let i = 0; i < animatableNavEls.length; i += 1) {
            const el = animatableNavEls[i];
            if (el && el.el) {
              if (transition === true) el.el.classList.add('navbar-page-transitioning');
              if (transition === false) el.el.classList.remove('navbar-page-transitioning');
            }
          }
        }
        if (reflow && animatableNavEls.length && animatableNavEls[0] && animatableNavEls[0].el) {
          // eslint-disable-next-line
          animatableNavEls[0].el._clientLeft = animatableNavEls[0].el.clientLeft;
        }
        for (let i = 0; i < animatableNavEls.length; i += 1) {
          const el = animatableNavEls[i];
          if (el && el.el) {
            if (el.className && !el.classNameSet && !reset) {
              el.el.classList.add(el.className);
              el.classNameSet = true;
            }
            if (el.className && reset) {
              el.el.classList.remove(el.className);
            }
            for (let j = 0; j < styles.length; j += 1) {
              const styleProp = styles[j];
              if (el[styleProp]) {
                if (reset) {
                  el.el.style[styleProp] = '';
                } else if (typeof el[styleProp] === 'function') {
                  el.el.style[styleProp] = el[styleProp](progress);
                } else {
                  el.el.style[styleProp] = el[styleProp];
                }
              }
            }
          }
        }
      }
      function handleTouchStart(e) {
        if (!e.isTrusted) return;
        const swipeBackEnabled = params[`${app.theme}SwipeBack`];
        if (!allowViewTouchMove || !swipeBackEnabled || isTouched || app.swipeout && app.swipeout.el || !router.allowPageChange) return;
        if ($(e.target).closest('.range-slider, .calendar-months').length > 0) return;
        if ($(e.target).closest('.page-master, .page-master-detail').length > 0 && params.masterDetailBreakpoint > 0 && app.width >= params.masterDetailBreakpoint) return;
        isMoved = false;
        isTouched = true;
        isScrolling = undefined;
        touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
        touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
        touchStartTime = now$1();
        dynamicNavbar = router.dynamicNavbar;
      }
      function handleTouchMove(e) {
        if (!e.isTrusted) return;
        if (!isTouched) return;
        const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
        const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
        if (typeof isScrolling === 'undefined') {
          isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x)) || pageX < touchesStart.x && !app.rtl || pageX > touchesStart.x && app.rtl;
        }
        if (isScrolling || e.f7PreventSwipeBack || app.preventSwipeBack) {
          isTouched = false;
          return;
        }
        if (!isMoved) {
          // Calc values during first move fired
          let cancel = false;
          const target = $(e.target);
          const swipeout = target.closest('.swipeout');
          if (swipeout.length > 0) {
            if (!app.rtl && swipeout.find('.swipeout-actions-left').length > 0) cancel = true;
            if (app.rtl && swipeout.find('.swipeout-actions-right').length > 0) cancel = true;
          }
          $currentPageEl = target.closest('.page');
          if ($currentPageEl.hasClass('no-swipeback') || target.closest('.no-swipeback, .card-opened').length > 0) cancel = true;
          $previousPageEl = $el.find('.page-previous');
          if ($previousPageEl.length > 1) {
            $previousPageEl = $previousPageEl.eq($previousPageEl.length - 1);
          }
          let notFromBorder = touchesStart.x - $el.offset().left > paramsSwipeBackActiveArea;
          viewContainerWidth = $el.width();
          if (app.rtl) {
            notFromBorder = touchesStart.x < $el.offset().left - $el[0].scrollLeft + (viewContainerWidth - paramsSwipeBackActiveArea);
          } else {
            notFromBorder = touchesStart.x - $el.offset().left > paramsSwipeBackActiveArea;
          }
          if (notFromBorder) cancel = true;
          if ($previousPageEl.length === 0 || $currentPageEl.length === 0) cancel = true;
          if (cancel) {
            isTouched = false;
            return;
          }
          if (paramsSwipeBackAnimateShadow) {
            $pageShadowEl = $currentPageEl.find('.page-shadow-effect');
            if ($pageShadowEl.length === 0) {
              $pageShadowEl = $('<div class="page-shadow-effect"></div>');
              $currentPageEl.append($pageShadowEl);
            }
          }
          if (paramsSwipeBackAnimateOpacity) {
            $pageOpacityEl = $previousPageEl.find('.page-opacity-effect');
            if ($pageOpacityEl.length === 0) {
              $pageOpacityEl = $('<div class="page-opacity-effect"></div>');
              $previousPageEl.append($pageOpacityEl);
            }
          }
          if (dynamicNavbar) {
            $currentNavbarEl = $navbarsEl.find('.navbar-current');
            $previousNavbarEl = $navbarsEl.find('.navbar-previous');
            if ($previousNavbarEl.length > 1) {
              $previousNavbarEl = $previousNavbarEl.eq($previousNavbarEl.length - 1);
            }
            animatableNavEls = animatableNavElements($previousNavbarEl, $currentNavbarEl);
          }

          // Close/Hide Any Picker
          if ($('.sheet.modal-in').length > 0 && app.sheet) {
            app.sheet.close($('.sheet.modal-in'));
          }
        }
        e.f7PreventSwipePanel = true;
        isMoved = true;
        app.preventSwipePanelBySwipeBack = true;
        e.preventDefault();

        // RTL inverter
        const inverter = app.rtl ? -1 : 1;

        // Touches diff
        touchesDiff = (pageX - touchesStart.x - paramsSwipeBackThreshold) * inverter;
        if (touchesDiff < 0) touchesDiff = 0;
        const percentage = Math.min(Math.max(touchesDiff / viewContainerWidth, 0), 1);

        // Swipe Back Callback
        const callbackData = {
          percentage,
          progress: percentage,
          currentPageEl: $currentPageEl[0],
          previousPageEl: $previousPageEl[0],
          currentNavbarEl: $currentNavbarEl[0],
          previousNavbarEl: $previousNavbarEl[0]
        };
        $el.trigger('swipeback:move', callbackData);
        router.emit('swipebackMove', callbackData);

        // Transform pages
        let currentPageTranslate = touchesDiff * inverter;
        let previousPageTranslate = (touchesDiff / 5 - viewContainerWidth / 5) * inverter;
        if (!app.rtl) {
          currentPageTranslate = Math.min(currentPageTranslate, viewContainerWidth);
          previousPageTranslate = Math.min(previousPageTranslate, 0);
        } else {
          currentPageTranslate = Math.max(currentPageTranslate, -viewContainerWidth);
          previousPageTranslate = Math.max(previousPageTranslate, 0);
        }
        if (device.pixelRatio === 1) {
          currentPageTranslate = Math.round(currentPageTranslate);
          previousPageTranslate = Math.round(previousPageTranslate);
        }
        router.swipeBackActive = true;
        $([$currentPageEl[0], $previousPageEl[0]]).addClass('page-swipeback-active');
        $currentPageEl.transform(`translate3d(${currentPageTranslate}px,0,0)`);
        if (paramsSwipeBackAnimateShadow) $pageShadowEl[0].style.opacity = 1 - 1 * percentage;
        if (app.theme === 'ios') {
          $previousPageEl.transform(`translate3d(${previousPageTranslate}px,0,0)`);
        }
        if (paramsSwipeBackAnimateOpacity) $pageOpacityEl[0].style.opacity = 1 - 1 * percentage;

        // Dynamic Navbars Animation
        if (!dynamicNavbar) return;
        setAnimatableNavElements({
          progress: percentage
        });
      }
      function handleTouchEnd(e) {
        if (!e.isTrusted) return;
        app.preventSwipePanelBySwipeBack = false;
        if (!isTouched || !isMoved) {
          isTouched = false;
          isMoved = false;
          return;
        }
        isTouched = false;
        isMoved = false;
        router.swipeBackActive = false;
        const $pages = $([$currentPageEl[0], $previousPageEl[0]]);
        $pages.removeClass('page-swipeback-active');
        if (touchesDiff === 0) {
          $pages.transform('');
          if ($pageShadowEl && $pageShadowEl.length > 0) $pageShadowEl.remove();
          if ($pageOpacityEl && $pageOpacityEl.length > 0) $pageOpacityEl.remove();
          if (dynamicNavbar) {
            setAnimatableNavElements({
              reset: true
            });
          }
          return;
        }
        const timeDiff = now$1() - touchStartTime;
        let pageChanged = false;
        // Swipe back to previous page
        if (timeDiff < 300 && touchesDiff > 10 || timeDiff >= 300 && touchesDiff > viewContainerWidth / 2) {
          $currentPageEl.removeClass('page-current').addClass(`page-next${app.theme !== 'ios' ? ' page-next-on-right' : ''}`);
          $previousPageEl.removeClass('page-previous').addClass('page-current').removeAttr('aria-hidden');
          if ($pageShadowEl) $pageShadowEl[0].style.opacity = '';
          if ($pageOpacityEl) $pageOpacityEl[0].style.opacity = '';
          if (dynamicNavbar) {
            router.setNavbarPosition($currentNavbarEl, 'next');
            router.setNavbarPosition($previousNavbarEl, 'current', false);
          }
          pageChanged = true;
        }
        // Reset custom styles
        // Add transitioning class for transition-duration
        $pages.addClass('page-transitioning page-transitioning-swipeback');
        if (device.ios) {
          // eslint-disable-next-line
          $currentPageEl[0]._clientLeft = $currentPageEl[0].clientLeft;
        }
        $pages.transform('');
        if (dynamicNavbar) {
          setAnimatableNavElements({
            progress: pageChanged ? 1 : 0,
            transition: true,
            reflow: !!device.ios
          });
        }
        allowViewTouchMove = false;
        router.allowPageChange = false;

        // Swipe Back Callback
        const callbackData = {
          currentPageEl: $currentPageEl[0],
          previousPageEl: $previousPageEl[0],
          currentNavbarEl: $currentNavbarEl[0],
          previousNavbarEl: $previousNavbarEl[0]
        };
        if (pageChanged) {
          // Update Route
          router.currentRoute = $previousPageEl[0].f7Page.route;
          router.currentPage = $previousPageEl[0];

          // Page before animation callback
          router.pageCallback('beforeOut', $currentPageEl, $currentNavbarEl, 'current', 'next', {
            route: $currentPageEl[0].f7Page.route,
            swipeBack: true
          });
          router.pageCallback('beforeIn', $previousPageEl, $previousNavbarEl, 'previous', 'current', {
            route: $previousPageEl[0].f7Page.route,
            swipeBack: true
          }, $currentPageEl[0]);
          $el.trigger('swipeback:beforechange', callbackData);
          router.emit('swipebackBeforeChange', callbackData);
        } else {
          $el.trigger('swipeback:beforereset', callbackData);
          router.emit('swipebackBeforeReset', callbackData);
        }
        $currentPageEl.transitionEnd(() => {
          $pages.removeClass('page-transitioning page-transitioning-swipeback');
          if (dynamicNavbar) {
            setAnimatableNavElements({
              reset: true,
              transition: false
            });
          }
          allowViewTouchMove = true;
          router.allowPageChange = true;
          if (pageChanged) {
            // Update History
            if (router.history.length === 1) {
              router.history.unshift(router.url);
            }
            router.history.pop();
            router.saveHistory();

            // Update push state
            if (params.browserHistory) {
              History$1.back();
            }

            // Page after animation callback
            router.pageCallback('afterOut', $currentPageEl, $currentNavbarEl, 'current', 'next', {
              route: $currentPageEl[0].f7Page.route,
              swipeBack: true
            });
            router.pageCallback('afterIn', $previousPageEl, $previousNavbarEl, 'previous', 'current', {
              route: $previousPageEl[0].f7Page.route,
              swipeBack: true
            });

            // Remove Old Page

            router.pageCallback('beforeRemove', $currentPageEl, $currentNavbarEl, 'next', {
              swipeBack: true
            });
            router.removePage($currentPageEl);
            if (dynamicNavbar) {
              router.removeNavbar($currentNavbarEl);
            }
            $el.trigger('swipeback:afterchange', callbackData);
            router.emit('swipebackAfterChange', callbackData);
            router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
            if (params.preloadPreviousPage) {
              router.back(router.history[router.history.length - 2], {
                preload: true
              });
            }
          } else {
            $el.trigger('swipeback:afterreset', callbackData);
            router.emit('swipebackAfterReset', callbackData);
          }
          if ($pageShadowEl && $pageShadowEl.length > 0) $pageShadowEl.remove();
          if ($pageOpacityEl && $pageOpacityEl.length > 0) $pageOpacityEl.remove();
        });
      }
      function attachEvents() {
        const passiveListener = app.touchEvents.start === 'touchstart' && support.passiveListener ? {
          passive: true,
          capture: false
        } : false;
        $el.on(app.touchEvents.start, handleTouchStart, passiveListener);
        app.on('touchmove:active', handleTouchMove);
        app.on('touchend:passive', handleTouchEnd);
      }
      function detachEvents() {
        const passiveListener = app.touchEvents.start === 'touchstart' && support.passiveListener ? {
          passive: true,
          capture: false
        } : false;
        $el.off(app.touchEvents.start, handleTouchStart, passiveListener);
        app.off('touchmove:active', handleTouchMove);
        app.off('touchend:passive', handleTouchEnd);
      }
      attachEvents();
      router.on('routerDestroy', detachEvents);
    }

    function redirect(direction, route, options) {
      const router = this;
      const r = route.route.redirect;
      const method = direction === 'forward' ? 'navigate' : 'back';
      if (options.initial && router.params.browserHistory) {
        options.replaceState = true; // eslint-disable-line
        options.history = true; // eslint-disable-line
      }

      function redirectResolve(redirectUrl, redirectOptions) {
        if (redirectOptions === void 0) {
          redirectOptions = {};
        }
        router.allowPageChange = true;
        router[method](redirectUrl, extend$3({}, options, redirectOptions));
      }
      function redirectReject() {
        router.allowPageChange = true;
      }
      if (typeof r === 'function') {
        router.allowPageChange = false;
        const redirectUrl = r.call(router, {
          router,
          to: route,
          resolve: redirectResolve,
          reject: redirectReject,
          direction,
          app: router.app
        });
        if (redirectUrl && typeof redirectUrl === 'string') {
          router.allowPageChange = true;
          return router[method](redirectUrl, options);
        }
        return router;
      }
      return router[method](r, options);
    }

    function processQueue(router, routerQueue, routeQueue, to, from, resolve, reject, direction) {
      const queue = [];
      if (Array.isArray(routeQueue)) {
        queue.push(...routeQueue);
      } else if (routeQueue && typeof routeQueue === 'function') {
        queue.push(routeQueue);
      }
      if (routerQueue) {
        if (Array.isArray(routerQueue)) {
          queue.push(...routerQueue);
        } else {
          queue.push(routerQueue);
        }
      }
      function next() {
        if (queue.length === 0) {
          resolve();
          return;
        }
        const queueItem = queue.shift();
        queueItem.call(router, {
          router,
          to,
          from,
          resolve() {
            next();
          },
          reject() {
            reject();
          },
          direction,
          app: router.app
        });
      }
      next();
    }
    function processRouteQueue(to, from, resolve, reject, direction) {
      const router = this;
      function enterNextRoute() {
        if (to && to.route && (router.params.routesBeforeEnter || to.route.beforeEnter)) {
          router.allowPageChange = false;
          processQueue(router, router.params.routesBeforeEnter, to.route.beforeEnter, to, from, () => {
            router.allowPageChange = true;
            resolve();
          }, () => {
            reject();
          }, direction);
        } else {
          resolve();
        }
      }
      function leaveCurrentRoute() {
        if (from && from.route && (router.params.routesBeforeLeave || from.route.beforeLeave)) {
          router.allowPageChange = false;
          processQueue(router, router.params.routesBeforeLeave, from.route.beforeLeave, to, from, () => {
            router.allowPageChange = true;
            enterNextRoute();
          }, () => {
            reject();
          }, direction);
        } else {
          enterNextRoute();
        }
      }
      leaveCurrentRoute();
    }

    function appRouterCheck(router, method) {
      if (!router.view) {
        throw new Error(`Framework7: it is not allowed to use router methods on global app router. Use router methods only on related View, e.g. app.views.main.router.${method}(...)`);
      }
    }

    function asyncComponent(router, component, resolve, reject) {
      function resolvePromise(componentPromise) {
        componentPromise.then(c => {
          // eslint-disable-next-line
          resolve({
            component: c.default || c._default || c
          });
        }).catch(err => {
          reject();
          throw new Error(err, {
            cause: err
          });
        });
      }
      if (component instanceof Promise) {
        resolvePromise(component);
        return;
      }
      const asyncComponentResult = component.call(router);
      if (asyncComponentResult instanceof Promise) {
        resolvePromise(asyncComponentResult);
      } else {
        resolve({
          component: asyncComponentResult
        });
      }
    }

    function refreshPage(props) {
      if (props === void 0) {
        props = {};
      }
      const router = this;
      appRouterCheck(router, 'refreshPage');
      return router.navigate(router.currentRoute.url, {
        ignoreCache: true,
        reloadCurrent: true,
        props
      });
    }
    function forward(router, el, forwardOptions) {
      if (forwardOptions === void 0) {
        forwardOptions = {};
      }
      const document = getDocument$1();
      const $el = $(el);
      const app = router.app;
      const view = router.view;
      const options = extend$3(false, {
        animate: router.params.animate,
        browserHistory: true,
        replaceState: false,
        history: true,
        reloadCurrent: router.params.reloadPages,
        reloadPrevious: false,
        reloadAll: false,
        clearPreviousHistory: false,
        reloadDetail: router.params.reloadDetail,
        on: {}
      }, forwardOptions);
      const masterDetailEnabled = router.params.masterDetailBreakpoint > 0;
      const isMaster = masterDetailEnabled && options.route && options.route.route && (options.route.route.master === true || typeof options.route.route.master === 'function' && options.route.route.master(app, router));
      let masterPageEl;
      let otherDetailPageEl;
      let detailsInBetweenRemoved = 0;
      let currentRouteIsModal = router.currentRoute.modal;
      let modalType;
      if (!currentRouteIsModal) {
        'popup popover sheet loginScreen actions customModal panel'.split(' ').forEach(modalLoadProp => {
          if (router.currentRoute && router.currentRoute.route && router.currentRoute.route[modalLoadProp]) {
            currentRouteIsModal = true;
            modalType = modalLoadProp;
          }
        });
      }
      if (currentRouteIsModal) {
        const modalToClose = router.currentRoute.modal || router.currentRoute.route.modalInstance || app[modalType].get();
        const previousUrl = router.history[router.history.length - 2];
        let previousRoute = router.findMatchingRoute(previousUrl);
        if (!previousRoute && previousUrl) {
          previousRoute = {
            url: previousUrl,
            path: previousUrl.split('?')[0],
            query: parseUrlQuery(previousUrl),
            route: {
              path: previousUrl.split('?')[0],
              url: previousUrl
            }
          };
        }
        router.modalRemove(modalToClose);
      }
      const dynamicNavbar = router.dynamicNavbar;
      const $viewEl = router.$el;
      const $newPage = $el;
      const reload = options.reloadPrevious || options.reloadCurrent || options.reloadAll;
      let $oldPage;
      let $navbarsEl;
      let $newNavbarEl;
      let $oldNavbarEl;
      router.allowPageChange = false;
      if ($newPage.length === 0) {
        router.allowPageChange = true;
        return router;
      }
      if ($newPage.length) {
        // Remove theme elements
        router.removeThemeElements($newPage);
      }
      if (dynamicNavbar) {
        $newNavbarEl = $newPage.children('.navbar');
        $navbarsEl = router.$navbarsEl;
        if ($newNavbarEl.length === 0 && $newPage[0] && $newPage[0].f7Page) {
          // Try from pageData
          $newNavbarEl = $newPage[0].f7Page.$navbarEl;
        }
      }

      // Save Keep Alive Cache
      if (options.route && options.route.route && options.route.route.keepAlive && !options.route.route.keepAliveData) {
        options.route.route.keepAliveData = {
          pageEl: $el[0]
        };
      }

      // Pages In View
      const $pagesInView = $viewEl.children('.page').filter(pageInView => pageInView !== $newPage[0]);

      // Navbars In View
      let $navbarsInView;
      if (dynamicNavbar) {
        $navbarsInView = $navbarsEl.children('.navbar').filter(navbarInView => navbarInView !== $newNavbarEl[0]);
      }

      // Exit when reload previous and only 1 page in view so nothing ro reload
      if (options.reloadPrevious && $pagesInView.length < 2) {
        router.allowPageChange = true;
        return router;
      }

      // Find Detail' master page
      let isDetail;
      let reloadDetail;
      let isDetailRoot;
      if (masterDetailEnabled && !options.reloadAll) {
        for (let i = 0; i < $pagesInView.length; i += 1) {
          if (!masterPageEl && $pagesInView[i].classList.contains('page-master')) {
            masterPageEl = $pagesInView[i];
            continue; // eslint-disable-line
          }
        }

        isDetail = !isMaster && masterPageEl;
        if (isDetail) {
          // Find Other Detail
          if (masterPageEl) {
            for (let i = 0; i < $pagesInView.length; i += 1) {
              if ($pagesInView[i].classList.contains('page-master-detail')) {
                otherDetailPageEl = $pagesInView[i];
                continue; // eslint-disable-line
              }
            }
          }
        }

        reloadDetail = isDetail && options.reloadDetail && app.width >= router.params.masterDetailBreakpoint && masterPageEl;
      }
      if (isDetail) {
        isDetailRoot = !otherDetailPageEl || reloadDetail || options.reloadAll || options.reloadCurrent;
      }

      // New Page
      let newPagePosition = 'next';
      if (options.reloadCurrent || options.reloadAll || reloadDetail) {
        newPagePosition = 'current';
      } else if (options.reloadPrevious) {
        newPagePosition = 'previous';
      }
      $newPage.removeClass('page-previous page-current page-next').addClass(`page-${newPagePosition}${isMaster ? ' page-master' : ''}${isDetail ? ' page-master-detail' : ''}${isDetailRoot ? ' page-master-detail-root' : ''}`).trigger('page:unstack').trigger('page:position', {
        position: newPagePosition
      });
      router.emit('pageUnstack', $newPage[0]);
      router.emit('pagePosition', $newPage[0], newPagePosition);
      if (isMaster || isDetail) {
        $newPage.trigger('page:role', {
          role: isMaster ? 'master' : 'detail',
          root: !!isDetailRoot
        });
        router.emit('pageRole', $newPage[0], {
          role: isMaster ? 'master' : 'detail',
          detailRoot: !!isDetailRoot
        });
      }
      if (dynamicNavbar && $newNavbarEl.length) {
        $newNavbarEl.removeClass('navbar-previous navbar-current navbar-next').addClass(`navbar-${newPagePosition}${isMaster ? ' navbar-master' : ''}${isDetail ? ' navbar-master-detail' : ''}${isDetailRoot ? ' navbar-master-detail-root' : ''}`);
        $newNavbarEl.trigger('navbar:position', {
          position: newPagePosition
        });
        router.emit('navbarPosition', $newNavbarEl[0], newPagePosition);
        if (isMaster || isDetail) {
          router.emit('navbarRole', $newNavbarEl[0], {
            role: isMaster ? 'master' : 'detail',
            detailRoot: !!isDetailRoot
          });
        }
      }

      // Find Old Page
      if (options.reloadCurrent || reloadDetail) {
        if (reloadDetail) {
          $oldPage = $pagesInView.filter(pageEl => !pageEl.classList.contains('page-master'));
          if (dynamicNavbar) {
            $oldNavbarEl = $($oldPage.map(pageEl => app.navbar.getElByPage(pageEl)));
          }
          if ($oldPage.length > 1 && masterPageEl) {
            detailsInBetweenRemoved = $oldPage.length - 1;
            $(masterPageEl).removeClass('page-master-stacked').trigger('page:masterunstack');
            router.emit('pageMasterUnstack', masterPageEl);
            if (dynamicNavbar) {
              $(app.navbar.getElByPage(masterPageEl)).removeClass('navbar-master-stacked');
              router.emit('navbarMasterUnstack', app.navbar.getElByPage(masterPageEl));
            }
          }
        } else {
          $oldPage = $pagesInView.eq($pagesInView.length - 1);
          if (dynamicNavbar) {
            $oldNavbarEl = $(app.navbar.getElByPage($oldPage));
          }
        }
      } else if (options.reloadPrevious) {
        $oldPage = $pagesInView.eq($pagesInView.length - 2);
        if (dynamicNavbar) {
          // $oldNavbarEl = $navbarsInView.eq($pagesInView.length - 2);
          $oldNavbarEl = $(app.navbar.getElByPage($oldPage));
        }
      } else if (options.reloadAll) {
        $oldPage = $pagesInView.filter(pageEl => pageEl !== $newPage[0]);
        if (dynamicNavbar) {
          $oldNavbarEl = $navbarsInView.filter(navbarEl => navbarEl !== $newNavbarEl[0]);
        }
      } else {
        let removedPageEls = [];
        let removedNavbarEls = [];
        if ($pagesInView.length > 1) {
          let i = 0;
          for (i = 0; i < $pagesInView.length - 1; i += 1) {
            if (masterPageEl && $pagesInView[i] === masterPageEl) {
              $pagesInView.eq(i).addClass('page-master-stacked');
              $pagesInView.eq(i).trigger('page:masterstack');
              router.emit('pageMasterStack', $pagesInView[i]);
              if (dynamicNavbar) {
                $(app.navbar.getElByPage(masterPageEl)).addClass('navbar-master-stacked');
                router.emit('navbarMasterStack', app.navbar.getElByPage(masterPageEl));
              }
              continue; // eslint-disable-line
            }

            const oldNavbarEl = app.navbar.getElByPage($pagesInView.eq(i));

            // Page remove event
            removedPageEls.push($pagesInView[i]);
            router.pageCallback('beforeRemove', $pagesInView[i], $navbarsInView && $navbarsInView[i], 'previous', undefined, options);
            router.removePage($pagesInView[i]);
            if (dynamicNavbar && oldNavbarEl) {
              removedNavbarEls.push(oldNavbarEl);
              router.removeNavbar(oldNavbarEl);
            }
          }
        }
        $oldPage = $viewEl.children('.page').filter(pageEl => pageEl !== $newPage[0] && removedPageEls.indexOf(pageEl) < 0);
        if (dynamicNavbar) {
          $oldNavbarEl = $navbarsEl.children('.navbar').filter(navbarEl => navbarEl !== $newNavbarEl[0] && removedNavbarEls.indexOf(removedNavbarEls) < 0);
        }
        removedPageEls = [];
        removedNavbarEls = [];
      }
      if (isDetail && !options.reloadAll) {
        if ($oldPage.length > 1 || reloadDetail) {
          $oldPage = $oldPage.filter(pageEl => !pageEl.classList.contains('page-master'));
        }
        if ($oldNavbarEl && ($oldNavbarEl.length > 1 || reloadDetail)) {
          $oldNavbarEl = $oldNavbarEl.filter(navbarEl => !navbarEl.classList.contains('navbar-master'));
        }
      }

      // Push State
      if (router.params.browserHistory && (options.browserHistory || options.replaceState) && !options.reloadPrevious) {
        const browserHistoryRoot = router.params.browserHistoryRoot || '';
        History$1[options.reloadCurrent || reloadDetail && otherDetailPageEl || options.reloadAll || options.replaceState ? 'replace' : 'push'](view.id, {
          url: options.route.url
        }, browserHistoryRoot + router.params.browserHistorySeparator + options.route.url);
      }
      if (!options.reloadPrevious) {
        // Current Page & Navbar
        router.currentPageEl = $newPage[0];
        if (dynamicNavbar && $newNavbarEl.length) {
          router.currentNavbarEl = $newNavbarEl[0];
        } else {
          delete router.currentNavbarEl;
        }

        // Current Route
        router.currentRoute = options.route;
      }

      // Update router history
      const url = options.route.url;
      if (options.history) {
        if (((options.reloadCurrent || reloadDetail && otherDetailPageEl) && router.history.length) > 0 || options.replaceState) {
          if (reloadDetail && detailsInBetweenRemoved > 0) {
            router.history = router.history.slice(0, router.history.length - detailsInBetweenRemoved);
            router.propsHistory = router.propsHistory.slice(0, router.propsHistory.length - detailsInBetweenRemoved);
          }
          router.history[router.history.length - (options.reloadPrevious ? 2 : 1)] = url;
          router.propsHistory[router.propsHistory.length - (options.reloadPrevious ? 2 : 1)] = options.props || {};
        } else if (options.reloadPrevious) {
          router.history[router.history.length - 2] = url;
          router.propsHistory[router.propsHistory.length - 2] = options.props || {};
        } else if (options.reloadAll) {
          router.history = [url];
          router.propsHistory = [options.props || {}];
        } else {
          router.history.push(url);
          router.propsHistory.push(options.props || {});
        }
      }
      router.saveHistory();

      // Insert new page and navbar
      const newPageInDom = $newPage.parents(document).length > 0;
      const f7Component = $newPage[0].f7Component;
      if (options.reloadPrevious) {
        if (f7Component && !newPageInDom) {
          f7Component.mount(componentEl => {
            $(componentEl).insertBefore($oldPage);
          });
        } else {
          $newPage.insertBefore($oldPage);
        }
        if (dynamicNavbar && $newNavbarEl.length) {
          if ($newNavbarEl.find('.title-large').length) {
            $newNavbarEl.addClass('navbar-large');
          }
          if ($oldNavbarEl.length) {
            $newNavbarEl.insertBefore($oldNavbarEl);
          } else {
            if (!router.$navbarsEl.parents(document).length) {
              router.$el.prepend(router.$navbarsEl);
            }
            $navbarsEl.append($newNavbarEl);
          }
        }
      } else {
        if ($oldPage.next('.page')[0] !== $newPage[0]) {
          if (f7Component && !newPageInDom) {
            f7Component.mount(componentEl => {
              $viewEl.append(componentEl);
            });
          } else {
            $viewEl.append($newPage[0]);
          }
        }
        if (dynamicNavbar && $newNavbarEl.length) {
          if ($newNavbarEl.find('.title-large').length) {
            $newNavbarEl.addClass('navbar-large');
          }
          if (!router.$navbarsEl.parents(document).length) {
            router.$el.prepend(router.$navbarsEl);
          }
          $navbarsEl.append($newNavbarEl[0]);
        }
      }
      if (!newPageInDom) {
        router.pageCallback('mounted', $newPage, $newNavbarEl, newPagePosition, reload ? newPagePosition : 'current', options, $oldPage);
      } else if (options.route && options.route.route && options.route.route.keepAlive && !$newPage[0].f7PageMounted) {
        $newPage[0].f7PageMounted = true;
        router.pageCallback('mounted', $newPage, $newNavbarEl, newPagePosition, reload ? newPagePosition : 'current', options, $oldPage);
      }

      // Remove old page
      if ((options.reloadCurrent || reloadDetail) && $oldPage.length > 0) {
        // Page remove event
        router.pageCallback('beforeOut', $oldPage, $oldNavbarEl, 'current', undefined, options);
        router.pageCallback('afterOut', $oldPage, $oldNavbarEl, 'current', undefined, options);
        router.pageCallback('beforeRemove', $oldPage, $oldNavbarEl, 'current', undefined, options);
        router.removePage($oldPage);
        if (dynamicNavbar && $oldNavbarEl && $oldNavbarEl.length) {
          router.removeNavbar($oldNavbarEl);
        }
      } else if (options.reloadAll) {
        $oldPage.each((pageEl, index) => {
          const $oldPageEl = $(pageEl);
          const $oldNavbarElEl = $(app.navbar.getElByPage($oldPageEl));

          // Page remove event
          if ($oldPageEl.hasClass('page-current')) {
            router.pageCallback('beforeOut', $oldPage, $oldNavbarEl, 'current', undefined, options);
            router.pageCallback('afterOut', $oldPage, $oldNavbarEl, 'current', undefined, options);
          }
          router.pageCallback('beforeRemove', $oldPageEl, $oldNavbarEl && $oldNavbarEl.eq(index), 'previous', undefined, options);
          router.removePage($oldPageEl);
          if (dynamicNavbar && $oldNavbarElEl.length) {
            router.removeNavbar($oldNavbarElEl);
          }
        });
      } else if (options.reloadPrevious) {
        // Page remove event
        router.pageCallback('beforeRemove', $oldPage, $oldNavbarEl, 'previous', undefined, options);
        router.removePage($oldPage);
        if (dynamicNavbar && $oldNavbarEl && $oldNavbarEl.length) {
          router.removeNavbar($oldNavbarEl);
        }
      }

      // Load Tab
      if (options.route.route.tab) {
        router.tabLoad(options.route.route.tab, extend$3({}, options, {
          history: false,
          browserHistory: false
        }));
      }

      // Check master detail
      if (masterDetailEnabled) {
        view.checkMasterDetailBreakpoint();
      }

      // Page init and before init events
      router.pageCallback('init', $newPage, $newNavbarEl, newPagePosition, reload ? newPagePosition : 'current', options, $oldPage);
      if (options.reloadCurrent || options.reloadAll || reloadDetail) {
        router.allowPageChange = true;
        router.pageCallback('beforeIn', $newPage, $newNavbarEl, newPagePosition, 'current', options);
        $newPage.removeAttr('aria-hidden');
        if (dynamicNavbar && $newNavbarEl) {
          $newNavbarEl.removeAttr('aria-hidden');
        }
        router.pageCallback('afterIn', $newPage, $newNavbarEl, newPagePosition, 'current', options);
        if (options.reloadCurrent && options.clearPreviousHistory) router.clearPreviousHistory();
        if (reloadDetail) {
          router.setPagePosition($(masterPageEl), 'previous');
          if (masterPageEl.f7Page && masterPageEl.f7Page.navbarEl) {
            router.setNavbarPosition($(masterPageEl.f7Page.navbarEl), 'previous');
          }
        }
        return router;
      }
      if (options.reloadPrevious) {
        router.allowPageChange = true;
        return router;
      }

      // Before animation event
      router.pageCallback('beforeOut', $oldPage, $oldNavbarEl, 'current', 'previous', options);
      router.pageCallback('beforeIn', $newPage, $newNavbarEl, 'next', 'current', options);

      // Animation
      function afterAnimation() {
        router.setPagePosition($newPage, 'current', false);
        router.setPagePosition($oldPage, 'previous', !$oldPage.hasClass('page-master'));
        if (dynamicNavbar) {
          router.setNavbarPosition($newNavbarEl, 'current', false);
          router.setNavbarPosition($oldNavbarEl, 'previous', !$oldNavbarEl.hasClass('navbar-master'));
        }
        // After animation event
        router.allowPageChange = true;
        router.pageCallback('afterOut', $oldPage, $oldNavbarEl, 'current', 'previous', options);
        router.pageCallback('afterIn', $newPage, $newNavbarEl, 'next', 'current', options);
        let keepOldPage = (router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`]) && !isMaster;
        if (!keepOldPage) {
          if ($newPage.hasClass('smart-select-page') || $newPage.hasClass('photo-browser-page') || $newPage.hasClass('autocomplete-page') || $newPage.hasClass('color-picker-page')) {
            keepOldPage = true;
          }
        }
        if (!keepOldPage) {
          if (!($newPage.attr('data-name') && $newPage.attr('data-name') === 'smart-select-page')) {
            // Remove event
            router.pageCallback('beforeRemove', $oldPage, $oldNavbarEl, 'previous', undefined, options);
            router.removePage($oldPage);
            if (dynamicNavbar && $oldNavbarEl.length) {
              router.removeNavbar($oldNavbarEl);
            }
          }
        }
        if (options.clearPreviousHistory) router.clearPreviousHistory();
        router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
        if (router.params.browserHistory) {
          History$1.clearRouterQueue();
        }
      }
      function setPositionClasses() {
        router.setPagePosition($oldPage, 'current', false);
        router.setPagePosition($newPage, 'next', false);
        if (dynamicNavbar) {
          router.setNavbarPosition($oldNavbarEl, 'current', false);
          router.setNavbarPosition($newNavbarEl, 'next', false);
        }
      }
      if (options.animate && !(isMaster && app.width >= router.params.masterDetailBreakpoint)) {
        const delay = router.params[`${router.app.theme}PageLoadDelay`];
        let transition = router.params.transition;
        if (options.transition) transition = options.transition;
        if (!transition && router.currentRoute && router.currentRoute.route) {
          transition = router.currentRoute.route.transition;
        }
        if (!transition && router.currentRoute && router.currentRoute.route.options) {
          transition = router.currentRoute.route.options.transition;
        }
        if (transition) {
          $newPage[0].f7PageTransition = transition;
        }
        if (delay) {
          setTimeout(() => {
            setPositionClasses();
            router.animate($oldPage, $newPage, $oldNavbarEl, $newNavbarEl, 'forward', transition, () => {
              afterAnimation();
            });
          }, delay);
        } else {
          setPositionClasses();
          router.animate($oldPage, $newPage, $oldNavbarEl, $newNavbarEl, 'forward', transition, () => {
            afterAnimation();
          });
        }
      } else {
        afterAnimation();
      }
      return router;
    }
    function load(router, loadParams, loadOptions, ignorePageChange) {
      if (loadParams === void 0) {
        loadParams = {};
      }
      if (loadOptions === void 0) {
        loadOptions = {};
      }
      if (!router.allowPageChange && !ignorePageChange) return router;
      const params = loadParams;
      const options = loadOptions;
      const {
        url,
        content,
        el,
        pageName,
        component,
        componentUrl
      } = params;
      if (!options.reloadCurrent && options.route && options.route.route && options.route.route.parentPath && router.currentRoute.route && router.currentRoute.route.parentPath === options.route.route.parentPath) {
        // Do something nested
        if (options.route.url === router.url) {
          router.allowPageChange = true;
          return false;
        }
        // Check for same params
        let sameParams = Object.keys(options.route.params).length === Object.keys(router.currentRoute.params).length;
        if (sameParams) {
          // Check for equal params name
          Object.keys(options.route.params).forEach(paramName => {
            if (!(paramName in router.currentRoute.params) || router.currentRoute.params[paramName] !== options.route.params[paramName]) {
              sameParams = false;
            }
          });
        }
        if (sameParams) {
          if (options.route.route.tab) {
            return router.tabLoad(options.route.route.tab, options);
          }
          return false;
        }
        if (!sameParams && options.route.route.tab && router.currentRoute.route.tab && router.currentRoute.parentPath === options.route.parentPath) {
          return router.tabLoad(options.route.route.tab, options);
        }
      }
      if (options.route && options.route.url && router.url === options.route.url && !(options.reloadCurrent || options.reloadPrevious) && !router.params.allowDuplicateUrls) {
        router.allowPageChange = true;
        return false;
      }
      if (!options.route && url) {
        options.route = router.parseRouteUrl(url);
        extend$3(options.route, {
          route: {
            url,
            path: url
          }
        });
      }

      // Component Callbacks
      function resolve(pageEl, newOptions) {
        return forward(router, pageEl, extend$3(options, newOptions));
      }
      function reject() {
        router.allowPageChange = true;
        return router;
      }
      if (url || componentUrl || component) {
        router.allowPageChange = false;
      }

      // Proceed
      if (content) {
        forward(router, router.getPageEl(content), options);
      } else if (el) {
        // Load page from specified HTMLElement or by page name in pages container
        forward(router, router.getPageEl(el), options);
      } else if (pageName) {
        // Load page by page name in pages container
        forward(router, router.$el.children(`.page[data-name="${pageName}"]`).eq(0), options);
      } else if (component || componentUrl) {
        // Load from component (F7/Vue/React/...)
        try {
          router.pageComponentLoader({
            routerEl: router.el,
            component,
            componentUrl,
            options,
            resolve,
            reject
          });
        } catch (err) {
          router.allowPageChange = true;
          throw err;
        }
      } else if (url) {
        // Load using XHR
        if (router.xhrAbortController) {
          router.xhrAbortController.abort();
          router.xhrAbortController = false;
        }
        router.xhrRequest(url, options).then(pageContent => {
          forward(router, router.getPageEl(pageContent), options);
        }).catch(() => {
          router.allowPageChange = true;
        });
      }
      return router;
    }
    function navigate(navigateParams, navigateOptions) {
      if (navigateOptions === void 0) {
        navigateOptions = {};
      }
      const router = this;
      if (router.swipeBackActive) return router;
      let url;
      let createRoute;
      let name;
      let path;
      let query;
      let params;
      let route;
      if (typeof navigateParams === 'string') {
        url = navigateParams;
      } else {
        url = navigateParams.url;
        createRoute = navigateParams.route;
        name = navigateParams.name;
        path = navigateParams.path;
        query = navigateParams.query;
        params = navigateParams.params;
      }
      if (name || path) {
        url = router.generateUrl({
          path,
          name,
          params,
          query
        });
        if (url) {
          return router.navigate(url, navigateOptions);
        }
        return router;
      }
      const app = router.app;
      appRouterCheck(router, 'navigate');
      if (url === '#' || url === '') {
        return router;
      }
      let navigateUrl = url.replace('./', '');
      if (navigateUrl[0] !== '/' && navigateUrl.indexOf('#') !== 0) {
        const currentPath = router.currentRoute.parentPath || router.currentRoute.path;
        navigateUrl = ((currentPath ? `${currentPath}/` : '/') + navigateUrl).replace('///', '/').replace('//', '/');
      }
      if (createRoute) {
        route = extend$3(router.parseRouteUrl(navigateUrl), {
          route: extend$3({}, createRoute)
        });
      } else {
        route = router.findMatchingRoute(navigateUrl);
      }
      if (!route) {
        return router;
      }
      if (route.route && route.route.viewName) {
        const anotherViewName = route.route.viewName;
        const anotherView = app.views[anotherViewName];
        if (!anotherView) {
          throw new Error(`Framework7: There is no View with "${anotherViewName}" name that was specified in this route`);
        }
        if (anotherView !== router.view) {
          return anotherView.router.navigate(navigateParams, navigateOptions);
        }
      }
      if (route.route.redirect) {
        return redirect.call(router, 'forward', route, navigateOptions);
      }
      const options = {};
      if (route.route.options) {
        extend$3(options, route.route.options, navigateOptions);
      } else {
        extend$3(options, navigateOptions);
      }
      if (options.openIn && (!router.params.ignoreOpenIn || router.params.ignoreOpenIn && router.history.length > 0)) {
        return router.openIn(router, navigateUrl, options);
      }
      options.route = route;
      function resolve() {
        let routerLoaded = false;
        'popup popover sheet loginScreen actions customModal panel'.split(' ').forEach(modalLoadProp => {
          if (route.route[modalLoadProp] && !routerLoaded) {
            routerLoaded = true;
            router.modalLoad(modalLoadProp, route, options, 'forward');
          }
        });
        if (route.route.keepAlive && route.route.keepAliveData) {
          load(router, {
            el: route.route.keepAliveData.pageEl
          }, options, false);
          routerLoaded = true;
        }
        'url content component pageName el componentUrl'.split(' ').forEach(pageLoadProp => {
          if (route.route[pageLoadProp] && !routerLoaded) {
            routerLoaded = true;
            load(router, {
              [pageLoadProp]: route.route[pageLoadProp]
            }, options, false);
          }
        });
        if (routerLoaded) return;
        // Async
        function asyncResolve(resolveParams, resolveOptions) {
          router.allowPageChange = false;
          let resolvedAsModal = false;
          'popup popover sheet loginScreen actions customModal panel'.split(' ').forEach(modalLoadProp => {
            if (resolveParams[modalLoadProp]) {
              resolvedAsModal = true;
              const modalRoute = extend$3({}, route, {
                route: resolveParams
              });
              router.allowPageChange = true;
              router.modalLoad(modalLoadProp, modalRoute, extend$3(options, resolveOptions), 'forward');
            }
          });
          if (resolvedAsModal) return;
          load(router, resolveParams, extend$3(options, resolveOptions), true);
        }
        function asyncReject() {
          router.allowPageChange = true;
        }
        if (route.route.async) {
          router.allowPageChange = false;
          route.route.async.call(router, {
            router,
            to: options.route,
            from: router.currentRoute,
            resolve: asyncResolve,
            reject: asyncReject,
            direction: 'forward',
            app
          });
        }
        if (route.route.asyncComponent) {
          asyncComponent(router, route.route.asyncComponent, asyncResolve, asyncReject);
        }
      }
      function reject() {
        router.allowPageChange = true;
      }
      if (router.params.masterDetailBreakpoint > 0 && route.route.masterRoute) {
        // load detail route
        let preloadMaster = true;
        let masterLoaded = false;
        if (router.currentRoute && router.currentRoute.route) {
          if ((router.currentRoute.route.master === true || typeof router.currentRoute.route.master === 'function' && router.currentRoute.route.master(app, router)) && (router.currentRoute.route === route.route.masterRoute || router.currentRoute.route.path === route.route.masterRoute.path)) {
            preloadMaster = false;
          }
          if (router.currentRoute.route.masterRoute && (router.currentRoute.route.masterRoute === route.route.masterRoute || router.currentRoute.route.masterRoute.path === route.route.masterRoute.path)) {
            preloadMaster = false;
            masterLoaded = true;
          }
        }
        if (preloadMaster || masterLoaded && navigateOptions.reloadAll) {
          router.navigate({
            path: route.route.masterRoute.path,
            params: route.params || {}
          }, {
            animate: false,
            reloadAll: navigateOptions.reloadAll,
            reloadCurrent: navigateOptions.reloadCurrent,
            reloadPrevious: navigateOptions.reloadPrevious,
            browserHistory: !navigateOptions.initial,
            history: !navigateOptions.initial,
            once: {
              pageAfterIn() {
                router.navigate(navigateParams, extend$3({}, navigateOptions, {
                  animate: false,
                  reloadAll: false,
                  reloadCurrent: false,
                  reloadPrevious: false,
                  history: !navigateOptions.initial,
                  browserHistory: !navigateOptions.initial
                }));
              }
            }
          });
          return router;
        }
      }
      processRouteQueue.call(router, route, router.currentRoute, () => {
        if (route.route.modules) {
          app.loadModules(Array.isArray(route.route.modules) ? route.route.modules : [route.route.modules]).then(() => {
            resolve();
          }).catch(() => {
            reject();
          });
        } else {
          resolve();
        }
      }, () => {
        reject();
      }, 'forward');

      // Return Router
      return router;
    }

    function tabLoad(tabRoute, loadOptions) {
      if (loadOptions === void 0) {
        loadOptions = {};
      }
      const router = this;
      const options = extend$3({
        animate: router.params.animate,
        browserHistory: true,
        history: true,
        parentPageEl: null,
        preload: false,
        on: {}
      }, loadOptions);
      let currentRoute;
      let previousRoute;
      if (options.route) {
        // Set Route
        if (!options.preload && options.route !== router.currentRoute) {
          previousRoute = router.previousRoute;
          router.currentRoute = options.route;
        }
        if (options.preload) {
          currentRoute = options.route;
          previousRoute = router.currentRoute;
        } else {
          currentRoute = router.currentRoute;
          if (!previousRoute) previousRoute = router.previousRoute;
        }

        // Update Browser History
        if (router.params.browserHistory && options.browserHistory && !options.reloadPrevious) {
          History$1[router.params.browserHistoryTabs](router.view.id, {
            url: options.route.url
          }, (router.params.browserHistoryRoot || '') + router.params.browserHistorySeparator + options.route.url);
        }

        // Update Router History
        if (options.history) {
          router.history[Math.max(router.history.length - 1, 0)] = options.route.url;
          router.saveHistory();
        }
      }

      // Show Tab
      const $parentPageEl = $(options.parentPageEl || router.currentPageEl);
      let tabEl;
      if ($parentPageEl.length && $parentPageEl.find(`#${tabRoute.id}`).length) {
        tabEl = $parentPageEl.find(`#${tabRoute.id}`).eq(0);
      } else if (router.view.selector) {
        tabEl = `${router.view.selector} #${tabRoute.id}`;
      } else {
        tabEl = `#${tabRoute.id}`;
      }
      const tabShowResult = router.app.tab.show({
        tabEl,
        animate: options.animate,
        tabRoute: options.route
      });
      const {
        $newTabEl,
        $oldTabEl,
        animated,
        onTabsChanged
      } = tabShowResult;
      if ($newTabEl && $newTabEl.parents('.page').length > 0 && options.route) {
        const tabParentPageData = $newTabEl.parents('.page')[0].f7Page;
        if (tabParentPageData && options.route) {
          tabParentPageData.route = options.route;
        }
      }

      // Tab Content Loaded
      function onTabLoaded(contentEl) {
        // Remove theme elements
        router.removeThemeElements($newTabEl);
        let tabEventTarget = $newTabEl;
        if (typeof contentEl !== 'string') tabEventTarget = $(contentEl);
        tabEventTarget.trigger('tab:init tab:mounted', tabRoute);
        router.emit('tabInit tabMounted', $newTabEl[0], tabRoute);
        if ($oldTabEl && $oldTabEl.length) {
          if (animated) {
            onTabsChanged(() => {
              router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
              if (router.params.unloadTabContent) {
                router.tabRemove($oldTabEl, $newTabEl, tabRoute);
              }
            });
          } else {
            router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
            if (router.params.unloadTabContent) {
              router.tabRemove($oldTabEl, $newTabEl, tabRoute);
            }
          }
        }
      }
      if ($newTabEl[0].f7RouterTabLoaded) {
        if (!$oldTabEl || !$oldTabEl.length) return router;
        if (animated) {
          onTabsChanged(() => {
            router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
          });
        } else {
          router.emit('routeChanged', router.currentRoute, router.previousRoute, router);
        }
        return router;
      }

      // Load Tab Content
      function loadTab(loadTabParams, loadTabOptions) {
        // Load Tab Props
        const {
          url,
          content,
          el,
          component,
          componentUrl
        } = loadTabParams;
        // Component/Template Callbacks
        function resolve(contentEl) {
          router.allowPageChange = true;
          if (!contentEl) return;
          if (typeof contentEl === 'string') {
            $newTabEl.html(contentEl);
          } else {
            $newTabEl.html('');
            if (contentEl.f7Component) {
              contentEl.f7Component.mount(componentEl => {
                $newTabEl.append(componentEl);
              });
            } else {
              $newTabEl.append(contentEl);
            }
          }
          $newTabEl[0].f7RouterTabLoaded = true;
          onTabLoaded(contentEl);
        }
        function reject() {
          router.allowPageChange = true;
          return router;
        }
        if (content) {
          resolve(content);
        } else if (el) {
          resolve(el);
        } else if (component || componentUrl) {
          // Load from component (F7/Vue/React/...)
          try {
            router.tabComponentLoader({
              tabEl: $newTabEl[0],
              component,
              componentUrl,
              options: loadTabOptions,
              resolve,
              reject
            });
          } catch (err) {
            router.allowPageChange = true;
            throw err;
          }
        } else if (url) {
          // Load using XHR
          if (router.xhrAbortController) {
            router.xhrAbortController.abort();
            router.xhrAbortController = false;
          }
          router.xhrRequest(url, loadTabOptions).then(tabContent => {
            resolve(tabContent);
          }).catch(() => {
            router.allowPageChange = true;
          });
        }
      }
      let hasContentLoadProp;
      'url content component el componentUrl'.split(' ').forEach(tabLoadProp => {
        if (tabRoute[tabLoadProp]) {
          hasContentLoadProp = true;
          loadTab({
            [tabLoadProp]: tabRoute[tabLoadProp]
          }, options);
        }
      });

      // Async
      function asyncResolve(resolveParams, resolveOptions) {
        loadTab(resolveParams, extend$3(options, resolveOptions));
      }
      function asyncReject() {
        router.allowPageChange = true;
      }
      if (tabRoute.async) {
        tabRoute.async.call(router, {
          router,
          to: currentRoute,
          from: previousRoute,
          resolve: asyncResolve,
          reject: asyncReject,
          app: router.app
        });
      } else if (tabRoute.asyncComponent) {
        asyncComponent(router, tabRoute.asyncComponent, asyncResolve, asyncReject);
      } else if (!hasContentLoadProp) {
        router.allowPageChange = true;
      }
      return router;
    }
    function tabRemove($oldTabEl, $newTabEl, tabRoute) {
      const router = this;
      let hasTabComponentChild;
      if ($oldTabEl[0]) {
        $oldTabEl[0].f7RouterTabLoaded = false;
        delete $oldTabEl[0].f7RouterTabLoaded;
      }
      $oldTabEl.children().each(tabChild => {
        if (tabChild.f7Component) {
          hasTabComponentChild = true;
          $(tabChild).trigger('tab:beforeremove', tabRoute);
          tabChild.f7Component.destroy();
        }
      });
      if (!hasTabComponentChild) {
        $oldTabEl.trigger('tab:beforeremove', tabRoute);
      }
      router.emit('tabBeforeRemove', $oldTabEl[0], $newTabEl[0], tabRoute);
      router.removeTabContent($oldTabEl[0], tabRoute);
    }

    function modalLoad(modalType, route, loadOptions, direction) {
      if (loadOptions === void 0) {
        loadOptions = {};
      }
      const router = this;
      const app = router.app;
      const isPanel = modalType === 'panel';
      const modalOrPanel = isPanel ? 'panel' : 'modal';
      const options = extend$3({
        animate: router.params.animate,
        browserHistory: true,
        history: true,
        on: {},
        once: {}
      }, loadOptions);
      const modalParams = extend$3({}, route.route[modalType]);
      const modalRoute = route.route;
      const routeCallback = (modal, name) => {
        const {
          on,
          once
        } = options;
        let callback;
        if (name === 'open') {
          callback = on.modalOpen || once.modalOpen || on.panelOpen || once.panelOpen;
        }
        if (name === 'close') {
          callback = on.modalClose || once.modalClose || on.panelClose || once.panelClose;
        }
        if (name === 'closed') {
          callback = on.modalClosed || once.modalClosed || on.panelClosed || once.panelClosed;
        }
        if (callback) callback(modal);
      };
      function onModalLoaded() {
        // Create Modal
        const modal = app[modalType].create(modalParams);
        modalRoute.modalInstance = modal;
        const hasEl = modal.el;
        function closeOnSwipeBack() {
          modal.close();
        }
        modal.on(`${modalOrPanel}Open`, () => {
          if (!hasEl) {
            // Remove theme elements
            router.removeThemeElements(modal.el);

            // Emit events
            modal.$el.trigger(`${modalType.toLowerCase()}:init ${modalType.toLowerCase()}:mounted`, route, modal);
            router.emit(`${!isPanel ? 'modalInit' : ''} ${modalType}Init ${modalType}Mounted`, modal.el, route, modal);
          }
          router.once('swipeBackMove', closeOnSwipeBack);
          routeCallback(modal, 'open');
        });
        modal.on(`${modalOrPanel}Close`, () => {
          router.off('swipeBackMove', closeOnSwipeBack);
          if (!modal.closeByRouter) {
            router.back();
          }
          routeCallback(modal, 'close');
        });
        modal.on(`${modalOrPanel}Closed`, () => {
          modal.$el.trigger(`${modalType.toLowerCase()}:beforeremove`, route, modal);
          modal.emit(`${!isPanel ? 'modalBeforeRemove ' : ''}${modalType}BeforeRemove`, modal.el, route, modal);
          const modalComponent = modal.el.f7Component;
          routeCallback(modal, 'closed');
          if (modalComponent) {
            modalComponent.destroy();
          }
          nextTick$1(() => {
            if (modalComponent || modalParams.component || modalParams.asyncComponent || modalParams.async) {
              router.removeModal(modal.el);
            }
            modal.destroy();
            delete modal.route;
            delete modalRoute.modalInstance;
          });
        });
        if (options.route) {
          // Update Browser History
          if (router.params.browserHistory && options.browserHistory) {
            History$1.push(router.view.id, {
              url: options.route.url,
              modal: modalType
            }, (router.params.browserHistoryRoot || '') + router.params.browserHistorySeparator + options.route.url);
          }

          // Set Route
          if (options.route !== router.currentRoute) {
            modal.route = extend$3(options.route, {
              modal
            });
            router.currentRoute = modal.route;
          }

          // Update Router History
          if (options.history && !options.reloadCurrent) {
            router.history.push(options.route.url);
            router.saveHistory();
          }
        }
        if (hasEl) {
          // Remove theme elements
          router.removeThemeElements(modal.el);

          // Emit events
          modal.$el.trigger(`${modalType.toLowerCase()}:init ${modalType.toLowerCase()}:mounted`, route, modal);
          router.emit(`${modalOrPanel}Init ${modalType}Init ${modalType}Mounted`, modal.el, route, modal);
        }

        // Open
        modal.open(options.animate === false || options.animate === true ? options.animate : undefined);
      }

      // Load Modal Content
      function loadModal(loadModalParams, loadModalOptions) {
        // Load Modal Props
        const {
          url,
          content,
          component,
          componentUrl
        } = loadModalParams;

        // Component/Template Callbacks
        function resolve(contentEl) {
          if (contentEl) {
            if (typeof contentEl === 'string') {
              modalParams.content = contentEl;
            } else if (contentEl.f7Component) {
              contentEl.f7Component.mount(componentEl => {
                modalParams.el = componentEl;
                app.$el.append(componentEl);
              });
            } else {
              modalParams.el = contentEl;
            }
            onModalLoaded();
          }
        }
        function reject() {
          router.allowPageChange = true;
          return router;
        }
        if (content) {
          resolve(content);
        } else if (component || componentUrl) {
          // Load from component (F7/Vue/React/...)
          try {
            router.modalComponentLoader({
              rootEl: app.el,
              component,
              componentUrl,
              options: loadModalOptions,
              resolve,
              reject
            });
          } catch (err) {
            router.allowPageChange = true;
            throw err;
          }
        } else if (url) {
          // Load using XHR
          if (router.xhrAbortController) {
            router.xhrAbortController.abort();
            router.xhrAbortController = false;
          }
          router.xhrRequest(url, loadModalOptions).then(modalContent => {
            modalParams.content = modalContent;
            onModalLoaded();
          }).catch(() => {
            router.allowPageChange = true;
          });
        } else {
          onModalLoaded();
        }
      }
      let foundLoadProp;
      'url content component el componentUrl template'.split(' ').forEach(modalLoadProp => {
        if (modalParams[modalLoadProp] && !foundLoadProp) {
          foundLoadProp = true;
          loadModal({
            [modalLoadProp]: modalParams[modalLoadProp]
          }, options);
        }
      });
      if (!foundLoadProp && modalType === 'actions') {
        onModalLoaded();
      }

      // Async
      function asyncResolve(resolveParams, resolveOptions) {
        loadModal(resolveParams, extend$3(options, resolveOptions));
      }
      function asyncReject() {
        router.allowPageChange = true;
      }
      if (modalParams.async) {
        modalParams.async.call(router, {
          router,
          to: options.route,
          from: router.currentRoute,
          resolve: asyncResolve,
          reject: asyncReject,
          direction,
          app
        });
      }
      if (modalParams.asyncComponent) {
        asyncComponent(router, modalParams.asyncComponent, asyncResolve, asyncReject);
      }
      return router;
    }
    function modalRemove(modal) {
      extend$3(modal, {
        closeByRouter: true
      });
      modal.close();
    }

    function backward(router, el, backwardOptions) {
      const device = getDevice$1();
      const document = getDocument$1();
      const $el = $(el);
      const app = router.app;
      const view = router.view;
      const options = extend$3(false, {
        animate: router.params.animate,
        browserHistory: true,
        replaceState: false
      }, backwardOptions);
      const masterDetailEnabled = router.params.masterDetailBreakpoint > 0;
      const isMaster = masterDetailEnabled && options.route && options.route.route && (options.route.route.master === true || typeof options.route.route.master === 'function' && options.route.route.master(app, router));
      let masterPageEl;
      let masterPageRemoved;
      const dynamicNavbar = router.dynamicNavbar;
      const $newPage = $el;
      const $oldPage = router.$el.children('.page-current');
      const initialPreload = $oldPage.length === 0 && options.preload;
      const currentIsMaster = masterDetailEnabled && $oldPage.hasClass('page-master');
      if ($newPage.length) {
        // Remove theme elements
        router.removeThemeElements($newPage);
      }
      let $navbarsEl;
      let $newNavbarEl;
      let $oldNavbarEl;
      if (dynamicNavbar) {
        $newNavbarEl = $newPage.children('.navbar');
        $navbarsEl = router.$navbarsEl;
        if ($newNavbarEl.length === 0 && $newPage[0] && $newPage[0].f7Page) {
          // Try from pageData
          $newNavbarEl = $newPage[0].f7Page.$navbarEl;
        }
        $oldNavbarEl = $navbarsEl.find('.navbar-current');
      }
      router.allowPageChange = false;
      if ($newPage.length === 0 || $oldPage.length === 0 && !options.preload) {
        router.allowPageChange = true;
        return router;
      }

      // Remove theme elements
      router.removeThemeElements($newPage);

      // Save Keep Alive Cache
      if (options.route && options.route.route && options.route.route.keepAlive && !options.route.route.keepAliveData) {
        options.route.route.keepAliveData = {
          pageEl: $el[0]
        };
      }

      // Pages In View
      let isDetail;
      let isDetailRoot;
      if (masterDetailEnabled) {
        const $pagesInView = router.$el.children('.page').filter(pageInView => pageInView !== $newPage[0]);

        // Find Detail' master page
        for (let i = 0; i < $pagesInView.length; i += 1) {
          if (!masterPageEl && $pagesInView[i].classList.contains('page-master')) {
            masterPageEl = $pagesInView[i];
            continue; // eslint-disable-line
          }
        }

        isDetail = !isMaster && masterPageEl && router.history.indexOf(options.route.url) > router.history.indexOf(masterPageEl.f7Page.route.url);
        if (!isDetail && !isMaster && masterPageEl && masterPageEl.f7Page && options.route.route.masterRoute) {
          isDetail = options.route.route.masterRoute.path === masterPageEl.f7Page.route.route.path;
        }
      }
      if (isDetail && masterPageEl && masterPageEl.f7Page) {
        isDetailRoot = router.history.indexOf(options.route.url) - router.history.indexOf(masterPageEl.f7Page.route.url) === 1;
      }

      // New Page
      $newPage.addClass(`page-${initialPreload ? 'current' : 'previous'}${isMaster ? ' page-master' : ''}${isDetail ? ' page-master-detail' : ''}${isDetailRoot ? ' page-master-detail-root' : ''}`).removeAttr('aria-hidden').trigger('page:unstack').trigger('page:position', {
        position: initialPreload ? 'current' : 'previous'
      });
      router.emit('pageUnstack', $newPage[0]);
      router.emit('pagePosition', $newPage[0], initialPreload ? 'current' : 'previous');
      if (isMaster || isDetail) {
        $newPage.trigger('page:role', {
          role: isMaster ? 'master' : 'detail',
          root: !!isDetailRoot
        });
        router.emit('pageRole', $newPage[0], {
          role: isMaster ? 'master' : 'detail',
          detailRoot: !!isDetailRoot
        });
      }
      if (dynamicNavbar && $newNavbarEl.length > 0) {
        $newNavbarEl.addClass(`navbar-${initialPreload ? 'current' : 'previous'}${isMaster ? ' navbar-master' : ''}${isDetail ? ' navbar-master-detail' : ''}${isDetailRoot ? ' navbar-master-detail-root' : ''}`).removeAttr('aria-hidden');
        $newNavbarEl.trigger('navbar:position', {
          position: initialPreload ? 'current' : 'previous'
        });
        router.emit('navbarPosition', $newNavbarEl[0], initialPreload ? 'current' : 'previous');
        if (isMaster || isDetailRoot) {
          router.emit('navbarRole', $newNavbarEl[0], {
            role: isMaster ? 'master' : 'detail',
            detailRoot: !!isDetailRoot
          });
        }
      }

      // Remove previous page in case of "forced"
      let backIndex;
      if (options.force) {
        if ($oldPage.prev('.page-previous').length >= 0) {
          if (router.history.indexOf(options.route.url) >= 0) {
            backIndex = router.history.length - router.history.indexOf(options.route.url) - 1;
            router.history = router.history.slice(0, router.history.indexOf(options.route.url) + 2);
            router.propsHistory = router.propsHistory.slice(0, router.history.indexOf(options.route.url) + 2);
            view.history = router.history;
          } else if (router.history[[router.history.length - 2]]) {
            router.propsHistory[router.propsHistory.length - 2] = options.props || {};
          } else {
            router.history.unshift(router.url);
            router.propsHistory.unshift(options.props || {});
          }
          const $pageToRemove = $oldPage.prev('.page-previous');
          let $navbarToRemove;
          if (dynamicNavbar) {
            $navbarToRemove = $(app.navbar.getElByPage($pageToRemove));
          }
          if ($pageToRemove.length > 0) {
            router.pageCallback('beforeRemove', $pageToRemove, $navbarToRemove, 'previous', undefined, options);
            if ($pageToRemove[0] === masterPageEl) {
              masterPageRemoved = true;
            }
            router.removePage($pageToRemove);
            if (dynamicNavbar && $navbarToRemove.length) {
              router.removeNavbar($navbarToRemove);
            }
          }
        }
      }

      // Insert new page
      const newPageInDom = $newPage.parents(document).length > 0;
      const f7Component = $newPage[0].f7Component;
      function insertPage() {
        if (initialPreload) {
          if (!newPageInDom && f7Component) {
            f7Component.mount(componentEl => {
              router.$el.append(componentEl);
            });
          } else {
            router.$el.append($newPage);
          }
        }
        if ($newPage.next($oldPage).length === 0) {
          if (!newPageInDom && f7Component) {
            f7Component.mount(componentEl => {
              $(componentEl).insertBefore($oldPage);
            });
          } else {
            $newPage.insertBefore($oldPage);
          }
        }
        if (dynamicNavbar && $newNavbarEl.length) {
          if ($newNavbarEl.find('.title-large').length) {
            $newNavbarEl.addClass('navbar-large');
          }
          $newNavbarEl.insertBefore($oldNavbarEl);
          if ($oldNavbarEl.length > 0) {
            $newNavbarEl.insertBefore($oldNavbarEl);
          } else {
            if (!router.$navbarsEl.parents(document).length) {
              router.$el.prepend(router.$navbarsEl);
            }
            $navbarsEl.append($newNavbarEl);
          }
        }
        if (!newPageInDom) {
          router.pageCallback('mounted', $newPage, $newNavbarEl, 'previous', 'current', options, $oldPage);
        } else if (options.route && options.route.route && options.route.route.keepAlive && !$newPage[0].f7PageMounted) {
          $newPage[0].f7PageMounted = true;
          router.pageCallback('mounted', $newPage, $newNavbarEl, 'previous', 'current', options, $oldPage);
        }
      }
      if (options.preload) {
        // Insert Page
        insertPage();
        // Tab route
        if (options.route.route.tab) {
          router.tabLoad(options.route.route.tab, extend$3({}, options, {
            history: false,
            browserHistory: false,
            preload: true
          }));
        }
        if (isMaster) {
          $newPage.removeClass('page-master-stacked').trigger('page:masterunstack');
          router.emit('pageMasterUnstack', $newPage[0]);
          if (dynamicNavbar) {
            $(app.navbar.getElByPage($newPage)).removeClass('navbar-master-stacked');
            router.emit('navbarMasterUnstack', app.navbar.getElByPage($newPage));
          }
        }
        // Page init and before init events
        router.pageCallback('init', $newPage, $newNavbarEl, 'previous', 'current', options, $oldPage);
        if (initialPreload) {
          router.pageCallback('beforeIn', $newPage, $newNavbarEl, 'current', undefined, options);
          router.pageCallback('afterIn', $newPage, $newNavbarEl, 'current', undefined, options);
        }
        const $previousPages = $newPage.prevAll('.page-previous:not(.page-master)');
        if ($previousPages.length > 0) {
          $previousPages.each(pageToRemove => {
            const $pageToRemove = $(pageToRemove);
            let $navbarToRemove;
            if (dynamicNavbar) {
              $navbarToRemove = $(app.navbar.getElByPage($pageToRemove));
            }
            router.pageCallback('beforeRemove', $pageToRemove, $navbarToRemove, 'previous', undefined);
            router.removePage($pageToRemove);
            if (dynamicNavbar && $navbarToRemove.length) {
              router.removeNavbar($navbarToRemove);
            }
          });
        }
        router.allowPageChange = true;
        return router;
      }

      // History State
      if (!(device.ie || device.edge || device.firefox && !device.ios)) {
        if (router.params.browserHistory && options.browserHistory) {
          if (options.replaceState) {
            const browserHistoryRoot = router.params.browserHistoryRoot || '';
            History$1.replace(view.id, {
              url: options.route.url
            }, browserHistoryRoot + router.params.browserHistorySeparator + options.route.url);
          } else if (backIndex) {
            History$1.go(-backIndex);
          } else {
            History$1.back();
          }
        }
      }

      // Update History
      if (options.replaceState) {
        router.history[router.history.length - 1] = options.route.url;
        router.propsHistory[router.propsHistory.length - 1] = options.props || {};
      } else {
        if (router.history.length === 1) {
          router.history.unshift(router.url);
          router.propsHistory.unshift(options.props || {});
        }
        router.history.pop();
        router.propsHistory.pop();
      }
      router.saveHistory();

      // Current Page & Navbar
      router.currentPageEl = $newPage[0];
      if (dynamicNavbar && $newNavbarEl.length) {
        router.currentNavbarEl = $newNavbarEl[0];
      } else {
        delete router.currentNavbarEl;
      }

      // Current Route
      router.currentRoute = options.route;

      // History State
      if (device.ie || device.edge || device.firefox && !device.ios) {
        if (router.params.browserHistory && options.browserHistory) {
          if (options.replaceState) {
            const browserHistoryRoot = router.params.browserHistoryRoot || '';
            History$1.replace(view.id, {
              url: options.route.url
            }, browserHistoryRoot + router.params.browserHistorySeparator + options.route.url);
          } else if (backIndex) {
            History$1.go(-backIndex);
          } else {
            History$1.back();
          }
        }
      }

      // Insert Page
      insertPage();

      // Load Tab
      if (options.route.route.tab) {
        router.tabLoad(options.route.route.tab, extend$3({}, options, {
          history: false,
          browserHistory: false
        }));
      }

      // Check master detail

      if (masterDetailEnabled && (currentIsMaster || masterPageRemoved)) {
        view.checkMasterDetailBreakpoint(false);
      }

      // Page init and before init events
      router.pageCallback('init', $newPage, $newNavbarEl, 'previous', 'current', options, $oldPage);

      // Before animation callback
      router.pageCallback('beforeOut', $oldPage, $oldNavbarEl, 'current', 'next', options);
      router.pageCallback('beforeIn', $newPage, $newNavbarEl, 'previous', 'current', options);

      // Animation
      function afterAnimation() {
        // Set classes
        router.setPagePosition($newPage, 'current', false);
        router.setPagePosition($oldPage, 'next', true);
        if (dynamicNavbar) {
          router.setNavbarPosition($newNavbarEl, 'current', false);
          router.setNavbarPosition($oldNavbarEl, 'next', true);
        }

        // After animation event
        router.pageCallback('afterOut', $oldPage, $oldNavbarEl, 'current', 'next', options);
        router.pageCallback('afterIn', $newPage, $newNavbarEl, 'previous', 'current', options);

        // Remove Old Page

        router.pageCallback('beforeRemove', $oldPage, $oldNavbarEl, 'next', undefined, options);
        router.removePage($oldPage);
        if (dynamicNavbar && $oldNavbarEl.length) {
          router.removeNavbar($oldNavbarEl);
        }
        router.allowPageChange = true;
        router.emit('routeChanged', router.currentRoute, router.previousRoute, router);

        // Preload previous page
        const preloadPreviousPage = router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`];
        if (preloadPreviousPage && router.history[router.history.length - 2] && !isMaster) {
          router.back(router.history[router.history.length - 2], {
            preload: true,
            props: router.propsHistory[router.propsHistory.length - 2] || {}
          });
        }
        if (router.params.browserHistory) {
          History$1.clearRouterQueue();
        }
      }
      function setPositionClasses() {
        router.setPagePosition($oldPage, 'current');
        router.setPagePosition($newPage, 'previous', false);
        if (dynamicNavbar) {
          router.setNavbarPosition($oldNavbarEl, 'current');
          router.setNavbarPosition($newNavbarEl, 'previous', false);
        }
      }
      if (options.animate && !(currentIsMaster && app.width >= router.params.masterDetailBreakpoint)) {
        let transition = router.params.transition;
        if ($oldPage[0] && $oldPage[0].f7PageTransition) {
          transition = $oldPage[0].f7PageTransition;
          delete $oldPage[0].f7PageTransition;
        }
        if (options.transition) transition = options.transition;
        if (!transition && router.previousRoute && router.previousRoute.route) {
          transition = router.previousRoute.route.transition;
        }
        if (!transition && router.previousRoute && router.previousRoute.route && router.previousRoute.route.options) {
          transition = router.previousRoute.route.options.transition;
        }
        setPositionClasses();
        router.animate($oldPage, $newPage, $oldNavbarEl, $newNavbarEl, 'backward', transition, () => {
          afterAnimation();
        });
      } else {
        afterAnimation();
      }
      return router;
    }
    function loadBack(router, backParams, backOptions, ignorePageChange) {
      if (!router.allowPageChange && !ignorePageChange) return router;
      const params = backParams;
      const options = backOptions;
      const {
        url,
        content,
        el,
        pageName,
        component,
        componentUrl
      } = params;
      if (options.route.url && router.url === options.route.url && !(options.reloadCurrent || options.reloadPrevious) && !router.params.allowDuplicateUrls) {
        router.allowPageChange = true;
        return false;
      }
      if (!options.route && url) {
        options.route = router.parseRouteUrl(url);
      }

      // Component Callbacks
      function resolve(pageEl, newOptions) {
        return backward(router, pageEl, extend$3(options, newOptions));
      }
      function reject() {
        router.allowPageChange = true;
        return router;
      }
      if (url || componentUrl || component) {
        router.allowPageChange = false;
      }

      // Proceed
      if (content) {
        backward(router, router.getPageEl(content), options);
      } else if (el) {
        // Load page from specified HTMLElement or by page name in pages container
        backward(router, router.getPageEl(el), options);
      } else if (pageName) {
        // Load page by page name in pages container
        backward(router, router.$el.children(`.page[data-name="${pageName}"]`).eq(0), options);
      } else if (component || componentUrl) {
        // Load from component (F7/Vue/React/...)
        try {
          router.pageComponentLoader({
            routerEl: router.el,
            component,
            componentUrl,
            options,
            resolve,
            reject
          });
        } catch (err) {
          router.allowPageChange = true;
          throw err;
        }
      } else if (url) {
        // Load using XHR
        if (router.xhrAbortController) {
          router.xhrAbortController.abort();
          router.xhrAbortController = false;
        }
        router.xhrRequest(url, options).then(pageContent => {
          backward(router, router.getPageEl(pageContent), options);
        }).catch(() => {
          router.allowPageChange = true;
        });
      }
      return router;
    }
    function back() {
      const router = this;
      const device = getDevice$1();
      if (router.swipeBackActive) return router;
      let navigateUrl;
      let navigateOptions;
      let navigateProps;
      let route;
      if (typeof (arguments.length <= 0 ? undefined : arguments[0]) === 'object') {
        navigateOptions = (arguments.length <= 0 ? undefined : arguments[0]) || {};
      } else {
        navigateUrl = arguments.length <= 0 ? undefined : arguments[0];
        navigateOptions = (arguments.length <= 1 ? undefined : arguments[1]) || {};
      }
      const {
        name,
        params,
        query
      } = navigateOptions;
      if (name) {
        navigateUrl = router.generateUrl({
          name,
          params,
          query
        });
        if (navigateUrl) {
          return router.back(navigateUrl, extend$3({}, navigateOptions, {
            name: null,
            params: null,
            query: null
          }));
        }
        return router;
      }
      const app = router.app;
      appRouterCheck(router, 'back');
      let currentRouteIsModal = router.currentRoute.modal;
      let modalType;
      if (!currentRouteIsModal) {
        'popup popover sheet loginScreen actions customModal panel'.split(' ').forEach(modalLoadProp => {
          if (router.currentRoute.route[modalLoadProp]) {
            currentRouteIsModal = true;
            modalType = modalLoadProp;
          }
        });
      }
      if (currentRouteIsModal && !navigateOptions.preload) {
        const modalToClose = router.currentRoute.modal || router.currentRoute.route.modalInstance || app[modalType].get();
        const previousUrl = router.history[router.history.length - 2];
        let previousRoute;
        // check if previous route is modal too
        if (modalToClose && modalToClose.$el) {
          const prevOpenedModals = modalToClose.$el.prevAll('.modal-in');
          if (prevOpenedModals.length && prevOpenedModals[0].f7Modal) {
            const modalEl = prevOpenedModals[0];
            // check if current router not inside of the modalEl
            if (!router.$el.parents(modalEl).length) {
              previousRoute = modalEl.f7Modal.route;
            }
          }
        }
        if (!previousRoute) {
          previousRoute = router.findMatchingRoute(previousUrl);
        }
        if (!previousRoute && previousUrl) {
          previousRoute = {
            url: previousUrl,
            path: previousUrl.split('?')[0],
            query: parseUrlQuery(previousUrl),
            route: {
              path: previousUrl.split('?')[0],
              url: previousUrl
            }
          };
        }
        if (!navigateUrl || navigateUrl.replace(/[# ]/g, '').trim().length === 0) {
          if (!previousRoute || !modalToClose) {
            return router;
          }
        }
        const forceOtherUrl = navigateOptions.force && previousRoute && navigateUrl;
        if (previousRoute && modalToClose) {
          const isBrokenBrowserHistory = device.ie || device.edge || device.firefox && !device.ios;
          const needHistoryBack = router.params.browserHistory && navigateOptions.browserHistory !== false;
          const currentRouteWithoutBrowserHistory = router.currentRoute && router.currentRoute.route && router.currentRoute.route.options && router.currentRoute.route.options.browserHistory === false;
          if (needHistoryBack && !isBrokenBrowserHistory && !currentRouteWithoutBrowserHistory) {
            History$1.back();
          }
          router.currentRoute = previousRoute;
          router.history.pop();
          router.propsHistory.pop();
          router.saveHistory();
          if (needHistoryBack && isBrokenBrowserHistory && !currentRouteWithoutBrowserHistory) {
            History$1.back();
          }
          router.modalRemove(modalToClose);
          if (forceOtherUrl) {
            router.navigate(navigateUrl, {
              reloadCurrent: true
            });
          }
        } else if (modalToClose) {
          router.modalRemove(modalToClose);
          if (navigateUrl) {
            router.navigate(navigateUrl, {
              reloadCurrent: true
            });
          }
        }
        return router;
      }
      let $previousPage = router.$el.children('.page-current').prevAll('.page-previous:not(.page-master)').eq(0);
      let skipMaster;
      if (router.params.masterDetailBreakpoint > 0) {
        const classes = [];
        router.$el.children('.page').each(pageEl => {
          classes.push(pageEl.className);
        });
        const $previousMaster = router.$el.children('.page-current').prevAll('.page-master').eq(0);
        if ($previousMaster.length) {
          const expectedPreviousPageUrl = router.history[router.history.length - 2];
          const expectedPreviousPageRoute = router.findMatchingRoute(expectedPreviousPageUrl);
          if (expectedPreviousPageRoute && $previousMaster[0].f7Page && expectedPreviousPageRoute.route === $previousMaster[0].f7Page.route.route) {
            $previousPage = $previousMaster;
            if (!navigateOptions.preload) {
              skipMaster = app.width >= router.params.masterDetailBreakpoint;
            }
          }
        }
      }
      if (!navigateOptions.force && $previousPage.length && !skipMaster) {
        const previousPageObj = $previousPage[0].f7Page;
        if (router.params.browserHistory && previousPageObj && router.history[router.history.length - 2] !== previousPageObj.route.url) {
          router.back(router.history[router.history.length - 2], extend$3(navigateOptions, {
            force: true,
            props: router.propsHistory[router.propsHistory.length - 2] || {}
          }));
          return router;
        }
        if (previousPageObj) {
          const previousPageRoute = previousPageObj.route;
          processRouteQueue.call(router, previousPageRoute, router.currentRoute, () => {
            loadBack(router, {
              el: $previousPage
            }, extend$3(navigateOptions, {
              route: previousPageRoute
            }));
          }, () => {}, 'backward');
          return router;
        }
      }

      // Navigate URL
      if (navigateUrl === '#') {
        navigateUrl = undefined;
      }
      if (navigateUrl && navigateUrl[0] !== '/' && navigateUrl.indexOf('#') !== 0) {
        navigateUrl = ((router.path || '/') + navigateUrl).replace('//', '/');
      }
      if (!navigateUrl && router.history.length > 1) {
        navigateUrl = router.history[router.history.length - 2];
        navigateProps = router.propsHistory[router.propsHistory.length - 2] || {};
      }
      if (skipMaster && !navigateOptions.force && router.history[router.history.length - 3]) {
        return router.back(router.history[router.history.length - 3], extend$3({}, navigateOptions || {}, {
          force: true,
          animate: false,
          props: router.propsHistory[router.propsHistory.length - 3] || {}
        }));
      }
      if (skipMaster && !navigateOptions.force) {
        return router;
      }

      // Find route to load
      route = router.findMatchingRoute(navigateUrl);
      if (!route) {
        if (navigateUrl) {
          route = {
            url: navigateUrl,
            path: navigateUrl.split('?')[0],
            query: parseUrlQuery(navigateUrl),
            route: {
              path: navigateUrl.split('?')[0],
              url: navigateUrl
            }
          };
        }
      }
      if (!route) {
        return router;
      }
      if (route.route.redirect) {
        return redirect.call(router, 'backward', route, navigateOptions);
      }
      const options = {};
      if (route.route.options) {
        extend$3(options, route.route.options, navigateOptions, {
          props: navigateProps || {}
        });
      } else {
        extend$3(options, navigateOptions, {
          props: navigateProps || {}
        });
      }
      options.route = route;
      function resolve() {
        let routerLoaded = false;
        if (route.route.keepAlive && route.route.keepAliveData) {
          loadBack(router, {
            el: route.route.keepAliveData.pageEl
          }, options);
          routerLoaded = true;
        }
        'url content component pageName el componentUrl'.split(' ').forEach(pageLoadProp => {
          if (route.route[pageLoadProp] && !routerLoaded) {
            routerLoaded = true;
            loadBack(router, {
              [pageLoadProp]: route.route[pageLoadProp]
            }, options);
          }
        });
        if (routerLoaded) return;
        // Async
        function asyncResolve(resolveParams, resolveOptions) {
          router.allowPageChange = false;
          loadBack(router, resolveParams, extend$3(options, resolveOptions), true);
        }
        function asyncReject() {
          router.allowPageChange = true;
        }
        if (route.route.async) {
          router.allowPageChange = false;
          route.route.async.call(router, {
            router,
            to: route,
            from: router.currentRoute,
            resolve: asyncResolve,
            reject: asyncReject,
            direction: 'backward',
            app
          });
        }
        if (route.route.asyncComponent) {
          asyncComponent(router, route.route.asyncComponent, asyncResolve, asyncReject);
        }
      }
      function reject() {
        router.allowPageChange = true;
      }
      if (options.preload) {
        resolve();
      } else {
        processRouteQueue.call(router, route, router.currentRoute, () => {
          if (route.route.modules) {
            app.loadModules(Array.isArray(route.route.modules) ? route.route.modules : [route.route.modules]).then(() => {
              resolve();
            }).catch(() => {
              reject();
            });
          } else {
            resolve();
          }
        }, () => {
          reject();
        }, 'backward');
      }

      // Return Router
      return router;
    }

    function clearPreviousPages(router) {
      appRouterCheck(router, 'clearPreviousPages');
      const app = router.app;
      const dynamicNavbar = router.dynamicNavbar;
      const $pagesToRemove = router.$el.children('.page').filter(pageInView => {
        if (router.currentRoute && (router.currentRoute.modal || router.currentRoute.panel)) return true;
        return pageInView !== router.currentPageEl;
      });
      $pagesToRemove.each(pageEl => {
        const $oldPageEl = $(pageEl);
        const $oldNavbarEl = $(app.navbar.getElByPage($oldPageEl));
        // Page remove event
        router.pageCallback('beforeRemove', $oldPageEl, $oldNavbarEl, 'previous', undefined, {});
        router.removePage($oldPageEl);
        if (dynamicNavbar && $oldNavbarEl.length) {
          router.removeNavbar($oldNavbarEl);
        }
      });
    }
    function clearPreviousHistory() {
      const router = this;
      appRouterCheck(router, 'clearPreviousHistory');
      const url = router.history[router.history.length - 1];
      clearPreviousPages(router);
      router.history = [url];
      router.view.history = [url];
      router.saveHistory();
    }
     // eslint-disable-line

    class Router extends Framework7Class {
      constructor(app, view) {
        super({}, [typeof view === 'undefined' ? app : view]);
        const router = this;

        // Is App Router
        router.isAppRouter = typeof view === 'undefined';
        if (router.isAppRouter) {
          // App Router
          extend$3(false, router, {
            app,
            params: app.params.view,
            routes: app.routes || [],
            cache: app.cache
          });
        } else {
          // View Router
          extend$3(false, router, {
            app,
            view,
            viewId: view.id,
            id: view.params.routerId,
            params: view.params,
            routes: view.routes,
            history: view.history,
            propsHistory: [],
            scrollHistory: view.scrollHistory,
            cache: app.cache,
            dynamicNavbar: app.theme === 'ios' && view.params.iosDynamicNavbar,
            initialPages: [],
            initialNavbars: []
          });
        }

        // Install Modules
        router.useModules();

        // AllowPageChage
        router.allowPageChange = true;

        // Current Route
        let currentRoute = {};
        let previousRoute = {};
        Object.defineProperty(router, 'currentRoute', {
          enumerable: true,
          configurable: true,
          set(newRoute) {
            if (newRoute === void 0) {
              newRoute = {};
            }
            previousRoute = extend$3({}, currentRoute);
            currentRoute = newRoute;
            if (!currentRoute) return;
            router.url = currentRoute.url;
            router.emit('routeChange', newRoute, previousRoute, router);
          },
          get() {
            return currentRoute;
          }
        });
        Object.defineProperty(router, 'previousRoute', {
          enumerable: true,
          configurable: true,
          get() {
            return previousRoute;
          },
          set(newRoute) {
            previousRoute = newRoute;
          }
        });
        return router;
      }
      mount() {
        const router = this;
        const view = router.view;
        const document = getDocument$1();
        extend$3(false, router, {
          tempDom: document.createElement('div'),
          $el: view.$el,
          el: view.el,
          $navbarsEl: view.$navbarsEl,
          navbarsEl: view.navbarsEl
        });
        router.emit('local::mount routerMount', router);
      }
      animatableNavElements($newNavbarEl, $oldNavbarEl, toLarge, fromLarge, direction) {
        const router = this;
        const dynamicNavbar = router.dynamicNavbar;
        const animateIcon = router.params.iosAnimateNavbarBackIcon;
        let newNavEls;
        let oldNavEls;
        function animatableNavEl($el, $navbarInner) {
          const isSliding = $el.hasClass('sliding') || $navbarInner.hasClass('sliding');
          const isSubnavbar = $el.hasClass('subnavbar');
          const needsOpacityTransition = isSliding ? !isSubnavbar : true;
          const $iconEl = $el.find('.back .icon');
          let isIconLabel;
          if (isSliding && animateIcon && $el.hasClass('left') && $iconEl.length > 0 && $iconEl.next('span').length) {
            $el = $iconEl.next('span'); // eslint-disable-line
            isIconLabel = true;
          }
          return {
            $el,
            isIconLabel,
            leftOffset: $el[0].f7NavbarLeftOffset,
            rightOffset: $el[0].f7NavbarRightOffset,
            isSliding,
            isSubnavbar,
            needsOpacityTransition
          };
        }
        if (dynamicNavbar) {
          newNavEls = [];
          oldNavEls = [];
          $newNavbarEl.children('.navbar-inner').children('.left, .right, .title, .subnavbar').each(navEl => {
            const $navEl = $(navEl);
            if ($navEl.hasClass('left') && fromLarge && direction === 'forward') return;
            if ($navEl.hasClass('title') && toLarge) return;
            newNavEls.push(animatableNavEl($navEl, $newNavbarEl.children('.navbar-inner')));
          });
          if (!($oldNavbarEl.hasClass('navbar-master') && router.params.masterDetailBreakpoint > 0 && router.app.width >= router.params.masterDetailBreakpoint)) {
            $oldNavbarEl.children('.navbar-inner').children('.left, .right, .title, .subnavbar').each(navEl => {
              const $navEl = $(navEl);
              if ($navEl.hasClass('left') && toLarge && !fromLarge && direction === 'forward') return;
              if ($navEl.hasClass('left') && toLarge && direction === 'backward') return;
              if ($navEl.hasClass('title') && fromLarge) {
                return;
              }
              oldNavEls.push(animatableNavEl($navEl, $oldNavbarEl.children('.navbar-inner')));
            });
          }
          [oldNavEls, newNavEls].forEach(navEls => {
            navEls.forEach(navEl => {
              const n = navEl;
              const {
                isSliding,
                $el
              } = navEl;
              const otherEls = navEls === oldNavEls ? newNavEls : oldNavEls;
              if (!(isSliding && $el.hasClass('title') && otherEls)) return;
              otherEls.forEach(otherNavEl => {
                if (otherNavEl.isIconLabel) {
                  const iconTextEl = otherNavEl.$el[0];
                  n.leftOffset += iconTextEl ? iconTextEl.offsetLeft || 0 : 0;
                }
              });
            });
          });
        }
        return {
          newNavEls,
          oldNavEls
        };
      }
      animate($oldPageEl, $newPageEl, $oldNavbarEl, $newNavbarEl, direction, transition, callback) {
        const router = this;
        if (router.params.animateCustom) {
          router.params.animateCustom.apply(router, [$oldPageEl, $newPageEl, $oldNavbarEl, $newNavbarEl, direction, callback]);
          return;
        }
        const dynamicNavbar = router.dynamicNavbar;
        const ios = router.app.theme === 'ios';
        if (transition) {
          const routerCustomTransitionClass = `router-transition-custom router-transition-${transition}-${direction}`;
          // Animate
          const onCustomTransitionDone = () => {
            router.$el.removeClass(routerCustomTransitionClass);
            if (dynamicNavbar && router.$navbarsEl.length) {
              if ($newNavbarEl) {
                router.$navbarsEl.prepend($newNavbarEl);
              }
              if ($oldNavbarEl) {
                router.$navbarsEl.prepend($oldNavbarEl);
              }
            }
            if (callback) callback();
          };
          (direction === 'forward' ? $newPageEl : $oldPageEl).animationEnd(onCustomTransitionDone);
          if (dynamicNavbar) {
            if ($newNavbarEl && $newPageEl) {
              router.setNavbarPosition($newNavbarEl, '');
              $newNavbarEl.removeClass('navbar-next navbar-previous navbar-current');
              $newPageEl.prepend($newNavbarEl);
            }
            if ($oldNavbarEl && $oldPageEl) {
              router.setNavbarPosition($oldNavbarEl, '');
              $oldNavbarEl.removeClass('navbar-next navbar-previous navbar-current');
              $oldPageEl.prepend($oldNavbarEl);
            }
          }
          router.$el.addClass(routerCustomTransitionClass);
          return;
        }

        // Router Animation class
        const routerTransitionClass = `router-transition-${direction} router-transition`;
        let newNavEls;
        let oldNavEls;
        let fromLarge;
        let toLarge;
        let toDifferent;
        let oldIsLarge;
        let newIsLarge;
        if (ios && dynamicNavbar) {
          const betweenMasterAndDetail = router.params.masterDetailBreakpoint > 0 && router.app.width >= router.params.masterDetailBreakpoint && ($oldNavbarEl.hasClass('navbar-master') && $newNavbarEl.hasClass('navbar-master-detail') || $oldNavbarEl.hasClass('navbar-master-detail') && $newNavbarEl.hasClass('navbar-master'));
          if (!betweenMasterAndDetail) {
            oldIsLarge = $oldNavbarEl && $oldNavbarEl.hasClass('navbar-large');
            newIsLarge = $newNavbarEl && $newNavbarEl.hasClass('navbar-large');
            fromLarge = oldIsLarge && !$oldNavbarEl.hasClass('navbar-large-collapsed');
            toLarge = newIsLarge && !$newNavbarEl.hasClass('navbar-large-collapsed');
            toDifferent = fromLarge && !toLarge || toLarge && !fromLarge;
          }
          const navEls = router.animatableNavElements($newNavbarEl, $oldNavbarEl, toLarge, fromLarge, direction);
          newNavEls = navEls.newNavEls;
          oldNavEls = navEls.oldNavEls;
        }
        function animateNavbars(progress) {
          if (!(ios && dynamicNavbar)) return;
          if (progress === 1) {
            if (toLarge) {
              $newNavbarEl.addClass('router-navbar-transition-to-large');
              $oldNavbarEl.addClass('router-navbar-transition-to-large');
            }
            if (fromLarge) {
              $newNavbarEl.addClass('router-navbar-transition-from-large');
              $oldNavbarEl.addClass('router-navbar-transition-from-large');
            }
          }
          newNavEls.forEach(navEl => {
            const $el = navEl.$el;
            const offset = direction === 'forward' ? navEl.rightOffset : navEl.leftOffset;
            if (navEl.isSliding) {
              if (navEl.isSubnavbar && newIsLarge) {
                // prettier-ignore
                $el[0].style.setProperty('transform', `translate3d(${offset * (1 - progress)}px, calc(-1 * var(--f7-navbar-large-collapse-progress) * var(--f7-navbar-large-title-height)), 0)`, 'important');
              } else {
                $el.transform(`translate3d(${offset * (1 - progress)}px,0,0)`);
              }
            }
          });
          oldNavEls.forEach(navEl => {
            const $el = navEl.$el;
            const offset = direction === 'forward' ? navEl.leftOffset : navEl.rightOffset;
            if (navEl.isSliding) {
              if (navEl.isSubnavbar && oldIsLarge) {
                $el.transform(`translate3d(${offset * progress}px, calc(-1 * var(--f7-navbar-large-collapse-progress) * var(--f7-navbar-large-title-height)), 0)`);
              } else {
                $el.transform(`translate3d(${offset * progress}px,0,0)`);
              }
            }
          });
        }

        // AnimationEnd Callback
        function onDone() {
          if (router.dynamicNavbar) {
            if ($newNavbarEl) {
              $newNavbarEl.removeClass('router-navbar-transition-to-large router-navbar-transition-from-large');
              $newNavbarEl.addClass('navbar-no-title-large-transition');
              nextFrame$1(() => {
                $newNavbarEl.removeClass('navbar-no-title-large-transition');
              });
            }
            if ($oldNavbarEl) {
              $oldNavbarEl.removeClass('router-navbar-transition-to-large router-navbar-transition-from-large');
            }
            if ($newNavbarEl.hasClass('sliding') || $newNavbarEl.children('.navbar-inner.sliding').length) {
              $newNavbarEl.find('.title, .left, .right, .left .icon, .subnavbar').transform('');
            } else {
              $newNavbarEl.find('.sliding').transform('');
            }
            if ($oldNavbarEl.hasClass('sliding') || $oldNavbarEl.children('.navbar-inner.sliding').length) {
              $oldNavbarEl.find('.title, .left, .right, .left .icon, .subnavbar').transform('');
            } else {
              $oldNavbarEl.find('.sliding').transform('');
            }
          }
          router.$el.removeClass(routerTransitionClass);
          if (callback) callback();
        }

        // eslint-disable-next-line
        (direction === 'forward' ? $newPageEl : ios ? $oldPageEl : $newPageEl).animationEnd(() => {
          onDone();
        });

        // Animate
        if (dynamicNavbar) {
          // Prepare Navbars
          animateNavbars(0);
          nextFrame$1(() => {
            // Add class, start animation
            router.$el.addClass(routerTransitionClass);
            if (toDifferent) {
              // eslint-disable-next-line
              router.el._clientLeft = router.el.clientLeft;
            }
            animateNavbars(1);
          });
        } else {
          // Add class, start animation
          router.$el.addClass(routerTransitionClass);
        }
      }
      removeModal(modalEl) {
        const router = this;
        router.removeEl(modalEl);
      }
      // eslint-disable-next-line
      removeTabContent(tabEl) {
        const $tabEl = $(tabEl);
        $tabEl.html('');
      }
      removeNavbar(el) {
        const router = this;
        router.removeEl(el);
      }
      removePage(el) {
        const $el = $(el);
        const f7Page = $el && $el[0] && $el[0].f7Page;
        const router = this;
        if (f7Page && f7Page.route && f7Page.route.route && f7Page.route.route.keepAlive) {
          $el.remove();
          return;
        }
        router.removeEl(el);
      }
      removeEl(el) {
        if (!el) return;
        const router = this;
        const $el = $(el);
        if ($el.length === 0) return;
        $el.find('.tab').each(tabEl => {
          $(tabEl).children().each(tabChild => {
            if (tabChild.f7Component) {
              $(tabChild).trigger('tab:beforeremove');
              tabChild.f7Component.destroy();
            }
          });
        });
        if ($el[0].f7Component && $el[0].f7Component.destroy) {
          $el[0].f7Component.destroy();
        }
        if (!router.params.removeElements) {
          return;
        }
        if (router.params.removeElementsWithTimeout) {
          setTimeout(() => {
            $el.remove();
          }, router.params.removeElementsTimeout);
        } else {
          $el.remove();
        }
      }
      getPageEl(content) {
        const router = this;
        if (typeof content === 'string') {
          router.tempDom.innerHTML = content;
        } else {
          if ($(content).hasClass('page')) {
            return content;
          }
          router.tempDom.innerHTML = '';
          $(router.tempDom).append(content);
        }
        return router.findElement('.page', router.tempDom);
      }
      findElement(stringSelector, container) {
        const router = this;
        const view = router.view;
        const app = router.app;

        // Modals Selector
        const modalsSelector = '.popup, .dialog, .popover, .actions-modal, .sheet-modal, .login-screen, .page';
        const $container = $(container);
        const selector = stringSelector;
        let found = $container.find(selector).filter(el => $(el).parents(modalsSelector).length === 0);
        if (found.length > 1) {
          if (typeof view.selector === 'string') {
            // Search in related view
            found = $container.find(`${view.selector} ${selector}`);
          }
          if (found.length > 1) {
            // Search in main view
            found = $container.find(`.${app.params.viewMainClass} ${selector}`);
          }
        }
        if (found.length === 1) return found;
        found = router.findElement(selector, $container);
        if (found && found.length === 1) return found;
        if (found && found.length > 1) return $(found[0]);
        return undefined;
      }
      flattenRoutes(routes) {
        if (routes === void 0) {
          routes = this.routes;
        }
        const router = this;
        let flattenedRoutes = [];
        routes.forEach(route => {
          let hasTabRoutes = false;
          if ('tabs' in route && route.tabs) {
            const mergedPathsRoutes = route.tabs.map(tabRoute => {
              const tRoute = extend$3({}, route, {
                path: `${route.path}/${tabRoute.path}`.replace('///', '/').replace('//', '/'),
                parentPath: route.path,
                tab: tabRoute
              });
              delete tRoute.tabs;
              delete tRoute.routes;
              return tRoute;
            });
            hasTabRoutes = true;
            flattenedRoutes = flattenedRoutes.concat(router.flattenRoutes(mergedPathsRoutes));
          }
          if ('detailRoutes' in route) {
            const mergedPathsRoutes = route.detailRoutes.map(detailRoute => {
              const dRoute = extend$3({}, detailRoute);
              dRoute.masterRoute = route;
              dRoute.masterRoutePath = route.path;
              return dRoute;
            });
            flattenedRoutes = flattenedRoutes.concat(route, router.flattenRoutes(mergedPathsRoutes));
          }
          if ('routes' in route) {
            const mergedPathsRoutes = route.routes.map(childRoute => {
              const cRoute = extend$3({}, childRoute);
              cRoute.path = `${route.path}/${cRoute.path}`.replace('///', '/').replace('//', '/');
              return cRoute;
            });
            if (hasTabRoutes) {
              flattenedRoutes = flattenedRoutes.concat(router.flattenRoutes(mergedPathsRoutes));
            } else {
              flattenedRoutes = flattenedRoutes.concat(route, router.flattenRoutes(mergedPathsRoutes));
            }
          }
          if (!('routes' in route) && !('tabs' in route && route.tabs) && !('detailRoutes' in route)) {
            flattenedRoutes.push(route);
          }
        });
        return flattenedRoutes;
      }

      // eslint-disable-next-line
      parseRouteUrl(url) {
        if (!url) return {};
        const query = parseUrlQuery(url);
        const hash = url.split('#')[1];
        const params = {};
        const path = url.split('#')[0].split('?')[0];
        return {
          query,
          hash,
          params,
          url,
          path
        };
      }
      generateUrl(parameters) {
        if (parameters === void 0) {
          parameters = {};
        }
        if (typeof parameters === 'string') {
          return parameters;
        }
        const {
          name,
          path,
          params,
          query
        } = parameters;
        if (!name && !path) {
          throw new Error('Framework7: "name" or "path" parameter is required');
        }
        const router = this;
        const route = name ? router.findRouteByKey('name', name) : router.findRouteByKey('path', path);
        if (!route) {
          if (name) {
            throw new Error(`Framework7: route with name "${name}" not found`);
          } else {
            throw new Error(`Framework7: route with path "${path}" not found`);
          }
        }
        const url = router.constructRouteUrl(route, {
          params,
          query
        });
        if (url === '') {
          return '/';
        }
        if (!url) {
          throw new Error(`Framework7: can't construct URL for route with name "${name}"`);
        }
        return url;
      }

      // eslint-disable-next-line
      constructRouteUrl(route, _temp) {
        let {
          params,
          query
        } = _temp === void 0 ? {} : _temp;
        const {
          path
        } = route;
        const toUrl = compile(path);
        let url;
        try {
          url = toUrl(params || {});
        } catch (error) {
          throw new Error(`Framework7: error constructing route URL from passed params:\nRoute: ${path}\n${error.toString()}`);
        }
        if (query) {
          if (typeof query === 'string') url += `?${query}`;else if (Object.keys(query).length) url += `?${serializeObject(query)}`;
        }
        return url;
      }
      findTabRouteUrl(tabEl) {
        const router = this;
        const $tabEl = $(tabEl);
        const parentPath = router.currentRoute.route.parentPath;
        const tabId = $tabEl.attr('id');
        const flattenedRoutes = router.flattenRoutes(router.routes);
        let foundTabRouteUrl;
        flattenedRoutes.forEach(route => {
          if (route.parentPath === parentPath && route.tab && route.tab.id === tabId) {
            if (router.currentRoute.params && Object.keys(router.currentRoute.params).length > 0) {
              foundTabRouteUrl = router.constructRouteUrl(route, {
                params: router.currentRoute.params,
                query: router.currentRoute.query
              });
            } else {
              foundTabRouteUrl = route.path;
            }
          }
        });
        return foundTabRouteUrl;
      }
      findRouteByKey(key, value) {
        const router = this;
        const routes = router.routes;
        const flattenedRoutes = router.flattenRoutes(routes);
        let matchingRoute;
        flattenedRoutes.forEach(route => {
          if (matchingRoute) return;
          if (route[key] === value) {
            matchingRoute = route;
          }
        });
        return matchingRoute;
      }
      findMatchingRoute(url) {
        if (!url) return undefined;
        const router = this;
        const routes = router.routes;
        const flattenedRoutes = router.flattenRoutes(routes);
        const {
          path,
          query,
          hash,
          params
        } = router.parseRouteUrl(url);
        let matchingRoute;
        flattenedRoutes.forEach(route => {
          if (matchingRoute) return;
          const keys = [];
          const pathsToMatch = [route.path || '/'];
          if (route.alias) {
            if (typeof route.alias === 'string') pathsToMatch.push(route.alias);else if (Array.isArray(route.alias)) {
              route.alias.forEach(aliasPath => {
                pathsToMatch.push(aliasPath);
              });
            }
          }
          let matched;
          pathsToMatch.forEach(pathToMatch => {
            if (matched) return;
            matched = pathToRegexp(pathToMatch, keys).exec(path || '/');
          });
          if (matched) {
            keys.forEach((keyObj, index) => {
              if (typeof keyObj.name === 'number') return;
              const paramValue = matched[index + 1];
              if (typeof paramValue === 'undefined' || paramValue === null) {
                params[keyObj.name] = paramValue;
              } else {
                params[keyObj.name] = decodeURIComponent(paramValue);
              }
            });
            let parentPath;
            if (route.parentPath) {
              parentPath = (path || '/').split('/').slice(0, route.parentPath.split('/').length - 1).join('/');
            }
            matchingRoute = {
              query,
              hash,
              params,
              url,
              path: path || '/',
              parentPath,
              route,
              name: route.name
            };
          }
        });
        return matchingRoute;
      }

      // eslint-disable-next-line
      replaceRequestUrlParams(url, options) {
        if (url === void 0) {
          url = '';
        }
        if (options === void 0) {
          options = {};
        }
        let compiledUrl = url;
        if (typeof compiledUrl === 'string' && compiledUrl.indexOf('{{') >= 0 && options && options.route && options.route.params && Object.keys(options.route.params).length) {
          Object.keys(options.route.params).forEach(paramName => {
            const regExp = new RegExp(`{{${paramName}}}`, 'g');
            compiledUrl = compiledUrl.replace(regExp, options.route.params[paramName] || '');
          });
        }
        return compiledUrl;
      }
      removeFromXhrCache(url) {
        const router = this;
        const xhrCache = router.cache.xhr;
        let index = false;
        for (let i = 0; i < xhrCache.length; i += 1) {
          if (xhrCache[i].url === url) index = i;
        }
        if (index !== false) xhrCache.splice(index, 1);
      }
      xhrRequest(requestUrl, options) {
        const router = this;
        const params = router.params;
        const {
          ignoreCache
        } = options;
        let url = requestUrl;
        let hasQuery = url.indexOf('?') >= 0;
        if (params.passRouteQueryToRequest && options && options.route && options.route.query && Object.keys(options.route.query).length) {
          url += `${hasQuery ? '&' : '?'}${serializeObject(options.route.query)}`;
          hasQuery = true;
        }
        if (params.passRouteParamsToRequest && options && options.route && options.route.params && Object.keys(options.route.params).length) {
          url += `${hasQuery ? '&' : '?'}${serializeObject(options.route.params)}`;
          hasQuery = true;
        }
        if (url.indexOf('{{') >= 0) {
          url = router.replaceRequestUrlParams(url, options);
        }
        // should we ignore get params or not
        if (params.xhrCacheIgnoreGetParameters && url.indexOf('?') >= 0) {
          url = url.split('?')[0];
        }
        return new Promise((resolve, reject) => {
          if (params.xhrCache && !ignoreCache && url.indexOf('nocache') < 0 && params.xhrCacheIgnore.indexOf(url) < 0) {
            for (let i = 0; i < router.cache.xhr.length; i += 1) {
              const cachedUrl = router.cache.xhr[i];
              if (cachedUrl.url === url) {
                // Check expiration
                if (now$1() - cachedUrl.time < params.xhrCacheDuration) {
                  // Load from cache
                  resolve(cachedUrl.content);
                  return;
                }
              }
            }
          }
          router.xhrAbortController = new AbortController();
          let fetchRes;
          fetch(url, {
            signal: router.xhrAbortController.signal,
            method: 'GET'
          }).then(res => {
            fetchRes = res;
            return res.text();
          }).then(responseText => {
            const {
              status
            } = fetchRes;
            router.emit('routerAjaxComplete', fetchRes);
            if (status !== 'error' && status !== 'timeout' && status >= 200 && status < 300 || status === 0) {
              if (params.xhrCache && responseText !== '') {
                router.removeFromXhrCache(url);
                router.cache.xhr.push({
                  url,
                  time: now$1(),
                  content: responseText
                });
              }
              router.emit('routerAjaxSuccess', fetchRes, options);
              resolve(responseText);
            } else {
              router.emit('routerAjaxError', fetchRes, options);
              reject(fetchRes);
            }
          }).catch(err => {
            reject(err);
          });
        });
      }
      setNavbarPosition($el, position, ariaHidden) {
        const router = this;
        $el.removeClass('navbar-previous navbar-current navbar-next');
        if (position) {
          $el.addClass(`navbar-${position}`);
        }
        if (ariaHidden === false) {
          $el.removeAttr('aria-hidden');
        } else if (ariaHidden === true) {
          $el.attr('aria-hidden', 'true');
        }
        $el.trigger('navbar:position', {
          position
        });
        router.emit('navbarPosition', $el[0], position);
      }
      setPagePosition($el, position, ariaHidden) {
        const router = this;
        $el.removeClass('page-previous page-current page-next');
        $el.addClass(`page-${position}`);
        if (ariaHidden === false) {
          $el.removeAttr('aria-hidden');
        } else if (ariaHidden === true) {
          $el.attr('aria-hidden', 'true');
        }
        $el.trigger('page:position', {
          position
        });
        router.emit('pagePosition', $el[0], position);
      }

      // Remove theme elements
      removeThemeElements(el) {
        const router = this;
        const theme = router.app.theme;
        let toRemove;
        if (theme === 'ios') {
          toRemove = '.md-only, .if-md, .if-not-ios, .not-ios';
        } else if (theme === 'md') {
          toRemove = '.ios-only, .if-ios, .if-not-md, .not-md';
        }
        $(el).find(toRemove).remove();
      }
      getPageData(pageEl, navbarEl, from, to, route, pageFromEl) {
        if (route === void 0) {
          route = {};
        }
        const router = this;
        const $pageEl = $(pageEl).eq(0);
        const $navbarEl = $(navbarEl).eq(0);
        const currentPage = $pageEl[0].f7Page || {};
        let direction;
        let pageFrom;
        if (from === 'next' && to === 'current' || from === 'current' && to === 'previous') direction = 'forward';
        if (from === 'current' && to === 'next' || from === 'previous' && to === 'current') direction = 'backward';
        if (currentPage && !currentPage.fromPage) {
          const $pageFromEl = $(pageFromEl);
          if ($pageFromEl.length) {
            pageFrom = $pageFromEl[0].f7Page;
          }
        }
        pageFrom = currentPage.pageFrom || pageFrom;
        if (pageFrom && pageFrom.pageFrom) {
          pageFrom.pageFrom = null;
        }
        const page = {
          app: router.app,
          view: router.view,
          router,
          $el: $pageEl,
          el: $pageEl[0],
          $pageEl,
          pageEl: $pageEl[0],
          $navbarEl,
          navbarEl: $navbarEl[0],
          name: $pageEl.attr('data-name'),
          position: from,
          from,
          to,
          direction,
          route: currentPage.route ? currentPage.route : route,
          pageFrom
        };
        $pageEl[0].f7Page = page;
        return page;
      }

      // Callbacks
      pageCallback(callback, pageEl, navbarEl, from, to, options, pageFromEl) {
        if (options === void 0) {
          options = {};
        }
        if (!pageEl) return;
        const router = this;
        const $pageEl = $(pageEl);
        if (!$pageEl.length) return;
        const $navbarEl = $(navbarEl);
        const {
          route
        } = options;
        const restoreScrollTopOnBack = router.params.restoreScrollTopOnBack && !(router.params.masterDetailBreakpoint > 0 && $pageEl.hasClass('page-master') && router.app.width >= router.params.masterDetailBreakpoint);
        const keepAlive = $pageEl[0].f7Page && $pageEl[0].f7Page.route && $pageEl[0].f7Page.route.route && $pageEl[0].f7Page.route.route.keepAlive;
        if (callback === 'beforeRemove' && keepAlive) {
          callback = 'beforeUnmount'; // eslint-disable-line
        }

        const camelName = `page${callback[0].toUpperCase() + callback.slice(1, callback.length)}`;
        const colonName = `page:${callback.toLowerCase()}`;
        let page = {};
        if (callback === 'beforeRemove' && $pageEl[0].f7Page) {
          page = extend$3($pageEl[0].f7Page, {
            from,
            to,
            position: from
          });
        } else {
          page = router.getPageData($pageEl[0], $navbarEl[0], from, to, route, pageFromEl);
        }
        page.swipeBack = !!options.swipeBack;
        const {
          on = {},
          once = {}
        } = options.route ? options.route.route : {};
        if (options.on) {
          extend$3(on, options.on);
        }
        if (options.once) {
          extend$3(once, options.once);
        }
        function attachEvents() {
          if ($pageEl[0].f7RouteEventsAttached) return;
          $pageEl[0].f7RouteEventsAttached = true;
          if (on && Object.keys(on).length > 0) {
            $pageEl[0].f7RouteEventsOn = on;
            Object.keys(on).forEach(eventName => {
              on[eventName] = on[eventName].bind(router);
              $pageEl.on(eventNameToColonCase(eventName), on[eventName]);
            });
          }
          if (once && Object.keys(once).length > 0) {
            $pageEl[0].f7RouteEventsOnce = once;
            Object.keys(once).forEach(eventName => {
              once[eventName] = once[eventName].bind(router);
              $pageEl.once(eventNameToColonCase(eventName), once[eventName]);
            });
          }
        }
        function detachEvents() {
          if (!$pageEl[0].f7RouteEventsAttached) return;
          if ($pageEl[0].f7RouteEventsOn) {
            Object.keys($pageEl[0].f7RouteEventsOn).forEach(eventName => {
              $pageEl.off(eventNameToColonCase(eventName), $pageEl[0].f7RouteEventsOn[eventName]);
            });
          }
          if ($pageEl[0].f7RouteEventsOnce) {
            Object.keys($pageEl[0].f7RouteEventsOnce).forEach(eventName => {
              $pageEl.off(eventNameToColonCase(eventName), $pageEl[0].f7RouteEventsOnce[eventName]);
            });
          }
          $pageEl[0].f7RouteEventsAttached = null;
          $pageEl[0].f7RouteEventsOn = null;
          $pageEl[0].f7RouteEventsOnce = null;
          delete $pageEl[0].f7RouteEventsAttached;
          delete $pageEl[0].f7RouteEventsOn;
          delete $pageEl[0].f7RouteEventsOnce;
        }
        if (callback === 'mounted') {
          attachEvents();
        }
        if (callback === 'init') {
          if (restoreScrollTopOnBack && (from === 'previous' || !from) && to === 'current' && router.scrollHistory[page.route.url] && !$pageEl.hasClass('no-restore-scroll')) {
            let $pageContent = $pageEl.find('.page-content');
            if ($pageContent.length > 0) {
              // eslint-disable-next-line
              $pageContent = $pageContent.filter(pageContentEl => {
                return $(pageContentEl).parents('.tab:not(.tab-active)').length === 0 && !$(pageContentEl).is('.tab:not(.tab-active)');
              });
            }
            $pageContent.scrollTop(router.scrollHistory[page.route.url]);
          }
          attachEvents();
          if ($pageEl[0].f7PageInitialized) {
            $pageEl.trigger('page:reinit', page);
            router.emit('pageReinit', page);
            return;
          }
          $pageEl[0].f7PageInitialized = true;
        }
        if (restoreScrollTopOnBack && callback === 'beforeOut' && from === 'current' && to === 'previous') {
          // Save scroll position
          let $pageContent = $pageEl.find('.page-content');
          if ($pageContent.length > 0) {
            // eslint-disable-next-line
            $pageContent = $pageContent.filter(pageContentEl => {
              return $(pageContentEl).parents('.tab:not(.tab-active)').length === 0 && !$(pageContentEl).is('.tab:not(.tab-active)');
            });
          }
          router.scrollHistory[page.route.url] = $pageContent.scrollTop();
        }
        if (restoreScrollTopOnBack && callback === 'beforeOut' && from === 'current' && to === 'next') {
          // Delete scroll position
          delete router.scrollHistory[page.route.url];
        }
        $pageEl.trigger(colonName, page);
        router.emit(camelName, page);
        if (callback === 'beforeRemove' || callback === 'beforeUnmount') {
          detachEvents();
          if (!keepAlive) {
            if ($pageEl[0].f7Page && $pageEl[0].f7Page.navbarEl) {
              delete $pageEl[0].f7Page.navbarEl.f7Page;
            }
            $pageEl[0].f7Page = null;
          }
        }
      }
      saveHistory() {
        const router = this;
        const window = getWindow$1();
        router.view.history = router.history;
        if (router.params.browserHistory && router.params.browserHistoryStoreHistory && window.localStorage) {
          window.localStorage[`f7router-${router.view.id}-history`] = JSON.stringify(router.history);
        }
      }
      restoreHistory() {
        const router = this;
        const window = getWindow$1();
        if (router.params.browserHistory && router.params.browserHistoryStoreHistory && window.localStorage && window.localStorage[`f7router-${router.view.id}-history`]) {
          router.history = JSON.parse(window.localStorage[`f7router-${router.view.id}-history`]);
          router.view.history = router.history;
        }
      }
      clearHistory() {
        const router = this;
        router.history = [];
        if (router.view) router.view.history = [];
        router.saveHistory();
      }
      updateCurrentUrl(newUrl) {
        const router = this;
        appRouterCheck(router, 'updateCurrentUrl');
        // Update history
        if (router.history.length) {
          router.history[router.history.length - 1] = newUrl;
        } else {
          router.history.push(newUrl);
        }

        // Update current route params
        const {
          query,
          hash,
          params,
          url,
          path
        } = router.parseRouteUrl(newUrl);
        if (router.currentRoute) {
          extend$3(router.currentRoute, {
            query,
            hash,
            params,
            url,
            path
          });
        }
        if (router.params.browserHistory) {
          const browserHistoryRoot = router.params.browserHistoryRoot || '';
          History$1.replace(router.view.id, {
            url: newUrl
          }, browserHistoryRoot + router.params.browserHistorySeparator + newUrl);
        }

        // Save History
        router.saveHistory();
        router.emit('routeUrlUpdate', router.currentRoute, router);
      }
      getInitialUrl() {
        const router = this;
        if (router.initialUrl) {
          return {
            initialUrl: router.initialUrl,
            historyRestored: router.historyRestored
          };
        }
        const {
          app,
          view
        } = router;
        const document = getDocument$1();
        const window = getWindow$1();
        const location = app.params.url && typeof app.params.url === 'string' && typeof URL !== 'undefined' ? new URL(app.params.url) : document.location;
        let initialUrl = router.params.url;
        let documentUrl = location.href.split(location.origin)[1];
        let historyRestored;
        const {
          browserHistory,
          browserHistoryOnLoad,
          browserHistorySeparator
        } = router.params;
        let {
          browserHistoryRoot
        } = router.params;
        if ((window.cordova || window.Capacitor && window.Capacitor.isNative) && browserHistory && !browserHistorySeparator && !browserHistoryRoot && location.pathname.indexOf('index.html')) {
          // eslint-disable-next-line
          console.warn('Framework7: wrong or not complete browserHistory configuration, trying to guess browserHistoryRoot');
          browserHistoryRoot = location.pathname.split('index.html')[0];
        }
        if (!browserHistory || !browserHistoryOnLoad) {
          if (!initialUrl) {
            initialUrl = documentUrl;
          }
          if (location.search && initialUrl.indexOf('?') < 0) {
            initialUrl += location.search;
          }
          if (location.hash && initialUrl.indexOf('#') < 0) {
            initialUrl += location.hash;
          }
        } else {
          if (browserHistoryRoot && documentUrl.indexOf(browserHistoryRoot) >= 0) {
            documentUrl = documentUrl.substring(documentUrl.indexOf(browserHistoryRoot) + browserHistoryRoot.length);
            if (documentUrl === '') documentUrl = '/';
          }
          if (browserHistorySeparator.length > 0 && documentUrl.indexOf(browserHistorySeparator) >= 0) {
            initialUrl = documentUrl.substring(documentUrl.indexOf(browserHistorySeparator) + browserHistorySeparator.length);
          } else {
            initialUrl = documentUrl;
          }
          router.restoreHistory();
          if (router.history.indexOf(initialUrl) >= 0) {
            router.history = router.history.slice(0, router.history.indexOf(initialUrl) + 1);
          } else if (router.params.url === initialUrl) {
            router.history = [initialUrl];
          } else if (History$1.state && History$1.state[view.id] && History$1.state[view.id].url === router.history[router.history.length - 1]) {
            initialUrl = router.history[router.history.length - 1];
          } else {
            router.history = [documentUrl.split(browserHistorySeparator)[0] || '/', initialUrl];
          }
          if (router.history.length > 1) {
            historyRestored = true;
          } else {
            router.history = [];
          }
          router.saveHistory();
        }
        router.initialUrl = initialUrl;
        router.historyRestored = historyRestored;
        return {
          initialUrl,
          historyRestored
        };
      }
      init() {
        const router = this;
        const {
          app,
          view
        } = router;
        const document = getDocument$1();
        router.mount();
        const {
          initialUrl,
          historyRestored
        } = router.getInitialUrl();

        // Init Swipeback
        if (view && router.params.iosSwipeBack && app.theme === 'ios' || view && router.params.mdSwipeBack && app.theme === 'md') {
          SwipeBack(router);
        }
        const {
          browserHistory,
          browserHistoryOnLoad,
          browserHistoryAnimateOnLoad,
          browserHistoryInitialMatch
        } = router.params;
        let currentRoute;
        if (router.history.length > 1) {
          // Will load page
          const initUrl = browserHistoryInitialMatch ? initialUrl : router.history[0];
          currentRoute = router.findMatchingRoute(initUrl);
          if (!currentRoute) {
            currentRoute = extend$3(router.parseRouteUrl(initUrl), {
              route: {
                url: initUrl,
                path: initUrl.split('?')[0]
              }
            });
          }
        } else {
          // Don't load page
          currentRoute = router.findMatchingRoute(initialUrl);
          if (!currentRoute) {
            currentRoute = extend$3(router.parseRouteUrl(initialUrl), {
              route: {
                url: initialUrl,
                path: initialUrl.split('?')[0]
              }
            });
          }
        }
        if (router.$el.children('.page').length === 0 && initialUrl && router.params.loadInitialPage) {
          // No pages presented in DOM, reload new page
          router.navigate(initialUrl, {
            initial: true,
            reloadCurrent: true,
            browserHistory: false,
            animate: false,
            once: {
              modalOpen() {
                if (!historyRestored) return;
                const preloadPreviousPage = router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`];
                if (preloadPreviousPage && router.history.length > 1) {
                  router.back({
                    preload: true
                  });
                }
              },
              pageAfterIn() {
                if (!historyRestored) return;
                const preloadPreviousPage = router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`];
                if (preloadPreviousPage && router.history.length > 1) {
                  router.back({
                    preload: true
                  });
                }
              }
            }
          });
        } else if (router.$el.children('.page').length) {
          // Init current DOM page
          let hasTabRoute;
          router.currentRoute = currentRoute;
          router.$el.children('.page').each(pageEl => {
            const $pageEl = $(pageEl);
            let $navbarEl;
            router.setPagePosition($pageEl, 'current');
            if (router.dynamicNavbar) {
              $navbarEl = $pageEl.children('.navbar');
              if ($navbarEl.length > 0) {
                if (!router.$navbarsEl.parents(document).length) {
                  router.$el.prepend(router.$navbarsEl);
                }
                router.setNavbarPosition($navbarEl, 'current');
                router.$navbarsEl.append($navbarEl);
                if ($navbarEl.children('.title-large').length) {
                  $navbarEl.addClass('navbar-large');
                }
                $pageEl.children('.navbar').remove();
              } else {
                router.$navbarsEl.addClass('navbar-hidden');
                if ($navbarEl.children('.title-large').length) {
                  router.$navbarsEl.addClass('navbar-hidden navbar-large-hidden');
                }
              }
            }
            if (router.currentRoute && router.currentRoute.route && (router.currentRoute.route.master === true || typeof router.currentRoute.route.master === 'function' && router.currentRoute.route.master(app, router)) && router.params.masterDetailBreakpoint > 0) {
              $pageEl.addClass('page-master');
              $pageEl.trigger('page:role', {
                role: 'master'
              });
              if ($navbarEl && $navbarEl.length) {
                $navbarEl.addClass('navbar-master');
              }
              view.checkMasterDetailBreakpoint();
            }
            const initOptions = {
              route: router.currentRoute
            };
            if (router.currentRoute && router.currentRoute.route && router.currentRoute.route.options) {
              extend$3(initOptions, router.currentRoute.route.options);
            }
            router.currentPageEl = $pageEl[0];
            if (router.dynamicNavbar && $navbarEl.length) {
              router.currentNavbarEl = $navbarEl[0];
            }
            router.removeThemeElements($pageEl);
            if (router.dynamicNavbar && $navbarEl.length) {
              router.removeThemeElements($navbarEl);
            }
            if (initOptions.route.route.tab) {
              hasTabRoute = true;
              router.tabLoad(initOptions.route.route.tab, extend$3({}, initOptions));
            }
            router.pageCallback('init', $pageEl, $navbarEl, 'current', undefined, initOptions);
            router.pageCallback('beforeIn', $pageEl, $navbarEl, 'current', undefined, initOptions);
            router.pageCallback('afterIn', $pageEl, $navbarEl, 'current', undefined, initOptions);
          });
          if (historyRestored) {
            if (browserHistoryInitialMatch) {
              const preloadPreviousPage = router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`];
              if (preloadPreviousPage && router.history.length > 1) {
                router.back({
                  preload: true
                });
              }
            } else {
              router.navigate(initialUrl, {
                initial: true,
                browserHistory: false,
                history: false,
                animate: browserHistoryAnimateOnLoad,
                once: {
                  pageAfterIn() {
                    const preloadPreviousPage = router.params.preloadPreviousPage || router.params[`${app.theme}SwipeBack`];
                    if (preloadPreviousPage && router.history.length > 2) {
                      router.back({
                        preload: true
                      });
                    }
                  }
                }
              });
            }
          }
          if (!historyRestored && !hasTabRoute) {
            router.history.push(initialUrl);
            router.saveHistory();
          }
        }
        if (initialUrl && browserHistory && browserHistoryOnLoad && (!History$1.state || !History$1.state[view.id])) {
          History$1.initViewState(view.id, {
            url: initialUrl
          });
        }
        router.emit('local::init routerInit', router);
      }
      destroy() {
        let router = this;
        router.emit('local::destroy routerDestroy', router);

        // Delete props & methods
        Object.keys(router).forEach(routerProp => {
          router[routerProp] = null;
          delete router[routerProp];
        });
        router = null;
      }
    }

    // Load
    Router.prototype.navigate = navigate;
    Router.prototype.refreshPage = refreshPage;
    // Tab
    Router.prototype.tabLoad = tabLoad;
    Router.prototype.tabRemove = tabRemove;
    // Modal
    Router.prototype.modalLoad = modalLoad;
    Router.prototype.modalRemove = modalRemove;
    // Back
    Router.prototype.back = back;
    // Clear history
    Router.prototype.clearPreviousHistory = clearPreviousHistory;

    var RouterModule = {
      name: 'router',
      static: {
        Router
      },
      instance: {
        cache: {
          xhr: [],
          templates: [],
          components: []
        }
      },
      create() {
        const instance = this;
        if (instance.app) {
          // View Router
          if (instance.params.router) {
            instance.router = new Router(instance.app, instance);
          }
        } else {
          // App Router
          instance.router = new Router(instance);
        }
      }
    };

    function resizableView(view) {
      const app = view.app;
      const support = getSupport$1();
      if (view.resizableInitialized) return;
      extend$3(view, {
        resizable: true,
        resizableWidth: null,
        resizableInitialized: true
      });
      const $htmlEl = $('html');
      const {
        $el
      } = view;
      if (!$el) return;
      let $resizeHandlerEl;
      let isTouched;
      let isMoved;
      const touchesStart = {};
      let touchesDiff;
      let width;
      let minWidth;
      let maxWidth;
      function transformCSSWidth(v) {
        if (!v) return null;
        if (v.indexOf('%') >= 0 || v.indexOf('vw') >= 0) {
          return parseInt(v, 10) / 100 * app.width;
        }
        const newV = parseInt(v, 10);
        if (Number.isNaN(newV)) return null;
        return newV;
      }
      function isResizable() {
        return view.resizable && $el.hasClass('view-resizable') && $el.hasClass('view-master-detail');
      }
      function handleTouchStart(e) {
        if (!isResizable()) return;
        touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
        touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
        isMoved = false;
        isTouched = true;
        const $pageMasterEl = $el.children('.page-master');
        minWidth = transformCSSWidth($pageMasterEl.css('min-width'));
        maxWidth = transformCSSWidth($pageMasterEl.css('max-width'));
      }
      function handleTouchMove(e) {
        if (!isTouched) return;
        e.f7PreventSwipePanel = true;
        const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
        if (!isMoved) {
          width = $resizeHandlerEl[0].offsetLeft + $resizeHandlerEl[0].offsetWidth;
          $el.addClass('view-resizing');
          $htmlEl.css('cursor', 'col-resize');
        }
        isMoved = true;
        e.preventDefault();
        touchesDiff = pageX - touchesStart.x;
        let newWidth = width + touchesDiff;
        if (minWidth && !Number.isNaN(minWidth)) {
          newWidth = Math.max(newWidth, minWidth);
        }
        if (maxWidth && !Number.isNaN(maxWidth)) {
          newWidth = Math.min(newWidth, maxWidth);
        }
        newWidth = Math.min(Math.max(newWidth, 0), app.width);
        view.resizableWidth = newWidth;
        $htmlEl[0].style.setProperty('--f7-page-master-width', `${newWidth}px`);
        $el.trigger('view:resize', newWidth);
        view.emit('local::resize viewResize', view, newWidth);
      }
      function handleTouchEnd() {
        $('html').css('cursor', '');
        if (!isTouched || !isMoved) {
          isTouched = false;
          isMoved = false;
          return;
        }
        isTouched = false;
        isMoved = false;
        $htmlEl[0].style.setProperty('--f7-page-master-width', `${view.resizableWidth}px`);
        $el.removeClass('view-resizing');
      }
      function handleResize() {
        if (!view.resizableWidth) return;
        minWidth = transformCSSWidth($resizeHandlerEl.css('min-width'));
        maxWidth = transformCSSWidth($resizeHandlerEl.css('max-width'));
        if (minWidth && !Number.isNaN(minWidth) && view.resizableWidth < minWidth) {
          view.resizableWidth = Math.max(view.resizableWidth, minWidth);
        }
        if (maxWidth && !Number.isNaN(maxWidth) && view.resizableWidth > maxWidth) {
          view.resizableWidth = Math.min(view.resizableWidth, maxWidth);
        }
        view.resizableWidth = Math.min(Math.max(view.resizableWidth, 0), app.width);
        $htmlEl[0].style.setProperty('--f7-page-master-width', `${view.resizableWidth}px`);
      }
      $resizeHandlerEl = view.$el.children('.view-resize-handler');
      if (!$resizeHandlerEl.length) {
        view.$el.append('<div class="view-resize-handler"></div>');
        $resizeHandlerEl = view.$el.children('.view-resize-handler');
      }
      view.$resizeHandlerEl = $resizeHandlerEl;
      $el.addClass('view-resizable');

      // Add Events
      const passive = support.passiveListener ? {
        passive: true
      } : false;
      view.$el.on(app.touchEvents.start, '.view-resize-handler', handleTouchStart, passive);
      app.on('touchmove:active', handleTouchMove);
      app.on('touchend:passive', handleTouchEnd);
      app.on('resize', handleResize);
      view.on('beforeOpen', handleResize);
      view.once('viewDestroy', () => {
        $el.removeClass('view-resizable');
        view.$resizeHandlerEl.remove();
        view.$el.off(app.touchEvents.start, '.view-resize-handler', handleTouchStart, passive);
        app.off('touchmove:active', handleTouchMove);
        app.off('touchend:passive', handleTouchEnd);
        app.off('resize', handleResize);
        view.off('beforeOpen', handleResize);
      });
    }

    let View$1 = class View extends Framework7Class {
      constructor(app, el, viewParams) {
        if (viewParams === void 0) {
          viewParams = {};
        }
        super(viewParams, [app]);
        const view = this;
        const ssr = view.params.routerId;
        const defaults = {
          routes: [],
          routesAdd: []
        };
        if (!ssr) {
          const $el = $(el);
          if (!$el.length) {
            let message = "Framework7: can't create a View instance because ";
            message += typeof el === 'string' ? `the selector "${el}" didn't match any element` : 'el must be an HTMLElement or Dom7 object';
            throw new Error(message);
          }
        }

        // Default View params
        view.params = extend$3({
          el
        }, defaults, app.params.view, viewParams);

        // Routes
        if (view.params.routes.length > 0) {
          view.routes = view.params.routes;
        } else {
          view.routes = [].concat(app.routes, view.params.routesAdd);
        }

        // View Props
        extend$3(false, view, {
          app,
          name: view.params.name,
          main: view.params.main,
          history: [],
          scrollHistory: {}
        });

        // Install Modules
        view.useModules();

        // Add to app
        app.views.push(view);
        if (view.main) {
          app.views.main = view;
        }
        if (view.name) {
          app.views[view.name] = view;
        }

        // Index
        view.index = app.views.indexOf(view);

        // View ID
        let viewId;
        if (view.name) {
          viewId = `view_${view.name}`;
        } else if (view.main) {
          viewId = 'view_main';
        } else {
          viewId = `view_${view.index}`;
        }
        view.id = viewId;
        if (!view.params.init) {
          return view;
        }
        // Init View
        if (app.initialized) {
          view.init();
        } else {
          app.on('init', () => {
            view.init();
          });
        }
        return view;
      }
      destroy() {
        let view = this;
        const app = view.app;
        view.$el.trigger('view:beforedestroy');
        view.emit('local::beforeDestroy viewBeforeDestroy', view);
        app.off('resize', view.checkMasterDetailBreakpoint);
        if (view.main) {
          app.views.main = null;
          delete app.views.main;
        } else if (view.name) {
          app.views[view.name] = null;
          delete app.views[view.name];
        }
        view.$el[0].f7View = null;
        delete view.$el[0].f7View;
        app.views.splice(app.views.indexOf(view), 1);

        // Destroy Router
        if (view.params.router && view.router) {
          view.router.destroy();
        }
        view.emit('local::destroy viewDestroy', view);

        // Delete props & methods
        Object.keys(view).forEach(viewProp => {
          view[viewProp] = null;
          delete view[viewProp];
        });
        view = null;
      }
      checkMasterDetailBreakpoint(force) {
        const view = this;
        const app = view.app;
        const wasMasterDetail = view.$el.hasClass('view-master-detail');
        const isMasterDetail = app.width >= view.params.masterDetailBreakpoint && view.$el.children('.page-master').length;
        if (typeof force === 'undefined' && isMasterDetail || force === true) {
          view.$el.addClass('view-master-detail');
          if (!wasMasterDetail) {
            view.emit('local::masterDetailBreakpoint viewMasterDetailBreakpoint', view);
            view.$el.trigger('view:masterDetailBreakpoint');
          }
        } else {
          view.$el.removeClass('view-master-detail');
          if (wasMasterDetail) {
            view.emit('local::masterDetailBreakpoint viewMasterDetailBreakpoint', view);
            view.$el.trigger('view:masterDetailBreakpoint');
          }
        }
      }
      initMasterDetail() {
        const view = this;
        const app = view.app;
        view.checkMasterDetailBreakpoint = view.checkMasterDetailBreakpoint.bind(view);
        view.checkMasterDetailBreakpoint();
        if (view.params.masterDetailResizable) {
          resizableView(view);
        }
        app.on('resize', view.checkMasterDetailBreakpoint);
      }
      mount(viewEl) {
        const view = this;
        const app = view.app;
        const el = view.params.el || viewEl;
        const $el = $(el);

        // Selector
        let selector;
        if (typeof el === 'string') selector = el;else {
          // Supposed to be HTMLElement or Dom7
          selector = ($el.attr('id') ? `#${$el.attr('id')}` : '') + ($el.attr('class') ? `.${$el.attr('class').replace(/ /g, '.').replace('.active', '')}` : '');
        }

        // DynamicNavbar
        let $navbarsEl;
        if (app.theme === 'ios' && view.params.iosDynamicNavbar) {
          $navbarsEl = $el.children('.navbars').eq(0);
          if ($navbarsEl.length === 0) {
            $navbarsEl = $('<div class="navbars"></div>');
          }
        }
        extend$3(view, {
          $el,
          el: $el[0],
          main: view.main || $el.hasClass('view-main'),
          $navbarsEl,
          navbarsEl: $navbarsEl ? $navbarsEl[0] : undefined,
          selector
        });
        if (view.main) {
          app.views.main = view;
        }

        // Save in DOM
        if ($el && $el[0]) {
          $el[0].f7View = view;
        }
        view.emit('local::mount viewMount', view);
      }
      init(viewEl) {
        const view = this;
        view.mount(viewEl);
        if (view.params.router) {
          if (view.params.masterDetailBreakpoint > 0) {
            view.initMasterDetail();
          }
          if (view.params.initRouterOnTabShow && view.$el.hasClass('tab') && !view.$el.hasClass('tab-active')) {
            view.$el.once('tab:show', () => {
              view.router.init();
            });
          } else {
            view.router.init();
          }
          view.$el.trigger('view:init');
          view.emit('local::init viewInit', view);
        }
      }
    };

    // Use Router
    View$1.use(RouterModule);

    function initClicks(app) {
      function handleClicks(e) {
        const window = getWindow$1();
        const $clickedEl = $(e.target);
        const $clickedLinkEl = $clickedEl.closest('a');
        const isLink = $clickedLinkEl.length > 0;
        const url = isLink && $clickedLinkEl.attr('href');

        // Check if link is external
        if (isLink) {
          if ($clickedLinkEl.is(app.params.clicks.externalLinks) ||
          // eslint-disable-next-line
          url && url.indexOf('javascript:') >= 0) {
            const target = $clickedLinkEl.attr('target');
            if (url && window.cordova && window.cordova.InAppBrowser && (target === '_system' || target === '_blank')) {
              e.preventDefault();
              window.cordova.InAppBrowser.open(url, target);
            } else if (url && window.Capacitor && window.Capacitor.Plugins && window.Capacitor.Plugins.Browser && (target === '_system' || target === '_blank')) {
              e.preventDefault();
              window.Capacitor.Plugins.Browser.open({
                url
              });
            }
            return;
          }
        }

        // Modules Clicks
        Object.keys(app.modules).forEach(moduleName => {
          const moduleClicks = app.modules[moduleName].clicks;
          if (!moduleClicks) return;
          if (e.preventF7Router) return;
          Object.keys(moduleClicks).forEach(clickSelector => {
            const matchingClickedElement = $clickedEl.closest(clickSelector).eq(0);
            if (matchingClickedElement.length > 0) {
              moduleClicks[clickSelector].call(app, matchingClickedElement, matchingClickedElement.dataset(), e);
            }
          });
        });

        // Load Page
        let clickedLinkData = {};
        if (isLink) {
          e.preventDefault();
          clickedLinkData = $clickedLinkEl.dataset();
        }
        clickedLinkData.clickedEl = $clickedLinkEl[0];

        // Prevent Router
        if (e.preventF7Router) return;
        if ($clickedLinkEl.hasClass('prevent-router') || $clickedLinkEl.hasClass('router-prevent')) return;
        const validUrl = url && url.length > 0 && url[0] !== '#';
        if (validUrl || $clickedLinkEl.hasClass('back')) {
          let view;
          if (clickedLinkData.view && clickedLinkData.view === 'current') {
            view = app.views.current;
          } else if (clickedLinkData.view) {
            view = $(clickedLinkData.view)[0].f7View;
          } else {
            view = $clickedEl.parents('.view')[0] && $clickedEl.parents('.view')[0].f7View;
            if (view && view.params.linksView && (!$clickedLinkEl.hasClass('back') || $clickedLinkEl.hasClass('back') && view.router.history.length === 1)) {
              if (typeof view.params.linksView === 'string') view = $(view.params.linksView)[0].f7View;else if (view.params.linksView instanceof View$1) view = view.params.linksView;
            }
          }
          if (!view) {
            if (app.views.main) view = app.views.main;
          }
          if (!view || !view.router) return;
          if ($clickedLinkEl[0].f7RouteProps) {
            clickedLinkData.props = $clickedLinkEl[0].f7RouteProps;
          }
          if ($clickedLinkEl.hasClass('back')) view.router.back(url, clickedLinkData);else view.router.navigate(url, clickedLinkData);
        }
      }
      app.on('click', handleClicks);
    }
    var ClicksModule = {
      name: 'clicks',
      params: {
        clicks: {
          // External Links
          externalLinks: '.external'
        }
      },
      on: {
        init() {
          const app = this;
          initClicks(app);
        }
      }
    };

    var RouterComponentLoaderModule = {
      name: 'routerComponentLoader',
      proto: {
        openIn(router, url, options) {
          const navigateOptions = {
            url,
            route: {
              path: url,
              options: {
                ...options,
                openIn: undefined
              }
            }
          };
          const params = {
            ...options
          };
          const component = (props, ctx) => {
            const {
              $h,
              $onMounted,
              $el,
              $f7
            } = ctx;
            $onMounted(() => {
              const viewEl = $el.value.find('.view');
              const view = $f7.view.create(viewEl, {
                linksView: router.view.selector,
                ignoreOpenIn: true,
                loadInitialPage: false
              });
              view.router.navigate(url, {
                props: options.props,
                reloadAll: true
              });
            });
            // eslint-disable-next-line
            return () => {
              if (options.openIn === 'popup') {
                return $h`<div class="popup popup-router-open-in" data-url="${url}"><div class="view"></div></div>`;
              }
              if (options.openIn === 'loginScreen') {
                return $h`<div class="login-screen login-screen-router-open-in" data-url="${url}"><div class="view"></div></div>`;
              }
              if (options.openIn === 'sheet') {
                return $h`<div class="sheet-modal sheet-modal-router-open-in" data-url="${url}"><div class="sheet-modal-inner"><div class="view"></div></div></div>`;
              }
              if (options.openIn === 'popover') {
                return $h`<div class="popover popover-router-open-in" data-url="${url}"><div class="popover-inner"><div class="view"></div></div></div>`;
              }
              if (options.openIn.indexOf('panel') >= 0) {
                const parts = options.openIn.split(':');
                const side = parts[1] || 'left';
                const effect = parts[2] || 'cover';
                return $h`<div class="panel panel-router-open-in panel-${side} panel-${effect}" data-url="${url}"><div class="view"></div></div>`;
              }
            };
          };
          if (options.openIn === 'popup') {
            navigateOptions.route.popup = params;
          }
          if (options.openIn === 'loginScreen') {
            navigateOptions.route.loginScreen = params;
          }
          if (options.openIn === 'sheet') {
            navigateOptions.route.sheet = params;
          }
          if (options.openIn === 'popover') {
            params.targetEl = options.clickedEl || options.targetEl;
            navigateOptions.route.popover = params;
          }
          if (options.openIn.indexOf('panel') >= 0) {
            params.targetEl = options.clickedEl || options.targetEl;
            navigateOptions.route.panel = params;
          }
          params.component = component;
          return router.navigate(navigateOptions);
        },
        componentLoader(component, componentUrl, options, resolve, reject) {
          if (options === void 0) {
            options = {};
          }
          const router = this;
          const {
            app
          } = router;
          const url = typeof component === 'string' ? component : componentUrl;
          const compiledUrl = router.replaceRequestUrlParams(url, options);
          function compile(componentFunction) {
            let context = options.context || {};
            if (typeof context === 'function') context = context.call(router);else if (typeof context === 'string') {
              try {
                context = JSON.parse(context);
              } catch (err) {
                reject(err);
                throw err;
              }
            }
            const componentContext = merge({}, context, {
              f7route: options.route,
              f7router: router
            });
            const componentProps = merge(options.route ? options.route.params || {} : {}, options.props || {}, options.routeProps || {});
            let componentEl;
            let componentRoot;
            if (options.componentOptions && options.componentOptions.el) {
              componentEl = options.componentOptions.el;
            }
            if (options.componentOptions && options.componentOptions.root) {
              componentRoot = options.componentOptions.root;
            }
            app.component.create(componentFunction, componentProps, {
              context: componentContext,
              el: componentEl,
              root: componentRoot
            }).then(createdComponent => {
              resolve(createdComponent.el);
            }).catch(err => {
              reject(err);
              throw new Error(err, {
                cause: err
              });
            });
          }
          let cachedComponent;
          if (compiledUrl && router.params.componentCache) {
            router.cache.components.forEach(cached => {
              if (cached.url === compiledUrl) cachedComponent = cached.component;
            });
          }
          if (compiledUrl && cachedComponent) {
            compile(cachedComponent);
          } else if (compiledUrl && !cachedComponent) {
            // Load via XHR
            if (router.xhrAbortController) {
              router.xhrAbortController.abort();
              router.xhrAbortController = false;
            }
            router.xhrRequest(url, options).then(loadedComponent => {
              const parsedComponent = app.component.parse(loadedComponent);
              if (router.params.componentCache) {
                router.cache.components.push({
                  url: compiledUrl,
                  component: parsedComponent
                });
              }
              compile(parsedComponent);
            }).catch(err => {
              reject();
              throw err;
            });
          } else {
            compile(component);
          }
        },
        modalComponentLoader(_temp) {
          let {
            component,
            componentUrl,
            options,
            resolve,
            reject
          } = _temp === void 0 ? {} : _temp;
          const router = this;
          router.componentLoader(component, componentUrl, options, el => {
            resolve(el);
          }, reject);
        },
        tabComponentLoader(_temp2) {
          let {
            component,
            componentUrl,
            options,
            resolve,
            reject
          } = _temp2 === void 0 ? {} : _temp2;
          const router = this;
          router.componentLoader(component, componentUrl, options, el => {
            resolve(el);
          }, reject);
        },
        pageComponentLoader(_temp3) {
          let {
            component,
            componentUrl,
            options,
            resolve,
            reject
          } = _temp3 === void 0 ? {} : _temp3;
          const router = this;
          router.componentLoader(component, componentUrl, options, function (el, newOptions) {
            if (newOptions === void 0) {
              newOptions = {};
            }
            resolve(el, newOptions);
          }, reject);
        }
      }
    };

    var n = function (t, s, r, e) {
        var u;
        s[0] = 0;
        for (var h = 1; h < s.length; h++) {
          var p = s[h++],
            a = s[h] ? (s[0] |= p ? 1 : 2, r[s[h++]]) : s[++h];
          3 === p ? e[0] = a : 4 === p ? e[1] = Object.assign(e[1] || {}, a) : 5 === p ? (e[1] = e[1] || {})[s[++h]] = a : 6 === p ? e[1][s[++h]] += a + "" : p ? (u = t.apply(a, n(t, a, r, ["", null])), e.push(u), a[0] ? s[0] |= 2 : (s[h - 2] = 0, s[h] = u)) : e.push(a);
        }
        return e;
      },
      t = new Map();
    function htm (s) {
      var r = t.get(this);
      return r || (r = new Map(), t.set(this, r)), (r = n(this, r.get(s) || (r.set(s, r = function (n) {
        for (var t, s, r = 1, e = "", u = "", h = [0], p = function (n) {
            1 === r && (n || (e = e.replace(/^\s*\n\s*|\s*\n\s*$/g, ""))) ? h.push(0, n, e) : 3 === r && (n || e) ? (h.push(3, n, e), r = 2) : 2 === r && "..." === e && n ? h.push(4, n, 0) : 2 === r && e && !n ? h.push(5, 0, !0, e) : r >= 5 && ((e || !n && 5 === r) && (h.push(r, 0, e, s), r = 6), n && (h.push(r, n, 0, s), r = 6)), e = "";
          }, a = 0; a < n.length; a++) {
          a && (1 === r && p(), p(a));
          for (var l = 0; l < n[a].length; l++) t = n[a][l], 1 === r ? "<" === t ? (p(), h = [h], r = 3) : e += t : 4 === r ? "--" === e && ">" === t ? (r = 1, e = "") : e = t + e[0] : u ? t === u ? u = "" : e += t : '"' === t || "'" === t ? u = t : ">" === t ? (p(), r = 1) : r && ("=" === t ? (r = 5, s = e, e = "") : "/" === t && (r < 5 || ">" === n[a][l + 1]) ? (p(), 3 === r && (h = h[0]), r = h, (h = h[0]).push(2, 0, r), r = 0) : " " === t || "\t" === t || "\n" === t || "\r" === t ? (p(), r = 2) : e += t), 3 === r && "!--" === e && (r = 4, h = h[0]);
        }
        return p(), h;
      }(s)), r), arguments, [])).length > 1 ? r : r[0];
    }

    const ignoreChildren$1 = [false, null, '', undefined];
    const h$1 = function (type, props) {
      for (var _len = arguments.length, children = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        children[_key - 2] = arguments[_key];
      }
      return {
        type,
        props: props || {},
        children: flattenArray(children.filter(child => ignoreChildren$1.indexOf(child) < 0))
      };
    };
    const $h = htm.bind(h$1);

    function vnode(sel, data, children, text, elm) {
      var key = data === undefined ? undefined : data.key;
      return {
        sel: sel,
        data: data,
        children: children,
        text: text,
        elm: elm,
        key: key
      };
    }

    var array = Array.isArray;
    function primitive(s) {
      return typeof s === 'string' || typeof s === 'number';
    }

    function addNS(data, children, sel) {
      data.ns = 'http://www.w3.org/2000/svg';
      if (sel !== 'foreignObject' && children !== undefined) {
        for (var i = 0; i < children.length; ++i) {
          var childData = children[i].data;
          if (childData !== undefined) {
            addNS(childData, children[i].children, children[i].sel);
          }
        }
      }
    }
    function h(sel, b, c) {
      var data = {},
        children,
        text,
        i;
      if (c !== undefined) {
        data = b;
        if (array(c)) {
          children = c;
        } else if (primitive(c)) {
          text = c;
        } else if (c && c.sel) {
          children = [c];
        }
      } else if (b !== undefined) {
        if (array(b)) {
          children = b;
        } else if (primitive(b)) {
          text = b;
        } else if (b && b.sel) {
          children = [b];
        } else {
          data = b;
        }
      }
      if (array(children)) {
        for (i = 0; i < children.length; ++i) {
          if (primitive(children[i])) children[i] = vnode(undefined, undefined, undefined, children[i], undefined);
        }
      }
      if (sel[0] === 's' && sel[1] === 'v' && sel[2] === 'g' && (sel.length === 3 || sel[3] === '.' || sel[3] === '#')) {
        addNS(data, children, sel);
      }
      return vnode(sel, data, children, text, undefined);
    }
    ;

    var customComponents = {};

    /* eslint no-use-before-define: "off" */
    /* eslint import/no-named-as-default: "off" */
    const SELF_CLOSING = 'area base br col command embed hr img input keygen link menuitem meta param source track wbr'.split(' ');
    const PROPS_ATTRS = 'hidden checked disabled readonly selected autofocus autoplay required multiple value indeterminate routeProps innerHTML'.split(' ');
    const BOOLEAN_PROPS = 'hidden checked disabled readonly selected autofocus autoplay required multiple readOnly indeterminate'.split(' ');
    const getTagName = treeNode => {
      return typeof treeNode.type === 'function' ? treeNode.type.name || 'CustomComponent' : treeNode.type;
    };
    const toCamelCase = name => {
      return name.split('-').map((word, index) => {
        if (index === 0) return word.toLowerCase();
        return word[0].toUpperCase() + word.substr(1);
      }).join('');
    };
    const propsFromAttrs = function () {
      const context = {};
      for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }
      args.forEach(function (obj) {
        if (obj === void 0) {
          obj = {};
        }
        Object.keys(obj).forEach(key => {
          context[toCamelCase(key)] = obj[key];
        });
      });
      return context;
    };
    const createCustomComponent = _ref => {
      let {
        f7,
        treeNode,
        vnode,
        data
      } = _ref;
      const component = typeof treeNode.type === 'function' ? treeNode.type : customComponents[treeNode.type];
      f7.component.create(component, propsFromAttrs(data.attrs || {}, data.props || {}), {
        el: vnode.elm,
        children: treeNode.children
      }).then(c => {
        if (vnode.data && vnode.data.on && c && c.$el) {
          Object.keys(vnode.data.on).forEach(eventName => {
            c.$el.on(eventName, vnode.data.on[eventName]);
          });
        }
        // eslint-disable-next-line
        vnode.elm.__component__ = c;
      });
    };
    const updateCustomComponent = vnode => {
      // eslint-disable-next-line
      const component = vnode && vnode.elm && vnode.elm.__component__;
      if (!component) return;
      const newProps = propsFromAttrs(vnode.data.attrs || {}, vnode.data.props || {});
      component.children = vnode.data.treeNode.children;
      Object.assign(component.props, newProps);
      component.update();
    };
    const destroyCustomComponent = vnode => {
      // eslint-disable-next-line
      const component = vnode && vnode.elm && vnode.elm.__component__;
      if (component) {
        const {
          el,
          $el
        } = component;
        if (vnode.data && vnode.data.on && $el) {
          Object.keys(vnode.data.on).forEach(eventName => {
            $el.off(eventName, vnode.data.on[eventName]);
          });
        }
        if (component.destroy) component.destroy();
        if (el && el.parentNode) el.parentNode.removeChild(el);
        delete vnode.elm.__component__; // eslint-disable-line
      }
    };

    const isCustomComponent = treeNodeType => {
      return typeof treeNodeType === 'function' || treeNodeType && treeNodeType.indexOf('-') > 0 && customComponents[treeNodeType];
    };
    function getHooks(treeNode, data, f7, initial, isRoot) {
      const hooks = {};
      const insert = [];
      const destroy = [];
      const update = [];
      const postpatch = [];
      let isFakeElement = false;
      let tagName = getTagName(treeNode);
      if (data && data.attrs && data.attrs.component) {
        tagName = data.attrs.component;
        delete data.attrs.component;
        isFakeElement = true;
      }
      const isCustom = isCustomComponent(treeNode.type);
      if (isCustom) {
        insert.push(vnode => {
          if (vnode.sel !== tagName && !isFakeElement) return;
          createCustomComponent({
            f7,
            treeNode,
            vnode,
            data
          });
        });
        destroy.push(vnode => {
          destroyCustomComponent(vnode);
        });
        update.push((oldVnode, vnode) => {
          updateCustomComponent(vnode);
        });
      }
      if (!isCustom) {
        if (!data || !data.attrs || !data.attrs.class) return hooks;
        const classNames = data.attrs.class;
        classNames.split(' ').forEach(className => {
          if (!initial) {
            insert.push(...f7.getVnodeHooks('insert', className));
          }
          destroy.push(...f7.getVnodeHooks('destroy', className));
          update.push(...f7.getVnodeHooks('update', className));
          postpatch.push(...f7.getVnodeHooks('postpatch', className));
        });
      }
      if (isRoot && !initial) {
        postpatch.push((oldVnode, vnode) => {
          const vn = vnode || oldVnode;
          if (!vn) return;
          if (vn.data && vn.data.component) {
            vn.data.component.hook('onUpdated');
          }
        });
      }
      if (insert.length === 0 && destroy.length === 0 && update.length === 0 && postpatch.length === 0) {
        return hooks;
      }
      if (insert.length) {
        hooks.insert = vnode => {
          insert.forEach(f => f(vnode));
        };
      }
      if (destroy.length) {
        hooks.destroy = vnode => {
          destroy.forEach(f => f(vnode));
        };
      }
      if (update.length) {
        hooks.update = (oldVnode, vnode) => {
          update.forEach(f => f(oldVnode, vnode));
        };
      }
      if (postpatch.length) {
        hooks.postpatch = (oldVnode, vnode) => {
          postpatch.forEach(f => f(oldVnode, vnode));
        };
      }
      return hooks;
    }
    const getEventHandler = function (eventHandler, _temp) {
      let {
        stop,
        prevent,
        once
      } = _temp === void 0 ? {} : _temp;
      let fired = false;
      function handler() {
        const e = arguments.length <= 0 ? undefined : arguments[0];
        if (once && fired) return;
        if (stop) e.stopPropagation();
        if (prevent) e.preventDefault();
        fired = true;
        eventHandler(...arguments);
      }
      return handler;
    };
    const getData = (treeNode, component, f7, initial, isRoot) => {
      const data = {
        component,
        treeNode
      };
      const tagName = getTagName(treeNode);
      Object.keys(treeNode.props).forEach(attrName => {
        const attrValue = treeNode.props[attrName];
        if (typeof attrValue === 'undefined') return;
        if (PROPS_ATTRS.indexOf(attrName) >= 0) {
          // Props
          if (!data.props) data.props = {};
          if (attrName === 'readonly') {
            // eslint-disable-next-line
            attrName = 'readOnly';
          }
          if (attrName === 'routeProps') {
            // eslint-disable-next-line
            attrName = 'f7RouteProps';
          }
          if (tagName === 'option' && attrName === 'value') {
            if (!data.attrs) data.attrs = {};
            data.attrs.value = attrValue;
          }
          if (BOOLEAN_PROPS.indexOf(attrName) >= 0) {
            // eslint-disable-next-line
            data.props[attrName] = attrValue === false ? false : true;
          } else {
            data.props[attrName] = attrValue;
          }
        } else if (attrName === 'key') {
          // Key
          data.key = attrValue;
        } else if (attrName.indexOf('@') === 0 || attrName.indexOf('on') === 0 && attrName.length > 2) {
          // Events
          if (!data.on) data.on = {};
          let eventName = attrName.indexOf('@') === 0 ? attrName.substr(1) : eventNameToColonCase(attrName.substr(2));
          let stop = false;
          let prevent = false;
          let once = false;
          if (eventName.indexOf('.') >= 0) {
            eventName.split('.').forEach((eventNamePart, eventNameIndex) => {
              if (eventNameIndex === 0) eventName = eventNamePart;else {
                if (eventNamePart === 'stop') stop = true;
                if (eventNamePart === 'prevent') prevent = true;
                if (eventNamePart === 'once') once = true;
              }
            });
          }
          data.on[eventName] = getEventHandler(attrValue, {
            stop,
            prevent,
            once
          });
        } else if (attrName === 'style') {
          // Style
          if (typeof attrValue !== 'string') {
            data.style = attrValue;
          } else {
            if (!data.attrs) data.attrs = {};
            data.attrs.style = attrValue;
          }
        } else {
          // Rest of attribures
          if (!data.attrs) data.attrs = {};
          data.attrs[attrName] = attrValue;

          // ID -> Key
          if (attrName === 'id' && !data.key && !isRoot) {
            data.key = attrValue;
          }
        }
      });
      const hooks = getHooks(treeNode, data, f7, initial, isRoot);
      hooks.prepatch = (oldVnode, vnode) => {
        if (!oldVnode || !vnode) return;
        if (oldVnode && oldVnode.data && oldVnode.data.props) {
          Object.keys(oldVnode.data.props).forEach(key => {
            if (BOOLEAN_PROPS.indexOf(key) < 0) return;
            if (!vnode.data) vnode.data = {};
            if (!vnode.data.props) vnode.data.props = {};
            if (oldVnode.data.props[key] === true && !(key in vnode.data.props)) {
              vnode.data.props[key] = false;
            }
          });
        }
      };
      data.hook = hooks;
      return data;
    };
    const getChildren = (treeNode, component, f7, initial) => {
      if (treeNode && treeNode.type && SELF_CLOSING.indexOf(treeNode.type) >= 0) {
        return [];
      }
      const children = [];
      const nodes = treeNode.children;
      for (let i = 0; i < nodes.length; i += 1) {
        const childNode = nodes[i];
        const child = treeNodeToVNode(childNode, component, f7, initial, false);
        if (Array.isArray(child)) {
          children.push(...child);
        } else if (child) {
          children.push(child);
        }
      }
      return children;
    };
    const getSlots = (treeNode, component, f7, initial) => {
      const slotName = treeNode.props.name || 'default';
      const slotNodes = (component.children || []).filter(childTreeNode => {
        let childSlotName = 'default';
        if (childTreeNode.props) {
          childSlotName = childTreeNode.props.slot || 'default';
        }
        return childSlotName === slotName;
      });
      if (slotNodes.length === 0) {
        return getChildren(treeNode, component, f7, initial);
      }
      return slotNodes.map(subTreeNode => treeNodeToVNode(subTreeNode, component, f7, initial));
    };
    const isTreeNode = treeNode => {
      return isObject$3(treeNode) && 'props' in treeNode && 'type' in treeNode && 'children' in treeNode;
    };
    const treeNodeToVNode = (treeNode, component, f7, initial, isRoot) => {
      if (!isTreeNode(treeNode)) {
        return String(treeNode);
      }
      if (treeNode.type === 'slot') {
        return getSlots(treeNode, component, f7, initial);
      }
      const data = getData(treeNode, component, f7, initial, isRoot);
      const children = isCustomComponent(treeNode.type) ? [] : getChildren(treeNode, component, f7, initial);
      return h(getTagName(treeNode), data, children);
    };
    function vdom(tree, component, initial) {
      if (tree === void 0) {
        tree = {};
      }
      return treeNodeToVNode(tree, component, component.f7, initial, true);
    }

    function createElement$1(tagName) {
      return document.createElement(tagName);
    }
    function createElementNS(namespaceURI, qualifiedName) {
      return document.createElementNS(namespaceURI, qualifiedName);
    }
    function createTextNode(text) {
      return document.createTextNode(text);
    }
    function createComment(text) {
      return document.createComment(text);
    }
    function insertBefore(parentNode, newNode, referenceNode) {
      if (referenceNode && referenceNode.parentNode !== parentNode) {
        if (referenceNode.__component__) referenceNode = referenceNode.__component__.el;
      }
      parentNode.insertBefore(newNode, referenceNode);
    }
    function removeChild(node, child) {
      if (!node) return;
      node.removeChild(child);
    }
    function appendChild(node, child) {
      node.appendChild(child);
    }
    function parentNode(node) {
      return node.parentNode;
    }
    function nextSibling(node) {
      return node.nextSibling;
    }
    function tagName(elm) {
      return elm.tagName;
    }
    function setTextContent(node, text) {
      node.textContent = text;
    }
    function getTextContent(node) {
      return node.textContent;
    }
    function isElement(node) {
      return node.nodeType === 1;
    }
    function isText(node) {
      return node.nodeType === 3;
    }
    function isComment(node) {
      return node.nodeType === 8;
    }
    var htmlDomApi = {
      createElement: createElement$1,
      createElementNS: createElementNS,
      createTextNode: createTextNode,
      createComment: createComment,
      insertBefore: insertBefore,
      removeChild: removeChild,
      appendChild: appendChild,
      parentNode: parentNode,
      nextSibling: nextSibling,
      tagName: tagName,
      setTextContent: setTextContent,
      getTextContent: getTextContent,
      isElement: isElement,
      isText: isText,
      isComment: isComment
    };

    function copyToThunk(vnode, thunk) {
      thunk.elm = vnode.elm;
      vnode.data.fn = thunk.data.fn;
      vnode.data.args = thunk.data.args;
      thunk.data = vnode.data;
      thunk.children = vnode.children;
      thunk.text = vnode.text;
      thunk.elm = vnode.elm;
    }
    function init$1(thunk) {
      var cur = thunk.data;
      var vnode = cur.fn.apply(undefined, cur.args);
      copyToThunk(vnode, thunk);
    }
    function prepatch(oldVnode, thunk) {
      var i,
        old = oldVnode.data,
        cur = thunk.data;
      var oldArgs = old.args,
        args = cur.args;
      if (old.fn !== cur.fn || oldArgs.length !== args.length) {
        copyToThunk(cur.fn.apply(undefined, args), thunk);
        return;
      }
      for (i = 0; i < args.length; ++i) {
        if (oldArgs[i] !== args[i]) {
          copyToThunk(cur.fn.apply(undefined, args), thunk);
          return;
        }
      }
      copyToThunk(oldVnode, thunk);
    }
    var thunk = function thunk(sel, key, fn, args) {
      if (args === undefined) {
        args = fn;
        fn = key;
        key = undefined;
      }
      return h(sel, {
        key: key,
        hook: {
          init: init$1,
          prepatch: prepatch
        },
        fn: fn,
        args: args
      });
    };

    function isUndef(s) {
      return s === undefined;
    }
    function isDef(s) {
      return s !== undefined;
    }
    var emptyNode = vnode('', {}, [], undefined, undefined);
    function sameVnode(vnode1, vnode2) {
      return vnode1.key === vnode2.key && vnode1.sel === vnode2.sel;
    }
    function isVnode(vnode) {
      return vnode.sel !== undefined;
    }
    function createKeyToOldIdx(children, beginIdx, endIdx) {
      var i,
        map = {},
        key,
        ch;
      for (i = beginIdx; i <= endIdx; ++i) {
        ch = children[i];
        if (ch != null) {
          key = ch.key;
          if (key !== undefined) map[key] = i;
        }
      }
      return map;
    }
    var hooks = ['create', 'update', 'remove', 'destroy', 'pre', 'post'];
    function init(modules, domApi) {
      var i,
        j,
        cbs = {};
      var api = domApi !== undefined ? domApi : htmlDomApi;
      for (i = 0; i < hooks.length; ++i) {
        cbs[hooks[i]] = [];
        for (j = 0; j < modules.length; ++j) {
          var hook = modules[j][hooks[i]];
          if (hook !== undefined) {
            cbs[hooks[i]].push(hook);
          }
        }
      }
      function emptyNodeAt(elm) {
        var id = elm.id ? '#' + elm.id : '';
        var c = elm.className ? '.' + elm.className.split(' ').join('.') : '';
        return vnode(api.tagName(elm).toLowerCase() + id + c, {}, [], undefined, elm);
      }
      function createRmCb(childElm, listeners) {
        return function rmCb() {
          if (--listeners === 0) {
            var parent_1 = api.parentNode(childElm);
            api.removeChild(parent_1, childElm);
          }
        };
      }
      function createElm(vnode, insertedVnodeQueue) {
        var i,
          data = vnode.data;
        if (data !== undefined) {
          if (isDef(i = data.hook) && isDef(i = i.init)) {
            i(vnode);
            data = vnode.data;
          }
        }
        var children = vnode.children,
          sel = vnode.sel;
        if (sel === '!') {
          if (isUndef(vnode.text)) {
            vnode.text = '';
          }
          vnode.elm = api.createComment(vnode.text);
        } else if (sel !== undefined) {
          // Parse selector
          var hashIdx = sel.indexOf('#');
          var dotIdx = sel.indexOf('.', hashIdx);
          var hash = hashIdx > 0 ? hashIdx : sel.length;
          var dot = dotIdx > 0 ? dotIdx : sel.length;
          var tag = hashIdx !== -1 || dotIdx !== -1 ? sel.slice(0, Math.min(hash, dot)) : sel;
          var elm = vnode.elm = isDef(data) && isDef(i = data.ns) ? api.createElementNS(i, tag) : api.createElement(tag);
          if (hash < dot) elm.setAttribute('id', sel.slice(hash + 1, dot));
          if (dotIdx > 0) elm.setAttribute('class', sel.slice(dot + 1).replace(/\./g, ' '));
          for (i = 0; i < cbs.create.length; ++i) cbs.create[i](emptyNode, vnode);
          if (array(children)) {
            for (i = 0; i < children.length; ++i) {
              var ch = children[i];
              if (ch != null) {
                api.appendChild(elm, createElm(ch, insertedVnodeQueue));
              }
            }
          } else if (primitive(vnode.text)) {
            api.appendChild(elm, api.createTextNode(vnode.text));
          }
          i = vnode.data.hook; // Reuse variable
          if (isDef(i)) {
            if (i.create) i.create(emptyNode, vnode);
            if (i.insert) insertedVnodeQueue.push(vnode);
          }
        } else {
          vnode.elm = api.createTextNode(vnode.text);
        }
        return vnode.elm;
      }
      function addVnodes(parentElm, before, vnodes, startIdx, endIdx, insertedVnodeQueue) {
        for (; startIdx <= endIdx; ++startIdx) {
          var ch = vnodes[startIdx];
          if (ch != null) {
            api.insertBefore(parentElm, createElm(ch, insertedVnodeQueue), before);
          }
        }
      }
      function invokeDestroyHook(vnode) {
        var i,
          j,
          data = vnode.data;
        if (data !== undefined) {
          if (isDef(i = data.hook) && isDef(i = i.destroy)) i(vnode);
          for (i = 0; i < cbs.destroy.length; ++i) cbs.destroy[i](vnode);
          if (vnode.children !== undefined) {
            for (j = 0; j < vnode.children.length; ++j) {
              i = vnode.children[j];
              if (i != null && typeof i !== "string") {
                invokeDestroyHook(i);
              }
            }
          }
        }
      }
      function removeVnodes(parentElm, vnodes, startIdx, endIdx) {
        for (; startIdx <= endIdx; ++startIdx) {
          var i_1 = void 0,
            listeners = void 0,
            rm = void 0,
            ch = vnodes[startIdx];
          if (ch != null) {
            if (isDef(ch.sel)) {
              invokeDestroyHook(ch);
              listeners = cbs.remove.length + 1;
              rm = createRmCb(ch.elm, listeners);
              for (i_1 = 0; i_1 < cbs.remove.length; ++i_1) cbs.remove[i_1](ch, rm);
              if (isDef(i_1 = ch.data) && isDef(i_1 = i_1.hook) && isDef(i_1 = i_1.remove)) {
                i_1(ch, rm);
              } else {
                rm();
              }
            } else {
              api.removeChild(parentElm, ch.elm);
            }
          }
        }
      }
      function updateChildren(parentElm, oldCh, newCh, insertedVnodeQueue) {
        var oldStartIdx = 0,
          newStartIdx = 0;
        var oldEndIdx = oldCh.length - 1;
        var oldStartVnode = oldCh[0];
        var oldEndVnode = oldCh[oldEndIdx];
        var newEndIdx = newCh.length - 1;
        var newStartVnode = newCh[0];
        var newEndVnode = newCh[newEndIdx];
        var oldKeyToIdx;
        var idxInOld;
        var elmToMove;
        var before;
        while (oldStartIdx <= oldEndIdx && newStartIdx <= newEndIdx) {
          if (oldStartVnode == null) {
            oldStartVnode = oldCh[++oldStartIdx]; // Vnode might have been moved left
          } else if (oldEndVnode == null) {
            oldEndVnode = oldCh[--oldEndIdx];
          } else if (newStartVnode == null) {
            newStartVnode = newCh[++newStartIdx];
          } else if (newEndVnode == null) {
            newEndVnode = newCh[--newEndIdx];
          } else if (sameVnode(oldStartVnode, newStartVnode)) {
            patchVnode(oldStartVnode, newStartVnode, insertedVnodeQueue);
            oldStartVnode = oldCh[++oldStartIdx];
            newStartVnode = newCh[++newStartIdx];
          } else if (sameVnode(oldEndVnode, newEndVnode)) {
            patchVnode(oldEndVnode, newEndVnode, insertedVnodeQueue);
            oldEndVnode = oldCh[--oldEndIdx];
            newEndVnode = newCh[--newEndIdx];
          } else if (sameVnode(oldStartVnode, newEndVnode)) {
            patchVnode(oldStartVnode, newEndVnode, insertedVnodeQueue);
            api.insertBefore(parentElm, oldStartVnode.elm, api.nextSibling(oldEndVnode.elm));
            oldStartVnode = oldCh[++oldStartIdx];
            newEndVnode = newCh[--newEndIdx];
          } else if (sameVnode(oldEndVnode, newStartVnode)) {
            patchVnode(oldEndVnode, newStartVnode, insertedVnodeQueue);
            api.insertBefore(parentElm, oldEndVnode.elm, oldStartVnode.elm);
            oldEndVnode = oldCh[--oldEndIdx];
            newStartVnode = newCh[++newStartIdx];
          } else {
            if (oldKeyToIdx === undefined) {
              oldKeyToIdx = createKeyToOldIdx(oldCh, oldStartIdx, oldEndIdx);
            }
            idxInOld = oldKeyToIdx[newStartVnode.key];
            if (isUndef(idxInOld)) {
              api.insertBefore(parentElm, createElm(newStartVnode, insertedVnodeQueue), oldStartVnode.elm);
              newStartVnode = newCh[++newStartIdx];
            } else {
              elmToMove = oldCh[idxInOld];
              if (elmToMove.sel !== newStartVnode.sel) {
                api.insertBefore(parentElm, createElm(newStartVnode, insertedVnodeQueue), oldStartVnode.elm);
              } else {
                patchVnode(elmToMove, newStartVnode, insertedVnodeQueue);
                oldCh[idxInOld] = undefined;
                api.insertBefore(parentElm, elmToMove.elm, oldStartVnode.elm);
              }
              newStartVnode = newCh[++newStartIdx];
            }
          }
        }
        if (oldStartIdx <= oldEndIdx || newStartIdx <= newEndIdx) {
          if (oldStartIdx > oldEndIdx) {
            before = newCh[newEndIdx + 1] == null ? null : newCh[newEndIdx + 1].elm;
            addVnodes(parentElm, before, newCh, newStartIdx, newEndIdx, insertedVnodeQueue);
          } else {
            removeVnodes(parentElm, oldCh, oldStartIdx, oldEndIdx);
          }
        }
      }
      function patchVnode(oldVnode, vnode, insertedVnodeQueue) {
        var i, hook;
        if (isDef(i = vnode.data) && isDef(hook = i.hook) && isDef(i = hook.prepatch)) {
          i(oldVnode, vnode);
        }
        var elm = vnode.elm = oldVnode.elm;
        var oldCh = oldVnode.children;
        var ch = vnode.children;
        if (oldVnode === vnode) return;
        if (vnode.data !== undefined) {
          for (i = 0; i < cbs.update.length; ++i) cbs.update[i](oldVnode, vnode);
          i = vnode.data.hook;
          if (isDef(i) && isDef(i = i.update)) i(oldVnode, vnode);
        }
        if (isUndef(vnode.text)) {
          if (isDef(oldCh) && isDef(ch)) {
            if (oldCh !== ch) updateChildren(elm, oldCh, ch, insertedVnodeQueue);
          } else if (isDef(ch)) {
            if (isDef(oldVnode.text)) api.setTextContent(elm, '');
            addVnodes(elm, null, ch, 0, ch.length - 1, insertedVnodeQueue);
          } else if (isDef(oldCh)) {
            removeVnodes(elm, oldCh, 0, oldCh.length - 1);
          } else if (isDef(oldVnode.text)) {
            api.setTextContent(elm, '');
          }
        } else if (oldVnode.text !== vnode.text) {
          api.setTextContent(elm, vnode.text);
        }
        if (isDef(hook) && isDef(i = hook.postpatch)) {
          i(oldVnode, vnode);
        }
      }
      return function patch(oldVnode, vnode) {
        var i, elm, parent;
        var insertedVnodeQueue = [];
        for (i = 0; i < cbs.pre.length; ++i) cbs.pre[i]();
        if (!isVnode(oldVnode)) {
          oldVnode = emptyNodeAt(oldVnode);
        }
        if (sameVnode(oldVnode, vnode)) {
          patchVnode(oldVnode, vnode, insertedVnodeQueue);
        } else {
          elm = oldVnode.elm;
          parent = api.parentNode(elm);
          createElm(vnode, insertedVnodeQueue);
          if (parent !== null) {
            api.insertBefore(parent, vnode.elm, api.nextSibling(elm));
            removeVnodes(parent, [oldVnode], 0, 0);
          }
        }
        for (i = 0; i < insertedVnodeQueue.length; ++i) {
          insertedVnodeQueue[i].data.hook.insert(insertedVnodeQueue[i]);
        }
        for (i = 0; i < cbs.post.length; ++i) cbs.post[i]();
        return vnode;
      };
    }

    var xlinkNS = 'http://www.w3.org/1999/xlink';
    var xmlNS = 'http://www.w3.org/XML/1998/namespace';
    var colonChar = 58;
    var xChar = 120;
    function updateAttrs(oldVnode, vnode) {
      var key,
        elm = vnode.elm,
        oldAttrs = oldVnode.data.attrs,
        attrs = vnode.data.attrs;
      if (!oldAttrs && !attrs) return;
      if (oldAttrs === attrs) return;
      oldAttrs = oldAttrs || {};
      attrs = attrs || {};
      // update modified attributes, add new attributes
      for (key in attrs) {
        var cur = attrs[key];
        var old = oldAttrs[key];
        if (old !== cur) {
          if (cur === true) {
            elm.setAttribute(key, "");
          } else if (cur === false) {
            elm.removeAttribute(key);
          } else {
            if (key.charCodeAt(0) !== xChar) {
              elm.setAttribute(key, cur);
            } else if (key.charCodeAt(3) === colonChar) {
              // Assume xml namespace
              elm.setAttributeNS(xmlNS, key, cur);
            } else if (key.charCodeAt(5) === colonChar) {
              // Assume xlink namespace
              elm.setAttributeNS(xlinkNS, key, cur);
            } else {
              elm.setAttribute(key, cur);
            }
          }
        }
      }
      // remove removed attributes
      // use `in` operator since the previous `for` iteration uses it (.i.e. add even attributes with undefined value)
      // the other option is to remove all attributes with value == undefined
      for (key in oldAttrs) {
        if (!(key in attrs)) {
          elm.removeAttribute(key);
        }
      }
    }
    var attributesModule = {
      create: updateAttrs,
      update: updateAttrs
    };

    function updateProps(oldVnode, vnode) {
      var key,
        cur,
        old,
        elm = vnode.elm,
        oldProps = oldVnode.data.props,
        props = vnode.data.props;
      if (!oldProps && !props) return;
      if (oldProps === props) return;
      oldProps = oldProps || {};
      props = props || {};
      for (key in oldProps) {
        if (!props[key]) {
          delete elm[key];
        }
      }
      for (key in props) {
        cur = props[key];
        old = oldProps[key];
        if (old !== cur && (key !== 'value' || elm[key] !== cur)) {
          elm[key] = cur;
        }
      }
    }
    var propsModule = {
      create: updateProps,
      update: updateProps
    };

    var raf = typeof window !== 'undefined' && window.requestAnimationFrame || setTimeout;
    var nextFrame = function (fn) {
      raf(function () {
        raf(fn);
      });
    };
    function setNextFrame(obj, prop, val) {
      nextFrame(function () {
        obj[prop] = val;
      });
    }
    function updateStyle(oldVnode, vnode) {
      var cur,
        name,
        elm = vnode.elm,
        oldStyle = oldVnode.data.style,
        style = vnode.data.style;
      if (!oldStyle && !style) return;
      if (oldStyle === style) return;
      oldStyle = oldStyle || {};
      style = style || {};
      var oldHasDel = ('delayed' in oldStyle);
      for (name in oldStyle) {
        if (!style[name]) {
          if (name[0] === '-' && name[1] === '-') {
            elm.style.removeProperty(name);
          } else {
            elm.style[name] = '';
          }
        }
      }
      for (name in style) {
        cur = style[name];
        if (name === 'delayed' && style.delayed) {
          for (var name2 in style.delayed) {
            cur = style.delayed[name2];
            if (!oldHasDel || cur !== oldStyle.delayed[name2]) {
              setNextFrame(elm.style, name2, cur);
            }
          }
        } else if (name !== 'remove' && cur !== oldStyle[name]) {
          if (name[0] === '-' && name[1] === '-') {
            elm.style.setProperty(name, cur);
          } else {
            elm.style[name] = cur;
          }
        }
      }
    }
    function applyDestroyStyle(vnode) {
      var style,
        name,
        elm = vnode.elm,
        s = vnode.data.style;
      if (!s || !(style = s.destroy)) return;
      for (name in style) {
        elm.style[name] = style[name];
      }
    }
    function applyRemoveStyle(vnode, rm) {
      var s = vnode.data.style;
      if (!s || !s.remove) {
        rm();
        return;
      }
      var name,
        elm = vnode.elm,
        i = 0,
        compStyle,
        style = s.remove,
        amount = 0,
        applied = [];
      for (name in style) {
        applied.push(name);
        elm.style[name] = style[name];
      }
      compStyle = getComputedStyle(elm);
      var props = compStyle['transition-property'].split(', ');
      for (; i < props.length; ++i) {
        if (applied.indexOf(props[i]) !== -1) amount++;
      }
      elm.addEventListener('transitionend', function (ev) {
        if (ev.target === elm) --amount;
        if (amount === 0) rm();
      });
    }
    var styleModule = {
      create: updateStyle,
      update: updateStyle,
      destroy: applyDestroyStyle,
      remove: applyRemoveStyle
    };

    function invokeHandler(handler, event, args) {
      if (typeof handler === 'function') {
        // call function handler
        handler(event, ...args);
      }
    }
    function handleEvent(event, args, vnode) {
      const name = event.type;
      const on = vnode.data.on;
      // call event handler(s) if exists
      if (on && on[name]) {
        invokeHandler(on[name], event, args, vnode);
      }
    }
    function createListener() {
      return function handler(event) {
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }
        handleEvent(event, args, handler.vnode);
      };
    }
    function updateEvents(oldVnode, vnode) {
      const oldOn = oldVnode.data.on;
      const oldListener = oldVnode.listener;
      const oldElm = oldVnode.elm;
      const on = vnode && vnode.data.on;
      const elm = vnode && vnode.elm;
      // optimization for reused immutable handlers
      if (oldOn === on) {
        return;
      }
      // remove existing listeners which no longer used
      if (oldOn && oldListener) {
        // if element changed or deleted we remove all existing listeners unconditionally
        if (!on) {
          Object.keys(oldOn).forEach(name => {
            $(oldElm).off(name, oldListener);
          });
        } else {
          Object.keys(oldOn).forEach(name => {
            if (!on[name]) {
              $(oldElm).off(name, oldListener);
            }
          });
        }
      }
      // add new listeners which has not already attached
      if (on) {
        // reuse existing listener or create new
        const listener = oldVnode.listener || createListener();
        vnode.listener = listener;
        // update vnode for listener
        listener.vnode = vnode;
        // if element changed or added we add all needed listeners unconditionally
        if (!oldOn) {
          Object.keys(on).forEach(name => {
            $(elm).on(name, listener);
          });
        } else {
          Object.keys(on).forEach(name => {
            if (!oldOn[name]) {
              $(elm).on(name, listener);
            }
          });
        }
      }
    }
    var eventListenersModule = {
      create: updateEvents,
      update: updateEvents,
      destroy: updateEvents
    };

    /* eslint import/no-named-as-default: off */
    const patch = init([attributesModule, propsModule, styleModule, eventListenersModule]);

    const ignoreChildren = [false, null, '', undefined];
    const $jsx = function (type, props) {
      for (var _len = arguments.length, children = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        children[_key - 2] = arguments[_key];
      }
      const flatChildren = flattenArray((children || []).filter(child => ignoreChildren.indexOf(child) < 0));
      if (type === 'Fragment') {
        return flatChildren;
      }
      return {
        type,
        props: props || {},
        children: flatChildren
      };
    };

    /* eslint no-underscore-dangle: "off" */

    /* eslint-disable no-shadow */
    /* eslint-disable no-return-assign */
    /* eslint-disable no-plusplus */
    /* eslint-disable no-param-reassign */
    /* eslint-disable no-sequences */
    const types = [{
      name: 'array',
      init: i => i,
      type: i => [i].find(Array.isArray),
      update: (i, o) => [o].filter(Array.isArray).find(() => (i.length = 0, i.push(...o))),
      insert: function (i, x, o) {
        if (o === void 0) {
          o = [];
        }
        return i.splice(Math.max(x, 0), 0, ...[o].flat());
      },
      replace: function (i, x, o) {
        if (o === void 0) {
          o = [];
        }
        return i.splice(Math.max(x, 0), Math.min(++x, 1), ...[o].flat());
      },
      append: function (i, o) {
        if (o === void 0) {
          o = [];
        }
        return i.push(...[o].flat());
      },
      prepend: function (i, o) {
        if (o === void 0) {
          o = [];
        }
        return i.unshift(...[o].flat());
      },
      swap: (i, a, b) => {
        [i[a], i[b]] = [i[b], i[a]];
      },
      fromTo: function (i, a, b) {
        if (b === void 0) {
          b = a;
        }
        return i.splice(Math.max(b, 0), 0, ...i.splice(Math.max(a, 0), 1));
      },
      remove: function (i, o, a) {
        if (a === void 0) {
          a = i.map((_, x) => x);
        }
        return [o].flat().filter(i => a.includes(i)).sort((a, b) => b - a).forEach(x => i.splice(x, 1));
      },
      clear: i => i.length = 0
    }, {
      name: 'object',
      init: i => i,
      type: i => [i].filter(i => [i !== null, i !== undefined].every(i => i)).find(i => Object.getPrototypeOf(i) === Object.prototype),
      update: (i, o) => Object.assign(i, o),
      insert: () => {},
      replace: () => {},
      append: () => {},
      prepend: () => {},
      swap: () => ({}),
      // N/A
      fromTo: () => ({}),
      // N/A
      remove: (i, o) => [o].flat().forEach(k => delete i[k]),
      clear: i => Object.keys(i).forEach(k => delete i[k])
    }, {
      name: 'atoms',
      type: () => true,
      init: function (i, o) {
        if (o === void 0) {
          o = {};
        }
        return Object.defineProperty(o, 'value', {
          get: () => i,
          set: v => {
            // eslint-disable-next-line
            i = v;
          }
        }), o;
      },
      update: function (i, v) {
        if (v === void 0) {
          v = i.value;
        }
        i.value = v;
      },
      insert: () => ({}),
      // N/A
      replace: () => ({}),
      // N/A
      append: () => ({}),
      // N/A
      prepend: () => ({}),
      // N/A
      swap: () => ({}),
      // N/A
      fromTo: () => ({}),
      // N/A
      remove: () => ({}),
      // N/A
      clear: i => {
        i.value = undefined;
      }
    }];

    /* eslint-enable no-shadow */
    /* eslint-enable no-return-assign */
    /* eslint-enable no-plusplus */
    /* eslint-enable no-param-reassign */
    /* eslint-enable no-sequences */

    class Component {
      constructor(app, component, props, _temp) {
        if (props === void 0) {
          props = {};
        }
        let {
          el,
          context,
          children
        } = _temp === void 0 ? {} : _temp;
        const document = getDocument$1();
        merge(this, {
          f7: app,
          props: props || {},
          context: context || {},
          id: component.id || id$1(),
          children: children || [],
          theme: {
            ios: app.theme === 'ios',
            md: app.theme === 'md'
          },
          style: component.style,
          __updateQueue: [],
          __eventHandlers: [],
          __onceEventHandlers: [],
          __onBeforeMount: [],
          __onMounted: [],
          __onBeforeUpdate: [],
          __onUpdated: [],
          __onBeforeUnmount: [],
          __onUnmounted: []
        });
        const createComponent = () => {
          return component(this.props, this.getComponentContext(true));
        };
        const getRenderFuncion = componentResult => new Promise((resolve, reject) => {
          if (typeof componentResult === 'function') {
            resolve(componentResult);
          } else if (componentResult instanceof Promise) {
            componentResult.then(render => {
              resolve(render);
            }).catch(err => {
              reject(err);
            });
          } else {
            reject(new Error('Framework7: Component render function is not a "function" type. Didn\'t you forget to "return $render"?'));
          }
        });
        return new Promise((resolve, reject) => {
          const componentResult = createComponent();
          getRenderFuncion(componentResult).then(render => {
            this.renderFunction = render;
            const tree = this.render();
            if (el) {
              this.vnode = vdom(tree, this, true);
              if (this.style) {
                this.styleEl = document.createElement('style');
                this.styleEl.innerHTML = this.style;
              }
              this.el = el;
              patch(this.el, this.vnode);
              this.el = this.vnode.elm;
              this.$el = $(this.el);
              this.attachEvents();
              this.el.f7Component = this;
              this.mount();
              resolve(this);
              return;
            }
            // Make Dom
            if (tree) {
              this.vnode = vdom(tree, this, true);
              this.el = document.createElement(this.vnode.sel || 'div');
              patch(this.el, this.vnode);
              this.$el = $(this.el);
            }
            if (this.style) {
              this.styleEl = document.createElement('style');
              this.styleEl.innerHTML = this.style;
            }
            this.attachEvents();
            if (this.el) {
              this.el.f7Component = this;
            }
            resolve(this);
          }).catch(err => {
            reject(err);
          });
        });
      }
      on(eventName, handler) {
        if (!this.__eventHandlers) return;
        this.__eventHandlers.push({
          eventName,
          handler
        });
      }
      once(eventName, handler) {
        if (!this.__eventHandlers) return;
        this.__onceEventHandlers.push({
          eventName,
          handler
        });
      }
      getComponentRef() {
        const self = this;
        return initialValue => {
          let value = initialValue;
          const obj = {};
          Object.defineProperty(obj, 'value', {
            get() {
              return value;
            },
            set(v) {
              value = v;
              self.update();
            }
          });
          return obj;
        };
      }
      getComponentStore() {
        const {
          state,
          _gettersPlain,
          dispatch
        } = this.f7.store;
        const $store = {
          state,
          dispatch
        };
        $store.getters = new Proxy(_gettersPlain, {
          get: (target, prop) => {
            const obj = target[prop];
            const callback = v => {
              obj.value = v;
              this.update();
            };
            obj.onUpdated(callback);
            return obj;
          }
        });
        return $store;
      }

      /* eslint-disable no-sequences */
      getUseState() {
        var _this = this;
        return o => {
          const obj = [o].reduce(function (t, _i, _x, _a, i) {
            if (i === void 0) {
              i = t.init(_i);
            }
            return {
              state: i,
              update: v => (t.update(i, v), _this.update()),
              remove: v => (t.remove(i, v), _this.update()),
              clear: () => (t.clear(i), _this.update()),
              insert: (x, v) => (t.insert(i, x, v), _this.update()),
              replace: (x, v) => (t.replace(i, x, v), _this.update()),
              append: v => (t.append(i, v), _this.update()),
              prepend: v => (t.prepend(i, v), _this.update()),
              swap: (a, b) => (t.swap(i, a, b), _this.update()),
              fromTo: (a, b) => (t.fromTo(i, a, b), _this.update()),
              method: function (f) {
                if (f === void 0) {
                  f = () => ({});
                }
                return f(i), _this.update();
              },
              async: function (f) {
                if (f === void 0) {
                  f = () => Promise.reject(i);
                }
                return f(i).then(() => _this.update());
              }
            };
          }, types.find(i => i.type(o)));
          obj.length = 12;
          obj[Symbol.iterator] = function Iterate() {
            const values = Object.values(this);
            values.splice(values.indexOf(12), 1);
            let index = 0;
            return {
              next() {
                if (index < values.length) {
                  const val = values[index];
                  index += 1;
                  return {
                    value: val,
                    done: false
                  };
                }
                return {
                  done: true
                };
              }
            };
          };
          return obj;
        };
      }
      /* eslint-enable no-sequences */

      getComponentContext(includeHooks) {
        const ctx = {
          $f7route: this.context.f7route,
          $f7router: this.context.f7router,
          $h,
          $,
          $id: this.id,
          $f7: this.f7,
          $f7ready: this.f7ready.bind(this),
          $theme: this.theme,
          $tick: this.tick.bind(this),
          $update: this.update.bind(this),
          $emit: this.emit.bind(this),
          $store: this.getComponentStore(),
          $ref: this.getComponentRef(),
          $el: {},
          $useState: this.getUseState()
        };
        Object.defineProperty(ctx.$el, 'value', {
          get: () => {
            return this.$el;
          }
        });
        if (includeHooks) Object.assign(ctx, {
          $on: this.on.bind(this),
          $once: this.once.bind(this),
          $onBeforeMount: handler => this.__onBeforeMount.push(handler),
          $onMounted: handler => this.__onMounted.push(handler),
          $onBeforeUpdate: handler => this.__onBeforeUpdate.push(handler),
          $onUpdated: handler => this.__onUpdated.push(handler),
          $onBeforeUnmount: handler => this.__onBeforeUnmount.push(handler),
          $onUnmounted: handler => this.__onUnmounted.push(handler)
        });
        return ctx;
      }
      render() {
        return this.renderFunction(this.getComponentContext());
      }
      emit(name, data) {
        if (!this.el) return;
        this.$el.trigger(name, data);
      }
      attachEvents() {
        const {
          $el
        } = this;
        if (!this.__eventHandlers) return;
        this.__eventHandlers.forEach(_ref => {
          let {
            eventName,
            handler
          } = _ref;
          $el.on(eventNameToColonCase(eventName), handler);
        });
        this.__onceEventHandlers.forEach(_ref2 => {
          let {
            eventName,
            handler
          } = _ref2;
          $el.once(eventNameToColonCase(eventName), handler);
        });
      }
      detachEvents() {
        const {
          $el
        } = this;
        if (!this.__eventHandlers) return;
        this.__eventHandlers.forEach(_ref3 => {
          let {
            eventName,
            handler
          } = _ref3;
          $el.on(eventNameToColonCase(eventName), handler);
        });
        this.__onceEventHandlers.forEach(_ref4 => {
          let {
            eventName,
            handler
          } = _ref4;
          $el.once(eventNameToColonCase(eventName), handler);
        });
      }
      startUpdateQueue() {
        const window = getWindow$1();
        if (this.__requestAnimationFrameId) return;
        const update = () => {
          this.hook('onBeforeUpdate');
          const tree = this.render();

          // Make Dom
          if (tree) {
            const newVNode = vdom(tree, this, false);
            this.vnode = patch(this.vnode, newVNode);
          }
        };
        this.__requestAnimationFrameId = window.requestAnimationFrame(() => {
          if (this.__updateIsPending) update();
          let resolvers = [...this.__updateQueue];
          this.__updateQueue = [];
          this.__updateIsPending = false;
          window.cancelAnimationFrame(this.__requestAnimationFrameId);
          delete this.__requestAnimationFrameId;
          delete this.__updateIsPending;
          resolvers.forEach(resolver => resolver());
          resolvers = [];
        });
      }
      tick(callback) {
        return new Promise(resolve => {
          function resolver() {
            resolve();
            if (callback) callback();
          }
          this.__updateQueue.push(resolver);
          this.startUpdateQueue();
        });
      }
      update(callback) {
        if (this.__destroyed) return new Promise(() => {});
        return new Promise(resolve => {
          const resolver = () => {
            resolve();
            if (callback) callback();
          };
          this.__updateIsPending = true;
          this.__updateQueue.push(resolver);
          this.startUpdateQueue();
        });
      }
      setState(callback) {
        return this.update(callback);
      }
      f7ready(callback) {
        if (this.f7.initialized) {
          callback(this.f7);
          return;
        }
        this.f7.once('init', () => {
          callback(this.f7);
        });
      }
      mount(mountMethod) {
        this.hook('onBeforeMount', this.$el);
        if (this.styleEl) $('head').append(this.styleEl);
        if (mountMethod) mountMethod(this.el);
        this.hook('onMounted', this.$el);
      }
      destroy() {
        if (this.__destroyed) return;
        const window = getWindow$1();
        this.hook('onBeforeUnmount');
        if (this.styleEl) $(this.styleEl).remove();
        this.detachEvents();
        this.hook('onUnmounted');
        // Delete component instance
        if (this.el && this.el.f7Component) {
          this.el.f7Component = null;
          delete this.el.f7Component;
        }
        // Patch with empty node
        if (this.vnode) {
          this.vnode = patch(this.vnode, {
            sel: this.vnode.sel,
            data: {}
          });
        }
        // Clear update queue
        window.cancelAnimationFrame(this.__requestAnimationFrameId);
        this.__updateQueue = [];
        this.__eventHandlers = [];
        this.__onceEventHandlers = [];
        this.__onBeforeMount = [];
        this.__onMounted = [];
        this.__onBeforeUpdate = [];
        this.__onUpdated = [];
        this.__onBeforeUnmount = [];
        this.__onUnmounted = [];
        // Delete all props
        deleteProps$1(this);
        this.__destroyed = true;
      }
      hook(name) {
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }
        if (this.__destroyed) return;
        this[`__${name}`].forEach(handler => {
          handler(...args);
        });
      }
    }
    Component.$jsx = $jsx;

    function parseComponent(componentString) {
      const window = getWindow$1();
      const document = getDocument$1();
      const componentId = id$1();
      const callbackCreateName = `f7_component_create_callback_${componentId}`;

      // Template
      let template;
      const hasTemplate = componentString.match(/<template([ ]?)([a-z0-9-]*)>/);
      if (hasTemplate) {
        template = componentString.split(/<template[ ]?[a-z0-9-]*>/).filter((item, index) => index > 0).join('<template>').split('</template>').filter((item, index, arr) => index < arr.length - 1).join('</template>').replace(/{{#raw}}([ \n]*)<template/g, '{{#raw}}<template').replace(/\/template>([ \n]*){{\/raw}}/g, '/template>{{/raw}}').replace(/([ \n])<template/g, '$1{{#raw}}<template').replace(/\/template>([ \n])/g, '/template>{{/raw}}$1');
      }

      // Parse Styles
      let style = null;
      if (componentString.indexOf('<style>') >= 0) {
        style = componentString.split('<style>')[1].split('</style>')[0];
      }
      if (componentString.indexOf('<style scoped>') >= 0) {
        style = componentString.split('<style scoped>')[1].split('</style>')[0];
      }

      // Parse Script
      let scriptContent;
      if (componentString.indexOf('<script>') >= 0) {
        const scripts = componentString.split('<script>');
        scriptContent = scripts[scripts.length - 1].split('</script>')[0].trim();
      } else {
        scriptContent = 'return () => {return $render}';
      }
      if (!scriptContent || !scriptContent.trim()) scriptContent = 'return () => {return $render}';

      // Parse Template
      if (template) {
        scriptContent = scriptContent.replace('$render', `function ($$ctx) {
          var $ = $$ctx.$$;
          var $h = $$ctx.$h;
          var $root = $$ctx.$root;
          var $f7 = $$ctx.$f7;
          var $f7route = $$ctx.$f7route;
          var $f7router = $$ctx.$f7router;
          var $theme = $$ctx.$theme;
          var $update = $$ctx.$update;
          var $store = $$ctx.$store;
          var $ref = $$ctx.$ref;
          var $useState = $$ctx.$useState;

          return $h\`${template}\`
        }
        `).replace(/export default/g, 'return');
      }

      // Execute Script
      scriptContent = `window.${callbackCreateName} = function () {${scriptContent}}`;

      // Insert Script El
      const scriptEl = document.createElement('script');
      scriptEl.innerHTML = scriptContent;
      $('head').append(scriptEl);
      const component = window[callbackCreateName]();

      // Remove Script El
      $(scriptEl).remove();
      window[callbackCreateName] = null;
      delete window[callbackCreateName];

      // Assign Style
      if (style) {
        component.style = style;
      }

      // Component ID
      component.id = componentId;
      return component;
    }

    function registerComponent(tagName, component) {
      customComponents[tagName] = component;
    }
    function unregisterComponent(tagName) {
      delete customComponents[tagName];
    }
    var ComponentModule = {
      name: 'component',
      static: {
        Component,
        parseComponent,
        registerComponent,
        unregisterComponent
      },
      create() {
        const app = this;
        app.component = {
          registerComponent,
          unregisterComponent,
          parse(componentString) {
            return parseComponent(componentString);
          },
          create(component, props, _ref) {
            let {
              root,
              el,
              context,
              children
            } = _ref;
            return new Component(app, component, props, {
              root,
              el,
              context,
              children
            });
          }
        };
      }
    };

    var HistoryModule = {
      name: 'history',
      static: {
        history: History$1
      },
      on: {
        init() {
          History$1.init(this);
        }
      }
    };

    const SW = {
      registrations: [],
      register(path, scope) {
        const app = this;
        const window = getWindow$1();
        if (!('serviceWorker' in window.navigator) || !app.serviceWorker.container) {
          return new Promise((resolve, reject) => {
            reject(new Error('Service worker is not supported'));
          });
        }
        return new Promise((resolve, reject) => {
          app.serviceWorker.container.register(path, scope ? {
            scope
          } : {}).then(reg => {
            SW.registrations.push(reg);
            app.emit('serviceWorkerRegisterSuccess', reg);
            resolve(reg);
          }).catch(error => {
            app.emit('serviceWorkerRegisterError', error);
            reject(error);
          });
        });
      },
      unregister(registration) {
        const app = this;
        const window = getWindow$1();
        if (!('serviceWorker' in window.navigator) || !app.serviceWorker.container) {
          return new Promise((resolve, reject) => {
            reject(new Error('Service worker is not supported'));
          });
        }
        let registrations;
        if (!registration) registrations = SW.registrations;else if (Array.isArray(registration)) registrations = registration;else registrations = [registration];
        return Promise.all(registrations.map(reg => new Promise((resolve, reject) => {
          reg.unregister().then(() => {
            if (SW.registrations.indexOf(reg) >= 0) {
              SW.registrations.splice(SW.registrations.indexOf(reg), 1);
            }
            app.emit('serviceWorkerUnregisterSuccess', reg);
            resolve();
          }).catch(error => {
            app.emit('serviceWorkerUnregisterError', reg, error);
            reject(error);
          });
        })));
      }
    };
    var ServiceWorkerModule = {
      name: 'sw',
      params: {
        serviceWorker: {
          path: undefined,
          scope: undefined
        }
      },
      create() {
        const app = this;
        const window = getWindow$1();
        extend$3(app, {
          serviceWorker: {
            container: 'serviceWorker' in window.navigator ? window.navigator.serviceWorker : undefined,
            registrations: SW.registrations,
            register: SW.register.bind(app),
            unregister: SW.unregister.bind(app)
          }
        });
      },
      on: {
        init() {
          const window = getWindow$1();
          if (!('serviceWorker' in window.navigator)) return;
          const app = this;
          if (app.device.cordova || window.Capacitor && window.Capacitor.isNative) return;
          if (!app.serviceWorker.container) return;
          const paths = app.params.serviceWorker.path;
          const scope = app.params.serviceWorker.scope;
          if (!paths || Array.isArray(paths) && !paths.length) return;
          const toRegister = Array.isArray(paths) ? paths : [paths];
          toRegister.forEach(path => {
            app.serviceWorker.register(path, scope);
          });
        }
      }
    };

    /* eslint-disable no-underscore-dangle */
    function createStore(storeParams) {
      if (storeParams === void 0) {
        storeParams = {};
      }
      const store = {
        __store: true
      };
      const originalState = {
        ...(storeParams.state || {})
      };
      const actions = {
        ...(storeParams.actions || {})
      };
      const getters = {
        ...(storeParams.getters || {})
      };
      const state = extend$3({}, originalState);
      let propsQueue = [];
      const gettersDependencies = {};
      const gettersCallbacks = {};
      Object.keys(getters).forEach(getterKey => {
        gettersDependencies[getterKey] = [];
        gettersCallbacks[getterKey] = [];
      });
      const getGetterValue = getterKey => {
        return getters[getterKey]({
          state: store.state
        });
      };
      const addGetterDependencies = (getterKey, deps) => {
        if (!gettersDependencies[getterKey]) gettersDependencies[getterKey] = [];
        deps.forEach(dep => {
          if (gettersDependencies[getterKey].indexOf(dep) < 0) {
            gettersDependencies[getterKey].push(dep);
          }
        });
      };
      const addGetterCallback = (getterKey, callback) => {
        if (!gettersCallbacks[getterKey]) gettersCallbacks[getterKey] = [];
        gettersCallbacks[getterKey].push(callback);
      };
      const runGetterCallbacks = stateKey => {
        const keys = Object.keys(gettersDependencies).filter(getterKey => {
          return gettersDependencies[getterKey].indexOf(stateKey) >= 0;
        });
        keys.forEach(getterKey => {
          if (!gettersCallbacks[getterKey] || !gettersCallbacks[getterKey].length) return;
          gettersCallbacks[getterKey].forEach(callback => {
            callback(getGetterValue(getterKey));
          });
        });
      };
      const removeGetterCallback = callback => {
        Object.keys(gettersCallbacks).forEach(stateKey => {
          const callbacks = gettersCallbacks[stateKey];
          if (callbacks.indexOf(callback) >= 0) {
            callbacks.splice(callbacks.indexOf(callback), 1);
          }
        });
      };

      // eslint-disable-next-line
      store.__removeCallback = callback => {
        removeGetterCallback(callback);
      };
      const getterValue = function (getterKey, addCallback) {
        if (addCallback === void 0) {
          addCallback = true;
        }
        if (getterKey === 'constructor') return undefined;
        propsQueue = [];
        const value = getGetterValue(getterKey);
        addGetterDependencies(getterKey, propsQueue);
        const onUpdated = callback => {
          addGetterCallback(getterKey, callback);
        };
        const obj = {
          value,
          onUpdated
        };
        if (!addCallback) {
          return obj;
        }
        const callback = v => {
          obj.value = v;
        };
        obj.__callback = callback;
        addGetterCallback(getterKey, callback);
        // eslint-disable-next-line
        return obj;
      };
      store.state = new Proxy(state, {
        set: (target, prop, value) => {
          target[prop] = value;
          runGetterCallbacks(prop);
          return true;
        },
        get: (target, prop) => {
          propsQueue.push(prop);
          return target[prop];
        }
      });
      store.getters = new Proxy(getters, {
        set: () => false,
        get: (target, prop) => {
          if (!target[prop]) {
            return undefined;
          }
          return getterValue(prop, true);
        }
      });
      store._gettersPlain = new Proxy(getters, {
        set: () => false,
        get: (target, prop) => {
          if (!target[prop]) {
            return undefined;
          }
          return getterValue(prop, false);
        }
      });
      store.dispatch = (actionName, data) => {
        return new Promise((resolve, reject) => {
          if (!actions[actionName]) {
            reject();
            throw new Error(`Framework7: Store action "${actionName}" is not found`);
          }
          const result = actions[actionName]({
            state: store.state,
            dispatch: store.dispatch
          }, data);
          resolve(result);
        });
      };
      return store;
    }

    var StoreModule = {
      name: 'store',
      static: {
        createStore
      },
      proto: {
        createStore
      }
    };

    const isCapacitor = () => {
      const window = getWindow$1();
      return window.Capacitor && window.Capacitor.isNative && window.Capacitor.Plugins && window.Capacitor.Plugins.StatusBar;
    };
    const Statusbar = {
      hide() {
        const window = getWindow$1();
        const device = getDevice$1();
        if (device.cordova && window.StatusBar) {
          window.StatusBar.hide();
        }
        if (isCapacitor()) {
          window.Capacitor.Plugins.StatusBar.hide();
        }
      },
      show() {
        const window = getWindow$1();
        const device = getDevice$1();
        if (device.cordova && window.StatusBar) {
          window.StatusBar.show();
        }
        if (isCapacitor()) {
          window.Capacitor.Plugins.StatusBar.show();
        }
      },
      onClick() {
        const app = this;
        let pageContent;
        if ($('.popup.modal-in').length > 0) {
          // Check for opened popup
          pageContent = $('.popup.modal-in').find('.page:not(.page-previous):not(.page-next):not(.cached)').find('.page-content');
        } else if ($('.panel.panel-in').length > 0) {
          // Check for opened panel
          pageContent = $('.panel.panel-in').find('.page:not(.page-previous):not(.page-next):not(.cached)').find('.page-content');
        } else if ($('.views > .view.tab-active').length > 0) {
          // View in tab bar app layout
          pageContent = $('.views > .view.tab-active').find('.page:not(.page-previous):not(.page-next):not(.cached)').find('.page-content');
        } else if ($('.views').length > 0) {
          pageContent = $('.views').find('.page:not(.page-previous):not(.page-next):not(.cached)').find('.page-content');
        } else {
          pageContent = app.$el.children('.view').find('.page:not(.page-previous):not(.page-next):not(.cached)').find('.page-content');
        }
        if (pageContent && pageContent.length > 0) {
          // Check for tab
          if (pageContent.hasClass('tab')) {
            pageContent = pageContent.parent('.tabs').children('.page-content.tab-active');
          }
          if (pageContent.length > 0) pageContent.scrollTop(0, 300);
        }
      },
      setTextColor(color) {
        const window = getWindow$1();
        const device = getDevice$1();
        if (device.cordova && window.StatusBar) {
          if (color === 'white') {
            window.StatusBar.styleLightContent();
          } else {
            window.StatusBar.styleDefault();
          }
        }
        if (isCapacitor()) {
          if (color === 'white') {
            window.Capacitor.Plugins.StatusBar.setStyle({
              style: 'DARK'
            });
          } else {
            window.Capacitor.Plugins.StatusBar.setStyle({
              style: 'LIGHT'
            });
          }
        }
      },
      setBackgroundColor(color) {
        const window = getWindow$1();
        const device = getDevice$1();
        if (device.cordova && window.StatusBar) {
          window.StatusBar.backgroundColorByHexString(color);
        }
        if (isCapacitor()) {
          window.Capacitor.Plugins.StatusBar.setBackgroundColor({
            color
          });
        }
      },
      isVisible() {
        const window = getWindow$1();
        const device = getDevice$1();
        return new Promise(resolve => {
          if (device.cordova && window.StatusBar) {
            resolve(window.StatusBar.isVisible);
          }
          if (isCapacitor()) {
            window.Capacitor.Plugins.StatusBar.getInfo().then(info => {
              resolve(info.visible);
            });
          }
          resolve(false);
        });
      },
      overlaysWebView(overlays) {
        if (overlays === void 0) {
          overlays = true;
        }
        const window = getWindow$1();
        const device = getDevice$1();
        if (device.cordova && window.StatusBar) {
          window.StatusBar.overlaysWebView(overlays);
        }
        if (isCapacitor()) {
          window.Capacitor.Plugins.StatusBar.setOverlaysWebView({
            overlay: overlays
          });
        }
      },
      init() {
        const app = this;
        const window = getWindow$1();
        const device = getDevice$1();
        const params = app.params.statusbar;
        if (!params.enabled) return;
        const isCordova = device.cordova && window.StatusBar;
        const isCap = isCapacitor();
        if (isCordova || isCap) {
          if (params.scrollTopOnClick) {
            $(window).on('statusTap', Statusbar.onClick.bind(app));
          }
          if (device.ios) {
            if (params.iosOverlaysWebView) {
              Statusbar.overlaysWebView(true);
            } else {
              Statusbar.overlaysWebView(false);
            }
            if (params.iosTextColor === 'white') {
              Statusbar.setTextColor('white');
            } else {
              Statusbar.setTextColor('black');
            }
          }
          if (device.android) {
            if (params.androidOverlaysWebView) {
              Statusbar.overlaysWebView(true);
            } else {
              Statusbar.overlaysWebView(false);
            }
            if (params.androidTextColor === 'white') {
              Statusbar.setTextColor('white');
            } else {
              Statusbar.setTextColor('black');
            }
          }
        }
        if (params.iosBackgroundColor && device.ios) {
          Statusbar.setBackgroundColor(params.iosBackgroundColor);
        }
        if (params.androidBackgroundColor && device.android) {
          Statusbar.setBackgroundColor(params.androidBackgroundColor);
        }
      }
    };
    var Statusbar$1 = {
      name: 'statusbar',
      params: {
        statusbar: {
          enabled: true,
          scrollTopOnClick: true,
          iosOverlaysWebView: true,
          iosTextColor: 'black',
          iosBackgroundColor: null,
          androidOverlaysWebView: false,
          androidTextColor: 'black',
          androidBackgroundColor: null
        }
      },
      create() {
        const app = this;
        bindMethods(app, {
          statusbar: Statusbar
        });
      },
      on: {
        init() {
          const app = this;
          Statusbar.init.call(app);
        }
      }
    };

    function getCurrentView(app) {
      const $popoverView = $('.popover.modal-in .view');
      const $popupView = $('.popup.modal-in .view');
      const $panelView = $('.panel.panel-in .view');
      let $viewsEl = $('.views');
      if ($viewsEl.length === 0) $viewsEl = app.$el;
      // Find active view as tab
      let $viewEl = $viewsEl.children('.view');
      if ($viewEl.length === 0) {
        $viewEl = $viewsEl.children('.tabs').children('.view');
      }
      // Propably in tabs or split view
      if ($viewEl.length > 1) {
        if ($viewEl.hasClass('tab')) {
          // Tabs
          $viewEl = $viewsEl.children('.view.tab-active');
          if ($viewEl.length === 0) {
            $viewEl = $viewsEl.children('.tabs').children('.view.tab-active');
          }
        } else {
          // Split View, leave appView intact
        }
      }
      if ($popoverView.length > 0 && $popoverView[0].f7View) return $popoverView[0].f7View;
      if ($popupView.length > 0 && $popupView[0].f7View) return $popupView[0].f7View;
      if ($panelView.length > 0 && $panelView[0].f7View) return $panelView[0].f7View;
      if ($viewEl.length > 0) {
        if ($viewEl.length === 1 && $viewEl[0].f7View) return $viewEl[0].f7View;
        if ($viewEl.length > 1) {
          return app.views.main;
        }
      }
      return undefined;
    }
    var View = {
      name: 'view',
      params: {
        view: {
          init: true,
          initRouterOnTabShow: false,
          name: undefined,
          main: false,
          router: true,
          linksView: null,
          xhrCache: true,
          xhrCacheIgnore: [],
          xhrCacheIgnoreGetParameters: false,
          xhrCacheDuration: 1000 * 60 * 10,
          // Ten minutes
          componentCache: true,
          preloadPreviousPage: true,
          allowDuplicateUrls: false,
          reloadPages: false,
          reloadDetail: false,
          masterDetailBreakpoint: 0,
          masterDetailResizable: false,
          removeElements: true,
          removeElementsWithTimeout: false,
          removeElementsTimeout: 0,
          restoreScrollTopOnBack: true,
          unloadTabContent: true,
          passRouteQueryToRequest: true,
          passRouteParamsToRequest: false,
          loadInitialPage: true,
          // Swipe Back
          iosSwipeBack: true,
          iosSwipeBackAnimateShadow: true,
          iosSwipeBackAnimateOpacity: true,
          iosSwipeBackActiveArea: 30,
          iosSwipeBackThreshold: 0,
          mdSwipeBack: false,
          mdSwipeBackAnimateShadow: true,
          mdSwipeBackAnimateOpacity: false,
          mdSwipeBackActiveArea: 30,
          mdSwipeBackThreshold: 0,
          // Push State
          browserHistory: false,
          browserHistoryRoot: undefined,
          browserHistoryAnimate: true,
          browserHistoryAnimateOnLoad: false,
          browserHistorySeparator: '#!',
          browserHistoryOnLoad: true,
          browserHistoryInitialMatch: false,
          browserHistoryStoreHistory: true,
          browserHistoryTabs: 'replace',
          // Animate Pages
          animate: true,
          // iOS Dynamic Navbar
          iosDynamicNavbar: true,
          // Animate iOS Navbar Back Icon
          iosAnimateNavbarBackIcon: true,
          // Delays
          iosPageLoadDelay: 0,
          mdPageLoadDelay: 0,
          // Routes hooks
          routesBeforeEnter: null,
          routesBeforeLeave: null
        }
      },
      static: {
        View: View$1
      },
      create() {
        const app = this;
        extend$3(app, {
          views: extend$3([], {
            create(el, params) {
              return new View$1(app, el, params);
            },
            get(viewEl) {
              const $viewEl = $(viewEl);
              if ($viewEl.length && $viewEl[0].f7View) return $viewEl[0].f7View;
              return undefined;
            }
          })
        });
        Object.defineProperty(app.views, 'current', {
          enumerable: true,
          configurable: true,
          get() {
            return getCurrentView(app);
          }
        });
        // Alias
        app.view = app.views;
      },
      on: {
        init() {
          const app = this;
          $('.view-init').each(viewEl => {
            if (viewEl.f7View) return;
            const viewParams = $(viewEl).dataset();
            app.views.create(viewEl, viewParams);
          });
        },
        'modalOpen panelOpen': function onOpen(instance) {
          const app = this;
          instance.$el.find('.view-init').each(viewEl => {
            if (viewEl.f7View) return;
            const viewParams = $(viewEl).dataset();
            app.views.create(viewEl, viewParams);
          });
        },
        'modalBeforeDestroy panelBeforeDestroy': function onClose(instance) {
          if (!instance || !instance.$el) return;
          instance.$el.find('.view-init').each(viewEl => {
            const view = viewEl.f7View;
            if (!view) return;
            view.destroy();
          });
        }
      },
      vnode: {
        'view-init': {
          insert(vnode) {
            const app = this;
            const viewEl = vnode.elm;
            if (viewEl.f7View) return;
            const viewParams = $(viewEl).dataset();
            app.views.create(viewEl, viewParams);
          },
          destroy(vnode) {
            const viewEl = vnode.elm;
            const view = viewEl.f7View;
            if (!view) return;
            view.destroy();
          }
        }
      }
    };

    const Navbar = {
      size(el) {
        const app = this;
        let $el = $(el);
        if ($el.hasClass('navbars')) {
          $el = $el.children('.navbar').each(navbarEl => {
            app.navbar.size(navbarEl);
          });
          return;
        }
        const $innerEl = $el.children('.navbar-inner');
        if (!$innerEl.length) return;
        const needCenterTitle = $innerEl.hasClass('navbar-inner-centered-title') || app.params.navbar[`${app.theme}CenterTitle`];
        const needLeftTitle = app.theme === 'ios' && !app.params.navbar[`${app.theme}CenterTitle`];
        if (!needCenterTitle && !needLeftTitle) return;
        if ($el.parents('.tab:not(.tab-active)').length > 0 || $el.parents('.popup:not(.modal-in)').length > 0) {
          return;
        }
        if (app.theme !== 'ios' && app.params.navbar[`${app.theme}CenterTitle`]) {
          $innerEl.addClass('navbar-inner-centered-title');
        }
        if (app.theme === 'ios' && !app.params.navbar.iosCenterTitle) {
          $innerEl.addClass('navbar-inner-left-title');
        }
        const $viewEl = $el.parents('.view').eq(0);
        const left = app.rtl ? $innerEl.children('.right') : $innerEl.children('.left');
        const right = app.rtl ? $innerEl.children('.left') : $innerEl.children('.right');
        const title = $innerEl.children('.title');
        const subnavbar = $innerEl.children('.subnavbar');
        const noLeft = left.length === 0;
        const noRight = right.length === 0;
        const leftWidth = noLeft ? 0 : left.outerWidth(true);
        const rightWidth = noRight ? 0 : right.outerWidth(true);
        const titleWidth = title.outerWidth(true);
        const navbarStyles = $innerEl.styles();
        const navbarWidth = $innerEl[0].offsetWidth;
        const navbarInnerWidth = navbarWidth - parseInt(navbarStyles.paddingLeft, 10) - parseInt(navbarStyles.paddingRight, 10);
        const isPrevious = $el.hasClass('navbar-previous');
        const sliding = $innerEl.hasClass('sliding');
        let router;
        let dynamicNavbar;
        if ($viewEl.length > 0 && $viewEl[0].f7View) {
          router = $viewEl[0].f7View.router;
          dynamicNavbar = router && router.dynamicNavbar;
        }
        let currLeft;
        let diff;
        if (noRight) {
          currLeft = navbarInnerWidth - titleWidth;
        }
        if (noLeft) {
          currLeft = 0;
        }
        if (!noLeft && !noRight) {
          currLeft = (navbarInnerWidth - rightWidth - titleWidth + leftWidth) / 2;
        }
        let requiredLeft = (navbarInnerWidth - titleWidth) / 2;
        if (navbarInnerWidth - leftWidth - rightWidth > titleWidth) {
          if (requiredLeft < leftWidth) {
            requiredLeft = leftWidth;
          }
          if (requiredLeft + titleWidth > navbarInnerWidth - rightWidth) {
            requiredLeft = navbarInnerWidth - rightWidth - titleWidth;
          }
          diff = requiredLeft - currLeft;
        } else {
          diff = 0;
        }

        // RTL inverter
        const inverter = app.rtl ? -1 : 1;
        if (dynamicNavbar && app.theme === 'ios') {
          if (title.hasClass('sliding') || title.length > 0 && sliding) {
            let titleLeftOffset = -(currLeft + diff) * inverter;
            const titleRightOffset = (navbarInnerWidth - currLeft - diff - titleWidth) * inverter;
            if (isPrevious) {
              if (router && router.params.iosAnimateNavbarBackIcon) {
                const activeNavbarBackLink = $el.parent().find('.navbar-current').children('.left.sliding').find('.back .icon ~ span');
                if (activeNavbarBackLink.length > 0) {
                  titleLeftOffset += activeNavbarBackLink[0].offsetLeft;
                }
              }
            }
            title[0].f7NavbarLeftOffset = titleLeftOffset;
            title[0].f7NavbarRightOffset = titleRightOffset;
          }
          if (!noLeft && (left.hasClass('sliding') || sliding)) {
            if (app.rtl) {
              left[0].f7NavbarLeftOffset = -(navbarInnerWidth - left[0].offsetWidth) / 2 * inverter;
              left[0].f7NavbarRightOffset = leftWidth * inverter;
            } else {
              left[0].f7NavbarLeftOffset = -leftWidth;
              left[0].f7NavbarRightOffset = (navbarInnerWidth - left[0].offsetWidth) / 2;
              if (router && router.params.iosAnimateNavbarBackIcon && left.find('.back .icon').length > 0) {
                if (left.find('.back .icon ~ span').length) {
                  const leftOffset = left[0].f7NavbarLeftOffset;
                  const rightOffset = left[0].f7NavbarRightOffset;
                  left[0].f7NavbarLeftOffset = 0;
                  left[0].f7NavbarRightOffset = 0;
                  left.find('.back .icon ~ span')[0].f7NavbarLeftOffset = leftOffset;
                  left.find('.back .icon ~ span')[0].f7NavbarRightOffset = rightOffset - left.find('.back .icon')[0].offsetWidth;
                }
              }
            }
          }
          if (!noRight && (right.hasClass('sliding') || sliding)) {
            if (app.rtl) {
              right[0].f7NavbarLeftOffset = -rightWidth * inverter;
              right[0].f7NavbarRightOffset = (navbarInnerWidth - right[0].offsetWidth) / 2 * inverter;
            } else {
              right[0].f7NavbarLeftOffset = -(navbarInnerWidth - right[0].offsetWidth) / 2;
              right[0].f7NavbarRightOffset = rightWidth;
            }
          }
          if (subnavbar.length && (subnavbar.hasClass('sliding') || sliding)) {
            subnavbar[0].f7NavbarLeftOffset = app.rtl ? subnavbar[0].offsetWidth : -subnavbar[0].offsetWidth;
            subnavbar[0].f7NavbarRightOffset = -subnavbar[0].f7NavbarLeftOffset;
          }
        }

        // Center title
        if (needCenterTitle) {
          let titleLeft = diff;
          if (app.rtl && noLeft && noRight && title.length > 0) titleLeft = -titleLeft;
          title.css({
            left: `${titleLeft}px`
          });
        }
      },
      hide(el, animate, hideStatusbar, hideOnlyCurrent) {
        if (animate === void 0) {
          animate = true;
        }
        if (hideStatusbar === void 0) {
          hideStatusbar = false;
        }
        if (hideOnlyCurrent === void 0) {
          hideOnlyCurrent = false;
        }
        const app = this;
        let $el = $(el);
        const isDynamic = $el.hasClass('navbar') && $el.parent('.navbars').length && !hideOnlyCurrent;
        if (isDynamic) $el = $el.parents('.navbars');
        if (!$el.length) return;
        if ($el.hasClass('navbar-hidden')) return;
        let className = `navbar-hidden${animate ? ' navbar-transitioning' : ''}`;
        const currentIsLarge = isDynamic ? $el.find('.navbar-current .title-large').length : $el.find('.title-large').length;
        if (currentIsLarge) {
          className += ' navbar-large-hidden';
        }
        if (hideStatusbar) {
          className += ' navbar-hidden-statusbar';
        }
        $el.transitionEnd(() => {
          $el.removeClass('navbar-transitioning');
        });
        $el.addClass(className);
        if (isDynamic) {
          $el.children('.navbar').each(subEl => {
            $(subEl).trigger('navbar:hide');
            app.emit('navbarHide', subEl);
          });
        } else {
          $el.trigger('navbar:hide');
          app.emit('navbarHide', $el[0]);
        }
      },
      show(el, animate, hideOnlyCurrent) {
        if (el === void 0) {
          el = '.navbar-hidden';
        }
        if (animate === void 0) {
          animate = true;
        }
        if (hideOnlyCurrent === void 0) {
          hideOnlyCurrent = false;
        }
        const app = this;
        let $el = $(el);
        const isDynamic = $el.hasClass('navbar') && $el.parent('.navbars').length && !hideOnlyCurrent;
        if (isDynamic) $el = $el.parents('.navbars');
        if (!$el.length) return;
        if (!$el.hasClass('navbar-hidden')) return;
        if (animate) {
          $el.addClass('navbar-transitioning');
          $el.transitionEnd(() => {
            $el.removeClass('navbar-transitioning');
          });
        }
        $el.removeClass('navbar-hidden navbar-large-hidden navbar-hidden-statusbar');
        if (isDynamic) {
          $el.children('.navbar').each(subEl => {
            $(subEl).trigger('navbar:show');
            app.emit('navbarShow', subEl);
          });
        } else {
          $el.trigger('navbar:show');
          app.emit('navbarShow', $el[0]);
        }
      },
      getElByPage(page) {
        let $pageEl;
        let $navbarEl;
        let pageData;
        if (page.$navbarEl || page.$el) {
          pageData = page;
          $pageEl = page.$el;
        } else {
          $pageEl = $(page);
          if ($pageEl.length > 0) pageData = $pageEl[0].f7Page;
        }
        if (pageData && pageData.$navbarEl && pageData.$navbarEl.length > 0) {
          $navbarEl = pageData.$navbarEl;
        } else if ($pageEl) {
          $navbarEl = $pageEl.children('.navbar');
        }
        if (!$navbarEl || $navbarEl && $navbarEl.length === 0) return undefined;
        return $navbarEl[0];
      },
      getPageByEl(navbarEl) {
        const $navbarEl = $(navbarEl);
        if ($navbarEl.parents('.page').length) {
          return $navbarEl.parents('.page')[0];
        }
        let pageEl;
        $navbarEl.parents('.view').find('.page').each(el => {
          if (el && el.f7Page && el.f7Page.navbarEl && $navbarEl[0] === el.f7Page.navbarEl) {
            pageEl = el;
          }
        });
        return pageEl;
      },
      collapseLargeTitle(navbarEl) {
        const app = this;
        let $navbarEl = $(navbarEl);
        if ($navbarEl.hasClass('navbars')) {
          $navbarEl = $navbarEl.find('.navbar');
          if ($navbarEl.length > 1) {
            $navbarEl = $(navbarEl).find('.navbar-large.navbar-current');
          }
          if ($navbarEl.length > 1 || !$navbarEl.length) {
            return;
          }
        }
        const $pageEl = $(app.navbar.getPageByEl($navbarEl));
        $navbarEl.addClass('navbar-large-collapsed');
        $pageEl.eq(0).addClass('page-with-navbar-large-collapsed').trigger('page:navbarlargecollapsed');
        app.emit('pageNavbarLargeCollapsed', $pageEl[0]);
        $navbarEl.trigger('navbar:collapse');
        app.emit('navbarCollapse', $navbarEl[0]);
      },
      expandLargeTitle(navbarEl) {
        const app = this;
        let $navbarEl = $(navbarEl);
        if ($navbarEl.hasClass('navbars')) {
          $navbarEl = $navbarEl.find('.navbar-large');
          if ($navbarEl.length > 1) {
            $navbarEl = $(navbarEl).find('.navbar-large.navbar-current');
          }
          if ($navbarEl.length > 1 || !$navbarEl.length) {
            return;
          }
        }
        const $pageEl = $(app.navbar.getPageByEl($navbarEl));
        $navbarEl.removeClass('navbar-large-collapsed');
        $pageEl.eq(0).removeClass('page-with-navbar-large-collapsed').trigger('page:navbarlargeexpanded');
        app.emit('pageNavbarLargeExpanded', $pageEl[0]);
        $navbarEl.trigger('navbar:expand');
        app.emit('navbarExpand', $navbarEl[0]);
      },
      toggleLargeTitle(navbarEl) {
        const app = this;
        let $navbarEl = $(navbarEl);
        if ($navbarEl.hasClass('navbars')) {
          $navbarEl = $navbarEl.find('.navbar-large');
          if ($navbarEl.length > 1) {
            $navbarEl = $(navbarEl).find('.navbar-large.navbar-current');
          }
          if ($navbarEl.length > 1 || !$navbarEl.length) {
            return;
          }
        }
        if ($navbarEl.hasClass('navbar-large-collapsed')) {
          app.navbar.expandLargeTitle($navbarEl);
        } else {
          app.navbar.collapseLargeTitle($navbarEl);
        }
      },
      initNavbarOnScroll(pageEl, navbarEl, needHide, needCollapse, needTransparent) {
        const app = this;
        const support = getSupport$1();
        const $pageEl = $(pageEl);
        const $navbarEl = $(navbarEl);
        const $titleLargeEl = $navbarEl.find('.title-large');
        const isLarge = $titleLargeEl.length || $navbarEl.hasClass('.navbar-large');
        let navbarHideHeight = 44;
        const snapPageScrollToLargeTitle = app.params.navbar.snapPageScrollToLargeTitle;
        const snapPageScrollToTransparentNavbar = app.params.navbar.snapPageScrollToTransparentNavbar;
        let previousScrollTop;
        let currentScrollTop;
        let scrollHeight;
        let offsetHeight;
        let reachEnd;
        let action;
        let navbarHidden;
        let navbarCollapsed;
        let navbarTitleLargeHeight;
        let navbarOffsetHeight;
        if (needCollapse || needHide && isLarge) {
          navbarTitleLargeHeight = $navbarEl.css('--f7-navbar-large-title-height');
          if (navbarTitleLargeHeight && navbarTitleLargeHeight.indexOf('px') >= 0) {
            navbarTitleLargeHeight = parseInt(navbarTitleLargeHeight, 10);
            if (Number.isNaN(navbarTitleLargeHeight) && $titleLargeEl.length) {
              navbarTitleLargeHeight = $titleLargeEl[0].offsetHeight;
            } else if (Number.isNaN(navbarTitleLargeHeight)) {
              if (app.theme === 'ios') navbarTitleLargeHeight = 52;else if (app.theme === 'md') navbarTitleLargeHeight = 88;
            }
          } else if ($titleLargeEl.length) {
            navbarTitleLargeHeight = $titleLargeEl[0].offsetHeight;
          } else {
            // eslint-disable-next-line
            if (app.theme === 'ios') navbarTitleLargeHeight = 52;else if (app.theme === 'md') navbarTitleLargeHeight = 88;
          }
        }
        if (needHide && isLarge) {
          navbarHideHeight += navbarTitleLargeHeight;
        }
        let scrollChanged;
        let scrollContent;
        let scrollTimeoutId;
        let touchEndTimeoutId;
        const touchSnapTimeout = 70;
        const desktopSnapTimeout = 300;
        function calcScrollableDistance() {
          $pageEl.find('.page-content').each(pageContentEl => {
            pageContentEl.f7ScrollableDistance = pageContentEl.scrollHeight - pageContentEl.offsetHeight;
          });
        }
        function snapLargeNavbar() {
          const inSearchbarExpanded = $navbarEl.hasClass('with-searchbar-expandable-enabled');
          if (inSearchbarExpanded) return;
          if (!scrollContent || currentScrollTop < 0) return;
          if (currentScrollTop >= navbarTitleLargeHeight / 2 && currentScrollTop < navbarTitleLargeHeight) {
            $(scrollContent).scrollTop(navbarTitleLargeHeight, 100);
          } else if (currentScrollTop < navbarTitleLargeHeight) {
            $(scrollContent).scrollTop(0, 200);
          }
        }
        function snapTransparentNavbar() {
          const inSearchbarExpanded = $navbarEl.hasClass('with-searchbar-expandable-enabled');
          if (inSearchbarExpanded) return;
          if (!scrollContent || currentScrollTop < 0) return;
          if (currentScrollTop >= navbarOffsetHeight / 2 && currentScrollTop < navbarOffsetHeight) {
            $(scrollContent).scrollTop(navbarOffsetHeight, 100);
          } else if (currentScrollTop < navbarOffsetHeight) {
            $(scrollContent).scrollTop(0, 200);
          }
        }
        function handleNavbarTransparent() {
          const isHidden = $navbarEl.hasClass('navbar-hidden') || $navbarEl.parent('.navbars').hasClass('navbar-hidden');
          const inSearchbarExpanded = $navbarEl.hasClass('with-searchbar-expandable-enabled');
          if (inSearchbarExpanded || isHidden) return;
          if (!navbarOffsetHeight) {
            navbarOffsetHeight = navbarEl.offsetHeight;
          }
          let opacity = currentScrollTop / navbarOffsetHeight;
          const notTransparent = $navbarEl.hasClass('navbar-transparent-visible');
          opacity = Math.max(Math.min(opacity, 1), 0);
          if (notTransparent && opacity === 1 || !notTransparent && opacity === 0) {
            $navbarEl.find('.navbar-bg, .title').css('opacity', '');
            return;
          }
          if (notTransparent && opacity === 0) {
            $navbarEl.trigger('navbar:transparenthide');
            app.emit('navbarTransparentHide', $navbarEl[0]);
            $navbarEl.removeClass('navbar-transparent-visible');
            $navbarEl.find('.navbar-bg, .title').css('opacity', '');
            return;
          }
          if (!notTransparent && opacity === 1) {
            $navbarEl.trigger('navbar:transparentshow');
            app.emit('navbarTransparentShow', $navbarEl[0]);
            $navbarEl.addClass('navbar-transparent-visible');
            $navbarEl.find('.navbar-bg, .title').css('opacity', '');
            return;
          }
          $navbarEl.find('.navbar-bg, .title').css('opacity', opacity);
          if (snapPageScrollToTransparentNavbar) {
            if (!support.touch) {
              clearTimeout(scrollTimeoutId);
              scrollTimeoutId = setTimeout(() => {
                snapTransparentNavbar();
              }, desktopSnapTimeout);
            } else if (touchEndTimeoutId) {
              clearTimeout(touchEndTimeoutId);
              touchEndTimeoutId = null;
              touchEndTimeoutId = setTimeout(() => {
                snapTransparentNavbar();
                clearTimeout(touchEndTimeoutId);
                touchEndTimeoutId = null;
              }, touchSnapTimeout);
            }
          }
        }
        let previousCollapseProgress = null;
        let collapseProgress = null;
        function handleLargeNavbarCollapse(pageContentEl) {
          const isHidden = $navbarEl.hasClass('navbar-hidden') || $navbarEl.parent('.navbars').hasClass('navbar-hidden');
          if (isHidden) return;
          const isLargeTransparent = $navbarEl.hasClass('navbar-large-transparent') || $navbarEl.hasClass('navbar-large') && $navbarEl.hasClass('navbar-transparent');
          previousCollapseProgress = collapseProgress;
          const scrollableDistance = Math.min(navbarTitleLargeHeight, pageContentEl.f7ScrollableDistance || navbarTitleLargeHeight);
          collapseProgress = Math.min(Math.max(currentScrollTop / scrollableDistance, 0), 1);
          const previousCollapseWasInMiddle = previousCollapseProgress > 0 && previousCollapseProgress < 1;
          const inSearchbarExpanded = $navbarEl.hasClass('with-searchbar-expandable-enabled');
          if (inSearchbarExpanded) return;
          navbarCollapsed = $navbarEl.hasClass('navbar-large-collapsed');
          const $bgEl = $navbarEl.find('.navbar-bg');
          if (collapseProgress === 0 && navbarCollapsed) {
            app.navbar.expandLargeTitle($navbarEl[0]);
          } else if (collapseProgress === 1 && !navbarCollapsed) {
            app.navbar.collapseLargeTitle($navbarEl[0]);
          }
          if (collapseProgress === 0 && navbarCollapsed || collapseProgress === 0 && previousCollapseWasInMiddle || collapseProgress === 1 && !navbarCollapsed || collapseProgress === 1 && previousCollapseWasInMiddle) {
            if (app.theme === 'md') {
              $navbarEl.find('.navbar-inner').css('overflow', '');
            }
            $navbarEl.find('.title').css('opacity', '');
            $navbarEl.find('.title-large-text, .subnavbar').css('transform', '');
            $navbarEl.find('.title-large-text').css('opacity', '');
            if (isLargeTransparent) {
              $bgEl.css('opacity', '');
            }
            $bgEl.css('transform', '');
          } else if (collapseProgress > 0 && collapseProgress < 1) {
            if (app.theme === 'md') {
              $navbarEl.find('.navbar-inner').css('overflow', 'visible');
            }
            $navbarEl.find('.title').css('opacity', -0.5 + collapseProgress * 1.5);
            $navbarEl.find('.title-large-text, .subnavbar').css('transform', `translate3d(0px, ${-1 * collapseProgress * navbarTitleLargeHeight}px, 0)`);
            $navbarEl.find('.title-large-text').css('opacity', 1 - collapseProgress * 2);
            if (isLargeTransparent) {
              $bgEl.css('opacity', collapseProgress);
            }
            $bgEl.css('transform', `translate3d(0px, ${-1 * collapseProgress * navbarTitleLargeHeight}px, 0)`);
          }
          if (snapPageScrollToLargeTitle) {
            if (!support.touch) {
              clearTimeout(scrollTimeoutId);
              scrollTimeoutId = setTimeout(() => {
                snapLargeNavbar();
              }, desktopSnapTimeout);
            } else if (touchEndTimeoutId) {
              clearTimeout(touchEndTimeoutId);
              touchEndTimeoutId = null;
              touchEndTimeoutId = setTimeout(() => {
                snapLargeNavbar();
                clearTimeout(touchEndTimeoutId);
                touchEndTimeoutId = null;
              }, touchSnapTimeout);
            }
          }
        }
        function handleTitleHideShow() {
          if ($pageEl.hasClass('page-with-card-opened')) return;
          scrollHeight = scrollContent.scrollHeight;
          offsetHeight = scrollContent.offsetHeight;
          reachEnd = currentScrollTop + offsetHeight >= scrollHeight;
          navbarHidden = $navbarEl.hasClass('navbar-hidden') || $navbarEl.parent('.navbars').hasClass('navbar-hidden');
          if (reachEnd) {
            if (app.params.navbar.showOnPageScrollEnd) {
              action = 'show';
            }
          } else if (previousScrollTop > currentScrollTop) {
            if (app.params.navbar.showOnPageScrollTop || currentScrollTop <= navbarHideHeight) {
              action = 'show';
            } else {
              action = 'hide';
            }
          } else if (currentScrollTop > navbarHideHeight) {
            action = 'hide';
          } else {
            action = 'show';
          }
          if (action === 'show' && navbarHidden) {
            app.navbar.show($navbarEl, true, true);
            navbarHidden = false;
          } else if (action === 'hide' && !navbarHidden) {
            app.navbar.hide($navbarEl, true, false, true);
            navbarHidden = true;
          }
          previousScrollTop = currentScrollTop;
        }
        function handleScroll(e) {
          scrollContent = this;
          if (e && e.target && e.target !== scrollContent) {
            return;
          }
          currentScrollTop = scrollContent.scrollTop;
          scrollChanged = currentScrollTop;
          if (needCollapse) {
            handleLargeNavbarCollapse(scrollContent);
          } else if (needTransparent) {
            handleNavbarTransparent();
          }
          if ($pageEl.hasClass('page-previous')) return;
          if (needHide) {
            handleTitleHideShow();
          }
        }
        function handeTouchStart() {
          scrollChanged = false;
        }
        function handleTouchEnd() {
          clearTimeout(touchEndTimeoutId);
          touchEndTimeoutId = null;
          touchEndTimeoutId = setTimeout(() => {
            if (scrollChanged !== false) {
              if (needTransparent && !needCollapse) {
                snapTransparentNavbar();
              } else {
                snapLargeNavbar();
              }
              clearTimeout(touchEndTimeoutId);
              touchEndTimeoutId = null;
            }
          }, touchSnapTimeout);
        }
        $pageEl.on('scroll', '.page-content', handleScroll, true);
        if (support.touch && (needCollapse && snapPageScrollToLargeTitle || needTransparent && snapPageScrollToTransparentNavbar)) {
          app.on('touchstart:passive', handeTouchStart);
          app.on('touchend:passive', handleTouchEnd);
        }
        calcScrollableDistance();
        if (needCollapse || needTransparent) {
          $pageEl.find('.page-content').each(pageContentEl => {
            if (pageContentEl.scrollTop > 0) handleScroll.call(pageContentEl);
          });
        }
        app.on('resize', calcScrollableDistance);
        $pageEl[0].f7DetachNavbarScrollHandlers = function f7DetachNavbarScrollHandlers() {
          app.off('resize', calcScrollableDistance);
          delete $pageEl[0].f7DetachNavbarScrollHandlers;
          $pageEl.off('scroll', '.page-content', handleScroll, true);
          if (support.touch && (needCollapse && snapPageScrollToLargeTitle || needTransparent && snapPageScrollToTransparentNavbar)) {
            app.off('touchstart:passive', handeTouchStart);
            app.off('touchend:passive', handleTouchEnd);
          }
        };
      }
    };
    var Navbar$1 = {
      name: 'navbar',
      create() {
        const app = this;
        bindMethods(app, {
          navbar: Navbar
        });
      },
      params: {
        navbar: {
          scrollTopOnTitleClick: true,
          iosCenterTitle: true,
          mdCenterTitle: false,
          hideOnPageScroll: false,
          showOnPageScrollEnd: true,
          showOnPageScrollTop: true,
          collapseLargeTitleOnScroll: true,
          snapPageScrollToLargeTitle: true,
          snapPageScrollToTransparentNavbar: true
        }
      },
      on: {
        'panelBreakpoint panelCollapsedBreakpoint panelResize viewResize resize viewMasterDetailBreakpoint': function onPanelResize() {
          const app = this;
          $('.navbar').each(navbarEl => {
            app.navbar.size(navbarEl);
          });
        },
        pageBeforeRemove(page) {
          if (page.$el[0].f7DetachNavbarScrollHandlers) {
            page.$el[0].f7DetachNavbarScrollHandlers();
          }
        },
        pageBeforeIn(page) {
          const app = this;
          if (app.theme !== 'ios') return;
          let $navbarsEl;
          const view = page.$el.parents('.view')[0].f7View;
          const navbarEl = app.navbar.getElByPage(page);
          if (!navbarEl) {
            $navbarsEl = page.$el.parents('.view').children('.navbars');
          } else {
            $navbarsEl = $(navbarEl).parents('.navbars');
          }
          if (page.$el.hasClass('no-navbar') || view.router.dynamicNavbar && !navbarEl) {
            const animate = !!(page.pageFrom && page.router.history.length > 0);
            app.navbar.hide($navbarsEl, animate);
          } else {
            app.navbar.show($navbarsEl);
          }
        },
        pageReinit(page) {
          const app = this;
          const $navbarEl = $(app.navbar.getElByPage(page));
          if (!$navbarEl || $navbarEl.length === 0) return;
          app.navbar.size($navbarEl);
        },
        pageInit(page) {
          const app = this;
          const $navbarEl = $(app.navbar.getElByPage(page));
          if (!$navbarEl || $navbarEl.length === 0) return;

          // Size
          app.navbar.size($navbarEl);

          // Need Collapse On Scroll
          let needCollapseOnScrollHandler;
          if ($navbarEl.find('.title-large').length > 0) {
            $navbarEl.addClass('navbar-large');
          }
          if ($navbarEl.hasClass('navbar-large')) {
            if (app.params.navbar.collapseLargeTitleOnScroll) needCollapseOnScrollHandler = true;
            page.$el.addClass('page-with-navbar-large');
          }

          // Need transparent on scroll
          let needTransparentOnScroll;
          if (!needCollapseOnScrollHandler && $navbarEl.hasClass('navbar-transparent')) {
            needTransparentOnScroll = true;
          }

          // Need Hide On Scroll
          let needHideOnScrollHandler;
          if (app.params.navbar.hideOnPageScroll || page.$el.find('.hide-navbar-on-scroll').length || page.$el.hasClass('hide-navbar-on-scroll') || page.$el.find('.hide-bars-on-scroll').length || page.$el.hasClass('hide-bars-on-scroll')) {
            if (page.$el.find('.keep-navbar-on-scroll').length || page.$el.hasClass('keep-navbar-on-scroll') || page.$el.find('.keep-bars-on-scroll').length || page.$el.hasClass('keep-bars-on-scroll')) {
              needHideOnScrollHandler = false;
            } else {
              needHideOnScrollHandler = true;
            }
          }
          if (needCollapseOnScrollHandler || needHideOnScrollHandler || needTransparentOnScroll) {
            app.navbar.initNavbarOnScroll(page.el, $navbarEl[0], needHideOnScrollHandler, needCollapseOnScrollHandler, needTransparentOnScroll);
          }
        },
        'panelOpen panelSwipeOpen modalOpen': function onPanelModalOpen(instance) {
          const app = this;
          instance.$el.find('.navbar:not(.navbar-previous)').each(navbarEl => {
            app.navbar.size(navbarEl);
          });
        },
        tabShow(tabEl) {
          const app = this;
          $(tabEl).find('.navbar:not(.navbar-previous)').each(navbarEl => {
            app.navbar.size(navbarEl);
          });
        }
      },
      clicks: {
        '.navbar .title': function onTitleClick($clickedEl, clickedData, e) {
          const app = this;
          if (!app.params.navbar.scrollTopOnTitleClick) return;
          if ($(e.target).closest('a, button').length > 0) {
            return;
          }
          let $pageContentEl;

          // Find active page
          const $navbarEl = $clickedEl.parents('.navbar');
          const $navbarsEl = $navbarEl.parents('.navbars');

          // Static Layout
          $pageContentEl = $navbarEl.parents('.page-content');
          if ($pageContentEl.length === 0) {
            // Fixed Layout
            if ($navbarEl.parents('.page').length > 0) {
              $pageContentEl = $navbarEl.parents('.page').find('.page-content');
            }
            // Through Layout iOS
            if ($pageContentEl.length === 0 && $navbarsEl.length) {
              if ($navbarsEl.nextAll('.page-current').length > 0) {
                $pageContentEl = $navbarsEl.nextAll('.page-current').find('.page-content');
              }
            }
            // Through Layout
            if ($pageContentEl.length === 0) {
              if ($navbarEl.nextAll('.page-current').length > 0) {
                $pageContentEl = $navbarEl.nextAll('.page-current').find('.page-content');
              }
            }
          }
          if ($pageContentEl && $pageContentEl.length > 0) {
            // Check for tab
            if ($pageContentEl.hasClass('tab')) {
              $pageContentEl = $pageContentEl.parent('.tabs').children('.page-content.tab-active');
            }
            if ($pageContentEl.length > 0) $pageContentEl.scrollTop(0, 300);
          }
        }
      },
      vnode: {
        navbar: {
          postpatch(vnode) {
            const app = this;
            app.navbar.size(vnode.elm);
          }
        }
      }
    };

    const Toolbar = {
      setHighlight(tabbarEl) {
        const app = this;
        const $tabbarEl = $(tabbarEl);
        if (app.theme === 'ios' && !$tabbarEl.hasClass('tabbar-highlight')) return;
        if ($tabbarEl.length === 0 || !($tabbarEl.hasClass('tabbar') || $tabbarEl.hasClass('tabbar-icons'))) return;
        let $highlightEl = $tabbarEl.find('.tab-link-highlight');
        const tabLinksCount = $tabbarEl.find('.tab-link').length;
        if (tabLinksCount === 0) {
          $highlightEl.remove();
          return;
        }
        if ($highlightEl.length === 0) {
          $tabbarEl.children('.toolbar-inner').append('<span class="tab-link-highlight"></span>');
          $highlightEl = $tabbarEl.find('.tab-link-highlight');
        } else if ($highlightEl.next().length) {
          $tabbarEl.children('.toolbar-inner').append($highlightEl);
        }
        const $activeLink = $tabbarEl.find('.tab-link-active');
        let highlightWidth;
        let highlightTranslate;
        if ($tabbarEl.hasClass('tabbar-scrollable') && $activeLink && $activeLink[0]) {
          highlightWidth = `${$activeLink[0].offsetWidth}px`;
          highlightTranslate = `${$activeLink[0].offsetLeft}px`;
        } else {
          const activeIndex = $activeLink.index();
          highlightWidth = `${100 / tabLinksCount}%`;
          highlightTranslate = `${(app.rtl ? -activeIndex : activeIndex) * 100}%`;
        }
        nextFrame$1(() => {
          $highlightEl.css('width', highlightWidth).transform(`translate3d(${highlightTranslate},0,0)`);
        });
      },
      init(tabbarEl) {
        const app = this;
        app.toolbar.setHighlight(tabbarEl);
      },
      hide(el, animate) {
        if (animate === void 0) {
          animate = true;
        }
        const app = this;
        const $el = $(el);
        if ($el.hasClass('toolbar-hidden')) return;
        const className = `toolbar-hidden${animate ? ' toolbar-transitioning' : ''}`;
        $el.transitionEnd(() => {
          $el.removeClass('toolbar-transitioning');
        });
        $el.addClass(className);
        $el.trigger('toolbar:hide');
        app.emit('toolbarHide', $el[0]);
      },
      show(el, animate) {
        if (animate === void 0) {
          animate = true;
        }
        const app = this;
        const $el = $(el);
        if (!$el.hasClass('toolbar-hidden')) return;
        if (animate) {
          $el.addClass('toolbar-transitioning');
          $el.transitionEnd(() => {
            $el.removeClass('toolbar-transitioning');
          });
        }
        $el.removeClass('toolbar-hidden');
        $el.trigger('toolbar:show');
        app.emit('toolbarShow', $el[0]);
      },
      initToolbarOnScroll(pageEl) {
        const app = this;
        const $pageEl = $(pageEl);
        let $toolbarEl = $pageEl.parents('.view').children('.toolbar');
        if ($toolbarEl.length === 0) {
          $toolbarEl = $pageEl.find('.toolbar');
        }
        if ($toolbarEl.length === 0) {
          $toolbarEl = $pageEl.parents('.views').children('.tabbar, .tabbar-icons');
        }
        if ($toolbarEl.length === 0) {
          return;
        }
        let previousScrollTop;
        let currentScrollTop;
        let scrollHeight;
        let offsetHeight;
        let reachEnd;
        let action;
        let toolbarHidden;
        function handleScroll(e) {
          if ($pageEl.hasClass('page-with-card-opened')) return;
          if ($pageEl.hasClass('page-previous')) return;
          const scrollContent = this;
          if (e && e.target && e.target !== scrollContent) {
            return;
          }
          currentScrollTop = scrollContent.scrollTop;
          scrollHeight = scrollContent.scrollHeight;
          offsetHeight = scrollContent.offsetHeight;
          reachEnd = currentScrollTop + offsetHeight >= scrollHeight;
          toolbarHidden = $toolbarEl.hasClass('toolbar-hidden');
          if (reachEnd) {
            if (app.params.toolbar.showOnPageScrollEnd) {
              action = 'show';
            }
          } else if (previousScrollTop > currentScrollTop) {
            if (app.params.toolbar.showOnPageScrollTop || currentScrollTop <= 44) {
              action = 'show';
            } else {
              action = 'hide';
            }
          } else if (currentScrollTop > 44) {
            action = 'hide';
          } else {
            action = 'show';
          }
          if (action === 'show' && toolbarHidden) {
            app.toolbar.show($toolbarEl);
            toolbarHidden = false;
          } else if (action === 'hide' && !toolbarHidden) {
            app.toolbar.hide($toolbarEl);
            toolbarHidden = true;
          }
          previousScrollTop = currentScrollTop;
        }
        $pageEl.on('scroll', '.page-content', handleScroll, true);
        $pageEl[0].f7ScrollToolbarHandler = handleScroll;
      }
    };
    var Toolbar$1 = {
      name: 'toolbar',
      create() {
        const app = this;
        bindMethods(app, {
          toolbar: Toolbar
        });
      },
      params: {
        toolbar: {
          hideOnPageScroll: false,
          showOnPageScrollEnd: true,
          showOnPageScrollTop: true
        }
      },
      on: {
        pageBeforeRemove(page) {
          if (page.$el[0].f7ScrollToolbarHandler) {
            page.$el.off('scroll', '.page-content', page.$el[0].f7ScrollToolbarHandler, true);
          }
        },
        pageBeforeIn(page) {
          const app = this;
          let $toolbarEl = page.$el.parents('.view').children('.toolbar');
          if ($toolbarEl.length === 0) {
            $toolbarEl = page.$el.parents('.views').children('.tabbar, .tabbar-icons');
          }
          if ($toolbarEl.length === 0) {
            $toolbarEl = page.$el.find('.toolbar');
          }
          if ($toolbarEl.length === 0) {
            return;
          }
          if (page.$el.hasClass('no-toolbar')) {
            app.toolbar.hide($toolbarEl);
          } else {
            app.toolbar.show($toolbarEl);
          }
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.tabbar, .tabbar-icons').each(tabbarEl => {
            app.toolbar.init(tabbarEl);
          });
          if (app.params.toolbar.hideOnPageScroll || page.$el.find('.hide-toolbar-on-scroll').length || page.$el.hasClass('hide-toolbar-on-scroll') || page.$el.find('.hide-bars-on-scroll').length || page.$el.hasClass('hide-bars-on-scroll')) {
            if (page.$el.find('.keep-toolbar-on-scroll').length || page.$el.hasClass('keep-toolbar-on-scroll') || page.$el.find('.keep-bars-on-scroll').length || page.$el.hasClass('keep-bars-on-scroll')) {
              return;
            }
            app.toolbar.initToolbarOnScroll(page.el);
          }
        },
        init() {
          const app = this;
          app.$el.find('.tabbar, .tabbar-icons').each(tabbarEl => {
            app.toolbar.init(tabbarEl);
          });
        }
      },
      vnode: {
        tabbar: {
          insert(vnode) {
            const app = this;
            app.toolbar.init(vnode.elm);
          }
        }
      }
    };

    var Subnavbar = {
      name: 'subnavbar',
      on: {
        pageInit(page) {
          if (page.$navbarEl && page.$navbarEl.length && page.$navbarEl.find('.subnavbar').length) {
            page.$el.addClass('page-with-subnavbar');
          }
          const $innerSubnavbars = page.$el.find('.subnavbar').filter(subnavbarEl => {
            return $(subnavbarEl).parents('.page')[0] === page.$el[0];
          });
          if ($innerSubnavbars.length) {
            page.$el.addClass('page-with-subnavbar');
          }
        }
      }
    };

    let TouchRipple$1 = class TouchRipple {
      constructor(app, $el, x, y) {
        const ripple = this;
        if (!$el) return undefined;
        const {
          left,
          top,
          width,
          height
        } = $el[0].getBoundingClientRect();
        const center = {
          x: x - left,
          y: y - top
        };
        let diameter = Math.max((height ** 2 + width ** 2) ** 0.5, 48);
        let isInset = false;
        const insetElements = app.params.touch.touchRippleInsetElements || '';
        if (insetElements && $el.is(insetElements)) {
          isInset = true;
        }
        if (isInset) {
          diameter = Math.max(Math.min(width, height), 48);
        }
        if (!isInset && $el.css('overflow') === 'hidden') {
          const distanceFromCenter = ((center.x - width / 2) ** 2 + (center.y - height / 2) ** 2) ** 0.5;
          const scale = (diameter / 2 + distanceFromCenter) / (diameter / 2);
          ripple.rippleTransform = `translate3d(0px, 0px, 0) scale(${scale * 2})`;
        } else {
          // prettier-ignore
          ripple.rippleTransform = `translate3d(${-center.x + width / 2}px, ${-center.y + height / 2}px, 0) scale(1)`;
        }
        if (isInset) {
          $el.addClass('ripple-inset');
        }
        ripple.$rippleWaveEl = $(`<div class="ripple-wave${isInset ? ' ripple-wave-inset' : ''}" style="width: ${diameter}px; height: ${diameter}px; margin-top:-${diameter / 2}px; margin-left:-${diameter / 2}px; left:${center.x}px; top:${center.y}px; --f7-ripple-transform: ${ripple.rippleTransform}"></div>`);
        $el.prepend(ripple.$rippleWaveEl);
        ripple.$rippleWaveEl.animationEnd(() => {
          if (!ripple.$rippleWaveEl) return;
          if (ripple.$rippleWaveEl.hasClass('ripple-wave-out')) return;
          ripple.$rippleWaveEl.addClass('ripple-wave-in');
          if (ripple.shouldBeRemoved) {
            ripple.out();
          }
        });
        return ripple;
      }
      destroy() {
        let ripple = this;
        if (ripple.$rippleWaveEl) {
          ripple.$rippleWaveEl.remove();
        }
        Object.keys(ripple).forEach(key => {
          ripple[key] = null;
          delete ripple[key];
        });
        ripple = null;
      }
      out() {
        const ripple = this;
        const {
          $rippleWaveEl
        } = this;
        clearTimeout(ripple.removeTimeout);
        $rippleWaveEl.addClass('ripple-wave-out');
        ripple.removeTimeout = setTimeout(() => {
          ripple.destroy();
        }, 300);
        $rippleWaveEl.animationEnd(() => {
          clearTimeout(ripple.removeTimeout);
          ripple.destroy();
        });
      }
      remove() {
        const ripple = this;
        if (ripple.shouldBeRemoved) return;
        ripple.removeTimeout = setTimeout(() => {
          ripple.destroy();
        }, 400);
        ripple.shouldBeRemoved = true;
        if (ripple.$rippleWaveEl.hasClass('ripple-wave-in')) {
          ripple.out();
        }
      }
    };

    var TouchRipple = {
      name: 'touch-ripple',
      static: {
        TouchRipple: TouchRipple$1
      },
      create() {
        const app = this;
        app.touchRipple = {
          create() {
            for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
              args[_key] = arguments[_key];
            }
            return new TouchRipple$1(...args);
          }
        };
      }
    };

    const openedModals = [];
    const dialogsQueue = [];
    function clearDialogsQueue() {
      if (dialogsQueue.length === 0) return;
      const dialog = dialogsQueue.shift();
      dialog.open();
    }
    let Modal$1 = class Modal extends Framework7Class {
      constructor(app, params) {
        super(params, [app]);
        const modal = this;
        const defaults = {};

        // Extend defaults with modules params
        modal.useModulesParams(defaults);
        modal.params = extend$3(defaults, params);
        modal.opened = false;
        let $containerEl = modal.params.containerEl ? $(modal.params.containerEl).eq(0) : app.$el;
        if (!$containerEl.length) $containerEl = app.$el;
        modal.$containerEl = $containerEl;
        modal.containerEl = $containerEl[0];

        // Install Modules
        modal.useModules();
        return this;
      }
      onOpen() {
        const modal = this;
        modal.opened = true;
        openedModals.push(modal);
        $('html').addClass(`with-modal-${modal.type.toLowerCase()}`);
        modal.$el.trigger(`modal:open ${modal.type.toLowerCase()}:open`);
        modal.emit(`local::open modalOpen ${modal.type}Open`, modal);
      }
      onOpened() {
        const modal = this;
        modal.$el.trigger(`modal:opened ${modal.type.toLowerCase()}:opened`);
        modal.emit(`local::opened modalOpened ${modal.type}Opened`, modal);
      }
      onClose() {
        const modal = this;
        modal.opened = false;
        if (!modal.type || !modal.$el) return;
        openedModals.splice(openedModals.indexOf(modal), 1);
        $('html').removeClass(`with-modal-${modal.type.toLowerCase()}`);
        modal.$el.trigger(`modal:close ${modal.type.toLowerCase()}:close`);
        modal.emit(`local::close modalClose ${modal.type}Close`, modal);
      }
      onClosed() {
        const modal = this;
        if (!modal.type || !modal.$el) return;
        modal.$el.removeClass('modal-out');
        modal.$el.hide();
        if (modal.params.backdrop && (modal.params.backdropUnique || modal.forceBackdropUnique) && modal.$backdropEl) {
          modal.$backdropEl.remove();
        }
        modal.$el.trigger(`modal:closed ${modal.type.toLowerCase()}:closed`);
        modal.emit(`local::closed modalClosed ${modal.type}Closed`, modal);
      }
      open(animateModal, force) {
        const modal = this;
        const document = getDocument$1();
        const app = modal.app;
        const $el = modal.$el;
        const $backdropEl = modal.$backdropEl;
        const type = modal.type;
        let animate = true;
        if (typeof animateModal !== 'undefined') animate = animateModal;else if (typeof modal.params.animate !== 'undefined') {
          animate = modal.params.animate;
        }
        if (!$el || $el.hasClass('modal-in')) {
          if (animateModal === false && $el[0] && type !== 'dialog') {
            $el[0].style.display = 'block';
          }
          if (!force) return modal;
        }
        if (type === 'dialog' && app.params.modal.queueDialogs) {
          let pushToQueue;
          if ($('.dialog.modal-in').length > 0) {
            pushToQueue = true;
          } else if (openedModals.length > 0) {
            openedModals.forEach(openedModal => {
              if (openedModal.type === 'dialog') pushToQueue = true;
            });
          }
          if (pushToQueue) {
            dialogsQueue.push(modal);
            return modal;
          }
        }
        const $modalParentEl = $el.parent();
        const wasInDom = $el.parents(document).length > 0;
        if (!$modalParentEl.is(modal.$containerEl)) {
          modal.$containerEl.append($el);
          modal.once(`${type}Closed`, () => {
            if (wasInDom) {
              $modalParentEl.append($el);
            } else {
              $el.remove();
            }
          });
        }
        // Show Modal
        $el.show();
        if (modal.params.backdrop && (modal.params.backdropUnique || modal.forceBackdropUnique) && modal.$backdropEl) {
          modal.$backdropEl.insertBefore($el);
        }

        /* eslint no-underscore-dangle: ["error", { "allow": ["_clientLeft"] }] */
        modal._clientLeft = $el[0].clientLeft;

        // Modal
        function transitionEnd() {
          if ($el.hasClass('modal-out')) {
            modal.onClosed();
          } else if ($el.hasClass('modal-in')) {
            modal.onOpened();
          }
        }
        if (animate) {
          if ($backdropEl) {
            $backdropEl.removeClass('not-animated');
            $backdropEl.addClass('backdrop-in');
          }
          $el.animationEnd(() => {
            transitionEnd();
          });
          $el.transitionEnd(() => {
            transitionEnd();
          });
          $el.removeClass('modal-out not-animated').addClass('modal-in');
          modal.onOpen();
        } else {
          if ($backdropEl) {
            $backdropEl.addClass('backdrop-in not-animated');
          }
          $el.removeClass('modal-out').addClass('modal-in not-animated');
          modal.onOpen();
          modal.onOpened();
        }
        return modal;
      }
      close(animateModal) {
        const modal = this;
        const $el = modal.$el;
        const $backdropEl = modal.$backdropEl;
        let animate = true;
        if (typeof animateModal !== 'undefined') animate = animateModal;else if (typeof modal.params.animate !== 'undefined') {
          animate = modal.params.animate;
        }
        if (!$el || !$el.hasClass('modal-in')) {
          if (dialogsQueue.indexOf(modal) >= 0) {
            dialogsQueue.splice(dialogsQueue.indexOf(modal), 1);
          }
          return modal;
        }

        // backdrop
        if ($backdropEl) {
          let needToHideBackdrop = true;
          if (modal.type === 'popup') {
            modal.$el.prevAll('.popup.modal-in').add(modal.$el.nextAll('.popup.modal-in')).each(popupEl => {
              const popupInstance = popupEl.f7Modal;
              if (!popupInstance) return;
              if (popupInstance.params.closeByBackdropClick && popupInstance.params.backdrop && popupInstance.backdropEl === modal.backdropEl) {
                needToHideBackdrop = false;
              }
            });
          }
          if (needToHideBackdrop) {
            $backdropEl[animate ? 'removeClass' : 'addClass']('not-animated');
            $backdropEl.removeClass('backdrop-in');
          }
        }

        // Modal
        $el[animate ? 'removeClass' : 'addClass']('not-animated');
        function transitionEnd() {
          if ($el.hasClass('modal-out')) {
            modal.onClosed();
          } else if ($el.hasClass('modal-in')) {
            modal.onOpened();
          }
        }
        if (animate) {
          $el.animationEnd(() => {
            transitionEnd();
          });
          $el.transitionEnd(() => {
            transitionEnd();
          });
          $el.removeClass('modal-in').addClass('modal-out');
          // Emit close
          modal.onClose();
        } else {
          $el.addClass('not-animated').removeClass('modal-in').addClass('modal-out');
          // Emit close
          modal.onClose();
          modal.onClosed();
        }
        if (modal.type === 'dialog') {
          clearDialogsQueue();
        }
        return modal;
      }
      destroy() {
        const modal = this;
        if (modal.destroyed) return;
        modal.emit(`local::beforeDestroy modalBeforeDestroy ${modal.type}BeforeDestroy`, modal);
        if (modal.$el) {
          modal.$el.trigger(`modal:beforedestroy ${modal.type.toLowerCase()}:beforedestroy`);
          if (modal.$el.length && modal.$el[0].f7Modal) {
            delete modal.$el[0].f7Modal;
          }
        }
        deleteProps$1(modal);
        modal.destroyed = true;
      }
    };

    class CustomModal extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          backdrop: true,
          closeByBackdropClick: true,
          on: {}
        }, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const customModal = this;
        customModal.params = extendedParams;

        // Find Element
        let $el;
        if (!customModal.params.el) {
          $el = $(customModal.params.content);
        } else {
          $el = $(customModal.params.el);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return customModal.destroy();
        }
        let $backdropEl;
        if (customModal.params.backdrop) {
          $backdropEl = app.$el.children('.custom-modal-backdrop');
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="custom-modal-backdrop"></div>');
            app.$el.append($backdropEl);
          }
        }
        function handleClick(e) {
          if (!customModal || customModal.destroyed) return;
          if ($backdropEl && e.target === $backdropEl[0]) {
            customModal.close();
          }
        }
        customModal.on('customModalOpened', () => {
          if (customModal.params.closeByBackdropClick && customModal.params.backdrop) {
            app.on('click', handleClick);
          }
        });
        customModal.on('customModalClose', () => {
          if (customModal.params.closeByBackdropClick && customModal.params.backdrop) {
            app.off('click', handleClick);
          }
        });
        extend$3(customModal, {
          app,
          $el,
          el: $el[0],
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'customModal'
        });
        $el[0].f7Modal = customModal;
        return customModal;
      }
    }

    var Modal = {
      name: 'modal',
      static: {
        Modal: Modal$1,
        CustomModal
      },
      create() {
        const app = this;
        app.customModal = {
          create(params) {
            return new CustomModal(app, params);
          }
        };
      },
      params: {
        modal: {
          queueDialogs: true
        }
      }
    };

    let Dialog$1 = class Dialog extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          title: app.params.dialog.title,
          text: undefined,
          content: '',
          buttons: [],
          verticalButtons: false,
          onClick: undefined,
          cssClass: undefined,
          destroyOnClose: false,
          on: {}
        }, params);
        if (typeof extendedParams.closeByBackdropClick === 'undefined') {
          extendedParams.closeByBackdropClick = app.params.dialog.closeByBackdropClick;
        }
        if (typeof extendedParams.backdrop === 'undefined') {
          extendedParams.backdrop = app.params.dialog.backdrop;
        }

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const dialog = this;
        const device = getDevice$1();
        const document = getDocument$1();
        const {
          title,
          text,
          content,
          buttons,
          verticalButtons,
          cssClass,
          backdrop
        } = extendedParams;
        dialog.params = extendedParams;

        // Find Element
        let $el;
        if (!dialog.params.el) {
          const dialogClasses = ['dialog'];
          if (buttons.length === 0) dialogClasses.push('dialog-no-buttons');
          if (buttons.length > 0) dialogClasses.push(`dialog-buttons-${buttons.length}`);
          if (verticalButtons) dialogClasses.push('dialog-buttons-vertical');
          if (cssClass) dialogClasses.push(cssClass);
          let buttonsHTML = '';
          if (buttons.length > 0) {
            buttonsHTML = $jsx$1("div", {
              class: "dialog-buttons"
            }, buttons.map(button => $jsx$1("span", {
              class: `dialog-button${button.strong ? ' dialog-button-strong' : ''}${button.color ? ` color-${button.color}` : ''}${button.cssClass ? ` ${button.cssClass}` : ''}`
            }, button.text)));
          }
          const dialogHtml = $jsx$1("div", {
            class: dialogClasses.join(' ')
          }, $jsx$1("div", {
            class: "dialog-inner"
          }, title && $jsx$1("div", {
            class: "dialog-title"
          }, title), text && $jsx$1("div", {
            class: "dialog-text"
          }, text), content), buttonsHTML);
          $el = $(dialogHtml);
        } else {
          $el = $(dialog.params.el);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return dialog.destroy();
        }
        let $backdropEl;
        if (backdrop) {
          $backdropEl = app.$el.children('.dialog-backdrop');
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="dialog-backdrop"></div>');
            app.$el.append($backdropEl);
          }
        }

        // Assign events
        function buttonOnClick(e) {
          const buttonEl = this;
          const index = $(buttonEl).index();
          const button = buttons[index];
          if (button.onClick) button.onClick(dialog, e);
          if (dialog.params.onClick) dialog.params.onClick(dialog, index);
          if (button.close !== false) dialog.close();
        }
        let addKeyboardHander;
        function onKeyDown(e) {
          const keyCode = e.keyCode;
          buttons.forEach((button, index) => {
            if (button.keyCodes && button.keyCodes.indexOf(keyCode) >= 0) {
              if (document.activeElement) document.activeElement.blur();
              if (button.onClick) button.onClick(dialog, e);
              if (dialog.params.onClick) dialog.params.onClick(dialog, index);
              if (button.close !== false) dialog.close();
            }
          });
        }
        if (buttons && buttons.length > 0) {
          dialog.on('open', () => {
            $el.find('.dialog-button').each((buttonEl, index) => {
              const button = buttons[index];
              if (button.keyCodes) addKeyboardHander = true;
              $(buttonEl).on('click', buttonOnClick);
            });
            if (addKeyboardHander && !device.ios && !device.android && !device.cordova && !device.capacitor) {
              $(document).on('keydown', onKeyDown);
            }
          });
          dialog.on('close', () => {
            $el.find('.dialog-button').each(buttonEl => {
              $(buttonEl).off('click', buttonOnClick);
            });
            if (addKeyboardHander && !device.ios && !device.android && !device.cordova && !device.capacitor) {
              $(document).off('keydown', onKeyDown);
            }
            addKeyboardHander = false;
          });
        }
        extend$3(dialog, {
          app,
          $el,
          el: $el[0],
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'dialog',
          setProgress(progress, duration) {
            app.progressbar.set($el.find('.progressbar'), progress, duration);
            return dialog;
          },
          setText(newText) {
            let $textEl = $el.find('.dialog-text');
            if ($textEl.length === 0) {
              $textEl = $('<div class="dialog-text"></div>');
              if (typeof title !== 'undefined') {
                $textEl.insertAfter($el.find('.dialog-title'));
              } else {
                $el.find('.dialog-inner').prepend($textEl);
              }
            }
            $textEl.html(newText);
            dialog.params.text = newText;
            return dialog;
          },
          setTitle(newTitle) {
            let $titleEl = $el.find('.dialog-title');
            if ($titleEl.length === 0) {
              $titleEl = $('<div class="dialog-title"></div>');
              $el.find('.dialog-inner').prepend($titleEl);
            }
            $titleEl.html(newTitle);
            dialog.params.title = newTitle;
            return dialog;
          }
        });
        function handleClick(e) {
          const target = e.target;
          const $target = $(target);
          if ($target.closest(dialog.el).length === 0) {
            if (dialog.params.closeByBackdropClick && dialog.backdropEl && dialog.backdropEl === target) {
              dialog.close();
            }
          }
        }
        dialog.on('opened', () => {
          if (dialog.params.closeByBackdropClick) {
            app.on('click', handleClick);
          }
        });
        dialog.on('close', () => {
          if (dialog.params.closeByBackdropClick) {
            app.off('click', handleClick);
          }
        });
        $el[0].f7Modal = dialog;
        if (dialog.params.destroyOnClose) {
          dialog.once('closed', () => {
            setTimeout(() => {
              dialog.destroy();
            }, 0);
          });
        }
        return dialog;
      }
    };

    var Dialog = {
      name: 'dialog',
      params: {
        dialog: {
          title: undefined,
          buttonOk: 'OK',
          buttonCancel: 'Cancel',
          usernamePlaceholder: 'Username',
          passwordPlaceholder: 'Password',
          preloaderTitle: 'Loading... ',
          progressTitle: 'Loading... ',
          backdrop: true,
          closeByBackdropClick: false,
          destroyPredefinedDialogs: true,
          keyboardActions: true,
          autoFocus: true
        }
      },
      static: {
        Dialog: Dialog$1
      },
      create() {
        const app = this;
        function defaultDialogTitle() {
          return app.params.dialog.title || app.name;
        }
        const destroyOnClose = app.params.dialog.destroyPredefinedDialogs;
        const keyboardActions = app.params.dialog.keyboardActions;
        const autoFocus = app.params.dialog.autoFocus;
        const autoFocusHandler = autoFocus ? {
          on: {
            opened(dialog) {
              dialog.$el.find('input').eq(0).focus();
            }
          }
        } : {};
        const isIosTheme = app.theme === 'ios';
        app.dialog = extend$3(ModalMethods({
          app,
          constructor: Dialog$1,
          defaultSelector: '.dialog.modal-in'
        }), {
          // Shortcuts
          alert() {
            for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
              args[_key] = arguments[_key];
            }
            let [text, title, callbackOk] = args;
            if (args.length === 2 && typeof args[1] === 'function') {
              [text, callbackOk, title] = args;
            }
            return new Dialog$1(app, {
              title: typeof title === 'undefined' ? defaultDialogTitle() : title,
              text,
              buttons: [{
                text: app.params.dialog.buttonOk,
                strong: isIosTheme,
                onClick: callbackOk,
                keyCodes: keyboardActions ? [13, 27] : null
              }],
              destroyOnClose
            }).open();
          },
          prompt() {
            for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
              args[_key2] = arguments[_key2];
            }
            let [text, title, callbackOk, callbackCancel, defaultValue] = args;
            if (typeof args[1] === 'function') {
              [text, callbackOk, callbackCancel, defaultValue, title] = args;
            }
            defaultValue = typeof defaultValue === 'undefined' || defaultValue === null ? '' : defaultValue;
            return new Dialog$1(app, {
              title: typeof title === 'undefined' ? defaultDialogTitle() : title,
              text,
              content: `<div class="dialog-input-field input"><input type="text" class="dialog-input" value="${defaultValue}"></div>`,
              buttons: [{
                text: app.params.dialog.buttonCancel,
                keyCodes: keyboardActions ? [27] : null,
                color: null
              }, {
                text: app.params.dialog.buttonOk,
                strong: isIosTheme,
                keyCodes: keyboardActions ? [13] : null
              }],
              onClick(dialog, index) {
                const inputValue = dialog.$el.find('.dialog-input').val();
                if (index === 0 && callbackCancel) callbackCancel(inputValue);
                if (index === 1 && callbackOk) callbackOk(inputValue);
              },
              destroyOnClose,
              ...autoFocusHandler
            }).open();
          },
          confirm() {
            for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
              args[_key3] = arguments[_key3];
            }
            let [text, title, callbackOk, callbackCancel] = args;
            if (typeof args[1] === 'function') {
              [text, callbackOk, callbackCancel, title] = args;
            }
            return new Dialog$1(app, {
              title: typeof title === 'undefined' ? defaultDialogTitle() : title,
              text,
              buttons: [{
                text: app.params.dialog.buttonCancel,
                onClick: callbackCancel,
                keyCodes: keyboardActions ? [27] : null,
                color: null
              }, {
                text: app.params.dialog.buttonOk,
                strong: isIosTheme,
                onClick: callbackOk,
                keyCodes: keyboardActions ? [13] : null
              }],
              destroyOnClose
            }).open();
          },
          login() {
            for (var _len4 = arguments.length, args = new Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
              args[_key4] = arguments[_key4];
            }
            let [text, title, callbackOk, callbackCancel] = args;
            if (typeof args[1] === 'function') {
              [text, callbackOk, callbackCancel, title] = args;
            }
            return new Dialog$1(app, {
              title: typeof title === 'undefined' ? defaultDialogTitle() : title,
              text,
              // prettier-ignore
              content: `
              <div class="dialog-input-field dialog-input-double input">
                <input type="text" name="dialog-username" placeholder="${app.params.dialog.usernamePlaceholder}" class="dialog-input">
              </div>
              <div class="dialog-input-field dialog-input-double input">
                <input type="password" name="dialog-password" placeholder="${app.params.dialog.passwordPlaceholder}" class="dialog-input">
              </div>`,
              buttons: [{
                text: app.params.dialog.buttonCancel,
                keyCodes: keyboardActions ? [27] : null,
                color: null
              }, {
                text: app.params.dialog.buttonOk,
                strong: isIosTheme,
                keyCodes: keyboardActions ? [13] : null
              }],
              onClick(dialog, index) {
                const username = dialog.$el.find('[name="dialog-username"]').val();
                const password = dialog.$el.find('[name="dialog-password"]').val();
                if (index === 0 && callbackCancel) callbackCancel(username, password);
                if (index === 1 && callbackOk) callbackOk(username, password);
              },
              destroyOnClose,
              ...autoFocusHandler
            }).open();
          },
          password() {
            for (var _len5 = arguments.length, args = new Array(_len5), _key5 = 0; _key5 < _len5; _key5++) {
              args[_key5] = arguments[_key5];
            }
            let [text, title, callbackOk, callbackCancel] = args;
            if (typeof args[1] === 'function') {
              [text, callbackOk, callbackCancel, title] = args;
            }
            return new Dialog$1(app, {
              title: typeof title === 'undefined' ? defaultDialogTitle() : title,
              text,
              // prettier-ignore
              content: `
              <div class="dialog-input-field input">
                <input type="password" name="dialog-password" placeholder="${app.params.dialog.passwordPlaceholder}" class="dialog-input">
              </div>`,
              buttons: [{
                text: app.params.dialog.buttonCancel,
                keyCodes: keyboardActions ? [27] : null,
                color: null
              }, {
                text: app.params.dialog.buttonOk,
                strong: isIosTheme,
                keyCodes: keyboardActions ? [13] : null
              }],
              onClick(dialog, index) {
                const password = dialog.$el.find('[name="dialog-password"]').val();
                if (index === 0 && callbackCancel) callbackCancel(password);
                if (index === 1 && callbackOk) callbackOk(password);
              },
              destroyOnClose,
              ...autoFocusHandler
            }).open();
          },
          preloader(title, color) {
            const preloaders = {
              iosPreloaderContent,
              mdPreloaderContent
            };
            const preloaderInner = preloaders[`${app.theme}PreloaderContent`] || '';
            return new Dialog$1(app, {
              title: typeof title === 'undefined' || title === null ? app.params.dialog.preloaderTitle : title,
              // prettier-ignore
              content: `<div class="preloader${color ? ` color-${color}` : ''}">${preloaderInner}</div>`,
              cssClass: 'dialog-preloader',
              destroyOnClose
            }).open();
          },
          progress() {
            for (var _len6 = arguments.length, args = new Array(_len6), _key6 = 0; _key6 < _len6; _key6++) {
              args[_key6] = arguments[_key6];
            }
            let [title, progress, color] = args;
            if (args.length === 2) {
              if (typeof args[0] === 'number') {
                [progress, color, title] = args;
              } else if (typeof args[0] === 'string' && typeof args[1] === 'string') {
                [title, color, progress] = args;
              }
            } else if (args.length === 1) {
              if (typeof args[0] === 'number') {
                [progress, title, color] = args;
              }
            }
            const infinite = typeof progress === 'undefined';
            const dialog = new Dialog$1(app, {
              title: typeof title === 'undefined' ? app.params.dialog.progressTitle : title,
              cssClass: 'dialog-progress',
              // prettier-ignore
              content: `
              <div class="progressbar${infinite ? '-infinite' : ''}${color ? ` color-${color}` : ''}">
                ${!infinite ? '<span></span>' : ''}
              </div>
            `,
              destroyOnClose
            });
            if (!infinite) dialog.setProgress(progress);
            return dialog.open();
          }
        });
      }
    };

    let Popup$1 = class Popup extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, app.params.popup, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const popup = this;
        const window = getWindow$1();
        const document = getDocument$1();
        const support = getSupport$1();
        const device = getDevice$1();
        popup.params = extendedParams;

        // Find Element
        let $el;
        if (!popup.params.el) {
          $el = $(popup.params.content).filter(node => node.nodeType === 1).eq(0);
        } else {
          $el = $(popup.params.el).eq(0);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return popup.destroy();
        }
        let $backdropEl;
        if (popup.params.backdrop && popup.params.backdropEl) {
          $backdropEl = $(popup.params.backdropEl);
        } else if (popup.params.backdrop) {
          if (popup.params.backdropUnique) {
            $backdropEl = $('<div class="popup-backdrop popup-backdrop-unique"></div>');
            popup.$containerEl.append($backdropEl);
          } else {
            $backdropEl = popup.$containerEl.children('.popup-backdrop');
          }
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="popup-backdrop"></div>');
            popup.$containerEl.append($backdropEl);
          }
        }
        extend$3(popup, {
          app,
          push: $el.hasClass('popup-push') || popup.params.push,
          $el,
          el: $el[0],
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'popup',
          $htmlEl: $('html')
        });
        if (popup.params.push) {
          $el.addClass('popup-push');
        }
        function handleClick(e) {
          const target = e.target;
          const $target = $(target);
          const keyboardOpened = !device.desktop && device.cordova && (window.Keyboard && window.Keyboard.isVisible || window.cordova.plugins && window.cordova.plugins.Keyboard && window.cordova.plugins.Keyboard.isVisible);
          if (keyboardOpened) return;
          if ($target.closest(popup.el).length === 0) {
            if (popup.params && popup.params.closeByBackdropClick && popup.params.backdrop && popup.backdropEl && popup.backdropEl === target) {
              let needToClose = true;
              popup.$el.nextAll('.popup.modal-in').each(popupEl => {
                const popupInstance = popupEl.f7Modal;
                if (!popupInstance) return;
                if (popupInstance.params.closeByBackdropClick && popupInstance.params.backdrop && popupInstance.backdropEl === popup.backdropEl) {
                  needToClose = false;
                }
              });
              if (needToClose) {
                popup.close();
              }
            }
          }
        }
        function onKeyDown(e) {
          const keyCode = e.keyCode;
          if (keyCode === 27 && popup.params.closeOnEscape) {
            popup.close();
          }
        }
        let pushOffset;
        let isPush;
        function pushViewScale(offset) {
          return (app.height - offset * 2) / app.height;
        }
        let allowSwipeToClose = true;
        let isTouched = false;
        let startTouch;
        let currentTouch;
        let isScrolling;
        let touchStartTime;
        let touchesDiff;
        let isMoved = false;
        let pageContentEl;
        let pageContentScrollTop;
        let pageContentOffsetHeight;
        let pageContentScrollHeight;
        let popupHeight;
        let $pushEl;
        function handleTouchStart(e) {
          if (isTouched || !allowSwipeToClose || !popup.params.swipeToClose || !e.isTrusted) return;
          if (popup.params.swipeHandler && $(e.target).closest(popup.params.swipeHandler).length === 0) {
            return;
          }
          if ($(e.target).closest('.sortable-handler').length > 0) return;
          isTouched = true;
          isMoved = false;
          startTouch = {
            x: e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX,
            y: e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY
          };
          touchStartTime = now$1();
          isScrolling = undefined;
          if (!popup.params.swipeHandler && e.type === 'touchstart') {
            pageContentEl = $(e.target).closest('.page-content')[0];
          }
        }
        function handleTouchMove(e) {
          if (!isTouched || !e.isTrusted) return;
          currentTouch = {
            x: e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX,
            y: e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY
          };
          if (typeof isScrolling === 'undefined') {
            isScrolling = !!(isScrolling || Math.abs(currentTouch.x - startTouch.x) > Math.abs(currentTouch.y - startTouch.y));
          }
          if (isScrolling) {
            isTouched = false;
            isMoved = false;
            return;
          }
          touchesDiff = startTouch.y - currentTouch.y;
          if (isPush && pushOffset && touchesDiff > 0) {
            touchesDiff = 0;
          }
          const direction = touchesDiff < 0 ? 'to-bottom' : 'to-top';
          $el.transition(0);
          if (typeof popup.params.swipeToClose === 'string' && direction !== popup.params.swipeToClose) {
            $el.transform('');
            $el.transition('');
            return;
          }
          if (!isMoved) {
            if (isPush && pushOffset) {
              popupHeight = $el[0].offsetHeight;
              $pushEl = $el.prevAll('.popup.modal-in').eq(0);
              if ($pushEl.length === 0) {
                $pushEl = app.$el.children('.view, .views');
              }
            }
            if (pageContentEl) {
              pageContentScrollTop = pageContentEl.scrollTop;
              pageContentScrollHeight = pageContentEl.scrollHeight;
              pageContentOffsetHeight = pageContentEl.offsetHeight;
              if (!(pageContentScrollHeight === pageContentOffsetHeight) && !(direction === 'to-bottom' && pageContentScrollTop === 0) && !(direction === 'to-top' && pageContentScrollTop === pageContentScrollHeight - pageContentOffsetHeight)) {
                $el.transform('');
                $el.transition('');
                isTouched = false;
                isMoved = false;
                return;
              }
            }
            isMoved = true;
            popup.emit('local::swipeStart popupSwipeStart', popup);
            popup.$el.trigger('popup:swipestart');
          } else {
            popup.emit('local::swipeMove popupSwipeMove', popup);
            popup.$el.trigger('popup:swipemove');
          }
          e.preventDefault();
          if (isPush && pushOffset) {
            const pushProgress = 1 - Math.abs(touchesDiff / popupHeight);
            const scale = 1 - (1 - pushViewScale(pushOffset)) * pushProgress;
            if ($pushEl.hasClass('popup')) {
              if ($pushEl.hasClass('popup-push')) {
                $pushEl.transition(0).forEach(el => {
                  el.style.setProperty('transform', `translate3d(0, calc(-1 * ${pushProgress} * (var(--f7-popup-push-offset) + 10px)) , 0px) scale(${scale})`, 'important');
                });
              } else {
                $pushEl.transition(0).forEach(el => {
                  el.style.setProperty('transform', `translate3d(0, 0px , 0px) scale(${scale})`, 'important');
                });
              }
            } else {
              $pushEl.transition(0).forEach(el => {
                el.style.setProperty('transform', `translate3d(0,0,0) scale(${scale})`, 'important');
              });
            }
          }
          $el.transition(0).transform(`translate3d(0,${-touchesDiff}px,0)`);
        }
        function handleTouchEnd(e) {
          if (!e.isTrusted) return;
          isTouched = false;
          if (!isMoved) {
            return;
          }
          popup.emit('local::swipeEnd popupSwipeEnd', popup);
          popup.$el.trigger('popup:swipeend');
          isMoved = false;
          allowSwipeToClose = false;
          $el.transition('');
          if (isPush && pushOffset) {
            $pushEl.transition('').transform('');
          }
          const direction = touchesDiff <= 0 ? 'to-bottom' : 'to-top';
          if (typeof popup.params.swipeToClose === 'string' && direction !== popup.params.swipeToClose) {
            $el.transform('');
            allowSwipeToClose = true;
            return;
          }
          const diff = Math.abs(touchesDiff);
          const timeDiff = new Date().getTime() - touchStartTime;
          if (timeDiff < 300 && diff > 20 || timeDiff >= 300 && diff > 100) {
            nextTick$1(() => {
              if (direction === 'to-bottom') {
                $el.addClass('swipe-close-to-bottom');
              } else {
                $el.addClass('swipe-close-to-top');
              }
              $el.transform('');
              popup.emit('local::swipeclose popupSwipeClose', popup);
              popup.$el.trigger('popup:swipeclose');
              popup.close();
              allowSwipeToClose = true;
            });
            return;
          }
          allowSwipeToClose = true;
          $el.transform('');
        }
        const passive = support.passiveListener ? {
          passive: true
        } : false;
        if (popup.params.swipeToClose) {
          $el.on(app.touchEvents.start, handleTouchStart, passive);
          app.on('touchmove', handleTouchMove);
          app.on('touchend:passive', handleTouchEnd);
          popup.once('popupDestroy', () => {
            $el.off(app.touchEvents.start, handleTouchStart, passive);
            app.off('touchmove', handleTouchMove);
            app.off('touchend:passive', handleTouchEnd);
          });
        }
        let hasPreviousPushPopup;
        const updatePushOffset = () => {
          const wasPush = isPush;
          if (popup.push) {
            isPush = popup.push && (app.width < 630 || app.height < 630 || $el.hasClass('popup-tablet-fullscreen'));
          }
          if (isPush && !wasPush) {
            // eslint-disable-next-line
            setPushOffset();
          } else if (isPush && wasPush) {
            popup.$htmlEl[0].style.setProperty('--f7-popup-push-scale', pushViewScale(pushOffset));
          } else if (!isPush && wasPush) {
            popup.$htmlEl.removeClass('with-modal-popup-push');
            popup.$htmlEl[0].style.removeProperty('--f7-popup-push-scale');
          }
        };
        const setPushOffset = () => {
          app.off('resize', updatePushOffset);
          if (popup.push) {
            isPush = popup.push && (app.width < 630 || app.height < 630 || $el.hasClass('popup-tablet-fullscreen'));
          }
          if (isPush) {
            pushOffset = parseInt($el.css('--f7-popup-push-offset'), 10);
            if (Number.isNaN(pushOffset)) {
              pushOffset = 0;
            }
            if (!pushOffset) pushOffset = app.theme === 'ios' ? 44 : 48;
            popup.$htmlEl[0].style.setProperty('--f7-popup-push-offset', `${pushOffset}px`);
            $el.addClass('popup-push');
            popup.$htmlEl.addClass('with-modal-popup-push');
            popup.$htmlEl[0].style.setProperty('--f7-popup-push-scale', pushViewScale(pushOffset));
          }
          app.on('resize', updatePushOffset);
        };
        popup.on('open', () => {
          hasPreviousPushPopup = false;
          if (popup.params.closeOnEscape) {
            $(document).on('keydown', onKeyDown);
          }
          $el.prevAll('.popup.modal-in').addClass('popup-behind');
          setPushOffset();
        });
        popup.on('opened', () => {
          $el.removeClass('swipe-close-to-bottom swipe-close-to-top');
          if (popup.params.closeByBackdropClick) {
            app.on('click', handleClick);
          }
        });
        popup.on('close', () => {
          hasPreviousPushPopup = popup.$el.prevAll('.popup-push.modal-in').length > 0;
          if (popup.params.closeOnEscape) {
            $(document).off('keydown', onKeyDown);
          }
          if (popup.params.closeByBackdropClick) {
            app.off('click', handleClick);
          }
          $el.prevAll('.popup.modal-in').eq(0).removeClass('popup-behind');
          if (isPush && pushOffset && !hasPreviousPushPopup) {
            popup.$htmlEl.removeClass('with-modal-popup-push');
            popup.$htmlEl.addClass('with-modal-popup-push-closing');
          }
          app.off('resize', updatePushOffset);
        });
        popup.on('closed', () => {
          $el.removeClass('popup-behind');
          if (isPush && pushOffset && !hasPreviousPushPopup) {
            popup.$htmlEl.removeClass('with-modal-popup-push-closing');
            popup.$htmlEl[0].style.removeProperty('--f7-popup-push-scale');
            popup.$htmlEl[0].style.removeProperty('--f7-popup-push-offset');
          }
        });
        $el[0].f7Modal = popup;
        return popup;
      }
    };

    var Popup = {
      name: 'popup',
      params: {
        popup: {
          backdrop: true,
          backdropEl: undefined,
          backdropUnique: false,
          closeByBackdropClick: true,
          closeOnEscape: false,
          swipeToClose: false,
          swipeHandler: null,
          push: false,
          containerEl: null
        }
      },
      static: {
        Popup: Popup$1
      },
      create() {
        const app = this;
        app.popup = ModalMethods({
          app,
          constructor: Popup$1,
          defaultSelector: '.popup.modal-in',
          parentSelector: '.popup'
        });
      },
      clicks: {
        '.popup-open': function openPopup($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.popup.open(data.popup, data.animate, $clickedEl);
        },
        '.popup-close': function closePopup($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.popup.close(data.popup, data.animate, $clickedEl);
        }
      }
    };

    let LoginScreen$1 = class LoginScreen extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const loginScreen = this;
        loginScreen.params = extendedParams;

        // Find Element
        let $el;
        if (!loginScreen.params.el) {
          $el = $(loginScreen.params.content).filter(node => node.nodeType === 1).eq(0);
        } else {
          $el = $(loginScreen.params.el).eq(0);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return loginScreen.destroy();
        }
        extend$3(loginScreen, {
          app,
          $el,
          el: $el[0],
          type: 'loginScreen'
        });
        $el[0].f7Modal = loginScreen;
        return loginScreen;
      }
    };

    var LoginScreen = {
      name: 'loginScreen',
      static: {
        LoginScreen: LoginScreen$1
      },
      create() {
        const app = this;
        app.loginScreen = ModalMethods({
          app,
          constructor: LoginScreen$1,
          defaultSelector: '.login-screen.modal-in'
        });
      },
      clicks: {
        '.login-screen-open': function openLoginScreen($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.loginScreen.open(data.loginScreen, data.animate, $clickedEl);
        },
        '.login-screen-close': function closeLoginScreen($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.loginScreen.close(data.loginScreen, data.animate, $clickedEl);
        }
      }
    };

    let Popover$1 = class Popover extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, app.params.popover, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const popover = this;
        const device = getDevice$1();
        const window = getWindow$1();
        const document = getDocument$1();
        popover.params = extendedParams;

        // Find Element
        let $el;
        if (!popover.params.el) {
          $el = $(popover.params.content).filter(node => node.nodeType === 1).eq(0);
        } else {
          $el = $(popover.params.el).eq(0);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }

        // Find Target
        const $targetEl = $(popover.params.targetEl).eq(0);
        if ($el.length === 0) {
          return popover.destroy();
        }

        // Backdrop
        let $backdropEl;
        const forceBackdropUnique = popover.params.backdrop && app.$el.find('.popover.modal-in').filter(anotherPopoverEl => anotherPopoverEl !== $el[0]).length > 0;
        if (popover.params.backdrop && popover.params.backdropEl) {
          $backdropEl = $(popover.params.backdropEl);
        } else if (popover.params.backdrop) {
          if (popover.params.backdropUnique || forceBackdropUnique) {
            $backdropEl = $('<div class="popover-backdrop popover-backdrop-unique"></div>');
            $backdropEl[0].f7PopoverRef = popover;
            popover.$containerEl.append($backdropEl);
          } else {
            $backdropEl = popover.$containerEl.children('.popover-backdrop');
          }
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="popover-backdrop"></div>');
            popover.$containerEl.append($backdropEl);
          }
        }

        // Find Arrow
        let $arrowEl;
        if ($el.find('.popover-arrow').length === 0 && popover.params.arrow) {
          $arrowEl = $('<div class="popover-arrow"></div>');
          $el.prepend($arrowEl);
        } else {
          $arrowEl = $el.find('.popover-arrow');
        }

        // Open
        const originalOpen = popover.open;
        extend$3(popover, {
          app,
          $el,
          el: $el[0],
          $targetEl,
          targetEl: $targetEl[0],
          $arrowEl,
          arrowEl: $arrowEl[0],
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'popover',
          forceBackdropUnique,
          open() {
            for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
              args[_key] = arguments[_key];
            }
            let [targetEl, animate] = args;
            if (typeof args[0] === 'boolean') [animate, targetEl] = args;
            if (targetEl) {
              popover.$targetEl = $(targetEl);
              popover.targetEl = popover.$targetEl[0];
            }
            return originalOpen.call(popover, animate);
          }
        });
        function handleResize() {
          popover.resize();
        }
        popover.on('popoverOpen', () => {
          popover.resize();
          app.on('resize', handleResize);
          $(window).on('keyboardDidShow keyboardDidHide', handleResize);
          popover.on('popoverClose popoverBeforeDestroy', () => {
            app.off('resize', handleResize);
            $(window).off('keyboardDidShow keyboardDidHide', handleResize);
          });
        });
        let touchStartTarget = null;
        function handleTouchStart(e) {
          touchStartTarget = e.target;
        }
        function handleClick(e) {
          const target = e.target;
          const $target = $(target);
          const keyboardOpened = !device.desktop && device.cordova && (window.Keyboard && window.Keyboard.isVisible || window.cordova.plugins && window.cordova.plugins.Keyboard && window.cordova.plugins.Keyboard.isVisible);
          if (keyboardOpened) return;
          if ($target.closest(popover.el).length === 0) {
            if (popover.params.closeByBackdropClick && popover.params.backdrop && popover.backdropEl && popover.backdropEl === target && touchStartTarget === target) {
              popover.close();
            } else if (popover.params.closeByOutsideClick && touchStartTarget === target) {
              const isAnotherPopoverBackdrop = $target.hasClass('popover-backdrop-unique') && target.f7PopoverRef !== popover || $target.hasClass('popover-backdrop') && target !== popover.backdropEl;
              const isAnotherPopoverTarget = target.closest('.popover') && target.closest('.popover') !== popover.$el[0];
              if (!isAnotherPopoverBackdrop && !isAnotherPopoverTarget) {
                popover.close();
              }
            }
          }
        }
        function onKeyDown(e) {
          const keyCode = e.keyCode;
          if (keyCode === 27 && popover.params.closeOnEscape) {
            popover.close();
          }
        }
        if (popover.params.closeOnEscape) {
          popover.on('popoverOpen', () => {
            $(document).on('keydown', onKeyDown);
          });
          popover.on('popoverClose', () => {
            $(document).off('keydown', onKeyDown);
          });
        }
        popover.on('popoverOpened', () => {
          if (popover.params.closeByOutsideClick || popover.params.closeByBackdropClick) {
            app.on('touchstart', handleTouchStart);
            app.on('click', handleClick);
          }
        });
        popover.on('popoverClose', () => {
          if (popover.params.closeByOutsideClick || popover.params.closeByBackdropClick) {
            app.off('touchstart', handleTouchStart);
            app.off('click', handleClick);
          }
        });
        $el[0].f7Modal = popover;
        return popover;
      }
      resize() {
        const popover = this;
        const {
          app,
          $el,
          $targetEl,
          $arrowEl
        } = popover;
        const {
          targetX,
          targetY,
          verticalPosition
        } = popover.params;
        $el.css({
          left: '',
          top: ''
        });
        const [width, height] = [$el.width(), $el.height()];
        let arrowSize = 0;
        let arrowLeft;
        let arrowTop;
        const hasArrow = $arrowEl.length > 0;
        const arrowMin = app.theme === 'ios' ? 13 : 24;
        if (hasArrow) {
          $arrowEl.removeClass('on-left on-right on-top on-bottom').css({
            left: '',
            top: ''
          });
          arrowSize = $arrowEl.width() / 2;
        }
        $el.removeClass('popover-on-left popover-on-right popover-on-top popover-on-bottom popover-on-middle').css({
          left: '',
          top: ''
        });
        let targetWidth;
        let targetHeight;
        let targetOffsetLeft;
        let targetOffsetTop;
        let safeAreaTop = parseInt($('html').css('--f7-safe-area-top'), 10);
        let safeAreaLeft = parseInt($('html').css('--f7-safe-area-left'), 10);
        let safeAreaRight = parseInt($('html').css('--f7-safe-area-right'), 10);
        if (Number.isNaN(safeAreaTop)) safeAreaTop = 0;
        if (Number.isNaN(safeAreaLeft)) safeAreaLeft = 0;
        if (Number.isNaN(safeAreaRight)) safeAreaRight = 0;
        if ($targetEl && $targetEl.length > 0) {
          targetWidth = $targetEl.outerWidth();
          targetHeight = $targetEl.outerHeight();
          const targetOffset = $targetEl.offset();
          targetOffsetLeft = targetOffset.left - app.left;
          targetOffsetTop = targetOffset.top - app.top;
          const targetParentPage = $targetEl.parents('.page');
          if (targetParentPage.length > 0) {
            targetOffsetTop -= targetParentPage[0].scrollTop;
          }
        } else if (typeof targetX !== 'undefined' && targetY !== 'undefined') {
          targetOffsetLeft = targetX;
          targetOffsetTop = targetY;
          targetWidth = popover.params.targetWidth || 0;
          targetHeight = popover.params.targetHeight || 0;
        }
        let [left, top, diff] = [0, 0, 0];
        // Top Position
        const forcedPosition = verticalPosition === 'auto' ? false : verticalPosition;
        let position = forcedPosition || 'top';
        if (forcedPosition === 'top' || !forcedPosition && height + arrowSize < targetOffsetTop - safeAreaTop) {
          // On top
          top = targetOffsetTop - height - arrowSize;
        } else if (forcedPosition === 'bottom' || !forcedPosition && height + arrowSize < app.height - targetOffsetTop - targetHeight) {
          // On bottom
          position = 'bottom';
          top = targetOffsetTop + targetHeight + arrowSize;
        } else {
          // On middle
          position = 'middle';
          top = targetHeight / 2 + targetOffsetTop - height / 2;
          diff = top;
          top = Math.max(5, Math.min(top, app.height - height - 5));
          diff -= top;
        }

        // Horizontal Position
        if (position === 'top' || position === 'bottom') {
          left = targetWidth / 2 + targetOffsetLeft - width / 2;
          diff = left;
          left = Math.max(5, Math.min(left, app.width - width - 5));
          if (safeAreaLeft) {
            left = Math.max(left, safeAreaLeft);
          }
          if (safeAreaRight && left + width > app.width - 5 - safeAreaRight) {
            left = app.width - 5 - safeAreaRight - width;
          }
          diff -= left;
          if (hasArrow) {
            if (position === 'top') {
              $arrowEl.addClass('on-bottom');
            }
            if (position === 'bottom') {
              $arrowEl.addClass('on-top');
            }
            arrowLeft = width / 2 - arrowSize + diff;
            arrowLeft = Math.max(Math.min(arrowLeft, width - arrowSize * 2 - arrowMin), arrowMin);
            $arrowEl.css({
              left: `${arrowLeft}px`
            });
          }
        } else if (position === 'middle') {
          left = targetOffsetLeft - width - arrowSize;
          if (hasArrow) $arrowEl.addClass('on-right');
          if (left < 5 || left + width + safeAreaRight > app.width || left < safeAreaLeft) {
            if (left < 5) left = targetOffsetLeft + targetWidth + arrowSize;
            if (left + width + safeAreaRight > app.width) left = app.width - width - 5 - safeAreaRight;
            if (left < safeAreaLeft) left = safeAreaLeft;
            if (hasArrow) $arrowEl.removeClass('on-right').addClass('on-left');
          }
          if (hasArrow) {
            arrowTop = height / 2 - arrowSize + diff;
            arrowTop = Math.max(Math.min(arrowTop, height - arrowSize * 2 - arrowMin), arrowMin);
            $arrowEl.css({
              top: `${arrowTop}px`
            });
          }
        }

        // Horizontal Position
        let hPosition;
        if (targetOffsetLeft < app.width / 2) {
          hPosition = 'right';
        } else {
          hPosition = 'left';
        }
        $el.addClass(`popover-on-${position} popover-on-${hPosition}`);

        // Apply Styles
        $el.css({
          top: `${top}px`,
          left: `${left}px`
        });
      }
    };

    var Popover = {
      name: 'popover',
      params: {
        popover: {
          verticalPosition: 'auto',
          arrow: true,
          backdrop: true,
          backdropEl: undefined,
          backdropUnique: false,
          closeByBackdropClick: true,
          closeByOutsideClick: true,
          closeOnEscape: false,
          containerEl: null
        }
      },
      static: {
        Popover: Popover$1
      },
      create() {
        const app = this;
        app.popover = extend$3(ModalMethods({
          app,
          constructor: Popover$1,
          defaultSelector: '.popover.modal-in'
        }), {
          open(popoverEl, targetEl, animate) {
            let $popoverEl = $(popoverEl);
            if ($popoverEl.length > 1) {
              // check if same popover in other page
              const $targetPage = $(targetEl).parents('.page');
              if ($targetPage.length) {
                $popoverEl.each(el => {
                  const $el = $(el);
                  if ($el.parents($targetPage)[0] === $targetPage[0]) {
                    $popoverEl = $el;
                  }
                });
              }
            }
            if ($popoverEl.length > 1) {
              $popoverEl = $popoverEl.eq($popoverEl.length - 1);
            }
            let popover = $popoverEl[0].f7Modal;
            const data = $popoverEl.dataset();
            if (!popover) {
              popover = new Popover$1(app, Object.assign({
                el: $popoverEl,
                targetEl
              }, data));
            }
            return popover.open(targetEl, animate);
          }
        });
      },
      clicks: {
        '.popover-open': function openPopover($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.popover.open(data.popover, $clickedEl, data.animate);
        },
        '.popover-close': function closePopover($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.popover.close(data.popover, data.animate, $clickedEl);
        }
      }
    };

    /* eslint indent: ["off"] */
    let Actions$1 = class Actions extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, app.params.actions, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const actions = this;
        const device = getDevice$1();
        const window = getWindow$1();
        const document = getDocument$1();
        actions.params = extendedParams;

        // Buttons
        let groups;
        if (actions.params.buttons) {
          groups = actions.params.buttons;
          if (!Array.isArray(groups[0])) groups = [groups];
        }
        actions.groups = groups;

        // Find Element
        let $el;
        if (actions.params.el) {
          $el = $(actions.params.el).eq(0);
        } else if (actions.params.content) {
          $el = $(actions.params.content).filter(node => node.nodeType === 1).eq(0);
        } else if (actions.params.buttons) {
          if (actions.params.convertToPopover) {
            actions.popoverHtml = actions.renderPopover();
          }
          actions.actionsHtml = actions.render();
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el && $el.length === 0 && !(actions.actionsHtml || actions.popoverHtml)) {
          return actions.destroy();
        }

        // Backdrop
        let $backdropEl;
        if (actions.params.backdrop && actions.params.backdropEl) {
          $backdropEl = $(actions.params.backdropEl);
        } else if (actions.params.backdrop) {
          if (actions.params.backdropUnique) {
            $backdropEl = $('<div class="popup-backdrop popup-backdrop-unique"></div>');
            actions.$containerEl.append($backdropEl);
          } else {
            $backdropEl = actions.$containerEl.children('.actions-backdrop');
          }
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="actions-backdrop"></div>');
            actions.$containerEl.append($backdropEl);
          }
        }
        const originalOpen = actions.open;
        const originalClose = actions.close;
        let popover;
        function buttonOnClick(e) {
          const $buttonEl = $(this);
          let buttonIndex;
          let groupIndex;
          if ($buttonEl.hasClass('list-button') || $buttonEl.hasClass('item-link')) {
            buttonIndex = $buttonEl.parents('li').index();
            groupIndex = $buttonEl.parents('.list').index();
          } else {
            buttonIndex = $buttonEl.index();
            groupIndex = $buttonEl.parents('.actions-group').index();
          }
          if (typeof groups !== 'undefined') {
            const button = groups[groupIndex][buttonIndex];
            if (button.onClick) button.onClick(actions, e);
            if (actions.params.onClick) actions.params.onClick(actions, e);
            if (button.close !== false) actions.close();
          }
        }
        actions.open = function open(animate) {
          let convertToPopover = false;
          const {
            targetEl,
            targetX,
            targetY,
            targetWidth,
            targetHeight
          } = actions.params;
          if (actions.params.convertToPopover && (targetEl || targetX !== undefined && targetY !== undefined)) {
            // Popover
            if (actions.params.forceToPopover || device.ios && device.ipad || app.width >= 768 || device.desktop) {
              convertToPopover = true;
            }
          }
          if (convertToPopover && actions.popoverHtml) {
            popover = app.popover.create({
              containerEl: actions.params.containerEl,
              content: actions.popoverHtml,
              backdrop: actions.params.backdrop,
              targetEl,
              targetX,
              targetY,
              targetWidth,
              targetHeight,
              on: {
                open() {
                  if (!actions.$el) {
                    actions.$el = popover.$el;
                  }
                  actions.$el.trigger(`modal:open ${actions.type.toLowerCase()}:open`);
                  actions.emit(`local::open modalOpen ${actions.type}Open`, actions);
                },
                opened() {
                  if (!actions.$el) {
                    actions.$el = popover.$el;
                  }
                  actions.$el.trigger(`modal:opened ${actions.type.toLowerCase()}:opened`);
                  actions.emit(`local::opened modalOpened ${actions.type}Opened`, actions);
                },
                close() {
                  if (!actions.$el) {
                    actions.$el = popover.$el;
                  }
                  actions.$el.trigger(`modal:close ${actions.type.toLowerCase()}:close`);
                  actions.emit(`local::close modalClose ${actions.type}Close`, actions);
                },
                closed() {
                  if (!actions.$el) {
                    actions.$el = popover.$el;
                  }
                  actions.$el.trigger(`modal:closed ${actions.type.toLowerCase()}:closed`);
                  actions.emit(`local::closed modalClosed ${actions.type}Closed`, actions);
                }
              }
            });
            popover.open(animate);
            popover.once('popoverOpened', () => {
              popover.$el.find('.list-button, .item-link').each(buttonEl => {
                $(buttonEl).on('click', buttonOnClick);
              });
            });
            popover.once('popoverClosed', () => {
              popover.$el.find('.list-button, .item-link').each(buttonEl => {
                $(buttonEl).off('click', buttonOnClick);
              });
              nextTick$1(() => {
                popover.destroy();
                popover = undefined;
              });
            });
          } else {
            actions.$el = actions.actionsHtml ? $(actions.actionsHtml) : actions.$el;
            actions.$el[0].f7Modal = actions;
            if (actions.groups) {
              actions.$el.find('.actions-button').each(buttonEl => {
                $(buttonEl).on('click', buttonOnClick);
              });
              actions.once('actionsClosed', () => {
                actions.$el.find('.actions-button').each(buttonEl => {
                  $(buttonEl).off('click', buttonOnClick);
                });
              });
            }
            actions.el = actions.$el[0];
            originalOpen.call(actions, animate);
          }
          return actions;
        };
        actions.close = function close(animate) {
          if (popover) {
            popover.close(animate);
          } else {
            originalClose.call(actions, animate);
          }
          return actions;
        };
        extend$3(actions, {
          app,
          $el,
          el: $el ? $el[0] : undefined,
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'actions'
        });
        function handleClick(e) {
          const target = e.target;
          const $target = $(target);
          const keyboardOpened = !device.desktop && device.cordova && (window.Keyboard && window.Keyboard.isVisible || window.cordova.plugins && window.cordova.plugins.Keyboard && window.cordova.plugins.Keyboard.isVisible);
          if (keyboardOpened) return;
          if ($target.closest(actions.el).length === 0) {
            if (actions.params.closeByBackdropClick && actions.params.backdrop && actions.backdropEl && actions.backdropEl === target) {
              actions.close();
            } else if (actions.params.closeByOutsideClick) {
              actions.close();
            }
          }
        }
        function onKeyDown(e) {
          const keyCode = e.keyCode;
          if (keyCode === 27 && actions.params.closeOnEscape) {
            actions.close();
          }
        }
        if (actions.params.closeOnEscape) {
          actions.on('open', () => {
            $(document).on('keydown', onKeyDown);
          });
          actions.on('close', () => {
            $(document).off('keydown', onKeyDown);
          });
        }
        actions.on('opened', () => {
          if (actions.params.closeByBackdropClick || actions.params.closeByOutsideClick) {
            app.on('click', handleClick);
          }
        });
        actions.on('close', () => {
          if (actions.params.closeByBackdropClick || actions.params.closeByOutsideClick) {
            app.off('click', handleClick);
          }
        });
        if ($el) {
          $el[0].f7Modal = actions;
        }
        return actions;
      }
      render() {
        const actions = this;
        if (actions.params.render) return actions.params.render.call(actions, actions);
        const {
          groups
        } = actions;
        const cssClass = actions.params.cssClass;
        return $jsx$1("div", {
          class: `actions-modal${actions.params.grid ? ' actions-grid' : ''} ${cssClass || ''}`
        }, groups.map(group => $jsx$1("div", {
          class: "actions-group"
        }, group.map(button => {
          const buttonClasses = [`actions-${button.label ? 'label' : 'button'}`];
          const {
            color,
            bg,
            strong,
            disabled,
            label,
            text,
            icon
          } = button;
          if (color) buttonClasses.push(`color-${color}`);
          if (bg) buttonClasses.push(`bg-color-${bg}`);
          if (strong) buttonClasses.push('actions-button-strong');
          if (disabled) buttonClasses.push('disabled');
          if (label) {
            return $jsx$1("div", {
              class: buttonClasses.join(' ')
            }, text);
          }
          return $jsx$1("div", {
            class: buttonClasses.join(' ')
          }, icon && $jsx$1("div", {
            class: "actions-button-media"
          }, icon), $jsx$1("div", {
            class: "actions-button-text"
          }, text));
        }))));
      }
      renderPopover() {
        const actions = this;
        if (actions.params.renderPopover) return actions.params.renderPopover.call(actions, actions);
        const {
          groups
        } = actions;
        const cssClass = actions.params.cssClass;
        return $jsx$1("div", {
          class: `popover popover-from-actions ${cssClass || ''}`
        }, $jsx$1("div", {
          class: "popover-inner"
        }, groups.map(group => $jsx$1("div", {
          class: "list"
        }, $jsx$1("ul", null, group.map(button => {
          const itemClasses = [];
          const {
            color,
            bg,
            strong,
            disabled,
            label,
            text,
            icon
          } = button;
          if (color) itemClasses.push(`color-${color}`);
          if (bg) itemClasses.push(`bg-color-${bg}`);
          if (strong) itemClasses.push('popover-from-actions-strong');
          if (disabled) itemClasses.push('disabled');
          if (label) {
            itemClasses.push('popover-from-actions-label');
            return `<li class="${itemClasses.join(' ')}">${text}</li>`;
          }
          if (icon) {
            itemClasses.push('item-link item-content');
            return $jsx$1("li", null, $jsx$1("a", {
              class: itemClasses.join(' ')
            }, $jsx$1("div", {
              class: "item-media"
            }, icon), $jsx$1("div", {
              class: "item-inner"
            }, $jsx$1("div", {
              class: "item-title"
            }, text))));
          }
          itemClasses.push('list-button');
          return $jsx$1("li", null, $jsx$1("a", {
            class: itemClasses.join(' ')
          }, text));
        }))))));
      }
    };

    var Actions = {
      name: 'actions',
      params: {
        actions: {
          convertToPopover: true,
          forceToPopover: false,
          backdrop: true,
          backdropEl: undefined,
          backdropUnique: false,
          cssClass: null,
          closeByBackdropClick: true,
          closeOnEscape: false,
          render: null,
          renderPopover: null,
          containerEl: null
        }
      },
      static: {
        Actions: Actions$1
      },
      create() {
        const app = this;
        app.actions = ModalMethods({
          app,
          constructor: Actions$1,
          defaultSelector: '.actions-modal.modal-in'
        });
      },
      clicks: {
        '.actions-open': function openActions($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.actions.open(data.actions, data.animate, $clickedEl);
        },
        '.actions-close': function closeActions($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.actions.close(data.actions, data.animate, $clickedEl);
        }
      }
    };

    let Sheet$1 = class Sheet extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, app.params.sheet, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const sheet = this;
        const window = getWindow$1();
        const document = getDocument$1();
        const support = getSupport$1();
        const device = getDevice$1();
        sheet.params = extendedParams;
        if (typeof sheet.params.backdrop === 'undefined') {
          sheet.params.backdrop = app.theme !== 'ios';
        }

        // Find Element
        let $el;
        if (!sheet.params.el) {
          $el = $(sheet.params.content).filter(node => node.nodeType === 1).eq(0);
        } else {
          $el = $(sheet.params.el).eq(0);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return sheet.destroy();
        }
        let $backdropEl;
        if (sheet.params.backdrop && sheet.params.backdropEl) {
          $backdropEl = $(sheet.params.backdropEl);
        } else if (sheet.params.backdrop) {
          if (sheet.params.backdropUnique) {
            $backdropEl = $('<div class="sheet-backdrop sheet-backdrop-unique"></div>');
            sheet.$containerEl.append($backdropEl);
          } else {
            $backdropEl = sheet.$containerEl.children('.sheet-backdrop');
          }
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="sheet-backdrop"></div>');
            sheet.$containerEl.append($backdropEl);
          }
        }
        extend$3(sheet, {
          app,
          push: $el.hasClass('sheet-modal-push') || sheet.params.push,
          $el,
          el: $el[0],
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0],
          type: 'sheet',
          $htmlEl: $('html')
        });
        if (sheet.params.push) {
          $el.addClass('sheet-modal-push');
        }
        let $pageContentEl;
        function scrollToElementOnOpen() {
          const $scrollEl = $(sheet.params.scrollToEl).eq(0);
          if ($scrollEl.length === 0) return;
          $pageContentEl = $scrollEl.parents('.page-content');
          if ($pageContentEl.length === 0) return;
          const paddingTop = parseInt($pageContentEl.css('padding-top'), 10);
          const paddingBottom = parseInt($pageContentEl.css('padding-bottom'), 10);
          const pageHeight = $pageContentEl[0].offsetHeight - paddingTop - $el.height();
          const pageScrollHeight = $pageContentEl[0].scrollHeight - paddingTop - $el.height();
          const pageScroll = $pageContentEl.scrollTop();
          let newPaddingBottom;
          const scrollElTop = $scrollEl.offset().top - paddingTop + $scrollEl[0].offsetHeight;
          if (scrollElTop > pageHeight) {
            const scrollTop = pageScroll + scrollElTop - pageHeight;
            if (scrollTop + pageHeight > pageScrollHeight) {
              newPaddingBottom = scrollTop + pageHeight - pageScrollHeight + paddingBottom;
              if (pageHeight === pageScrollHeight) {
                newPaddingBottom = $el.height();
              }
              $pageContentEl.css({
                'padding-bottom': `${newPaddingBottom}px`
              });
            }
            $pageContentEl.scrollTop(scrollTop, 300);
          }
        }
        function scrollToElementOnClose() {
          if ($pageContentEl && $pageContentEl.length > 0) {
            $pageContentEl.css({
              'padding-bottom': ''
            });
          }
        }
        function handleClick(e) {
          const target = e.target;
          const $target = $(target);
          const keyboardOpened = !device.desktop && device.cordova && (window.Keyboard && window.Keyboard.isVisible || window.cordova.plugins && window.cordova.plugins.Keyboard && window.cordova.plugins.Keyboard.isVisible);
          if (keyboardOpened) return;
          if ($target.closest(sheet.el).length === 0) {
            if (sheet.params.closeByBackdropClick && sheet.params.backdrop && sheet.backdropEl && sheet.backdropEl === target) {
              sheet.close();
            } else if (sheet.params.closeByOutsideClick) {
              sheet.close();
            }
          }
        }
        function onKeyDown(e) {
          const keyCode = e.keyCode;
          if (keyCode === 27 && sheet.params.closeOnEscape) {
            sheet.close();
          }
        }
        let pushOffset;
        function pushViewScale(offset) {
          return (app.height - offset * 2) / app.height;
        }
        const useBreakpoints = sheet.params.breakpoints && sheet.params.breakpoints.length > 0;
        let isTouched = false;
        let startTouch;
        let currentTouch;
        let isScrolling;
        let touchStartTime;
        let touchesDiff;
        let isMoved = false;
        let isTopSheetModal;
        let swipeStepTranslate;
        let startTranslate;
        let currentTranslate;
        let sheetElOffsetHeight;
        let minTranslate;
        let maxTranslate;
        let $pushViewEl;
        let pushBorderRadius;
        let sheetPageContentEl;
        let sheetPageContentScrollTop;
        let sheetPageContentScrollHeight;
        let sheetPageContentOffsetHeight;
        let breakpointsTranslate = [];
        let currentBreakpointIndex;
        let backdropBreakpointSet = true;
        function handleTouchStart(e) {
          if (isTouched || !(sheet.params.swipeToClose || sheet.params.swipeToStep) || !e.isTrusted) return;
          if (sheet.params.swipeHandler && $(e.target).closest(sheet.params.swipeHandler).length === 0) {
            return;
          }
          if ($(e.target).closest('.sortable-handler').length > 0) return;
          isTouched = true;
          isMoved = false;
          startTouch = {
            x: e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX,
            y: e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY
          };
          touchStartTime = now$1();
          isScrolling = undefined;
          isTopSheetModal = $el.hasClass('sheet-modal-top');
          if (!sheet.params.swipeHandler && e.type === 'touchstart') {
            sheetPageContentEl = $(e.target).closest('.page-content')[0];
          }
        }
        function handleTouchMove(e) {
          if (!isTouched || !e.isTrusted) return;
          currentTouch = {
            x: e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX,
            y: e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY
          };
          if (typeof isScrolling === 'undefined') {
            isScrolling = !!(isScrolling || Math.abs(currentTouch.x - startTouch.x) > Math.abs(currentTouch.y - startTouch.y));
          }
          if (isScrolling) {
            isTouched = false;
            isMoved = false;
            return;
          }
          touchesDiff = startTouch.y - currentTouch.y;
          const direction = touchesDiff < 0 ? 'to-bottom' : 'to-top';
          if (!isMoved) {
            if (sheetPageContentEl && !$el.hasClass('modal-in-swipe-step')) {
              sheetPageContentScrollTop = sheetPageContentEl.scrollTop;
              sheetPageContentScrollHeight = sheetPageContentEl.scrollHeight;
              sheetPageContentOffsetHeight = sheetPageContentEl.offsetHeight;
              if (!(sheetPageContentScrollHeight === sheetPageContentOffsetHeight) && !(direction === 'to-bottom' && sheetPageContentScrollTop === 0) && !(direction === 'to-top' && sheetPageContentScrollTop === sheetPageContentScrollHeight - sheetPageContentOffsetHeight)) {
                $el.transform('');
                isTouched = false;
                isMoved = false;
                return;
              }
            }
            if (sheet.push && pushOffset) {
              $pushViewEl = app.$el.children('.view, .views');
            }
            sheetElOffsetHeight = $el[0].offsetHeight;
            startTranslate = getTranslate$1($el[0], 'y');
            if (isTopSheetModal) {
              minTranslate = sheet.params.swipeToClose ? -sheetElOffsetHeight : -swipeStepTranslate;
              maxTranslate = 0;
            } else {
              minTranslate = 0;
              maxTranslate = sheet.params.swipeToClose ? sheetElOffsetHeight : useBreakpoints ? breakpointsTranslate[0] : swipeStepTranslate;
            }
            isMoved = true;
          }
          currentTranslate = startTranslate - touchesDiff;
          currentTranslate = Math.min(Math.max(currentTranslate, minTranslate), maxTranslate);
          e.preventDefault();
          if (useBreakpoints) {
            let progress = isTopSheetModal ? 1 + currentTranslate / sheetElOffsetHeight : 1 - currentTranslate / sheetElOffsetHeight;
            progress = Math.abs(progress);
            progress = Math.min(Math.max(progress, 0), 1);
            // eslint-disable-next-line
            setBackdropBreakpoint(progress);
            // eslint-disable-next-line
            setPushBreakpoint(progress);
          }
          if (sheet.push && pushOffset && !useBreakpoints) {
            let progress = (currentTranslate - startTranslate) / sheetElOffsetHeight;
            if (sheet.params.swipeToStep) {
              if (isTopSheetModal) {
                progress = currentTranslate / swipeStepTranslate;
              } else {
                progress = 1 - (swipeStepTranslate - currentTranslate) / swipeStepTranslate;
              }
            }
            progress = Math.abs(progress);
            progress = Math.min(Math.max(progress, 0), 1);
            const pushProgress = 1 - progress;
            const scale = 1 - (1 - pushViewScale(pushOffset)) * pushProgress;
            $pushViewEl.transition(0).forEach(el => {
              el.style.setProperty('transform', `translate3d(0,0,0) scale(${scale})`, 'important');
            });
            if (sheet.params.swipeToStep) {
              $pushViewEl.css('border-radius', `${pushBorderRadius * pushProgress}px`);
            }
          }
          $el.transition(0).transform(`translate3d(0,${currentTranslate}px,0)`);
          if (sheet.params.swipeToStep) {
            let progress;
            if (isTopSheetModal) {
              progress = 1 - currentTranslate / swipeStepTranslate;
            } else {
              progress = (swipeStepTranslate - currentTranslate) / swipeStepTranslate;
            }
            progress = Math.min(Math.max(progress, 0), 1);
            $el.trigger('sheet:stepprogress', progress);
            sheet.emit('local::stepProgress sheetStepProgress', sheet, progress);
          }
        }
        function handleTouchEnd() {
          isTouched = false;
          if (!isMoved) {
            return;
          }
          isMoved = false;
          $el.transform('').transition('');
          if (sheet.push && pushOffset) {
            $pushViewEl.transition('');
            if (!useBreakpoints) {
              $pushViewEl.transform('');
              $pushViewEl.css('border-radius', '');
            }
          }
          const direction = touchesDiff < 0 ? 'to-bottom' : 'to-top';
          const diff = Math.abs(touchesDiff);
          if (diff === 0 || currentTranslate === startTranslate) return;
          const timeDiff = new Date().getTime() - touchStartTime;
          if (!sheet.params.swipeToStep && !useBreakpoints) {
            if (direction !== (isTopSheetModal ? 'to-top' : 'to-bottom')) {
              return;
            }
            if (timeDiff < 300 && diff > 20 || timeDiff >= 300 && diff > sheetElOffsetHeight / 2) {
              sheet.close();
            }
            return;
          }
          const openDirection = isTopSheetModal ? 'to-bottom' : 'to-top';
          const closeDirection = isTopSheetModal ? 'to-top' : 'to-bottom';
          const absCurrentTranslate = Math.abs(currentTranslate);
          const absSwipeStepTranslate = Math.abs(swipeStepTranslate);
          if (timeDiff < 300 && diff > 10 && useBreakpoints) {
            // SHORT SWIPES BREAKPOINTS
            if (direction === openDirection && typeof currentBreakpointIndex !== 'undefined') {
              if (currentBreakpointIndex === params.breakpoints.length - 1) {
                // open
                sheet.setBreakpoint(1);
              } else {
                // move to next breakpoint
                currentBreakpointIndex = Math.min(breakpointsTranslate.length - 1, currentBreakpointIndex + 1);
                sheet.setBreakpoint(params.breakpoints[currentBreakpointIndex]);
              }
            }
            if (direction === closeDirection) {
              if (currentBreakpointIndex === 0) {
                // close
                sheet.close();
              } else {
                // move to prev breakpoint
                if (typeof currentBreakpointIndex === 'undefined') {
                  currentBreakpointIndex = params.breakpoints.length - 1;
                } else {
                  currentBreakpointIndex = Math.max(0, currentBreakpointIndex - 1);
                }
                sheet.setBreakpoint(params.breakpoints[currentBreakpointIndex]);
              }
            }
          } else if (timeDiff < 300 && diff > 10) {
            // SHORT SWIPES SWIPE STEP
            if (direction === openDirection && absCurrentTranslate < absSwipeStepTranslate) {
              // open step
              $el.removeClass('modal-in-swipe-step');
              $el.trigger('sheet:stepprogress', 1);
              sheet.emit('local::stepProgress sheetStepProgress', sheet, 1);
              sheet.emit('local::_swipeStep', false);
              $el.trigger('sheet:stepopen');
              sheet.emit('local::stepOpen sheetStepOpen', sheet);
              if (sheet.push && pushOffset) {
                sheet.$htmlEl[0].style.setProperty('--f7-sheet-push-scale', pushViewScale(pushOffset));
                $pushViewEl.css('border-radius', '');
              }
            }
            if (direction === closeDirection && absCurrentTranslate > absSwipeStepTranslate) {
              // close sheet
              if (sheet.params.swipeToClose) {
                sheet.close();
              } else {
                // close step
                $el.addClass('modal-in-swipe-step');
                $el.trigger('sheet:stepprogress', 0);
                sheet.emit('local::stepProgress sheetStepProgress', sheet, 0);
                sheet.emit('local::_swipeStep', true);
                $el.trigger('sheet:stepclose');
                sheet.emit('local::stepClose sheetStepClose', sheet);
                if (sheet.push && pushOffset) {
                  sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-scale');
                  $pushViewEl.css('border-radius', '0px');
                }
              }
            }
            if (direction === closeDirection && absCurrentTranslate <= absSwipeStepTranslate) {
              // close step
              $el.addClass('modal-in-swipe-step');
              $el.trigger('sheet:stepprogress', 0);
              sheet.emit('local::stepProgress sheetStepProgress', sheet, 0);
              sheet.emit('local::_swipeStep', true);
              $el.trigger('sheet:stepclose');
              sheet.emit('local::stepClose sheetStepClose', sheet);
              if (sheet.push && pushOffset) {
                sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-scale');
                $pushViewEl.css('border-radius', '0px');
              }
            }
            return;
          }
          if (timeDiff >= 300 && useBreakpoints) {
            // LONG SWIPES BREAKPOINTS
            const allBreakpoints = [sheetElOffsetHeight, ...breakpointsTranslate, 0];
            const closestTranslate = allBreakpoints.reduce((prev, curr) => {
              return Math.abs(curr - currentTranslate) < Math.abs(prev - currentTranslate) ? curr : prev;
            });
            const closestIndex = allBreakpoints.indexOf(closestTranslate);
            if (closestTranslate === 0) {
              // open
              sheet.setBreakpoint(1);
            } else if (closestIndex === 0) {
              // close
              sheet.close();
            } else {
              // set bp
              currentBreakpointIndex = closestIndex - 1;
              sheet.setBreakpoint(params.breakpoints[currentBreakpointIndex]);
            }
          } else if (timeDiff >= 300) {
            // LONG SWIPES SWIPE STEP
            const stepOpened = !$el.hasClass('modal-in-swipe-step');
            if (!stepOpened) {
              if (absCurrentTranslate < absSwipeStepTranslate / 2) {
                // open step
                $el.removeClass('modal-in-swipe-step');
                $el.trigger('sheet:stepprogress', 1);
                sheet.emit('local::stepProgress sheetStepProgress', sheet, 1);
                sheet.emit('local::_swipeStep', false);
                $el.trigger('sheet:stepopen');
                sheet.emit('local::stepOpen sheetStepOpen', sheet);
                if (sheet.push && pushOffset) {
                  sheet.$htmlEl[0].style.setProperty('--f7-sheet-push-scale', pushViewScale(pushOffset));
                  $pushViewEl.css('border-radius', '');
                }
              } else if (absCurrentTranslate - absSwipeStepTranslate > (sheetElOffsetHeight - absSwipeStepTranslate) / 2) {
                // close sheet
                if (sheet.params.swipeToClose) sheet.close();
              }
            } else if (stepOpened) {
              if (absCurrentTranslate > absSwipeStepTranslate + (sheetElOffsetHeight - absSwipeStepTranslate) / 2) {
                // close sheet
                if (sheet.params.swipeToClose) sheet.close();
              } else if (absCurrentTranslate > absSwipeStepTranslate / 2) {
                // close step
                $el.addClass('modal-in-swipe-step');
                $el.trigger('sheet:stepprogress', 0);
                sheet.emit('local::stepProgress sheetStepProgress', sheet, 0);
                sheet.emit('local::_swipeStep', true);
                $el.trigger('sheet:stepclose');
                sheet.emit('local::stepClose sheetStepClose', sheet);
                if (sheet.push && pushOffset) {
                  sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-scale');
                  $pushViewEl.css('border-radius', '0px');
                }
              }
            }
          }
        }
        const setPushBreakpoint = breakpoint => {
          const {
            pushBreakpoint
          } = params;
          if (pushBreakpoint === null || typeof pushBreakpoint === 'undefined' || !sheet.push || !pushOffset) return;
          if (breakpoint >= pushBreakpoint) {
            sheet.$htmlEl.addClass('with-modal-sheet-push').removeClass('with-modal-sheet-push-closing');
            $pushViewEl.transition('').forEach(el => {
              el.style.setProperty('transform', `translate3d(0,0,0) scale(${pushViewScale(pushOffset)})`, 'important');
            });
            $pushViewEl.css('border-radius', `${pushBorderRadius * 1}px`);
          } else {
            const pushBreakpoints = [0, ...params.breakpoints, 1];
            const pushTransparentBreakpoint = pushBreakpoints[pushBreakpoints.indexOf(pushBreakpoint) - 1];
            if (breakpoint <= pushTransparentBreakpoint) {
              $pushViewEl.transition('').css('transform', '');
              $pushViewEl.css('border-radius', '');
              sheet.$htmlEl.removeClass('with-modal-sheet-push');
              if (breakpoint === pushTransparentBreakpoint) {
                sheet.$htmlEl.addClass('with-modal-sheet-push-closing');
              }
            } else {
              const progress = (breakpoint - pushTransparentBreakpoint) / (pushBreakpoint - pushTransparentBreakpoint);
              sheet.$htmlEl.addClass('with-modal-sheet-push').removeClass('with-modal-sheet-push-closing');
              $pushViewEl.transition(0).forEach(el => {
                el.style.setProperty('transform', `translate3d(0,0,0) scale(${1 - (1 - pushViewScale(pushOffset)) * progress})`, 'important');
              });
              $pushViewEl.css('border-radius', `${pushBorderRadius * progress}px`);
            }
          }
        };
        const setBackdropBreakpoint = breakpoint => {
          const {
            backdrop,
            backdropBreakpoint
          } = params;
          if (!backdropBreakpoint || !backdrop || !$backdropEl.length) return;
          if (breakpoint >= backdropBreakpoint) {
            if (!backdropBreakpointSet) {
              $backdropEl.transition('').css({
                opacity: '',
                pointerEvents: ''
              });
            }
            backdropBreakpointSet = true;
          } else {
            const backdropBreakpoints = [0, ...params.breakpoints, 1];
            const backdropTransparentBreakpoint = backdropBreakpoints[backdropBreakpoints.indexOf(backdropBreakpoint) - 1];
            if (breakpoint <= backdropTransparentBreakpoint) {
              if (backdropBreakpointSet) {
                $backdropEl.transition('').css({
                  opacity: 0,
                  pointerEvents: 'none'
                });
              }
              backdropBreakpointSet = false;
            } else {
              const progress = (breakpoint - backdropTransparentBreakpoint) / (backdropBreakpoint - backdropTransparentBreakpoint);
              $backdropEl.transition(0).css({
                opacity: progress,
                pointerEvents: 'auto'
              });
            }
          }
        };
        sheet.calcBreakpoints = () => {
          if (!useBreakpoints) {
            return;
          }
          const fullSize = $el[0].offsetHeight;
          // eslint-disable-next-line
          const isTopSheetModal = $el.hasClass('sheet-modal-top');
          breakpointsTranslate = [];
          sheet.params.breakpoints.forEach(ratio => {
            breakpointsTranslate.push((fullSize - fullSize * ratio) * (isTopSheetModal ? -1 : 1));
          });
        };
        sheet.setBreakpoint = value => {
          if (!useBreakpoints) {
            return sheet;
          }
          if (value === 1) {
            // open
            if (!sheet.opened) {
              sheet.open();
            }
            $el.removeClass('modal-in-breakpoint');
            currentBreakpointIndex = undefined;
            setBackdropBreakpoint(value);
            setPushBreakpoint(value);
            $el.trigger('sheet:breakpoint', value);
            sheet.emit('local::breakpoint sheetBreakpoint', sheet, value);
          } else if (value === 0) {
            // close
            $el.trigger('sheet:breakpoint', value);
            sheet.emit('local::breakpoint sheetBreakpoint', sheet, value);
            sheet.close();
          } else {
            const index = params.breakpoints.indexOf(value);
            if (index < 0) return sheet;
            if (!sheet.opened) {
              sheet.open();
            }
            setBackdropBreakpoint(value);
            setPushBreakpoint(value);
            $el.trigger('sheet:breakpoint', value);
            sheet.emit('local::breakpoint sheetBreakpoint', sheet, value);
            currentBreakpointIndex = index;
            $el[0].style.setProperty('--f7-sheet-breakpoint', `${breakpointsTranslate[index]}px`);
            $el.addClass('modal-in-breakpoint');
          }
          return sheet;
        };
        const setBreakpointsOnResize = () => {
          sheet.calcBreakpoints();
          if (typeof currentBreakpointIndex !== 'undefined') {
            sheet.setBreakpoint(params.breakpoints[currentBreakpointIndex]);
          }
        };
        sheet.setSwipeStep = function setSwipeStep(byResize) {
          if (byResize === void 0) {
            byResize = true;
          }
          const $swipeStepEl = $el.find('.sheet-modal-swipe-step').eq(0);
          if (!$swipeStepEl.length) return;

          // eslint-disable-next-line
          if ($el.hasClass('sheet-modal-top')) {
            swipeStepTranslate = -($swipeStepEl.offset().top - $el.offset().top + $swipeStepEl[0].offsetHeight);
          } else {
            swipeStepTranslate = $el[0].offsetHeight - ($swipeStepEl.offset().top - $el.offset().top + $swipeStepEl[0].offsetHeight);
          }
          $el[0].style.setProperty('--f7-sheet-swipe-step', `${swipeStepTranslate}px`);
          if (!byResize) {
            $el.addClass('modal-in-swipe-step');
            sheet.emit('local::_swipeStep', true);
          }
        };
        function onResize() {
          if (useBreakpoints) {
            setBreakpointsOnResize();
          } else {
            sheet.setSwipeStep(true);
          }
        }
        const passive = support.passiveListener ? {
          passive: true
        } : false;
        if (sheet.params.swipeToClose || sheet.params.swipeToStep || useBreakpoints) {
          $el.on(app.touchEvents.start, handleTouchStart, passive);
          app.on('touchmove', handleTouchMove);
          app.on('touchend:passive', handleTouchEnd);
          sheet.once('sheetDestroy', () => {
            $el.off(app.touchEvents.start, handleTouchStart, passive);
            app.off('touchmove', handleTouchMove);
            app.off('touchend:passive', handleTouchEnd);
          });
        }
        sheet.on('open', () => {
          if (sheet.params.closeOnEscape) {
            $(document).on('keydown', onKeyDown);
          }
          $el.prevAll('.popup.modal-in').addClass('popup-behind');
          app.on('resize', onResize);
          if (sheet.params.scrollToEl) {
            scrollToElementOnOpen();
          }
          if (sheet.push) {
            pushOffset = parseInt($el.css('--f7-sheet-push-offset'), 10);
            if (Number.isNaN(pushOffset)) pushOffset = 0;
            if (!pushOffset) pushOffset = app.theme === 'ios' ? 44 : 48;
            sheet.$htmlEl[0].style.setProperty('--f7-sheet-push-offset', `${pushOffset}px`);
            $el.addClass('sheet-modal-push');
            if (!useBreakpoints) {
              sheet.$htmlEl.addClass('with-modal-sheet-push');
            }
            if (!sheet.params.swipeToStep && !useBreakpoints) {
              sheet.$htmlEl[0].style.setProperty('--f7-sheet-push-scale', pushViewScale(pushOffset));
            } else {
              $pushViewEl = app.$el.children('.view, .views');
              pushBorderRadius = app.theme === 'ios' ? 10 : 16;
              $pushViewEl.css('border-radius', '0px');
            }
          }
          if (useBreakpoints) {
            sheet.calcBreakpoints();
            sheet.setBreakpoint(params.breakpoints[0]);
          } else if (sheet.params.swipeToStep) {
            sheet.setSwipeStep(false);
          }
        });
        sheet.on('opened', () => {
          if (sheet.params.closeByOutsideClick || sheet.params.closeByBackdropClick) {
            app.on('click', handleClick);
          }
        });
        sheet.on('close', () => {
          currentBreakpointIndex = undefined;
          if (sheet.params.swipeToStep || useBreakpoints) {
            $el.removeClass('modal-in-swipe-step modal-in-breakpoint');
            sheet.emit('local::_swipeStep', false);
          }
          app.off('resize', onResize);
          if (sheet.params.closeOnEscape) {
            $(document).off('keydown', onKeyDown);
          }
          if (sheet.params.scrollToEl) {
            scrollToElementOnClose();
          }
          if (sheet.params.closeByOutsideClick || sheet.params.closeByBackdropClick) {
            app.off('click', handleClick);
          }
          $el.prevAll('.popup.modal-in').eq(0).removeClass('popup-behind');
          if (sheet.push && pushOffset) {
            sheet.$htmlEl.removeClass('with-modal-sheet-push');
            sheet.$htmlEl.addClass('with-modal-sheet-push-closing');
            if ($pushViewEl) {
              $pushViewEl.transform('');
              $pushViewEl.css('border-radius', '');
            }
          }
        });
        sheet.on('closed', () => {
          if (sheet.push && pushOffset) {
            sheet.$htmlEl.removeClass('with-modal-sheet-push-closing');
            sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-scale');
            sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-offset');
          }
        });
        sheet.stepOpen = function stepOpen() {
          $el.removeClass('modal-in-swipe-step');
          sheet.emit('local::_swipeStep', false);
          if (sheet.push) {
            if (!pushOffset) {
              pushOffset = parseInt($el.css('--f7-sheet-push-offset'), 10);
              if (Number.isNaN(pushOffset)) pushOffset = 0;
            }
            if (pushOffset) {
              sheet.$htmlEl[0].style.setProperty('--f7-sheet-push-scale', pushViewScale(pushOffset));
            }
          }
        };
        sheet.stepClose = function stepClose() {
          $el.addClass('modal-in-swipe-step');
          sheet.emit('local::_swipeStep', true);
          if (sheet.push) {
            sheet.$htmlEl[0].style.removeProperty('--f7-sheet-push-scale');
          }
        };
        sheet.stepToggle = function stepToggle() {
          $el.toggleClass('modal-in-swipe-step');
          sheet.emit('local::_swipeStep', $el.hasClass('modal-in-swipe-step'));
        };
        $el[0].f7Modal = sheet;
        return sheet;
      }
    };

    var Sheet = {
      name: 'sheet',
      params: {
        sheet: {
          push: false,
          backdrop: undefined,
          backdropEl: undefined,
          backdropUnique: false,
          closeByBackdropClick: true,
          closeByOutsideClick: false,
          closeOnEscape: false,
          swipeToClose: false,
          swipeToStep: false,
          breakpoints: [],
          backdropBreakpoint: 0,
          pushBreakpoint: 0,
          swipeHandler: null,
          containerEl: null
        }
      },
      static: {
        Sheet: Sheet$1
      },
      create() {
        const app = this;
        app.sheet = extend$3({}, ModalMethods({
          app,
          constructor: Sheet$1,
          defaultSelector: '.sheet-modal.modal-in'
        }), {
          stepOpen(sheet) {
            const sheetInstance = app.sheet.get(sheet);
            if (sheetInstance && sheetInstance.stepOpen) return sheetInstance.stepOpen();
            return undefined;
          },
          stepClose(sheet) {
            const sheetInstance = app.sheet.get(sheet);
            if (sheetInstance && sheetInstance.stepClose) return sheetInstance.stepClose();
            return undefined;
          },
          stepToggle(sheet) {
            const sheetInstance = app.sheet.get(sheet);
            if (sheetInstance && sheetInstance.stepToggle) return sheetInstance.stepToggle();
            return undefined;
          },
          setBreakpoint(sheet, breakpoint) {
            const sheetInstance = app.sheet.get(sheet);
            if (sheetInstance && sheetInstance.setBreakpoint) return sheetInstance.setBreakpoint(breakpoint);
            return undefined;
          }
        });
      },
      clicks: {
        '.sheet-open': function openSheet($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          if ($('.sheet-modal.modal-in').length > 0 && data.sheet && $(data.sheet)[0] !== $('.sheet-modal.modal-in')[0]) {
            app.sheet.close('.sheet-modal.modal-in');
          }
          app.sheet.open(data.sheet, data.animate, $clickedEl);
        },
        '.sheet-close': function closeSheet($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.sheet.close(data.sheet, data.animate, $clickedEl);
        }
      }
    };

    let Toast$1 = class Toast extends Modal$1 {
      constructor(app, params) {
        const extendedParams = extend$3({
          on: {}
        }, app.params.toast, params);

        // Extends with open/close Modal methods;
        super(app, extendedParams);
        const toast = this;
        const window = getWindow$1();
        toast.app = app;
        toast.params = extendedParams;
        const {
          closeButton,
          closeTimeout
        } = toast.params;
        let $el;
        if (!toast.params.el) {
          // Find Element
          const toastHtml = toast.render();
          $el = $(toastHtml);
        } else {
          $el = $(toast.params.el);
        }
        if ($el && $el.length > 0 && $el[0].f7Modal) {
          return $el[0].f7Modal;
        }
        if ($el.length === 0) {
          return toast.destroy();
        }
        extend$3(toast, {
          $el,
          el: $el[0],
          type: 'toast'
        });
        $el[0].f7Modal = toast;
        if (closeButton) {
          $el.find('.toast-button').on('click', () => {
            toast.emit('local::closeButtonClick toastCloseButtonClick', toast);
            toast.close();
          });
          toast.on('beforeDestroy', () => {
            $el.find('.toast-button').off('click');
          });
        }
        let timeoutId;
        toast.on('open', () => {
          $('.toast.modal-in').each(openedEl => {
            const toastInstance = app.toast.get(openedEl);
            if (openedEl !== toast.el && toastInstance) {
              toastInstance.close();
            }
          });
          if (closeTimeout) {
            timeoutId = nextTick$1(() => {
              toast.close();
            }, closeTimeout);
          }
        });
        toast.on('close', () => {
          window.clearTimeout(timeoutId);
        });
        if (toast.params.destroyOnClose) {
          toast.once('closed', () => {
            setTimeout(() => {
              toast.destroy();
            }, 0);
          });
        }
        return toast;
      }
      render() {
        const toast = this;
        if (toast.params.render) return toast.params.render.call(toast, toast);
        const {
          position,
          horizontalPosition,
          cssClass,
          icon,
          text,
          closeButton,
          closeButtonColor,
          closeButtonText
        } = toast.params;
        const horizontalClass = position === 'top' || position === 'bottom' ? `toast-horizontal-${horizontalPosition}` : '';
        return $jsx$1("div", {
          class: `toast toast-${position} ${horizontalClass} ${cssClass || ''} ${icon ? 'toast-with-icon' : ''}`
        }, $jsx$1("div", {
          class: "toast-content"
        }, icon && $jsx$1("div", {
          class: "toast-icon"
        }, icon), $jsx$1("div", {
          class: "toast-text"
        }, text), closeButton && !icon && $jsx$1("a", {
          class: `toast-button button ${closeButtonColor ? `color-${closeButtonColor}` : ''}`
        }, closeButtonText)));
      }
    };

    var Toast = {
      name: 'toast',
      static: {
        Toast: Toast$1
      },
      create() {
        const app = this;
        app.toast = extend$3({}, ModalMethods({
          app,
          constructor: Toast$1,
          defaultSelector: '.toast.modal-in'
        }), {
          // Shortcuts
          show(params) {
            extend$3(params, {
              destroyOnClose: true
            });
            return new Toast$1(app, params).open();
          }
        });
      },
      params: {
        toast: {
          icon: null,
          text: null,
          position: 'bottom',
          horizontalPosition: 'left',
          closeButton: false,
          closeButtonColor: null,
          closeButtonText: 'Ok',
          closeTimeout: null,
          cssClass: null,
          render: null,
          containerEl: null
        }
      }
    };

    const Preloader = {
      init(el) {
        const app = this;
        const preloaders = {
          iosPreloaderContent,
          mdPreloaderContent
        };
        const $el = $(el);
        if ($el.length === 0 || $el.children('.preloader-inner').length > 0 || $el.children('.preloader-inner-line').length > 0) return;
        $el.append(preloaders[`${app.theme}PreloaderContent`]);
      },
      // Modal
      visible: false,
      show(color) {
        if (color === void 0) {
          color = 'white';
        }
        const app = this;
        if (Preloader.visible) return;
        const preloaders = {
          iosPreloaderContent,
          mdPreloaderContent
        };
        const preloaderInner = preloaders[`${app.theme}PreloaderContent`] || '';
        $('html').addClass('with-modal-preloader');
        // prettier-ignore
        app.$el.append(`
      <div class="preloader-backdrop"></div>
      <div class="preloader-modal">
        <div class="preloader color-${color}">${preloaderInner}</div>
      </div>
    `);
        Preloader.visible = true;
      },
      showIn(el, color) {
        if (color === void 0) {
          color = 'white';
        }
        const app = this;
        const preloaders = {
          iosPreloaderContent,
          mdPreloaderContent
        };
        const preloaderInner = preloaders[`${app.theme}PreloaderContent`] || '';
        $(el || 'html').addClass('with-modal-preloader');
        // prettier-ignore
        $(el || app.$el).append(`
      <div class="preloader-backdrop"></div>
      <div class="preloader-modal">
        <div class="preloader color-${color}">${preloaderInner}</div>
      </div>
    `);
      },
      hide() {
        const app = this;
        if (!Preloader.visible) return;
        $('html').removeClass('with-modal-preloader');
        app.$el.find('.preloader-backdrop, .preloader-modal').remove();
        Preloader.visible = false;
      },
      hideIn(el) {
        const app = this;
        $(el || 'html').removeClass('with-modal-preloader');
        $(el || app.$el).find('.preloader-backdrop, .preloader-modal').remove();
      }
    };
    var Preloader$1 = {
      name: 'preloader',
      create() {
        const app = this;
        bindMethods(app, {
          preloader: Preloader
        });
      },
      on: {
        photoBrowserOpen(pb) {
          const app = this;
          pb.$el.find('.preloader').each(preloaderEl => {
            app.preloader.init(preloaderEl);
          });
        },
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.preloader').each(preloaderEl => {
            app.preloader.init(preloaderEl);
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.preloader').each(preloaderEl => {
            app.preloader.init(preloaderEl);
          });
        }
      },
      vnode: {
        preloader: {
          insert(vnode) {
            const app = this;
            const preloaderEl = vnode.elm;
            app.preloader.init(preloaderEl);
          }
        }
      }
    };

    const Progressbar = {
      set() {
        const app = this;
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }
        let [el, progress, duration] = args;
        if (typeof args[0] === 'number') {
          [progress, duration] = args;
          el = app.$el;
        }
        if (typeof progress === 'undefined' || progress === null) return el;
        if (!progress) progress = 0;
        const $el = $(el || app.$el);
        if ($el.length === 0) {
          return el;
        }
        const progressNormalized = Math.min(Math.max(progress, 0), 100);
        let $progressbarEl;
        if ($el.hasClass('progressbar')) $progressbarEl = $el.eq(0);else {
          $progressbarEl = $el.children('.progressbar');
        }
        if ($progressbarEl.length === 0 || $progressbarEl.hasClass('progressbar-infinite')) {
          return $progressbarEl;
        }
        let $progressbarLine = $progressbarEl.children('span');
        if ($progressbarLine.length === 0) {
          $progressbarLine = $('<span></span>');
          $progressbarEl.append($progressbarLine);
        }
        $progressbarLine.transition(typeof duration !== 'undefined' ? duration : '').transform(`translate3d(${(-100 + progressNormalized) * (app.rtl ? -1 : 1)}%,0,0)`);
        return $progressbarEl[0];
      },
      show() {
        const app = this;

        // '.page', 50, 'multi'
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        let [el, progress, color] = args;
        let type = 'determined';
        if (args.length === 2) {
          if ((typeof args[0] === 'string' || typeof args[0] === 'object') && typeof args[1] === 'string') {
            // '.page', 'multi'
            [el, color, progress] = args;
            type = 'infinite';
          } else if (typeof args[0] === 'number' && typeof args[1] === 'string') {
            // 50, 'multi'
            [progress, color] = args;
            el = app.$el;
          }
        } else if (args.length === 1) {
          if (typeof args[0] === 'number') {
            el = app.$el;
            progress = args[0];
          } else if (typeof args[0] === 'string') {
            type = 'infinite';
            el = app.$el;
            color = args[0];
          }
        } else if (args.length === 0) {
          type = 'infinite';
          el = app.$el;
        }
        const $el = $(el);
        if ($el.length === 0) return undefined;
        let $progressbarEl;
        if ($el.hasClass('progressbar') || $el.hasClass('progressbar-infinite')) {
          $progressbarEl = $el;
        } else {
          $progressbarEl = $el.children('.progressbar:not(.progressbar-out), .progressbar-infinite:not(.progressbar-out)');
          if ($progressbarEl.length === 0) {
            $progressbarEl = $(`
          <span class="progressbar${type === 'infinite' ? '-infinite' : ''}${color ? ` color-${color}` : ''} progressbar-in">
            ${type === 'infinite' ? '' : '<span></span>'}
          </span>`);
            $el.append($progressbarEl);
          }
        }
        if (typeof progress !== 'undefined') {
          app.progressbar.set($progressbarEl, progress);
        }
        return $progressbarEl[0];
      },
      hide(el, removeAfterHide) {
        if (removeAfterHide === void 0) {
          removeAfterHide = true;
        }
        const app = this;
        const $el = $(el || app.$el);
        if ($el.length === 0) return undefined;
        let $progressbarEl;
        if ($el.hasClass('progressbar') || $el.hasClass('progressbar-infinite')) {
          $progressbarEl = $el;
        } else {
          $progressbarEl = $el.children('.progressbar, .progressbar-infinite');
        }
        if ($progressbarEl.length === 0 || !$progressbarEl.hasClass('progressbar-in') || $progressbarEl.hasClass('progressbar-out')) {
          return $progressbarEl;
        }
        $progressbarEl.removeClass('progressbar-in').addClass('progressbar-out').animationEnd(() => {
          if (removeAfterHide) {
            $progressbarEl.remove();
          }
        });
        return $progressbarEl;
      }
    };
    var Progressbar$1 = {
      name: 'progressbar',
      create() {
        const app = this;
        bindMethods(app, {
          progressbar: Progressbar
        });
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.progressbar').each(progressbarEl => {
            const $progressbarEl = $(progressbarEl);
            app.progressbar.set($progressbarEl, $progressbarEl.attr('data-progress'));
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.progressbar').each(progressbarEl => {
            const $progressbarEl = $(progressbarEl);
            app.progressbar.set($progressbarEl, $progressbarEl.attr('data-progress'));
          });
        }
      },
      vnode: {
        progressbar: {
          insert(vnode) {
            const app = this;
            const el = vnode.elm;
            app.progressbar.set(el, el.getAttribute('data-progress'));
          },
          update(vnode) {
            const app = this;
            const el = vnode.elm;
            app.progressbar.set(el, el.getAttribute('data-progress'));
          }
        }
      }
    };

    const Sortable = {
      init() {
        const app = this;
        const document = getDocument$1();
        let isTouched;
        let isMoved;
        let touchStartY;
        let touchesDiff;
        let $sortingEl;
        let $sortingItems;
        let $sortableContainer;
        let sortingElHeight;
        let minTop;
        let maxTop;
        let $insertAfterEl;
        let $insertBeforeEl;
        let indexFrom;
        let $pageEl;
        let $pageContentEl;
        let pageHeight;
        let pageOffset;
        let sortingElOffsetLocal;
        let sortingElOffsetTop;
        let initialScrollTop;
        let wasTapHold;
        function handleTouchStart(e, isTapHold) {
          isMoved = false;
          isTouched = true;
          wasTapHold = false;
          touchStartY = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          $sortingEl = $(e.target).closest('li').eq(0);
          indexFrom = $sortingEl.index();
          $sortableContainer = $sortingEl.parents('.sortable');
          const $listGroup = $sortingEl.parents('.list-group');
          if ($listGroup.length && $listGroup.parents($sortableContainer).length) {
            $sortableContainer = $listGroup;
          }
          $sortingItems = $sortableContainer.children('ul').children('li:not(.disallow-sorting):not(.no-sorting)');
          if (app.panel) app.panel.allowOpen = false;
          if (app.swipeout) app.swipeout.allow = false;
          if (isTapHold) {
            $sortingEl.addClass('sorting');
            $sortableContainer.addClass('sortable-sorting');
            wasTapHold = true;
          }
        }
        function handleTouchMove(e) {
          if (!isTouched || !$sortingEl) return;
          const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
          if (!isMoved) {
            $pageEl = $sortingEl.parents('.page');
            $pageContentEl = $sortingEl.parents('.page-content');
            const paddingTop = parseInt($pageContentEl.css('padding-top'), 10);
            const paddingBottom = parseInt($pageContentEl.css('padding-bottom'), 10);
            initialScrollTop = $pageContentEl[0].scrollTop;
            pageOffset = $pageEl.offset().top + paddingTop;
            pageHeight = $pageEl.height() - paddingTop - paddingBottom;
            $sortingEl.addClass('sorting');
            $sortableContainer.addClass('sortable-sorting');
            sortingElOffsetLocal = $sortingEl[0].offsetTop;
            minTop = $sortingEl[0].offsetTop;
            maxTop = $sortingEl.parent().height() - sortingElOffsetLocal - $sortingEl.height();
            sortingElHeight = $sortingEl[0].offsetHeight;
            sortingElOffsetTop = $sortingEl.offset().top;
          }
          isMoved = true;
          e.preventDefault();
          e.f7PreventSwipePanel = true;
          touchesDiff = pageY - touchStartY;
          const translateScrollOffset = $pageContentEl[0].scrollTop - initialScrollTop;
          const translate = Math.min(Math.max(touchesDiff + translateScrollOffset, -minTop), maxTop);
          $sortingEl.transform(`translate3d(0,${translate}px,0)`);
          const scrollAddition = 44;
          let allowScroll = true;
          if (touchesDiff + translateScrollOffset + scrollAddition < -minTop) {
            allowScroll = false;
          }
          if (touchesDiff + translateScrollOffset - scrollAddition > maxTop) {
            allowScroll = false;
          }
          $insertBeforeEl = undefined;
          $insertAfterEl = undefined;
          let scrollDiff;
          if (allowScroll) {
            if (sortingElOffsetTop + touchesDiff + sortingElHeight + scrollAddition > pageOffset + pageHeight) {
              // To Bottom
              scrollDiff = sortingElOffsetTop + touchesDiff + sortingElHeight + scrollAddition - (pageOffset + pageHeight);
            }
            if (sortingElOffsetTop + touchesDiff < pageOffset + scrollAddition) {
              // To Top
              scrollDiff = sortingElOffsetTop + touchesDiff - pageOffset - scrollAddition;
            }
            if (scrollDiff) {
              $pageContentEl[0].scrollTop += scrollDiff;
            }
          }
          $sortingItems.each(el => {
            const $currentEl = $(el);
            if ($currentEl[0] === $sortingEl[0]) return;
            const currentElOffset = $currentEl[0].offsetTop;
            const currentElHeight = $currentEl.height();
            const sortingElOffset = sortingElOffsetLocal + translate;
            let currentTranslate;
            const prevTranslate = $currentEl[0].f7Translate;
            if (sortingElOffset >= currentElOffset - currentElHeight / 2 && $sortingEl.index() < $currentEl.index()) {
              currentTranslate = -sortingElHeight;
              $currentEl.transform(`translate3d(0, ${currentTranslate}px,0)`);
              $insertAfterEl = $currentEl;
              $insertBeforeEl = undefined;
            } else if (sortingElOffset <= currentElOffset + currentElHeight / 2 && $sortingEl.index() > $currentEl.index()) {
              currentTranslate = sortingElHeight;
              $currentEl[0].f7Translate = currentTranslate;
              $currentEl.transform(`translate3d(0, ${currentTranslate}px,0)`);
              $insertAfterEl = undefined;
              if (!$insertBeforeEl) $insertBeforeEl = $currentEl;
            } else {
              currentTranslate = undefined;
              $currentEl.transform('translate3d(0, 0%,0)');
            }
            if (prevTranslate !== currentTranslate) {
              $currentEl.trigger('sortable:move');
              app.emit('sortableMove', $currentEl[0], $sortableContainer[0]);
            }
            $currentEl[0].f7Translate = currentTranslate;
          });
        }
        function handleTouchEnd() {
          if (!isTouched || !isMoved) {
            if (isTouched && !isMoved) {
              if (app.panel) app.panel.allowOpen = true;
              if (app.swipeout) app.swipeout.allow = true;
              if (wasTapHold) {
                $sortingEl.removeClass('sorting');
                $sortableContainer.removeClass('sortable-sorting');
              }
            }
            isTouched = false;
            isMoved = false;
            return;
          }
          if (app.panel) app.panel.allowOpen = true;
          if (app.swipeout) app.swipeout.allow = true;
          $sortingItems.transform('');
          $sortingEl.removeClass('sorting');
          $sortableContainer.removeClass('sortable-sorting');
          let indexTo;
          if ($insertAfterEl) indexTo = $insertAfterEl.index();else if ($insertBeforeEl) indexTo = $insertBeforeEl.index();
          let moveElements = $sortableContainer.dataset().sortableMoveElements;
          if (typeof moveElements === 'undefined') {
            moveElements = app.params.sortable.moveElements;
          }
          if (moveElements) {
            if ($insertAfterEl) {
              $sortingEl.insertAfter($insertAfterEl);
            }
            if ($insertBeforeEl) {
              $sortingEl.insertBefore($insertBeforeEl);
            }
          }
          if (($insertAfterEl || $insertBeforeEl) && $sortableContainer.hasClass('virtual-list')) {
            indexFrom = $sortingEl[0].f7VirtualListIndex;
            if (typeof indexFrom === 'undefined') indexFrom = $sortingEl.attr('data-virtual-list-index');
            if ($insertBeforeEl) {
              indexTo = $insertBeforeEl[0].f7VirtualListIndex;
              if (typeof indexTo === 'undefined') indexTo = $insertBeforeEl.attr('data-virtual-list-index');
            } else {
              indexTo = $insertAfterEl[0].f7VirtualListIndex;
              if (typeof indexTo === 'undefined') indexTo = $insertAfterEl.attr('data-virtual-list-index');
            }
            if (indexTo !== null) indexTo = parseInt(indexTo, 10);else indexTo = undefined;
            const virtualList = $sortableContainer[0].f7VirtualList;
            if (indexFrom) indexFrom = parseInt(indexFrom, 10);
            if (indexTo) indexTo = parseInt(indexTo, 10);
            if (virtualList) virtualList.moveItem(indexFrom, indexTo);
          }
          if (typeof indexTo !== 'undefined' && !Number.isNaN(indexTo) && indexTo !== indexFrom) {
            $sortingEl.trigger('sortable:sort', {
              from: indexFrom,
              to: indexTo
            });
            app.emit('sortableSort', $sortingEl[0], {
              from: indexFrom,
              to: indexTo,
              el: $sortingEl[0]
            }, $sortableContainer[0]);
          }
          $insertBeforeEl = undefined;
          $insertAfterEl = undefined;
          isTouched = false;
          isMoved = false;
        }
        const activeListener = getSupport$1().passiveListener ? {
          passive: false,
          capture: false
        } : false;
        $(document).on(app.touchEvents.start, '.list.sortable .sortable-handler', handleTouchStart, activeListener);
        app.on('touchmove:active', handleTouchMove);
        app.on('touchend:passive', handleTouchEnd);
        $(document).on('taphold', '.sortable-tap-hold', (e, pointerEvent) => {
          handleTouchStart(pointerEvent, true);
        });
      },
      enable(el) {
        if (el === void 0) {
          el = '.list.sortable';
        }
        const app = this;
        const $el = $(el);
        if ($el.length === 0) return;
        $el.addClass('sortable-enabled');
        $el.trigger('sortable:enable');
        app.emit('sortableEnable', $el[0]);
      },
      disable(el) {
        if (el === void 0) {
          el = '.list.sortable';
        }
        const app = this;
        const $el = $(el);
        if ($el.length === 0) return;
        $el.removeClass('sortable-enabled');
        $el.trigger('sortable:disable');
        app.emit('sortableDisable', $el[0]);
      },
      toggle(el) {
        if (el === void 0) {
          el = '.list.sortable';
        }
        const app = this;
        const $el = $(el);
        if ($el.length === 0) return;
        if ($el.hasClass('sortable-enabled')) {
          app.sortable.disable($el);
        } else {
          app.sortable.enable($el);
        }
      }
    };
    var Sortable$1 = {
      name: 'sortable',
      params: {
        sortable: {
          moveElements: true
        }
      },
      create() {
        const app = this;
        bindMethods(app, {
          sortable: Sortable
        });
      },
      on: {
        init() {
          const app = this;
          if (!app.params.sortable) return;
          app.sortable.init();
        }
      },
      clicks: {
        '.sortable-enable': function enable($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.sortable.enable(data.sortable);
        },
        '.sortable-disable': function disable($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.sortable.disable(data.sortable);
        },
        '.sortable-toggle': function toggle($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.sortable.toggle(data.sortable);
        }
      }
    };

    const Swipeout = {
      init() {
        const app = this;
        const document = getDocument$1();
        const touchesStart = {};
        let isTouched;
        let isMoved;
        let isScrolling;
        let touchStartTime;
        let touchesDiff;
        let $swipeoutEl;
        let $swipeoutContent;
        let $actionsRight;
        let $actionsLeft;
        let actionsLeftWidth;
        let actionsRightWidth;
        let translate;
        let opened;
        let openedActionsSide;
        let $leftButtons;
        let $rightButtons;
        let direction;
        let $overswipeLeftButton;
        let $overswipeRightButton;
        let overswipeLeft;
        let overswipeRight;
        function handleTouchStart(e) {
          if (!app.swipeout.allow) return;
          isMoved = false;
          isTouched = true;
          isScrolling = undefined;
          touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
          touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          touchStartTime = new Date().getTime();
          $swipeoutEl = $(this);
        }
        function handleTouchMove(e) {
          if (!isTouched) return;
          const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
          const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
          if (typeof isScrolling === 'undefined') {
            isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x));
          }
          if (isScrolling) {
            isTouched = false;
            return;
          }
          if (!isMoved) {
            if ($('.list.sortable-opened').length > 0) return;
            $swipeoutContent = $swipeoutEl.find('.swipeout-content');
            $actionsRight = $swipeoutEl.find('.swipeout-actions-right');
            $actionsLeft = $swipeoutEl.find('.swipeout-actions-left');
            actionsLeftWidth = null;
            actionsRightWidth = null;
            $leftButtons = null;
            $rightButtons = null;
            $overswipeRightButton = null;
            $overswipeLeftButton = null;
            if ($actionsLeft.length > 0) {
              actionsLeftWidth = $actionsLeft.outerWidth();
              $leftButtons = $actionsLeft.children('a');
              $overswipeLeftButton = $actionsLeft.find('.swipeout-overswipe');
            }
            if ($actionsRight.length > 0) {
              actionsRightWidth = $actionsRight.outerWidth();
              $rightButtons = $actionsRight.children('a');
              $overswipeRightButton = $actionsRight.find('.swipeout-overswipe');
            }
            opened = $swipeoutEl.hasClass('swipeout-opened');
            if (opened) {
              openedActionsSide = $swipeoutEl.find('.swipeout-actions-left.swipeout-actions-opened').length > 0 ? 'left' : 'right';
            }
            $swipeoutEl.removeClass('swipeout-transitioning');
            if (!app.params.swipeout.noFollow) {
              $swipeoutEl.find('.swipeout-actions-opened').removeClass('swipeout-actions-opened');
              $swipeoutEl.removeClass('swipeout-opened');
            }
          }
          isMoved = true;
          if (e.cancelable) {
            e.preventDefault();
          }
          touchesDiff = pageX - touchesStart.x;
          translate = touchesDiff;
          if (opened) {
            if (openedActionsSide === 'right') translate -= actionsRightWidth;else translate += actionsLeftWidth;
          }
          if (translate > 0 && $actionsLeft.length === 0 || translate < 0 && $actionsRight.length === 0) {
            if (!opened) {
              isTouched = false;
              isMoved = false;
              $swipeoutContent.transform('');
              if ($rightButtons && $rightButtons.length > 0) {
                $rightButtons.transform('');
              }
              if ($leftButtons && $leftButtons.length > 0) {
                $leftButtons.transform('');
              }
              return;
            }
            translate = 0;
          }
          if (translate < 0) direction = 'to-left';else if (translate > 0) direction = 'to-right';else if (!direction) direction = 'to-left';
          let buttonOffset;
          let progress;
          e.f7PreventSwipePanel = true;
          if (app.params.swipeout.noFollow) {
            if (opened) {
              if (openedActionsSide === 'right' && touchesDiff > 0) {
                app.swipeout.close($swipeoutEl);
              }
              if (openedActionsSide === 'left' && touchesDiff < 0) {
                app.swipeout.close($swipeoutEl);
              }
            } else {
              if (touchesDiff < 0 && $actionsRight.length > 0) {
                app.swipeout.open($swipeoutEl, 'right');
              }
              if (touchesDiff > 0 && $actionsLeft.length > 0) {
                app.swipeout.open($swipeoutEl, 'left');
              }
            }
            isTouched = false;
            isMoved = false;
            return;
          }
          overswipeLeft = false;
          overswipeRight = false;
          if ($actionsRight.length > 0) {
            // Show right actions
            let buttonTranslate = translate;
            progress = buttonTranslate / actionsRightWidth;
            if (buttonTranslate < -actionsRightWidth) {
              const ratio = buttonTranslate / -actionsRightWidth;
              buttonTranslate = -actionsRightWidth - (-buttonTranslate - actionsRightWidth) ** 0.8;
              translate = buttonTranslate;
              if ($overswipeRightButton.length > 0 && ratio > app.params.swipeout.overswipeRatio) {
                overswipeRight = true;
              }
            }
            if (direction !== 'to-left') {
              progress = 0;
              buttonTranslate = 0;
            }
            $rightButtons.each(buttonEl => {
              const $buttonEl = $(buttonEl);
              if (typeof buttonEl.f7SwipeoutButtonOffset === 'undefined') {
                $buttonEl[0].f7SwipeoutButtonOffset = buttonEl.offsetLeft;
              }
              buttonOffset = buttonEl.f7SwipeoutButtonOffset;
              if ($overswipeRightButton.length > 0 && $buttonEl.hasClass('swipeout-overswipe') && direction === 'to-left') {
                $buttonEl.css({
                  left: `${overswipeRight ? -buttonOffset : 0}px`
                });
                if (overswipeRight) {
                  if (!$buttonEl.hasClass('swipeout-overswipe-active')) {
                    $swipeoutEl.trigger('swipeout:overswipeenter');
                    app.emit('swipeoutOverswipeEnter', $swipeoutEl[0]);
                  }
                  $buttonEl.addClass('swipeout-overswipe-active');
                } else {
                  if ($buttonEl.hasClass('swipeout-overswipe-active')) {
                    $swipeoutEl.trigger('swipeout:overswipeexit');
                    app.emit('swipeoutOverswipeExit', $swipeoutEl[0]);
                  }
                  $buttonEl.removeClass('swipeout-overswipe-active');
                }
              }
              $buttonEl.transform(`translate3d(${buttonTranslate - buttonOffset * (1 + Math.max(progress, -1))}px,0,0)`);
            });
          }
          if ($actionsLeft.length > 0) {
            // Show left actions
            let buttonTranslate = translate;
            progress = buttonTranslate / actionsLeftWidth;
            if (buttonTranslate > actionsLeftWidth) {
              const ratio = buttonTranslate / actionsRightWidth;
              buttonTranslate = actionsLeftWidth + (buttonTranslate - actionsLeftWidth) ** 0.8;
              translate = buttonTranslate;
              if ($overswipeLeftButton.length > 0 && ratio > app.params.swipeout.overswipeRatio) {
                overswipeLeft = true;
              }
            }
            if (direction !== 'to-right') {
              buttonTranslate = 0;
              progress = 0;
            }
            $leftButtons.each((buttonEl, index) => {
              const $buttonEl = $(buttonEl);
              if (typeof buttonEl.f7SwipeoutButtonOffset === 'undefined') {
                $buttonEl[0].f7SwipeoutButtonOffset = actionsLeftWidth - buttonEl.offsetLeft - buttonEl.offsetWidth;
              }
              buttonOffset = buttonEl.f7SwipeoutButtonOffset;
              if ($overswipeLeftButton.length > 0 && $buttonEl.hasClass('swipeout-overswipe') && direction === 'to-right') {
                $buttonEl.css({
                  left: `${overswipeLeft ? buttonOffset : 0}px`
                });
                if (overswipeLeft) {
                  if (!$buttonEl.hasClass('swipeout-overswipe-active')) {
                    $swipeoutEl.trigger('swipeout:overswipeenter');
                    app.emit('swipeoutOverswipeEnter', $swipeoutEl[0]);
                  }
                  $buttonEl.addClass('swipeout-overswipe-active');
                } else {
                  if ($buttonEl.hasClass('swipeout-overswipe-active')) {
                    $swipeoutEl.trigger('swipeout:overswipeexit');
                    app.emit('swipeoutOverswipeExit', $swipeoutEl[0]);
                  }
                  $buttonEl.removeClass('swipeout-overswipe-active');
                }
              }
              if ($leftButtons.length > 1) {
                $buttonEl.css('z-index', $leftButtons.length - index);
              }
              $buttonEl.transform(`translate3d(${buttonTranslate + buttonOffset * (1 - Math.min(progress, 1))}px,0,0)`);
            });
          }
          $swipeoutEl.trigger('swipeout', progress);
          app.emit('swipeout', $swipeoutEl[0], progress);
          $swipeoutContent.transform(`translate3d(${translate}px,0,0)`);
        }
        function handleTouchEnd() {
          if (!isTouched || !isMoved) {
            isTouched = false;
            isMoved = false;
            return;
          }
          isTouched = false;
          isMoved = false;
          const timeDiff = new Date().getTime() - touchStartTime;
          const $actions = direction === 'to-left' ? $actionsRight : $actionsLeft;
          const actionsWidth = direction === 'to-left' ? actionsRightWidth : actionsLeftWidth;
          let action;
          let $buttons;
          let i;
          if (timeDiff < 300 && (touchesDiff < -10 && direction === 'to-left' || touchesDiff > 10 && direction === 'to-right') || timeDiff >= 300 && Math.abs(translate) > actionsWidth / 2) {
            action = 'open';
          } else {
            action = 'close';
          }
          if (timeDiff < 300) {
            if (Math.abs(translate) === 0) action = 'close';
            if (Math.abs(translate) === actionsWidth) action = 'open';
          }
          if (action === 'open') {
            Swipeout.el = $swipeoutEl[0];
            $swipeoutEl.trigger('swipeout:open');
            app.emit('swipeoutOpen', $swipeoutEl[0]);
            $swipeoutEl.addClass('swipeout-opened swipeout-transitioning');
            const newTranslate = direction === 'to-left' ? -actionsWidth : actionsWidth;
            $swipeoutContent.transform(`translate3d(${newTranslate}px,0,0)`);
            $actions.addClass('swipeout-actions-opened');
            $buttons = direction === 'to-left' ? $rightButtons : $leftButtons;
            if ($buttons) {
              for (i = 0; i < $buttons.length; i += 1) {
                $($buttons[i]).transform(`translate3d(${newTranslate}px,0,0)`);
              }
            }
            if (overswipeRight) {
              $actionsRight.find('.swipeout-overswipe').trigger('click', 'f7Overswipe');
            }
            if (overswipeLeft) {
              $actionsLeft.find('.swipeout-overswipe').trigger('click', 'f7Overswipe');
            }
          } else {
            $swipeoutEl.trigger('swipeout:close');
            app.emit('swipeoutClose', $swipeoutEl[0]);
            Swipeout.el = undefined;
            $swipeoutEl.addClass('swipeout-transitioning').removeClass('swipeout-opened');
            $swipeoutContent.transform('');
            $actions.removeClass('swipeout-actions-opened');
          }
          let buttonOffset;
          if ($leftButtons && $leftButtons.length > 0 && $leftButtons !== $buttons) {
            $leftButtons.each(buttonEl => {
              const $buttonEl = $(buttonEl);
              buttonOffset = buttonEl.f7SwipeoutButtonOffset;
              if (typeof buttonOffset === 'undefined') {
                $buttonEl[0].f7SwipeoutButtonOffset = actionsLeftWidth - buttonEl.offsetLeft - buttonEl.offsetWidth;
              }
              $buttonEl.transform(`translate3d(${buttonOffset}px,0,0)`);
            });
          }
          if ($rightButtons && $rightButtons.length > 0 && $rightButtons !== $buttons) {
            $rightButtons.each(buttonEl => {
              const $buttonEl = $(buttonEl);
              buttonOffset = buttonEl.f7SwipeoutButtonOffset;
              if (typeof buttonOffset === 'undefined') {
                $buttonEl[0].f7SwipeoutButtonOffset = buttonEl.offsetLeft;
              }
              $buttonEl.transform(`translate3d(${-buttonOffset}px,0,0)`);
            });
          }
          $swipeoutContent.transitionEnd(() => {
            if (opened && action === 'open' || !opened && action === 'close') return;
            $swipeoutEl.trigger(action === 'open' ? 'swipeout:opened' : 'swipeout:closed');
            app.emit(action === 'open' ? 'swipeoutOpened' : 'swipeoutClosed', $swipeoutEl[0]);
            $swipeoutEl.removeClass('swipeout-transitioning');
            if (opened && action === 'close') {
              if ($actionsRight.length > 0) {
                $rightButtons.transform('');
              }
              if ($actionsLeft.length > 0) {
                $leftButtons.transform('');
              }
            }
          });
        }
        const passiveListener = getSupport$1().passiveListener ? {
          passive: true
        } : false;
        app.on('touchstart', e => {
          if (Swipeout.el) {
            const $targetEl = $(e.target);
            if (!($(Swipeout.el).is($targetEl[0]) || $targetEl.parents('.swipeout').is(Swipeout.el) || $targetEl.hasClass('modal-in') || ($targetEl.attr('class') || '').indexOf('-backdrop') > 0 || $targetEl.hasClass('actions-modal') || $targetEl.parents('.actions-modal.modal-in, .dialog.modal-in').length > 0)) {
              app.swipeout.close(Swipeout.el);
            }
          }
        });
        $(document).on(app.touchEvents.start, 'li.swipeout', handleTouchStart, passiveListener);
        app.on('touchmove:active', handleTouchMove);
        app.on('touchend:passive', handleTouchEnd);
      },
      allow: true,
      el: undefined,
      open() {
        const app = this;
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }
        let [el, side, callback] = args;
        if (typeof args[1] === 'function') {
          [el, callback, side] = args;
        }
        const $el = $(el).eq(0);
        if ($el.length === 0) return;
        if (!$el.hasClass('swipeout') || $el.hasClass('swipeout-opened')) return;
        if (!side) {
          if ($el.find('.swipeout-actions-right').length > 0) side = 'right';else side = 'left';
        }
        const $swipeoutActions = $el.find(`.swipeout-actions-${side}`);
        const $swipeoutContent = $el.find('.swipeout-content');
        if ($swipeoutActions.length === 0) return;
        $el.trigger('swipeout:open').addClass('swipeout-opened').removeClass('swipeout-transitioning');
        app.emit('swipeoutOpen', $el[0]);
        $swipeoutActions.addClass('swipeout-actions-opened');
        const $buttons = $swipeoutActions.children('a');
        const swipeoutActionsWidth = $swipeoutActions.outerWidth();
        const translate = side === 'right' ? -swipeoutActionsWidth : swipeoutActionsWidth;
        if ($buttons.length > 1) {
          $buttons.each((buttonEl, buttonIndex) => {
            const $buttonEl = $(buttonEl);
            if (side === 'right') {
              $buttonEl.transform(`translate3d(${-buttonEl.offsetLeft}px,0,0)`);
            } else {
              $buttonEl.css('z-index', $buttons.length - buttonIndex).transform(`translate3d(${swipeoutActionsWidth - buttonEl.offsetWidth - buttonEl.offsetLeft}px,0,0)`);
            }
          });
        }
        $el.addClass('swipeout-transitioning');
        $swipeoutContent.transitionEnd(() => {
          $el.trigger('swipeout:opened');
          app.emit('swipeoutOpened', $el[0]);
          if (callback) callback.call($el[0]);
        });
        nextFrame$1(() => {
          $buttons.transform(`translate3d(${translate}px,0,0)`);
          $swipeoutContent.transform(`translate3d(${translate}px,0,0)`);
        });
        Swipeout.el = $el[0];
      },
      close(el, callback) {
        const app = this;
        const $el = $(el).eq(0);
        if ($el.length === 0) return;
        if (!$el.hasClass('swipeout-opened')) return;
        const side = $el.find('.swipeout-actions-opened').hasClass('swipeout-actions-right') ? 'right' : 'left';
        const $swipeoutActions = $el.find('.swipeout-actions-opened').removeClass('swipeout-actions-opened');
        const $buttons = $swipeoutActions.children('a');
        const swipeoutActionsWidth = $swipeoutActions.outerWidth();
        app.swipeout.allow = false;
        $el.trigger('swipeout:close');
        app.emit('swipeoutClose', $el[0]);
        $el.removeClass('swipeout-opened').addClass('swipeout-transitioning');
        let closeTimeout;
        function onSwipeoutClose() {
          app.swipeout.allow = true;
          if ($el.hasClass('swipeout-opened')) return;
          $el.removeClass('swipeout-transitioning');
          $buttons.transform('');
          $el.trigger('swipeout:closed');
          app.emit('swipeoutClosed', $el[0]);
          if (callback) callback.call($el[0]);
          if (closeTimeout) clearTimeout(closeTimeout);
        }
        $el.find('.swipeout-content').transform('').transitionEnd(onSwipeoutClose);
        closeTimeout = setTimeout(onSwipeoutClose, 500);
        $buttons.each(buttonEl => {
          const $buttonEl = $(buttonEl);
          if (side === 'right') {
            $buttonEl.transform(`translate3d(${-buttonEl.offsetLeft}px,0,0)`);
          } else {
            $buttonEl.transform(`translate3d(${swipeoutActionsWidth - buttonEl.offsetWidth - buttonEl.offsetLeft}px,0,0)`);
          }
          $buttonEl.css({
            left: '0px'
          }).removeClass('swipeout-overswipe-active');
        });
        if (Swipeout.el && Swipeout.el === $el[0]) Swipeout.el = undefined;
      },
      delete(el, callback) {
        const app = this;
        const $el = $(el).eq(0);
        if ($el.length === 0) return;
        Swipeout.el = undefined;
        $el.trigger('swipeout:delete');
        app.emit('swipeoutDelete', $el[0]);
        $el.css({
          height: `${$el.outerHeight()}px`
        });
        $el.transitionEnd(() => {
          $el.trigger('swipeout:deleted');
          app.emit('swipeoutDeleted', $el[0]);
          if (callback) callback.call($el[0]);
          if ($el.parents('.virtual-list').length > 0) {
            const virtualList = $el.parents('.virtual-list')[0].f7VirtualList;
            const virtualIndex = $el[0].f7VirtualListIndex;
            if (virtualList && typeof virtualIndex !== 'undefined') virtualList.deleteItem(virtualIndex);
          } else if (app.params.swipeout.removeElements) {
            if (app.params.swipeout.removeElementsWithTimeout) {
              setTimeout(() => {
                $el.remove();
              }, app.params.swipeout.removeElementsTimeout);
            } else {
              $el.remove();
            }
          } else {
            $el.removeClass('swipeout-deleting swipeout-transitioning');
          }
        });
        // eslint-disable-next-line
        // $el[0]._clientLeft = $el[0].clientLeft;
        nextFrame$1(() => {
          $el.addClass('swipeout-deleting swipeout-transitioning').css({
            height: '0px'
          }).find('.swipeout-content').transform('translate3d(-100%,0,0)');
        });
      }
    };
    var Swipeout$1 = {
      name: 'swipeout',
      params: {
        swipeout: {
          actionsNoFold: false,
          noFollow: false,
          removeElements: true,
          removeElementsWithTimeout: false,
          removeElementsTimeout: 0,
          overswipeRatio: 1.2
        }
      },
      create() {
        const app = this;
        bindMethods(app, {
          swipeout: Swipeout
        });
      },
      clicks: {
        '.swipeout-open': function openSwipeout($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.swipeout.open(data.swipeout, data.side);
        },
        '.swipeout-close': function closeSwipeout($clickedEl) {
          const app = this;
          const $swipeoutEl = $clickedEl.closest('.swipeout');
          if ($swipeoutEl.length === 0) return;
          app.swipeout.close($swipeoutEl);
        },
        '.swipeout-delete': function deleteSwipeout($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          const $swipeoutEl = $clickedEl.closest('.swipeout');
          if ($swipeoutEl.length === 0) return;
          const {
            confirm,
            confirmTitle
          } = data;
          if (data.confirm) {
            app.dialog.confirm(confirm, confirmTitle, () => {
              app.swipeout.delete($swipeoutEl);
            });
          } else {
            app.swipeout.delete($swipeoutEl);
          }
        }
      },
      on: {
        init() {
          const app = this;
          if (!app.params.swipeout) return;
          app.swipeout.init();
        }
      }
    };

    const Accordion = {
      toggleClicked($clickedEl) {
        const app = this;
        let $accordionItemEl = $clickedEl.closest('.accordion-item').eq(0);
        if (!$accordionItemEl.length) $accordionItemEl = $clickedEl.parents('li').eq(0);
        const $accordionContent = $clickedEl.parents('.accordion-item-content').eq(0);
        if ($accordionContent.length) {
          if ($accordionContent.parents($accordionItemEl).length) return;
        }
        if ($clickedEl.parents('li').length > 1 && $clickedEl.parents('li')[0] !== $accordionItemEl[0]) return;
        app.accordion.toggle($accordionItemEl);
      },
      open(el) {
        const app = this;
        const $el = $(el);
        let prevented = false;
        function prevent() {
          prevented = true;
        }
        $el.trigger('accordion:beforeopen', {
          prevent
        }, prevent);
        app.emit('accordionBeforeOpen', $el[0], prevent);
        if (prevented) return;
        const $list = $el.parents('.accordion-list').eq(0);
        let $contentEl = $el.children('.accordion-item-content');
        $contentEl.removeAttr('aria-hidden');
        if ($contentEl.length === 0) $contentEl = $el.find('.accordion-item-content');
        if ($contentEl.length === 0) return;
        const $openedItem = $list.length > 0 && $el.parent().children('.accordion-item-opened');
        if ($openedItem.length > 0) {
          app.accordion.close($openedItem);
        }
        $contentEl.transitionEnd(() => {
          if ($el.hasClass('accordion-item-opened')) {
            $contentEl.transition(0);
            $contentEl.css('height', 'auto');
            nextFrame$1(() => {
              $contentEl.transition('');
              $el.trigger('accordion:opened');
              app.emit('accordionOpened', $el[0]);
            });
          } else {
            $contentEl.css('height', '');
            $el.trigger('accordion:closed');
            app.emit('accordionClosed', $el[0]);
          }
        });
        $contentEl.css('height', `${$contentEl[0].scrollHeight}px`);
        $el.trigger('accordion:open');
        $el.addClass('accordion-item-opened');
        app.emit('accordionOpen', $el[0]);
      },
      close(el) {
        const app = this;
        const $el = $(el);
        let prevented = false;
        function prevent() {
          prevented = true;
        }
        $el.trigger('accordion:beforeclose', {
          prevent
        }, prevent);
        app.emit('accordionBeforeClose', $el[0], prevent);
        if (prevented) return;
        let $contentEl = $el.children('.accordion-item-content');
        if ($contentEl.length === 0) $contentEl = $el.find('.accordion-item-content');
        $el.removeClass('accordion-item-opened');
        $contentEl.attr('aria-hidden', true);
        $contentEl.transition(0);
        $contentEl.css('height', `${$contentEl[0].scrollHeight}px`);
        // Close
        $contentEl.transitionEnd(() => {
          if ($el.hasClass('accordion-item-opened')) {
            $contentEl.transition(0);
            $contentEl.css('height', 'auto');
            nextFrame$1(() => {
              $contentEl.transition('');
              $el.trigger('accordion:opened');
              app.emit('accordionOpened', $el[0]);
            });
          } else {
            $contentEl.css('height', '');
            $el.trigger('accordion:closed');
            app.emit('accordionClosed', $el[0]);
          }
        });
        nextFrame$1(() => {
          $contentEl.transition('');
          $contentEl.css('height', '');
          $el.trigger('accordion:close');
          app.emit('accordionClose', $el[0]);
        });
      },
      toggle(el) {
        const app = this;
        const $el = $(el);
        if ($el.length === 0) return;
        if ($el.hasClass('accordion-item-opened')) app.accordion.close(el);else app.accordion.open(el);
      }
    };
    var Accordion$1 = {
      name: 'accordion',
      create() {
        const app = this;
        bindMethods(app, {
          accordion: Accordion
        });
      },
      clicks: {
        '.accordion-item .item-link, .accordion-item-toggle, .links-list.accordion-list > ul > li > a': function open($clickedEl) {
          const app = this;
          Accordion.toggleClicked.call(app, $clickedEl);
        }
      }
    };

    var ContactsList = {
      name: 'contactsList'
    };

    let VirtualList$1 = class VirtualList extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const vl = this;
        const device = getDevice$1();
        const document = getDocument$1();
        let defaultHeight;
        if (app.theme === 'md') {
          defaultHeight = 48;
        } else if (app.theme === 'ios') {
          defaultHeight = 44;
        }
        const defaults = {
          cols: 1,
          height: defaultHeight,
          cache: true,
          dynamicHeightBufferSize: 1,
          showFilteredItemsOnly: false,
          renderExternal: undefined,
          setListHeight: true,
          searchByItem: undefined,
          searchAll: undefined,
          ul: null,
          createUl: true,
          scrollableParentEl: undefined,
          renderItem(item) {
            return `
          <li>
            <div class="item-content">
              <div class="item-inner">
                <div class="item-title">${item}</div>
              </div>
            </div>
          </li>
        `.trim();
          },
          on: {}
        };

        // Extend defaults with modules params
        vl.useModulesParams(defaults);
        vl.params = extend$3(defaults, params);
        if (vl.params.height === undefined || !vl.params.height) {
          vl.params.height = defaultHeight;
        }
        vl.$el = $(params.el);
        vl.el = vl.$el[0];
        if (vl.$el.length === 0) return undefined;
        vl.$el[0].f7VirtualList = vl;
        vl.items = vl.params.items;
        if (vl.params.showFilteredItemsOnly) {
          vl.filteredItems = [];
        }
        if (vl.params.renderItem) {
          vl.renderItem = vl.params.renderItem;
        }
        vl.$pageContentEl = vl.$el.parents('.page-content');
        vl.pageContentEl = vl.$pageContentEl[0];
        vl.$scrollableParentEl = vl.params.scrollableParentEl ? $(vl.params.scrollableParentEl).eq(0) : vl.$pageContentEl;
        if (!vl.$scrollableParentEl.length && vl.$pageContentEl.length) {
          vl.$scrollableParentEl = vl.$pageContentEl;
        }
        vl.scrollableParentEl = vl.$scrollableParentEl[0];

        // Bad scroll
        if (typeof vl.params.updatableScroll !== 'undefined') {
          vl.updatableScroll = vl.params.updatableScroll;
        } else {
          vl.updatableScroll = true;
          if (device.ios && device.osVersion.split('.')[0] < 8) {
            vl.updatableScroll = false;
          }
        }

        // Append <ul>
        const ul = vl.params.ul;
        vl.$ul = ul ? $(vl.params.ul) : vl.$el.children('ul');
        if (vl.$ul.length === 0 && vl.params.createUl) {
          vl.$el.append('<ul></ul>');
          vl.$ul = vl.$el.children('ul');
        }
        vl.ul = vl.$ul[0];
        let $itemsWrapEl;
        if (!vl.ul && !vl.params.createUl) $itemsWrapEl = vl.$el;else $itemsWrapEl = vl.$ul;
        extend$3(vl, {
          $itemsWrapEl,
          itemsWrapEl: $itemsWrapEl[0],
          // DOM cached items
          domCache: {},
          // Temporary DOM Element
          tempDomElement: document.createElement('ul'),
          // Last repain position
          lastRepaintY: null,
          // Fragment
          fragment: document.createDocumentFragment(),
          // Props
          pageHeight: undefined,
          rowsPerScreen: undefined,
          rowsBefore: undefined,
          rowsAfter: undefined,
          rowsToRender: undefined,
          maxBufferHeight: 0,
          listHeight: undefined,
          dynamicHeight: typeof vl.params.height === 'function',
          autoHeight: vl.params.height === 'auto'
        });

        // Install Modules
        vl.useModules();

        // Attach events
        const handleScrollBound = vl.handleScroll.bind(vl);
        const handleResizeBound = vl.handleResize.bind(vl);
        let $pageEl;
        let $tabEl;
        let $panelEl;
        let $popupEl;
        vl.attachEvents = function attachEvents() {
          $pageEl = vl.$el.parents('.page').eq(0);
          $tabEl = vl.$el.parents('.tab').filter(tabEl => {
            return $(tabEl).parent('.tabs').parent('.tabs-animated-wrap, swiper-container.tabs').length === 0;
          }).eq(0);
          $panelEl = vl.$el.parents('.panel').eq(0);
          $popupEl = vl.$el.parents('.popup').eq(0);
          vl.$scrollableParentEl.on('scroll', handleScrollBound);
          if ($pageEl.length) $pageEl.on('page:reinit', handleResizeBound);
          if ($tabEl.length) $tabEl.on('tab:show', handleResizeBound);
          if ($panelEl.length) $panelEl.on('panel:open', handleResizeBound);
          if ($popupEl.length) $popupEl.on('popup:open', handleResizeBound);
          app.on('resize', handleResizeBound);
        };
        vl.detachEvents = function attachEvents() {
          vl.$scrollableParentEl.off('scroll', handleScrollBound);
          if ($pageEl.length) $pageEl.off('page:reinit', handleResizeBound);
          if ($tabEl.length) $tabEl.off('tab:show', handleResizeBound);
          if ($panelEl.length) $panelEl.off('panel:open', handleResizeBound);
          if ($popupEl.length) $popupEl.off('popup:open', handleResizeBound);
          app.off('resize', handleResizeBound);
        };
        // Init
        vl.init();
        return vl;
      }
      setListSize(autoHeightRerender) {
        const vl = this;
        const items = vl.filteredItems || vl.items;
        if (!autoHeightRerender) {
          vl.pageHeight = vl.$scrollableParentEl[0].offsetHeight;
        }
        if (vl.dynamicHeight) {
          vl.listHeight = 0;
          vl.heights = [];
          for (let i = 0; i < items.length; i += 1) {
            const itemHeight = vl.params.height(items[i]);
            vl.listHeight += itemHeight;
            vl.heights.push(itemHeight);
          }
        } else if (vl.autoHeight) {
          vl.listHeight = 0;
          if (!vl.heights) vl.heights = [];
          if (!vl.heightsCalculated) vl.heightsCalculated = [];
          const renderedItems = {};
          vl.$itemsWrapEl.find(`[data-virtual-list-index]`).forEach(el => {
            renderedItems[parseInt(el.getAttribute('data-virtual-list-index'), 10)] = el;
          });
          for (let i = 0; i < items.length; i += 1) {
            const itemIndex = vl.items.indexOf(items[i]);
            const renderedItem = renderedItems[itemIndex];
            if (renderedItem) {
              if (!vl.heightsCalculated.includes(itemIndex)) {
                vl.heights[itemIndex] = renderedItem.offsetHeight;
                vl.heightsCalculated.push(itemIndex);
              }
            }
            if (typeof vl.heights[i] === 'undefined') {
              vl.heights[itemIndex] = 40;
            }
            vl.listHeight += vl.heights[itemIndex];
          }
        } else {
          vl.listHeight = Math.ceil(items.length / vl.params.cols) * vl.params.height;
          vl.rowsPerScreen = Math.ceil(vl.pageHeight / vl.params.height);
          vl.rowsBefore = vl.params.rowsBefore || vl.rowsPerScreen * 2;
          vl.rowsAfter = vl.params.rowsAfter || vl.rowsPerScreen;
          vl.rowsToRender = vl.rowsPerScreen + vl.rowsBefore + vl.rowsAfter;
          vl.maxBufferHeight = vl.rowsBefore / 2 * vl.params.height;
        }
        if (vl.updatableScroll || vl.params.setListHeight) {
          vl.$itemsWrapEl.css({
            height: `${vl.listHeight}px`
          });
        }
      }
      render(force, forceScrollTop) {
        const vl = this;
        if (force) vl.lastRepaintY = null;
        let scrollTop = -(vl.$el[0].getBoundingClientRect().top - vl.$scrollableParentEl[0].getBoundingClientRect().top);
        if (typeof forceScrollTop !== 'undefined') scrollTop = forceScrollTop;
        if (vl.lastRepaintY === null || Math.abs(scrollTop - vl.lastRepaintY) > vl.maxBufferHeight || !vl.updatableScroll && vl.$scrollableParentEl[0].scrollTop + vl.pageHeight >= vl.$scrollableParentEl[0].scrollHeight) {
          vl.lastRepaintY = scrollTop;
        } else {
          return;
        }
        const items = vl.filteredItems || vl.items;
        let fromIndex;
        let toIndex;
        let heightBeforeFirstItem = 0;
        let heightBeforeLastItem = 0;
        if (vl.dynamicHeight || vl.autoHeight) {
          let itemTop = 0;
          let itemHeight;
          vl.maxBufferHeight = vl.pageHeight;
          for (let j = 0; j < vl.heights.length; j += 1) {
            itemHeight = vl.heights[j];
            if (typeof fromIndex === 'undefined') {
              if (itemTop + itemHeight >= scrollTop - vl.pageHeight * 2 * vl.params.dynamicHeightBufferSize) fromIndex = j;else heightBeforeFirstItem += itemHeight;
            }
            if (typeof toIndex === 'undefined') {
              if (itemTop + itemHeight >= scrollTop + vl.pageHeight * 2 * vl.params.dynamicHeightBufferSize || j === vl.heights.length - 1) toIndex = j + 1;
              heightBeforeLastItem += itemHeight;
            }
            itemTop += itemHeight;
          }
          toIndex = Math.min(toIndex, items.length);
        } else {
          fromIndex = (parseInt(scrollTop / vl.params.height, 10) - vl.rowsBefore) * vl.params.cols;
          if (fromIndex < 0) {
            fromIndex = 0;
          }
          toIndex = Math.min(fromIndex + vl.rowsToRender * vl.params.cols, items.length);
        }
        let topPosition;
        const renderExternalItems = [];
        vl.reachEnd = false;
        let i;
        for (i = fromIndex; i < toIndex; i += 1) {
          let itemEl;
          // Define real item index
          const index = vl.items.indexOf(items[i]);
          if (i === fromIndex) vl.currentFromIndex = index;
          if (i === toIndex - 1) vl.currentToIndex = index;
          if (vl.filteredItems) {
            if (vl.items[index] === vl.filteredItems[vl.filteredItems.length - 1]) vl.reachEnd = true;
          } else if (index === vl.items.length - 1) vl.reachEnd = true;

          // Find items
          if (vl.params.renderExternal) {
            renderExternalItems.push(items[i]);
          } else if (vl.domCache[index]) {
            itemEl = vl.domCache[index];
            itemEl.f7VirtualListIndex = index;
          } else {
            if (vl.renderItem) {
              vl.tempDomElement.innerHTML = vl.renderItem(items[i], index).trim();
            } else {
              vl.tempDomElement.innerHTML = items[i].toString().trim();
            }
            itemEl = vl.tempDomElement.childNodes[0];
            if (vl.params.cache) vl.domCache[index] = itemEl;
            itemEl.f7VirtualListIndex = index;
          }

          // Set item top position
          if (i === fromIndex) {
            if (vl.dynamicHeight || vl.autoHeight) {
              topPosition = heightBeforeFirstItem;
            } else {
              topPosition = i * vl.params.height / vl.params.cols;
            }
          }
          if (!vl.params.renderExternal) {
            itemEl.style.top = `${topPosition}px`;

            // Before item insert
            vl.emit('local::itemBeforeInsert vlItemBeforeInsert', vl, itemEl, items[i]);

            // Append item to fragment
            vl.fragment.appendChild(itemEl);
          }
        }

        // Update list height with not updatable scroll
        if (!vl.updatableScroll) {
          if (vl.dynamicHeight || vl.autoHeight) {
            vl.itemsWrapEl.style.height = `${heightBeforeLastItem}px`;
          } else {
            vl.itemsWrapEl.style.height = `${i * vl.params.height / vl.params.cols}px`;
          }
        }

        // Update list html
        if (vl.params.renderExternal) {
          if (items && items.length === 0) {
            vl.reachEnd = true;
          }
        } else {
          vl.emit('local::beforeClear vlBeforeClear', vl, vl.fragment);
          vl.itemsWrapEl.innerHTML = '';
          vl.emit('local::itemsBeforeInsert vlItemsBeforeInsert', vl, vl.fragment);
          if (items && items.length === 0) {
            vl.reachEnd = true;
            if (vl.params.emptyTemplate) vl.itemsWrapEl.innerHTML = vl.params.emptyTemplate;
          } else {
            vl.itemsWrapEl.appendChild(vl.fragment);
          }
          vl.emit('local::itemsAfterInsert vlItemsAfterInsert', vl, vl.fragment);
        }
        if (typeof forceScrollTop !== 'undefined' && force) {
          vl.$scrollableParentEl.scrollTop(forceScrollTop, 0);
        }
        if (vl.params.renderExternal) {
          vl.params.renderExternal(vl, {
            fromIndex,
            toIndex,
            listHeight: vl.listHeight,
            topPosition,
            items: renderExternalItems
          });
        }
        if (vl.autoHeight) {
          requestAnimationFrame(() => {
            vl.setListSize(true);
          });
        }
      }

      // Filter
      filterItems(indexes, resetScrollTop) {
        if (resetScrollTop === void 0) {
          resetScrollTop = true;
        }
        const vl = this;
        vl.filteredItems = [];
        for (let i = 0; i < indexes.length; i += 1) {
          vl.filteredItems.push(vl.items[indexes[i]]);
        }
        if (resetScrollTop) {
          vl.$scrollableParentEl[0].scrollTop = 0;
        }
        vl.update();
      }
      resetFilter() {
        const vl = this;
        if (vl.params.showFilteredItemsOnly) {
          vl.filteredItems = [];
        } else {
          vl.filteredItems = null;
          delete vl.filteredItems;
        }
        vl.update();
      }
      scrollToItem(index) {
        const vl = this;
        if (index > vl.items.length) return false;
        let itemTop = 0;
        if (vl.dynamicHeight || vl.autoHeight) {
          for (let i = 0; i < index; i += 1) {
            itemTop += vl.heights[i];
          }
        } else {
          itemTop = index * vl.params.height;
        }
        const listTop = vl.$el[0].offsetTop;
        vl.render(true, listTop + itemTop - parseInt(vl.$scrollableParentEl.css('padding-top'), 10));
        return true;
      }
      handleScroll() {
        const vl = this;
        vl.render();
      }

      // Handle resize event
      isVisible() {
        const vl = this;
        return !!(vl.el.offsetWidth || vl.el.offsetHeight || vl.el.getClientRects().length);
      }
      handleResize() {
        const vl = this;
        if (vl.isVisible()) {
          vl.heightsCalculated = [];
          vl.setListSize();
          vl.render(true);
        }
      }

      // Append
      appendItems(items) {
        const vl = this;
        for (let i = 0; i < items.length; i += 1) {
          vl.items.push(items[i]);
        }
        vl.update();
      }
      appendItem(item) {
        const vl = this;
        vl.appendItems([item]);
      }

      // Replace
      replaceAllItems(items) {
        const vl = this;
        vl.items = items;
        delete vl.filteredItems;
        vl.domCache = {};
        vl.update();
      }
      replaceItem(index, item) {
        const vl = this;
        vl.items[index] = item;
        if (vl.params.cache) delete vl.domCache[index];
        vl.update();
      }

      // Prepend
      prependItems(items) {
        const vl = this;
        for (let i = items.length - 1; i >= 0; i -= 1) {
          vl.items.unshift(items[i]);
        }
        if (vl.params.cache) {
          const newCache = {};
          Object.keys(vl.domCache).forEach(cached => {
            newCache[parseInt(cached, 10) + items.length] = vl.domCache[cached];
          });
          vl.domCache = newCache;
        }
        vl.update();
      }
      prependItem(item) {
        const vl = this;
        vl.prependItems([item]);
      }

      // Move
      moveItem(from, to) {
        const vl = this;
        const fromIndex = from;
        let toIndex = to;
        if (fromIndex === toIndex) return;
        // remove item from array
        const item = vl.items.splice(fromIndex, 1)[0];
        if (toIndex >= vl.items.length) {
          // Add item to the end
          vl.items.push(item);
          toIndex = vl.items.length - 1;
        } else {
          // Add item to new index
          vl.items.splice(toIndex, 0, item);
        }
        // Update cache
        if (vl.params.cache) {
          const newCache = {};
          Object.keys(vl.domCache).forEach(cached => {
            const cachedIndex = parseInt(cached, 10);
            const leftIndex = fromIndex < toIndex ? fromIndex : toIndex;
            const rightIndex = fromIndex < toIndex ? toIndex : fromIndex;
            const indexShift = fromIndex < toIndex ? -1 : 1;
            if (cachedIndex < leftIndex || cachedIndex > rightIndex) newCache[cachedIndex] = vl.domCache[cachedIndex];
            if (cachedIndex === leftIndex) newCache[rightIndex] = vl.domCache[cachedIndex];
            if (cachedIndex > leftIndex && cachedIndex <= rightIndex) newCache[cachedIndex + indexShift] = vl.domCache[cachedIndex];
          });
          vl.domCache = newCache;
        }
        vl.update();
      }

      // Insert before
      insertItemBefore(index, item) {
        const vl = this;
        if (index === 0) {
          vl.prependItem(item);
          return;
        }
        if (index >= vl.items.length) {
          vl.appendItem(item);
          return;
        }
        vl.items.splice(index, 0, item);
        // Update cache
        if (vl.params.cache) {
          const newCache = {};
          Object.keys(vl.domCache).forEach(cached => {
            const cachedIndex = parseInt(cached, 10);
            if (cachedIndex >= index) {
              newCache[cachedIndex + 1] = vl.domCache[cachedIndex];
            }
          });
          vl.domCache = newCache;
        }
        vl.update();
      }

      // Delete
      deleteItems(indexes) {
        const vl = this;
        let prevIndex;
        let indexShift = 0;
        for (let i = 0; i < indexes.length; i += 1) {
          let index = indexes[i];
          if (typeof prevIndex !== 'undefined') {
            if (index > prevIndex) {
              indexShift = -i;
            }
          }
          index += indexShift;
          prevIndex = indexes[i];
          // Delete item
          const deletedItem = vl.items.splice(index, 1)[0];

          // Delete from filtered
          if (vl.filteredItems && vl.filteredItems.indexOf(deletedItem) >= 0) {
            vl.filteredItems.splice(vl.filteredItems.indexOf(deletedItem), 1);
          }
          // Update cache
          if (vl.params.cache) {
            const newCache = {};
            Object.keys(vl.domCache).forEach(cached => {
              const cachedIndex = parseInt(cached, 10);
              if (cachedIndex === index) {
                delete vl.domCache[index];
              } else if (parseInt(cached, 10) > index) {
                newCache[cachedIndex - 1] = vl.domCache[cached];
              } else {
                newCache[cachedIndex] = vl.domCache[cached];
              }
            });
            vl.domCache = newCache;
          }
        }
        vl.update();
      }
      deleteAllItems() {
        const vl = this;
        vl.items = [];
        delete vl.filteredItems;
        if (vl.params.cache) vl.domCache = {};
        vl.update();
      }
      deleteItem(index) {
        const vl = this;
        vl.deleteItems([index]);
      }

      // Clear cache
      clearCache() {
        const vl = this;
        vl.domCache = {};
      }

      // Update Virtual List
      update(deleteCache) {
        const vl = this;
        if (deleteCache && vl.params.cache) {
          vl.domCache = {};
        }
        vl.heightsCalculated = [];
        vl.setListSize();
        vl.render(true);
      }
      init() {
        const vl = this;
        vl.attachEvents();
        vl.setListSize();
        vl.render();
      }
      destroy() {
        let vl = this;
        vl.detachEvents();
        vl.$el[0].f7VirtualList = null;
        delete vl.$el[0].f7VirtualList;
        deleteProps$1(vl);
        vl = null;
      }
    };

    var VirtualList = {
      name: 'virtualList',
      static: {
        VirtualList: VirtualList$1
      },
      create() {
        const app = this;
        app.virtualList = ConstructorMethods({
          defaultSelector: '.virtual-list',
          constructor: VirtualList$1,
          app,
          domProp: 'f7VirtualList'
        });
      }
    };

    let ListIndex$1 = class ListIndex extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const index = this;
        const defaults = {
          el: null,
          // where to render indexes
          listEl: null,
          // list el to generate indexes
          indexes: 'auto',
          // or array of indexes
          iosItemHeight: 14,
          mdItemHeight: 14,
          scrollList: true,
          label: false,
          // eslint-disable-next-line
          renderItem(itemContent, itemIndex) {
            return `
          <li>${itemContent}</li>
        `.trim();
          },
          renderSkipPlaceholder() {
            return '<li class="list-index-skip-placeholder"></li>';
          },
          on: {}
        };

        // Extend defaults with modules params
        index.useModulesParams(defaults);
        index.params = extend$3(defaults, params);
        let $el;
        let $listEl;
        let $pageContentEl;
        let $ul;
        if (index.params.el) {
          $el = $(index.params.el);
        } else {
          return index;
        }
        if ($el[0].f7ListIndex) {
          return $el[0].f7ListIndex;
        }
        $ul = $el.find('ul');
        if ($ul.length === 0) {
          $ul = $('<ul></ul>');
          $el.append($ul);
        }
        if (index.params.listEl) {
          $listEl = $(index.params.listEl);
        }
        if (index.params.indexes === 'auto' && !$listEl) {
          return index;
        }
        if ($listEl) {
          $pageContentEl = $listEl.parents('.page-content').eq(0);
        } else {
          $pageContentEl = $el.siblings('.page-content').eq(0);
          if ($pageContentEl.length === 0) {
            $pageContentEl = $el.parents('.page').eq(0).find('.page-content').eq(0);
          }
        }
        $el[0].f7ListIndex = index;
        extend$3(index, {
          app,
          $el,
          el: $el && $el[0],
          $ul,
          ul: $ul && $ul[0],
          $listEl,
          listEl: $listEl && $listEl[0],
          $pageContentEl,
          pageContentEl: $pageContentEl && $pageContentEl[0],
          indexes: params.indexes,
          height: 0,
          skipRate: 0
        });

        // Install Modules
        index.useModules();

        // Attach events
        function handleResize() {
          const height = {
            index
          };
          index.calcSize();
          if (height !== index.height) {
            index.render();
          }
        }
        function handleClick(e) {
          const $clickedLi = $(e.target).closest('li');
          if (!$clickedLi.length) return;
          let itemIndex = $clickedLi.index();
          if (index.skipRate > 0) {
            const percentage = itemIndex / ($clickedLi.siblings('li').length - 1);
            itemIndex = Math.round((index.indexes.length - 1) * percentage);
          }
          const itemContent = index.indexes[itemIndex];
          index.$el.trigger('listindex:click', {
            content: itemContent,
            index: itemIndex
          });
          index.emit('local::click listIndexClick', index, itemContent, itemIndex);
          index.$el.trigger('listindex:select', {
            content: itemContent,
            index: itemIndex
          });
          index.emit('local::select listIndexSelect', index, itemContent, itemIndex);
          if (index.$listEl && index.params.scrollList) {
            index.scrollListToIndex(itemContent, itemIndex);
          }
        }
        const touchesStart = {};
        let isTouched;
        let isMoved;
        let topPoint;
        let bottomPoint;
        let $labelEl;
        let previousIndex = null;
        function handleTouchStart(e) {
          const $children = $ul.children();
          if (!$children.length) return;
          topPoint = $children[0].getBoundingClientRect().top;
          bottomPoint = $children[$children.length - 1].getBoundingClientRect().top + $children[0].offsetHeight;
          touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
          touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          isTouched = true;
          isMoved = false;
          previousIndex = null;
        }
        function handleTouchMove(e) {
          if (!isTouched) return;
          if (!isMoved && index.params.label) {
            $labelEl = $('<span class="list-index-label"></span>');
            $el.append($labelEl);
          }
          isMoved = true;
          const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
          e.preventDefault();
          let percentage = (pageY - topPoint) / (bottomPoint - topPoint);
          percentage = Math.min(Math.max(percentage, 0), 1);
          const itemIndex = Math.round((index.indexes.length - 1) * percentage);
          const itemContent = index.indexes[itemIndex];
          const ulHeight = bottomPoint - topPoint;
          const bubbleBottom = (index.height - ulHeight) / 2 + (1 - percentage) * ulHeight;
          if (itemIndex !== previousIndex) {
            if (index.params.label) {
              $labelEl.html(itemContent).transform(`translateY(-${bubbleBottom}px)`);
            }
            if (index.$listEl && index.params.scrollList) {
              index.scrollListToIndex(itemContent, itemIndex);
            }
          }
          previousIndex = itemIndex;
          index.$el.trigger('listindex:select');
          index.emit('local::select listIndexSelect', index, itemContent, itemIndex);
        }
        function handleTouchEnd() {
          if (!isTouched) return;
          isTouched = false;
          isMoved = false;
          if (index.params.label) {
            if ($labelEl) $labelEl.remove();
            $labelEl = undefined;
          }
        }
        const passiveListener = getSupport$1().passiveListener ? {
          passive: true
        } : false;
        index.attachEvents = function attachEvents() {
          $el.parents('.tab').on('tab:show', handleResize);
          $el.parents('.page').on('page:reinit', handleResize);
          $el.parents('.panel').on('panel:open', handleResize);
          $el.parents('.sheet-modal, .actions-modal, .popup, .popover, .login-screen, .dialog, .toast').on('modal:open', handleResize);
          app.on('resize', handleResize);
          $el.on('click', handleClick);
          $el.on(app.touchEvents.start, handleTouchStart, passiveListener);
          app.on('touchmove:active', handleTouchMove);
          app.on('touchend:passive', handleTouchEnd);
        };
        index.detachEvents = function attachEvents() {
          $el.parents('.tab').off('tab:show', handleResize);
          $el.parents('.page').off('page:reinit', handleResize);
          $el.parents('.panel').off('panel:open', handleResize);
          $el.parents('.sheet-modal, .actions-modal, .popup, .popover, .login-screen, .dialog, .toast').off('modal:open', handleResize);
          app.off('resize', handleResize);
          $el.off('click', handleClick);
          $el.off(app.touchEvents.start, handleTouchStart, passiveListener);
          app.off('touchmove:active', handleTouchMove);
          app.off('touchend:passive', handleTouchEnd);
        };
        // Init
        index.init();
        return index;
      }
      // eslint-disable-next-line
      scrollListToIndex(itemContent, itemIndex) {
        const index = this;
        const {
          $listEl,
          $pageContentEl,
          app
        } = index;
        if (!$listEl || !$pageContentEl || $pageContentEl.length === 0) return index;
        let $scrollToEl;
        $listEl.find('.list-group-title').each(el => {
          if ($scrollToEl) return;
          const $el = $(el);
          if ($el.text() === itemContent) {
            $scrollToEl = $el;
          }
        });
        if (!$scrollToEl || $scrollToEl.length === 0) return index;
        const parentTop = $scrollToEl.parent().offset().top;
        let paddingTop = parseInt($pageContentEl.css('padding-top'), 10);
        const scrollTop = $pageContentEl[0].scrollTop;
        const scrollToElTop = $scrollToEl.offset().top;
        if ($pageContentEl.parents('.page-with-navbar-large').length) {
          const navbarInnerEl = app.navbar.getElByPage($pageContentEl.parents('.page-with-navbar-large').eq(0));
          const $titleLargeEl = $(navbarInnerEl).find('.title-large');
          if ($titleLargeEl.length) {
            paddingTop -= $titleLargeEl[0].offsetHeight || 0;
          }
        }
        if (parentTop <= paddingTop) {
          $pageContentEl.scrollTop(parentTop + scrollTop - paddingTop);
        } else {
          $pageContentEl.scrollTop(scrollToElTop + scrollTop - paddingTop);
        }
        return index;
      }
      renderSkipPlaceholder() {
        const index = this;
        return index.params.renderSkipPlaceholder.call(index);
      }
      renderItem(itemContent, itemIndex) {
        const index = this;
        return index.params.renderItem.call(index, itemContent, itemIndex);
      }
      render() {
        const index = this;
        const {
          $ul,
          indexes,
          skipRate
        } = index;
        let wasSkipped;
        const html = indexes.map((itemContent, itemIndex) => {
          if (itemIndex % skipRate !== 0 && skipRate > 0) {
            wasSkipped = true;
            return '';
          }
          let itemHtml = index.renderItem(itemContent, itemIndex);
          if (wasSkipped) {
            itemHtml = index.renderSkipPlaceholder() + itemHtml;
          }
          wasSkipped = false;
          return itemHtml;
        }).join('');
        $ul.html(html);
        return index;
      }
      calcSize() {
        const index = this;
        const {
          app,
          params,
          el,
          indexes
        } = index;
        const height = el.offsetHeight;
        const itemHeight = params[`${app.theme}ItemHeight`];
        const maxItems = Math.floor(height / itemHeight);
        const items = indexes.length;
        let skipRate = 0;
        if (items > maxItems) {
          skipRate = Math.ceil((items * 2 - 1) / maxItems);
        }
        index.height = height;
        index.skipRate = skipRate;
        return index;
      }
      calcIndexes() {
        const index = this;
        if (index.params.indexes === 'auto') {
          index.indexes = [];
          index.$listEl.find('.list-group-title').each(el => {
            const elContent = $(el).text();
            if (index.indexes.indexOf(elContent) < 0) {
              index.indexes.push(elContent);
            }
          });
        } else {
          index.indexes = index.params.indexes;
        }
        return index;
      }
      update() {
        const index = this;
        index.calcIndexes();
        index.calcSize();
        index.render();
        return index;
      }
      init() {
        const index = this;
        index.calcIndexes();
        index.calcSize();
        index.render();
        index.attachEvents();
      }
      destroy() {
        let index = this;
        index.$el.trigger('listindex:beforedestroy', index);
        index.emit('local::beforeDestroy listIndexBeforeDestroy');
        index.detachEvents();
        if (index.$el[0]) {
          index.$el[0].f7ListIndex = null;
          delete index.$el[0].f7ListIndex;
        }
        deleteProps$1(index);
        index = null;
      }
    };

    var ListIndex = {
      name: 'listIndex',
      static: {
        ListIndex: ListIndex$1
      },
      create() {
        const app = this;
        app.listIndex = ConstructorMethods({
          defaultSelector: '.list-index',
          constructor: ListIndex$1,
          app,
          domProp: 'f7ListIndex'
        });
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.list-index-init').each(listIndexEl => {
            const params = extend$3($(listIndexEl).dataset(), {
              el: listIndexEl
            });
            app.listIndex.create(params);
          });
        },
        tabBeforeRemove(tabEl) {
          $(tabEl).find('.list-index-init').each(listIndexEl => {
            if (listIndexEl.f7ListIndex) listIndexEl.f7ListIndex.destroy();
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.list-index-init').each(listIndexEl => {
            const params = extend$3($(listIndexEl).dataset(), {
              el: listIndexEl
            });
            app.listIndex.create(params);
          });
        },
        pageBeforeRemove(page) {
          page.$el.find('.list-index-init').each(listIndexEl => {
            if (listIndexEl.f7ListIndex) listIndexEl.f7ListIndex.destroy();
          });
        }
      },
      vnode: {
        'list-index-init': {
          insert(vnode) {
            const app = this;
            const listIndexEl = vnode.elm;
            const params = extend$3($(listIndexEl).dataset(), {
              el: listIndexEl
            });
            app.listIndex.create(params);
          },
          destroy(vnode) {
            const listIndexEl = vnode.elm;
            if (listIndexEl.f7ListIndex) listIndexEl.f7ListIndex.destroy();
          }
        }
      }
    };

    var Timeline = {
      name: 'timeline'
    };

    const Tab = {
      show() {
        const app = this;
        let tabEl;
        let tabLinkEl;
        let animate;
        let tabRoute;
        let animatedInit;
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }
        if (args.length === 1 && args[0] && args[0].constructor === Object) {
          tabEl = args[0].tabEl;
          tabLinkEl = args[0].tabLinkEl;
          animate = args[0].animate;
          tabRoute = args[0].tabRoute;
          animatedInit = args[0].animatedInit;
        } else {
          [tabEl, tabLinkEl, animate, tabRoute] = args;
          if (typeof args[1] === 'boolean') {
            [tabEl, animate, tabLinkEl, tabRoute] = args;
            if (args.length > 2 && tabLinkEl.constructor === Object) {
              [tabEl, animate, tabRoute, tabLinkEl] = args;
            }
          }
        }
        if (typeof animate === 'undefined') animate = true;
        const $newTabEl = $(tabEl);
        if (tabRoute && $newTabEl[0]) {
          $newTabEl[0].f7TabRoute = tabRoute;
        }
        if (!animatedInit && ($newTabEl.length === 0 || $newTabEl.hasClass('tab-active'))) {
          return {
            $newTabEl,
            newTabEl: $newTabEl[0]
          };
        }
        let $tabLinkEl;
        if (tabLinkEl) $tabLinkEl = $(tabLinkEl);
        const $tabsEl = $newTabEl.parent('.tabs');
        if ($tabsEl.length === 0) {
          return {
            $newTabEl,
            newTabEl: $newTabEl[0]
          };
        }

        // Release swipeouts in hidden tabs
        if (app.swipeout) app.swipeout.allowOpen = true;

        // Animated tabs
        const tabsChangedCallbacks = [];
        function onTabsChanged(callback) {
          tabsChangedCallbacks.push(callback);
        }
        function tabsChanged() {
          tabsChangedCallbacks.forEach(callback => {
            callback();
          });
        }
        let animated = false;
        if ($tabsEl.parent().hasClass('tabs-animated-wrap')) {
          $tabsEl.parent()[animate ? 'removeClass' : 'addClass']('not-animated');
          const transitionDuration = parseFloat($tabsEl.css('transition-duration').replace(',', '.'));
          if (animate && transitionDuration) {
            $tabsEl.transitionEnd(tabsChanged);
            animated = true;
          }
          const tabsTranslate = (app.rtl ? $newTabEl.index() : -$newTabEl.index()) * 100;
          $tabsEl.transform(`translate3d(${tabsTranslate}%,0,0)`);
        }

        // Swipeable tabs
        let swiper;
        if ($tabsEl[0].nodeName.toLowerCase() === 'swiper-container' && app.swiper) {
          swiper = $tabsEl[0].swiper;
          const newTabIndex = swiper.slides.indexOf($newTabEl[0]);
          if (swiper && swiper.activeIndex !== newTabIndex) {
            animated = true;
            swiper.once('slideChangeTransitionEnd', () => {
              tabsChanged();
            }).slideTo(newTabIndex, animate ? undefined : 0);
          } else if (swiper && swiper.animating) {
            animated = true;
            swiper.once('slideChangeTransitionEnd', () => {
              tabsChanged();
            });
          }
        }

        // Remove active class from old tabs
        const $oldTabEl = $tabsEl.children('.tab-active');
        $oldTabEl.removeClass('tab-active');
        if (!animatedInit && (!swiper || swiper && !swiper.animating || swiper && tabRoute)) {
          if ($oldTabEl.hasClass('view') && $oldTabEl.children('.page').length) {
            $oldTabEl.children('.page').each(pageEl => {
              $(pageEl).trigger('page:tabhide');
              app.emit('pageTabHide', pageEl);
            });
          }
          $oldTabEl.trigger('tab:hide');
          app.emit('tabHide', $oldTabEl[0]);
        }

        // Trigger 'show' event on new tab
        $newTabEl.addClass('tab-active');
        if (!animatedInit && (!swiper || swiper && !swiper.animating || swiper && tabRoute)) {
          if ($newTabEl.hasClass('view') && $newTabEl.children('.page').length) {
            $newTabEl.children('.page').each(pageEl => {
              $(pageEl).trigger('page:tabshow');
              app.emit('pageTabShow', pageEl);
            });
          }
          $newTabEl.trigger('tab:show');
          app.emit('tabShow', $newTabEl[0]);
        }

        // Find related link for new tab
        if (!$tabLinkEl) {
          // Search by id
          if (typeof tabEl === 'string') $tabLinkEl = $(`.tab-link[href="${tabEl}"]`);else $tabLinkEl = $(`.tab-link[href="#${$newTabEl.attr('id')}"]`);
          // Search by data-tab
          if (!$tabLinkEl || $tabLinkEl && $tabLinkEl.length === 0) {
            $('[data-tab]').each(el => {
              if ($newTabEl.is($(el).attr('data-tab'))) $tabLinkEl = $(el);
            });
          }
          if (tabRoute && (!$tabLinkEl || $tabLinkEl && $tabLinkEl.length === 0)) {
            $tabLinkEl = $(`[data-route-tab-id="${tabRoute.route.tab.id}"]`);
            if ($tabLinkEl.length === 0) {
              $tabLinkEl = $(`.tab-link[href="${tabRoute.url}"]`);
            }
          }
          if ($tabLinkEl.length > 1 && $newTabEl.parents('.page').length) {
            // eslint-disable-next-line
            $tabLinkEl = $tabLinkEl.filter(tabLinkElement => {
              return $(tabLinkElement).parents('.page')[0] === $newTabEl.parents('.page')[0];
            });
            if (app.theme === 'ios' && $tabLinkEl.length === 0 && tabRoute) {
              const $pageEl = $newTabEl.parents('.page');
              const $navbarEl = $(app.navbar.getElByPage($pageEl));
              $tabLinkEl = $navbarEl.find(`[data-route-tab-id="${tabRoute.route.tab.id}"]`);
              if ($tabLinkEl.length === 0) {
                $tabLinkEl = $navbarEl.find(`.tab-link[href="${tabRoute.url}"]`);
              }
            }
          }
        }
        if ($tabLinkEl.length > 0) {
          // Find related link for old tab
          let $oldTabLinkEl;
          if ($oldTabEl && $oldTabEl.length > 0) {
            // Search by id
            const oldTabId = $oldTabEl.attr('id');
            if (oldTabId) {
              $oldTabLinkEl = $(`.tab-link[href="#${oldTabId}"]`);
              // Search by data-route-tab-id
              if (!$oldTabLinkEl || $oldTabLinkEl && $oldTabLinkEl.length === 0) {
                $oldTabLinkEl = $(`.tab-link[data-route-tab-id="${oldTabId}"]`);
              }
            }
            // Search by data-tab
            if (!$oldTabLinkEl || $oldTabLinkEl && $oldTabLinkEl.length === 0) {
              $('[data-tab]').each(tabLinkElement => {
                if ($oldTabEl.is($(tabLinkElement).attr('data-tab'))) $oldTabLinkEl = $(tabLinkElement);
              });
            }
            if (!$oldTabLinkEl || $oldTabLinkEl && $oldTabLinkEl.length === 0) {
              $oldTabLinkEl = $tabLinkEl.siblings('.tab-link-active');
            }
          } else if (tabRoute) {
            $oldTabLinkEl = $tabLinkEl.siblings('.tab-link-active');
          }
          if ($oldTabLinkEl && $oldTabLinkEl.length > 1 && $oldTabEl && $oldTabEl.parents('.page').length) {
            // eslint-disable-next-line
            $oldTabLinkEl = $oldTabLinkEl.filter(tabLinkElement => {
              return $(tabLinkElement).parents('.page')[0] === $oldTabEl.parents('.page')[0];
            });
          }
          if ($oldTabLinkEl && $oldTabLinkEl.length > 0) $oldTabLinkEl.removeClass('tab-link-active');

          // Update links' classes
          if ($tabLinkEl && $tabLinkEl.length > 0) {
            $tabLinkEl.addClass('tab-link-active');
            // Material Highlight
            const $tabbarEl = $tabLinkEl.parents('.tabbar, .tabbar-icons');
            const hasHighlight = app.toolbar && $tabbarEl.length > 0 && ($tabbarEl.hasClass('tabbar-highlight') || app.theme !== 'ios');
            if (hasHighlight) {
              app.toolbar.setHighlight($tabbarEl);
            }
          }
        }
        return {
          $newTabEl,
          newTabEl: $newTabEl[0],
          $oldTabEl,
          oldTabEl: $oldTabEl[0],
          onTabsChanged,
          animated
        };
      }
    };
    var Tabs = {
      name: 'tabs',
      create() {
        const app = this;
        extend$3(app, {
          tab: {
            show: Tab.show.bind(app)
          }
        });
      },
      on: {
        'pageInit tabMounted': function onInit(pageOrTabEl) {
          const $el = $(pageOrTabEl.el || pageOrTabEl);
          const animatedTabEl = $el.find('.tabs-animated-wrap > .tabs > .tab-active')[0];
          if (!animatedTabEl) return;
          const app = this;
          app.tab.show({
            tabEl: animatedTabEl,
            animatedInit: true,
            animate: false
          });
        }
      },
      clicks: {
        '.tab-link': function tabLinkClick($clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          if ($clickedEl.attr('href') && $clickedEl.attr('href').indexOf('#') === 0 || $clickedEl.attr('data-tab')) {
            const app = this;
            app.tab.show({
              tabEl: data.tab || $clickedEl.attr('href'),
              tabLinkEl: $clickedEl,
              animate: data.animate
            });
          }
        }
      }
    };

    function swipePanel(panel) {
      const app = panel.app;
      if (panel.swipeInitialized) {
        return;
      }
      extend$3(panel, {
        swipeable: true,
        swipeInitialized: true
      });
      const params = panel.params;
      const {
        $el,
        $backdropEl,
        side,
        effect
      } = panel;
      let otherPanel;
      let isTouched;
      let isGestureStarted;
      let isMoved;
      let isScrolling;
      let isInterrupted;
      const touchesStart = {};
      let touchStartTime;
      let touchesDiff;
      let translate;
      let backdropOpacity;
      let panelWidth;
      let direction;
      let $viewEl;
      let touchMoves = 0;
      function handleTouchStart(e) {
        if (!panel.swipeable || isGestureStarted) return;
        if (!app.panel.allowOpen || !params.swipe && !params.swipeOnlyClose || isTouched) return;
        if ($('.modal-in:not(.toast):not(.notification), .photo-browser-in').length > 0) return;
        otherPanel = app.panel.get(side === 'left' ? 'right' : 'left') || {};
        const otherPanelOpened = otherPanel.opened && otherPanel.$el && !otherPanel.$el.hasClass('panel-in-breakpoint');
        if (!panel.opened && otherPanelOpened) {
          return;
        }
        if (!params.swipeOnlyClose) {
          if (otherPanelOpened) return;
        }
        if (e.target && e.target.nodeName.toLowerCase() === 'input' && e.target.type === 'range') return;
        if ($(e.target).closest('.range-slider, swiper-container.tabs, .calendar-months, .no-swipe-panel, .card-opened').length > 0) return;
        touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
        touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
        if (params.swipeOnlyClose && !panel.opened) {
          return;
        }
        if (params.swipeActiveArea && !panel.opened) {
          if (side === 'left') {
            if (touchesStart.x > params.swipeActiveArea) return;
          }
          if (side === 'right') {
            if (touchesStart.x < app.width - params.swipeActiveArea) return;
          }
        }
        touchMoves = 0;
        $viewEl = $(panel.getViewEl());
        isMoved = false;
        isTouched = true;
        isScrolling = undefined;
        isInterrupted = false;
        touchStartTime = now$1();
        direction = undefined;
      }
      function handleTouchMove(e) {
        if (!isTouched || isGestureStarted || isInterrupted) return;
        touchMoves += 1;
        if (touchMoves < 2) return;
        if (e.f7PreventSwipePanel || app.preventSwipePanelBySwipeBack || app.preventSwipePanel) {
          isTouched = false;
          return;
        }
        const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
        const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
        if (typeof isScrolling === 'undefined') {
          isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x));
        }
        if (isScrolling) {
          isTouched = false;
          return;
        }
        if (!direction) {
          if (pageX > touchesStart.x) {
            direction = 'to-right';
          } else {
            direction = 'to-left';
          }
          if (params.swipeActiveArea > 0 && !panel.opened) {
            if (side === 'left' && touchesStart.x > params.swipeActiveArea) {
              isTouched = false;
              return;
            }
            if (side === 'right' && touchesStart.x < app.width - params.swipeActiveArea) {
              isTouched = false;
              return;
            }
          }
          if ($el.hasClass('panel-in-breakpoint')) {
            isTouched = false;
            return;
          }
          if (side === 'left' && direction === 'to-left' && !$el.hasClass('panel-in') || side === 'right' && direction === 'to-right' && !$el.hasClass('panel-in')) {
            isTouched = false;
            return;
          }
        }
        let threshold = panel.opened ? 0 : -params.swipeThreshold;
        if (side === 'right') threshold = -threshold;
        if (!isMoved) {
          if (!panel.opened) {
            panel.insertToRoot();
            $el.addClass('panel-in-swipe');
            if ($backdropEl) $backdropEl.css('visibility', 'visible');
            $el.trigger('panel:swipeopen');
            panel.emit('local::swipeOpen panelSwipeOpen', panel);
          }
          panelWidth = $el[0].offsetWidth;
          if (effect === 'reveal' && $el.hasClass('panel-in-collapsed')) {
            panelWidth -= parseFloat($viewEl.css(`margin-${side}`));
          }
          $el.transition(0);
        }
        isMoved = true;
        if (e.cancelable) {
          e.preventDefault();
        }
        touchesDiff = pageX - touchesStart.x + threshold;
        const startTranslate = effect === 'floating' ? 8 : 0;
        if (side === 'right') {
          if (effect === 'cover' || effect === 'push' || effect === 'floating') {
            translate = touchesDiff + (panel.opened ? startTranslate : panelWidth);
            if (translate < 0 - startTranslate) translate = -startTranslate;
            if (translate > panelWidth) {
              translate = panelWidth;
            }
          } else {
            translate = touchesDiff - (panel.opened ? panelWidth : 0);
            if (translate > 0) translate = 0;
            if (translate < -panelWidth) {
              translate = -panelWidth;
            }
          }
        } else {
          translate = touchesDiff + (panel.opened ? panelWidth : startTranslate);
          if (translate < 0) translate = 0;
          if (translate > panelWidth + startTranslate) {
            translate = panelWidth + startTranslate;
          }
        }
        const noFollowProgress = Math.abs(translate / panelWidth);
        if (effect === 'reveal') {
          if (!params.swipeNoFollow) {
            $viewEl.transform(`translate3d(${translate}px,0,0)`).transition(0);
            if ($backdropEl) $backdropEl.transform(`translate3d(${translate}px,0,0)`).transition(0);
          }
          $el.trigger('panel:swipe', Math.abs(translate / panelWidth));
          panel.emit('local::swipe panelSwipe', panel, Math.abs(translate / panelWidth));
        } else {
          if (side === 'left') translate -= panelWidth;
          if (!params.swipeNoFollow) {
            backdropOpacity = 1 - Math.abs(translate / panelWidth);
            if ($backdropEl) {
              $backdropEl.transition(0);
              $backdropEl.css({
                opacity: backdropOpacity
              });
            }
            $el.transform(`translate3d(${translate}px,0,0)`).transition(0);
            if (effect === 'push') {
              const viewTranslate = side === 'left' ? translate + panelWidth : translate - panelWidth;
              $viewEl.transform(`translate3d(${viewTranslate}px,0,0)`).transition(0);
              if ($backdropEl) {
                $backdropEl.transform(`translate3d(${viewTranslate}px,0,0)`).transition(0);
              }
            }
          }
          $el.trigger('panel:swipe', Math.abs(translate / panelWidth));
          panel.emit('local::swipe panelSwipe', panel, Math.abs(translate / panelWidth));
        }
        if (params.swipeNoFollow) {
          const stateChanged = panel.opened && noFollowProgress === 0 || !panel.opened && noFollowProgress === 1;
          if (stateChanged) {
            isInterrupted = true;
            // eslint-disable-next-line
            handleTouchEnd(e);
          }
        }
      }
      function handleTouchEnd(e) {
        if (!isTouched || !isMoved) {
          isTouched = false;
          isMoved = false;
          return;
        }
        const isGesture = e.type === 'gesturestart' || isGestureStarted;
        isTouched = false;
        isMoved = false;
        const timeDiff = new Date().getTime() - touchStartTime;
        let action;
        const startTranslate = effect === 'floating' ? side === 'left' ? 8 : -8 : 0;
        const edge = (translate === startTranslate || Math.abs(translate) === panelWidth) && !params.swipeNoFollow;
        const threshold = params.swipeThreshold || 0;
        if (isGesture) {
          action = 'reset';
        } else if (!panel.opened) {
          if (Math.abs(touchesDiff) < threshold) {
            action = 'reset';
          } else if (effect === 'cover' || effect === 'push' || effect === 'floating') {
            if (translate === 0 + startTranslate) {
              action = 'swap'; // open
            } else if (timeDiff < 300 && Math.abs(translate) > 0) {
              action = 'swap'; // open
            } else if (timeDiff >= 300 && Math.abs(translate) < panelWidth / 2) {
              action = 'swap'; // open
            } else {
              action = 'reset'; // close
            }
          } else if (translate === 0) {
            action = 'reset';
          } else if (timeDiff < 300 && Math.abs(translate) > 0 || timeDiff >= 300 && Math.abs(translate) >= panelWidth / 2) {
            action = 'swap';
          } else {
            action = 'reset';
          }
        } else if (effect === 'cover' || effect === 'push' || effect === 'floating') {
          if (translate === 0) {
            action = 'reset'; // open
          } else if (timeDiff < 300 && Math.abs(translate) > 0) {
            action = 'swap'; // open
          } else if (timeDiff >= 300 && Math.abs(translate) < panelWidth / 2) {
            action = 'reset'; // open
          } else {
            action = 'swap'; // close
          }
        } else if (translate === -panelWidth) {
          action = 'reset';
        } else if (timeDiff < 300 && Math.abs(translate) >= 0 || timeDiff >= 300 && Math.abs(translate) <= panelWidth / 2) {
          if (side === 'left' && translate === panelWidth) action = 'reset';else action = 'swap';
        } else {
          action = 'reset';
        }
        if (action === 'swap') {
          if (panel.opened) {
            panel.close(!edge);
          } else {
            panel.open(!edge);
          }
        }
        let removePanelInClass = true;
        if (action === 'reset') {
          if (!panel.opened) {
            if (edge) {
              // edge position
              $el.removeClass('panel-in-swipe');
            } else {
              removePanelInClass = false;
              const target = effect === 'reveal' ? $viewEl : $el;
              panel.setStateClasses('before-closing');
              target.transitionEnd(() => {
                if ($el.hasClass('panel-in')) return;
                $el.removeClass('panel-in-swipe');
                panel.setStateClasses('after-closing');
              });
            }
          }
        }
        if (effect === 'reveal' || effect === 'push') {
          nextFrame$1(() => {
            $viewEl.transition('');
            $viewEl.transform('');
          });
        }
        if (removePanelInClass) {
          $el.removeClass('panel-in-swipe');
        }
        $el.transition('').transform('');
        if ($backdropEl) {
          $backdropEl.transform('').transition('').css({
            opacity: '',
            visibility: ''
          });
        }
      }
      function handleGestureStart(e) {
        isGestureStarted = true;
        handleTouchEnd(e);
      }
      function handleGestureEnd() {
        isGestureStarted = false;
      }

      // Add Events
      app.on('touchstart:passive', handleTouchStart);
      app.on('touchmove:active', handleTouchMove);
      app.on('touchend:passive', handleTouchEnd);
      app.on('gesturestart', handleGestureStart);
      app.on('gestureend', handleGestureEnd);
      panel.on('panelDestroy', () => {
        app.off('touchstart:passive', handleTouchStart);
        app.off('touchmove:active', handleTouchMove);
        app.off('touchend:passive', handleTouchEnd);
        app.off('gesturestart', handleGestureStart);
        app.off('gestureend', handleGestureEnd);
      });
    }

    function resizablePanel(panel) {
      const app = panel.app;
      const support = getSupport$1();
      if (panel.resizableInitialized) return;
      extend$3(panel, {
        resizable: true,
        resizableWidth: null,
        resizableInitialized: true
      });
      const $htmlEl = $('html');
      const {
        $el,
        $backdropEl,
        side,
        effect
      } = panel;
      if (!$el) return;
      let isTouched;
      let isMoved;
      const touchesStart = {};
      let touchesDiff;
      let panelWidth;
      let $viewEl;
      let panelMinWidth;
      let panelMaxWidth;
      let visibleByBreakpoint;
      const isPushingPanel = effect !== 'cover' && effect !== 'floating';
      function transformCSSWidth(v) {
        if (!v) return null;
        if (v.indexOf('%') >= 0 || v.indexOf('vw') >= 0) {
          return parseInt(v, 10) / 100 * app.width;
        }
        const newV = parseInt(v, 10);
        if (Number.isNaN(newV)) return null;
        return newV;
      }
      function isResizable() {
        return panel.resizable && $el.hasClass('panel-resizable');
      }
      function handleTouchStart(e) {
        if (!isResizable()) return;
        touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
        touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
        isMoved = false;
        isTouched = true;
        panelMinWidth = transformCSSWidth($el.css('min-width'));
        panelMaxWidth = transformCSSWidth($el.css('max-width'));
        visibleByBreakpoint = $el.hasClass('panel-in-breakpoint');
      }
      function handleTouchMove(e) {
        if (!isTouched) return;
        const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
        if (!isMoved) {
          panelWidth = $el[0].offsetWidth;
          $el.transition(0);
          $el.addClass('panel-resizing');
          $htmlEl.css('cursor', 'col-resize');
          if (isPushingPanel || visibleByBreakpoint) {
            $viewEl = $(panel.getViewEl());
            if (panel.$containerEl && panel.$containerEl.hasClass('page')) {
              $viewEl.add(panel.$containerEl.children('.page-content, .tabs, .fab'));
            }
          }
          if (isPushingPanel && !visibleByBreakpoint) {
            $backdropEl.transition(0);
            $viewEl.transition(0);
          }
        }
        isMoved = true;
        e.preventDefault();
        touchesDiff = pageX - touchesStart.x;
        let newPanelWidth = side === 'left' ? panelWidth + touchesDiff : panelWidth - touchesDiff;
        if (panelMinWidth && !Number.isNaN(panelMinWidth)) {
          newPanelWidth = Math.max(newPanelWidth, panelMinWidth);
        }
        if (panelMaxWidth && !Number.isNaN(panelMaxWidth)) {
          newPanelWidth = Math.min(newPanelWidth, panelMaxWidth);
        }
        newPanelWidth = Math.min(Math.max(newPanelWidth, 0), app.width);
        panel.resizableWidth = newPanelWidth;
        $el[0].style.width = `${newPanelWidth}px`;
        if (isPushingPanel && !visibleByBreakpoint) {
          if ($viewEl) {
            $viewEl.transform(`translate3d(${side === 'left' ? newPanelWidth : -newPanelWidth}px, 0, 0)`);
          }
          if ($backdropEl) {
            $backdropEl.transform(`translate3d(${side === 'left' ? newPanelWidth : -newPanelWidth}px, 0, 0)`);
          }
        } else if (visibleByBreakpoint && $viewEl) {
          $viewEl.css(`margin-${side}`, `${newPanelWidth}px`);
        }
        $el.trigger('panel:resize', newPanelWidth);
        panel.emit('local::resize panelResize', panel, newPanelWidth);
      }
      function handleTouchEnd() {
        $('html').css('cursor', '');
        if (!isTouched || !isMoved) {
          isTouched = false;
          isMoved = false;
          return;
        }
        isTouched = false;
        isMoved = false;
        $htmlEl[0].style.setProperty(`--f7-panel-${side}-width`, `${panel.resizableWidth}px`);
        $el[0].style.width = '';
        if (isPushingPanel && !visibleByBreakpoint) {
          $viewEl.transform('');
          $backdropEl.transform('');
        }
        $el.removeClass('panel-resizing');
        nextFrame$1(() => {
          $el.transition('');
          if (isPushingPanel) {
            $backdropEl.transition('');
            if ($viewEl) $viewEl.transition('');
          }
        });
      }
      function handleResize() {
        if (!panel.opened || !panel.resizableWidth) return;
        panelMinWidth = transformCSSWidth($el.css('min-width'));
        panelMaxWidth = transformCSSWidth($el.css('max-width'));
        if (panelMinWidth && !Number.isNaN(panelMinWidth) && panel.resizableWidth < panelMinWidth) {
          panel.resizableWidth = Math.max(panel.resizableWidth, panelMinWidth);
        }
        if (panelMaxWidth && !Number.isNaN(panelMaxWidth) && panel.resizableWidth > panelMaxWidth) {
          panel.resizableWidth = Math.min(panel.resizableWidth, panelMaxWidth);
        }
        panel.resizableWidth = Math.min(Math.max(panel.resizableWidth, 0), app.width);
        $htmlEl[0].style.setProperty(`--f7-panel-${side}-width`, `${panel.resizableWidth}px`);
      }
      if (panel.$el.find('.panel-resize-handler').length === 0) {
        panel.$el.append('<div class="panel-resize-handler"></div>');
      }
      panel.$resizeHandlerEl = panel.$el.children('.panel-resize-handler');
      $el.addClass('panel-resizable');

      // Add Events
      const passive = support.passiveListener ? {
        passive: true
      } : false;
      panel.$el.on(app.touchEvents.start, '.panel-resize-handler', handleTouchStart, passive);
      app.on('touchmove:active', handleTouchMove);
      app.on('touchend:passive', handleTouchEnd);
      app.on('resize', handleResize);
      panel.on('beforeOpen', handleResize);
      panel.once('panelDestroy', () => {
        $el.removeClass('panel-resizable');
        panel.$resizeHandlerEl.remove();
        panel.$el.off(app.touchEvents.start, '.panel-resize-handler', handleTouchStart, passive);
        app.off('touchmove:active', handleTouchMove);
        app.off('touchend:passive', handleTouchEnd);
        app.off('resize', handleResize);
        panel.off('beforeOpen', handleResize);
      });
    }

    let Panel$1 = class Panel extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        const extendedParams = extend$3({
          on: {}
        }, app.params.panel, params);
        super(extendedParams, [app]);
        const panel = this;
        panel.params = extendedParams;
        panel.$containerEl = panel.params.containerEl ? $(panel.params.containerEl).eq(0) : app.$el;
        panel.containerEl = panel.$containerEl[0];
        if (!panel.containerEl) {
          panel.$containerEl = app.$el;
          panel.containerEl = app.$el[0];
        }
        let $el;
        if (panel.params.el) {
          $el = $(panel.params.el).eq(0);
        } else if (panel.params.content) {
          $el = $(panel.params.content).filter(node => node.nodeType === 1).eq(0);
        }
        if ($el.length === 0) return panel;
        if ($el[0].f7Panel) return $el[0].f7Panel;
        $el[0].f7Panel = panel;
        let {
          side,
          effect,
          resizable
        } = panel.params;
        if (typeof side === 'undefined') side = $el.hasClass('panel-left') ? 'left' : 'right';
        if (typeof effect === 'undefined')
          // eslint-disable-next-line
          effect = $el.hasClass('panel-cover') ? 'cover' : $el.hasClass('panel-push') ? 'push' : $el.hasClass('panel-floating') ? 'floating' : 'reveal';
        if (typeof resizable === 'undefined') resizable = $el.hasClass('panel-resizable');
        let $backdropEl;
        if (panel.params.backdrop && panel.params.backdropEl) {
          $backdropEl = $(panel.params.backdropEl);
        } else if (panel.params.backdrop) {
          $backdropEl = panel.$containerEl.children('.panel-backdrop');
          if ($backdropEl.length === 0) {
            $backdropEl = $('<div class="panel-backdrop"></div>');
            panel.$containerEl.prepend($backdropEl);
          }
        }
        extend$3(panel, {
          app,
          side,
          effect,
          resizable,
          $el,
          el: $el[0],
          opened: false,
          $backdropEl,
          backdropEl: $backdropEl && $backdropEl[0]
        });

        // Install Modules
        panel.useModules();

        // Init
        panel.init();
        return panel;
      }
      getViewEl() {
        const panel = this;
        let viewEl;
        if (panel.$containerEl.children('.views').length > 0) {
          viewEl = panel.$containerEl.children('.views')[0];
        } else {
          viewEl = panel.$containerEl.children('.view')[0];
        }
        return viewEl;
      }
      setStateClasses(state) {
        const panel = this;
        const {
          side,
          el,
          effect
        } = panel;
        const viewEl = panel.getViewEl();
        const panelInView = viewEl && viewEl.contains(el);
        const $targetEl = !viewEl || panelInView ? panel.$containerEl : $('html');
        if (state === 'open') {
          $targetEl.addClass(`with-panel with-panel-${panel.side}-${panel.effect}`);
        }
        if (state === 'before-closing') {
          $targetEl.addClass('with-panel-closing');
        }
        if (state === 'closing') {
          $targetEl.addClass('with-panel-closing');
          $targetEl.removeClass(`with-panel with-panel-${panel.side}-${panel.effect}`);
        }
        if (state === 'after-closing') {
          $targetEl.removeClass('with-panel-closing');
        }
        if (state === 'closed') {
          $targetEl.removeClass(`with-panel-${side}-${effect}`);
        }
      }
      enableVisibleBreakpoint() {
        const panel = this;
        panel.visibleBreakpointDisabled = false;
        panel.setVisibleBreakpoint();
        return panel;
      }
      disableVisibleBreakpoint() {
        const panel = this;
        panel.visibleBreakpointDisabled = true;
        panel.setVisibleBreakpoint();
        return panel;
      }
      toggleVisibleBreakpoint() {
        const panel = this;
        panel.visibleBreakpointDisabled = !panel.visibleBreakpointDisabled;
        panel.setVisibleBreakpoint();
        return panel;
      }
      setVisibleBreakpoint(emitEvents) {
        if (emitEvents === void 0) {
          emitEvents = true;
        }
        const panel = this;
        const app = panel.app;
        if (!panel.visibleBreakpointResizeHandler) {
          panel.visibleBreakpointResizeHandler = function visibleBreakpointResizeHandler() {
            panel.setVisibleBreakpoint();
          };
          app.on('resize', panel.visibleBreakpointResizeHandler);
        }
        const {
          side,
          $el,
          $containerEl,
          params,
          visibleBreakpointDisabled
        } = panel;
        const breakpoint = params.visibleBreakpoint;
        const $viewEl = $(panel.getViewEl());
        const wasVisible = $el.hasClass('panel-in-breakpoint');
        if ($containerEl && $containerEl.hasClass('page')) {
          $viewEl.add($containerEl.children('.page-content, .tabs, .fab'));
        }
        if (app.width >= breakpoint && typeof breakpoint !== 'undefined' && breakpoint !== null && !visibleBreakpointDisabled) {
          if (!wasVisible) {
            panel.setStateClasses('closed');
            $el.addClass('panel-in-breakpoint').removeClass('panel-in panel-in-collapsed');
            panel.onOpen(false);
            panel.onOpened();
            $viewEl.css({
              [`margin-${side}`]: `${$el.width()}px`
            });
            app.allowPanelOpen = true;
            if (emitEvents) {
              panel.emit('local::breakpoint panelBreakpoint', panel);
              panel.$el.trigger('panel:breakpoint');
            }
          } else {
            $viewEl.css({
              [`margin-${side}`]: `${$el.width()}px`
            });
          }
        } else if (wasVisible) {
          $el.removeClass('panel-in-breakpoint panel-in');
          panel.onClose();
          panel.onClosed();
          $viewEl.css({
            [`margin-${side}`]: ''
          });
          if (emitEvents) {
            panel.emit('local::breakpoint panelBreakpoint', panel);
            panel.$el.trigger('panel:breakpoint');
          }
        }
      }
      enableCollapsedBreakpoint() {
        const panel = this;
        panel.collapsedBreakpointDisabled = false;
        panel.setCollapsedBreakpoint();
        return panel;
      }
      disableCollapsedBreakpoint() {
        const panel = this;
        panel.collapsedBreakpointDisabled = true;
        panel.setCollapsedBreakpoint();
        return panel;
      }
      toggleCollapsedBreakpoint() {
        const panel = this;
        panel.collapsedBreakpointDisabled = !panel.collapsedBreakpointDisabled;
        panel.setCollapsedBreakpoint();
        return panel;
      }
      setCollapsedBreakpoint(emitEvents) {
        if (emitEvents === void 0) {
          emitEvents = true;
        }
        const panel = this;
        const app = panel.app;
        if (!panel.collapsedBreakpointResizeHandler) {
          panel.collapsedBreakpointResizeHandler = function collapsedBreakpointResizeHandler() {
            panel.setCollapsedBreakpoint();
          };
          app.on('resize', panel.collapsedBreakpointResizeHandler);
        }
        const {
          $el,
          params,
          collapsedBreakpointDisabled
        } = panel;
        if ($el.hasClass('panel-in-breakpoint')) return;
        const breakpoint = params.collapsedBreakpoint;
        const wasVisible = $el.hasClass('panel-in-collapsed');
        if (app.width >= breakpoint && typeof breakpoint !== 'undefined' && breakpoint !== null && !collapsedBreakpointDisabled) {
          if (!wasVisible) {
            panel.setStateClasses('closed');
            $el.addClass('panel-in-collapsed').removeClass('panel-in');
            panel.collapsed = true;
            app.allowPanelOpen = true;
            if (emitEvents) {
              panel.emit('local::collapsedBreakpoint panelCollapsedBreakpoint', panel);
              panel.$el.trigger('panel:collapsedbreakpoint');
            }
          }
        } else if (wasVisible) {
          $el.removeClass('panel-in-collapsed panel-in');
          panel.collapsed = false;
          if (emitEvents) {
            panel.emit('local::collapsedBreakpoint panelCollapsedBreakpoint', panel);
            panel.$el.trigger('panel:collapsedbreakpoint');
          }
        }
      }
      enableResizable() {
        const panel = this;
        if (panel.resizableInitialized) {
          panel.resizable = true;
          panel.$el.addClass('panel-resizable');
        } else {
          resizablePanel(panel);
        }
        return panel;
      }
      disableResizable() {
        const panel = this;
        panel.resizable = false;
        panel.$el.removeClass('panel-resizable');
        return panel;
      }
      enableSwipe() {
        const panel = this;
        if (panel.swipeInitialized) {
          panel.swipeable = true;
        } else {
          swipePanel(panel);
        }
        return panel;
      }
      disableSwipe() {
        const panel = this;
        panel.swipeable = false;
        return panel;
      }
      onOpen(modifyHtmlClasses) {
        if (modifyHtmlClasses === void 0) {
          modifyHtmlClasses = true;
        }
        const panel = this;
        // eslint-disable-next-line
        panel._openTransitionStarted = false;
        const app = panel.app;
        panel.opened = true;
        app.panel.allowOpen = false;
        panel.$el.trigger('panel:beforeopen');
        panel.emit('local::beforeOpen panelBeforeOpen', panel);
        if (modifyHtmlClasses) {
          panel.setStateClasses('open');
        }
        panel.$el.trigger('panel:open');
        panel.emit('local::open panelOpen', panel);
      }
      onOpened() {
        const panel = this;
        const app = panel.app;
        app.panel.allowOpen = true;
        panel.$el.trigger('panel:opened');
        panel.emit('local::opened panelOpened', panel);
      }
      onClose() {
        const panel = this;
        const app = panel.app;
        panel.opened = false;
        app.panel.allowOpen = false;
        panel.$el.trigger('panel:beforeclose');
        panel.emit('local::beforeClose panelBeforeClose', panel);
        panel.setStateClasses('closing');
        panel.$el.trigger('panel:close');
        panel.emit('local::close panelClose', panel);
      }
      onClosed() {
        const panel = this;
        const app = panel.app;
        app.panel.allowOpen = true;
        panel.setStateClasses('after-closing');
        panel.$el.removeClass('panel-out');
        if (panel.$backdropEl) {
          const otherPanel = app.panel.get('.panel-in');
          const shouldHideBackdrop = !otherPanel || otherPanel && !otherPanel.$backdropEl;
          if (shouldHideBackdrop) {
            panel.$backdropEl.removeClass('panel-backdrop-in');
          }
        }
        panel.$el.trigger('panel:closed');
        panel.emit('local::closed panelClosed', panel);
      }
      toggle(animate) {
        if (animate === void 0) {
          animate = true;
        }
        const panel = this;
        const breakpoint = panel.params.visibleBreakpoint;
        const app = panel.app;
        if (app.width >= breakpoint && typeof breakpoint !== 'undefined' && breakpoint !== null) {
          return panel.toggleVisibleBreakpoint();
        }
        if (panel.opened) panel.close(animate);else panel.open(animate);
        return panel;
      }
      insertToRoot() {
        const panel = this;
        const document = getDocument$1();
        const {
          $el,
          $backdropEl,
          $containerEl
        } = panel;
        const $panelParentEl = $el.parent();
        const wasInDom = $el.parents(document).length > 0;
        if (!$panelParentEl.is($containerEl) || $el.prevAll('.views, .view').length) {
          const $insertBeforeEl = $containerEl.children('.panel, .views, .view').eq(0);
          const $insertAfterEl = $containerEl.children('.panel-backdrop').eq(0);
          if ($insertBeforeEl.length) {
            $el.insertBefore($insertBeforeEl);
          } else if ($insertAfterEl) {
            $el.insertBefore($insertAfterEl);
          } else {
            $containerEl.prepend($el);
          }
          if ($backdropEl && $backdropEl.length && (!$backdropEl.parent().is($containerEl) && $backdropEl.nextAll('.panel').length === 0 || $backdropEl.parent().is($containerEl) && $backdropEl.nextAll('.panel').length === 0)) {
            $backdropEl.insertBefore($el);
          }
          panel.once('panelClosed', () => {
            if (wasInDom) {
              $panelParentEl.append($el);
            } else {
              $el.remove();
            }
          });
        }
      }
      open(animate) {
        if (animate === void 0) {
          animate = true;
        }
        const panel = this;
        const app = panel.app;
        if (!app.panel.allowOpen) return false;
        const {
          effect,
          $el,
          $backdropEl,
          opened,
          $containerEl
        } = panel;
        if (!$el || $el.hasClass('panel-in')) {
          return panel;
        }
        panel.insertToRoot();

        // Ignore if opened
        if (opened || $el.hasClass('panel-in-breakpoint') || $el.hasClass('panel-in')) return false;

        // Close if some panel is opened
        const otherOpenedPanel = app.panel.get('.panel-in');
        if (otherOpenedPanel && otherOpenedPanel !== panel) {
          otherOpenedPanel.close(animate);
        }
        $el[animate ? 'removeClass' : 'addClass']('not-animated');
        $el.addClass('panel-in');
        if ($backdropEl) {
          $backdropEl.addClass('panel-backdrop-in');
          $backdropEl[animate ? 'removeClass' : 'addClass']('not-animated');
        }
        if (['cover', 'push', 'floating'].includes(panel.effect)) {
          /* eslint no-underscore-dangle: ["error", { "allow": ["_clientLeft"] }] */
          panel._clientLeft = $el[0].clientLeft;
        }

        // Transitionend
        const $viewEl = $(panel.getViewEl());
        if ($containerEl && $containerEl.hasClass('page')) {
          $viewEl.add($containerEl.children('.page-content, .tabs'));
        }
        const transitionEndTarget = effect === 'reveal' ? $viewEl : $el;
        function panelTransitionStart() {
          transitionEndTarget.transitionStart(() => {
            // eslint-disable-next-line
            panel._openTransitionStarted = true;
          });
        }
        function panelTransitionEnd() {
          transitionEndTarget.transitionEnd(e => {
            if ($(e.target).is(transitionEndTarget)) {
              if ($el.hasClass('panel-out')) {
                panel.onClosed();
              } else {
                panel.onOpened();
              }
            } else panelTransitionEnd();
          });
        }
        if (animate) {
          if ($backdropEl) {
            $backdropEl.removeClass('not-animated');
          }
          panelTransitionStart();
          panelTransitionEnd();
          $el.removeClass('panel-out not-animated').addClass('panel-in');
          panel.onOpen();
        } else {
          if ($backdropEl) {
            $backdropEl.addClass('not-animated');
          }
          $el.removeClass('panel-out').addClass('panel-in not-animated');
          panel.onOpen();
          panel.onOpened();
          // eslint-disable-next-line
          panel._openTransitionStarted = true;
        }
        return true;
      }
      close(animate) {
        if (animate === void 0) {
          animate = true;
        }
        const panel = this;
        const {
          effect,
          $el,
          $backdropEl,
          opened,
          $containerEl
        } = panel;
        if (!opened || $el.hasClass('panel-in-breakpoint') || !$el.hasClass('panel-in')) return panel;
        $el[animate ? 'removeClass' : 'addClass']('not-animated');
        if ($backdropEl) {
          $backdropEl[animate ? 'removeClass' : 'addClass']('not-animated');
        }
        const $viewEl = $(panel.getViewEl());
        if ($containerEl && $containerEl.hasClass('page')) {
          $viewEl.add($containerEl.children('.page-content, .tabs'));
        }
        const transitionEndTarget = effect === 'reveal' ? $viewEl : $el;
        // eslint-disable-next-line
        if (!panel._openTransitionStarted) {
          // eslint-disable-next-line
          animate = false;
        }
        function transitionEnd() {
          if ($el.hasClass('panel-out')) {
            panel.onClosed();
          } else if ($el.hasClass('panel-in')) {
            panel.onOpened();
          }
          panel.setStateClasses('after-closing');
        }
        if (animate) {
          transitionEndTarget.transitionEnd(() => {
            transitionEnd();
          });
          $el.removeClass('panel-in').addClass('panel-out');
          // Emit close
          panel.onClose();
        } else {
          $el.addClass('not-animated').removeClass('panel-in').addClass('panel-out');
          // Emit close
          panel.onClose();
          panel.onClosed();
        }
        return panel;
      }
      init() {
        const panel = this;
        // const app = panel.app;
        if (typeof panel.params.visibleBreakpoint !== 'undefined') {
          panel.setVisibleBreakpoint();
        }
        if (typeof panel.params.collapsedBreakpoint !== 'undefined') {
          panel.setCollapsedBreakpoint();
        }
        if (panel.params.swipe) {
          panel.enableSwipe();
        }
        if (panel.resizable) {
          panel.enableResizable();
        }
      }
      destroy() {
        let panel = this;
        const app = panel.app;
        const {
          $containerEl
        } = panel;
        if (!panel.$el) {
          // Panel already destroyed
          return;
        }
        panel.emit('local::beforeDestroy panelBeforeDestroy', panel);
        panel.$el.trigger('panel:beforedestroy');
        if (panel.visibleBreakpointResizeHandler) {
          app.off('resize', panel.visibleBreakpointResizeHandler);
        }
        if (panel.collapsedBreakpointResizeHandler) {
          app.off('resize', panel.collapsedBreakpointResizeHandler);
        }
        if (panel.$el.hasClass('panel-in-breakpoint') || panel.$el.hasClass('panel-in-collapsed')) {
          const $viewEl = $(panel.getViewEl());
          if ($containerEl && $containerEl.hasClass('page')) {
            $viewEl.add($containerEl.children('.page-content, .tabs'));
          }
          panel.$el.removeClass('panel-in-breakpoint panel-in-collapsed panel-in');
          $viewEl.css({
            [`margin-${panel.side}`]: ''
          });
          panel.emit('local::breakpoint panelBreakpoint', panel);
          panel.$el.trigger('panel:breakpoint');
        }
        panel.$el.trigger('panel:destroy');
        panel.emit('local::destroy panelDestroy', panel);
        if (panel.el) {
          panel.el.f7Panel = null;
          delete panel.el.f7Panel;
        }
        deleteProps$1(panel);
        panel = null;
      }
    };

    var Panel = {
      name: 'panel',
      params: {
        panel: {
          opened: undefined,
          // default based on panel-in class
          side: undefined,
          // default based on panel class
          effect: undefined,
          // default based on panel class
          resizable: undefined,
          // default based on panel-resizable class
          backdrop: true,
          backdropEl: undefined,
          visibleBreakpoint: undefined,
          collapsedBreakpoint: undefined,
          swipe: false,
          // or true
          swipeNoFollow: false,
          // or true
          swipeOnlyClose: false,
          swipeActiveArea: 0,
          swipeThreshold: 0,
          closeByBackdropClick: true,
          containerEl: undefined
        }
      },
      static: {
        Panel: Panel$1
      },
      create() {
        const app = this;
        extend$3(app, {
          panel: {
            allowOpen: true,
            create(params) {
              return new Panel$1(app, params);
            },
            get(el) {
              if (el === void 0) {
                el = '.panel';
              }
              if (el instanceof Panel$1) return el;
              if (el === 'left' || el === 'right') el = `.panel-${el}`; // eslint-disable-line
              const $el = $(el);
              if ($el.length === 0 || $el.length > 1) return undefined;
              return $el[0].f7Panel;
            },
            destroy(el) {
              if (el === void 0) {
                el = '.panel';
              }
              const panel = app.panel.get(el);
              if (panel && panel.destroy) return panel.destroy();
              return undefined;
            },
            open(el, animate) {
              if (el === void 0) {
                el = '.panel';
              }
              if (el === 'left' || el === 'right') el = `.panel-${el}`; // eslint-disable-line
              let panel = app.panel.get(el);
              if (panel && panel.open) return panel.open(animate);
              if (!panel) {
                panel = app.panel.create({
                  el
                });
                return panel.open(animate);
              }
              return undefined;
            },
            close(el, animate) {
              if (el === void 0) {
                el = '.panel-in';
              }
              if (el === 'left' || el === 'right') el = `.panel-${el}`; // eslint-disable-line
              let panel = app.panel.get(el);
              if (panel && panel.open) return panel.close(animate);
              if (!panel) {
                panel = app.panel.create({
                  el
                });
                return panel.close(animate);
              }
              return undefined;
            },
            toggle(el, animate) {
              if (el === void 0) {
                el = '.panel';
              }
              if (el === 'left' || el === 'right') el = `.panel-${el}`; // eslint-disable-line
              let panel = app.panel.get(el);
              if (panel && panel.toggle) return panel.toggle(animate);
              if (!panel) {
                panel = app.panel.create({
                  el
                });
                return panel.toggle(animate);
              }
              return undefined;
            }
          }
        });
      },
      on: {
        init() {
          const app = this;
          $('.panel-init').each(panelEl => {
            const params = Object.assign({
              el: panelEl
            }, $(panelEl).dataset() || {});
            app.panel.create(params);
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.panel-init').each(panelEl => {
            const params = Object.assign({
              el: panelEl
            }, $(panelEl).dataset() || {});
            app.panel.create(params);
          });
        },
        pageBeforeRemove(page) {
          const app = this;
          page.$el.find('.panel-init').each(panelEl => {
            const panel = app.panel.get(panelEl);
            if (panel && panel.destroy) panel.destroy();
          });
        }
      },
      vnode: {
        'panel-init': {
          insert(vnode) {
            const app = this;
            const panelEl = vnode.elm;
            const params = Object.assign({
              el: panelEl
            }, $(panelEl).dataset() || {});
            app.panel.create(params);
          },
          destroy(vnode) {
            const app = this;
            const panelEl = vnode.elm;
            const panel = app.panel.get(panelEl);
            if (panel && panel.destroy) panel.destroy();
          }
        }
      },
      clicks: {
        '.panel-open': function open(clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.panel.open(data.panel, data.animate);
        },
        '.panel-close': function close(clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.panel.close(data.panel, data.animate);
        },
        '.panel-toggle': function close(clickedEl, data) {
          if (data === void 0) {
            data = {};
          }
          const app = this;
          app.panel.toggle(data.panel, data.animate);
        },
        '.panel-backdrop': function close() {
          const app = this;
          const $panelEl = $('.panel-in:not(.panel-out)');
          if (!$panelEl.length) return;
          const instance = $panelEl[0] && $panelEl[0].f7Panel;
          $panelEl.trigger('panel:backdrop-click');
          if (instance) {
            instance.emit('backdropClick', instance);
          }
          app.emit('panelBackdropClick', instance || $panelEl[0]);
          if (instance && instance.params.closeByBackdropClick === false) return;
          if (app.params.panel.closeByBackdropClick) app.panel.close();
        }
      }
    };

    /* eslint no-param-reassign: "off" */
    const CardExpandable = {
      open(cardEl, animate) {
        if (cardEl === void 0) {
          cardEl = '.card-expandable';
        }
        if (animate === void 0) {
          animate = true;
        }
        const app = this;
        const device = getDevice$1();
        const document = getDocument$1();
        const support = getSupport$1();
        const $cardEl = $(cardEl).eq(0);
        if (!$cardEl || !$cardEl.length) return;
        if ($cardEl.hasClass('card-opened') || $cardEl.hasClass('card-opening') || $cardEl.hasClass('card-closing')) return;
        const $pageEl = $cardEl.parents('.page').eq(0);
        if (!$pageEl.length) return;
        if ($pageEl.find('.card-opened').length) {
          return;
        }
        let prevented;
        function prevent() {
          prevented = true;
        }
        $cardEl.trigger('card:beforeopen', {
          prevent
        });
        app.emit('cardBeforeOpen', $cardEl[0], prevent);
        if (prevented) return;
        const cardParams = Object.assign({
          animate
        }, app.params.card, $cardEl.dataset());
        const $pageContentEl = $cardEl.parents('.page-content');
        let $backdropEl;
        if ($cardEl.attr('data-backdrop-el')) {
          $backdropEl = $($cardEl.attr('data-backdrop-el'));
        }
        if (!$backdropEl && cardParams.backdrop) {
          $backdropEl = $pageContentEl.find('.card-backdrop');
          if (!$backdropEl.length) {
            $backdropEl = $('<div class="card-backdrop"></div>');
            $pageContentEl.append($backdropEl);
          }
        }
        let $navbarEl;
        let $toolbarEl;
        if (cardParams.hideNavbarOnOpen) {
          $navbarEl = $pageEl.children('.navbar');
          if (!$navbarEl.length) {
            if ($pageEl[0].f7Page) $navbarEl = $pageEl[0].f7Page.$navbarEl;
          }
        }
        if (cardParams.hideToolbarOnOpen) {
          $toolbarEl = $pageEl.children('.toolbar');
          if (!$toolbarEl.length) {
            $toolbarEl = $pageEl.parents('.view').children('.toolbar');
          }
          if (!$toolbarEl.length) {
            $toolbarEl = $pageEl.parents('.views').children('.toolbar');
          }
        }
        const currTransform = $cardEl.css('transform');
        let hasTransform;
        if (currTransform && currTransform.match(/[2-9]/)) {
          hasTransform = true;
        }
        const $cardContentEl = $cardEl.children('.card-content');
        const $cardSizeEl = $(document.createElement('div')).addClass('card-expandable-size');
        $cardEl.append($cardSizeEl);
        let cardWidth = $cardEl[0].offsetWidth;
        let cardHeight = $cardEl[0].offsetHeight;
        let pageWidth = $pageEl[0].offsetWidth;
        let pageHeight = $pageEl[0].offsetHeight;
        let maxWidth = $cardSizeEl[0].offsetWidth || pageWidth;
        let maxHeight = $cardSizeEl[0].offsetHeight || pageHeight;
        let statusbarHeight;
        if ($navbarEl && !cardParams.hideStatusbarOnOpen && maxHeight === pageHeight) {
          statusbarHeight = parseInt($navbarEl.css('--f7-safe-area-top'), 10);
          if (Number.isNaN(statusbarHeight)) statusbarHeight = 0;
        }
        if (statusbarHeight) {
          maxHeight -= statusbarHeight;
        }
        let scaleX = maxWidth / cardWidth;
        let scaleY = maxHeight / cardHeight;
        let offset = $cardEl.offset();
        let pageOffset = $pageEl.offset();
        if (statusbarHeight) {
          pageOffset.top += statusbarHeight / 2;
        }
        offset.left -= pageOffset.left;
        let cardLeftOffset;
        let cardTopOffset;
        if (hasTransform) {
          const transformValues = currTransform.replace(/matrix\(|\)/g, '').split(',').map(el => el.trim());
          if (transformValues && transformValues.length > 1) {
            const scale = parseFloat(transformValues[0]);
            cardLeftOffset = offset.left - cardWidth * (1 - scale) / 2;
            cardTopOffset = offset.top - pageOffset.top - cardHeight * (1 - scale) / 2;
            if (app.rtl) cardLeftOffset -= $cardEl[0].scrollLeft;
          } else {
            cardLeftOffset = $cardEl[0].offsetLeft;
            cardTopOffset = $cardEl[0].offsetTop - ($pageContentEl.length ? $pageContentEl[0].scrollTop : 0);
          }
        } else {
          cardLeftOffset = offset.left;
          cardTopOffset = offset.top - pageOffset.top;
          if (app.rtl) cardLeftOffset -= $cardEl[0].scrollLeft;
        }
        cardLeftOffset -= (pageWidth - maxWidth) / 2;
        cardTopOffset -= (pageHeight - maxHeight) / 2;
        let cardRightOffset = maxWidth - cardWidth - cardLeftOffset;
        if (app.rtl) {
          [cardLeftOffset, cardRightOffset] = [cardRightOffset, cardLeftOffset];
        }
        let cardBottomOffset = maxHeight - cardHeight - cardTopOffset;
        let translateX = (cardRightOffset - cardLeftOffset) / 2;
        let translateY = (cardBottomOffset - cardTopOffset) / 2;
        if (cardParams.hideNavbarOnOpen && $navbarEl && $navbarEl.length) {
          if ($navbarEl.closest('.navbar-hidden').length) {
            // Was hidden
            $cardEl[0].f7KeepNavbarOnClose = true;
          } else {
            delete $cardEl[0].f7KeepNavbarOnClose;
            app.navbar.hide($navbarEl, cardParams.animate, cardParams.hideStatusbarOnOpen, true);
          }
        }
        if (cardParams.hideToolbarOnOpen && $toolbarEl && $toolbarEl.length) {
          if ($toolbarEl.closest('.toolbar-hidden').length) {
            // Was hidden
            $cardEl[0].f7KeepToolbarOnClose = true;
          } else {
            delete $cardEl[0].f7KeepToolbarOnClose;
            app.toolbar.hide($toolbarEl, cardParams.animate);
          }
        }
        if ($backdropEl) {
          $backdropEl.removeClass('card-backdrop-out').addClass('card-backdrop-in');
        }
        $cardEl.removeClass('card-transitioning');
        if (cardParams.animate) {
          $cardEl.addClass('card-opening');
        }
        $cardEl.trigger('card:open');
        app.emit('cardOpen', $cardEl[0]);
        function transitionEnd() {
          $pageEl.addClass('page-with-card-opened');
          if (device.ios && $pageContentEl.length) {
            $pageContentEl.css('height', `${$pageContentEl[0].offsetHeight + 1}px`);
            setTimeout(() => {
              $pageContentEl.css('height', '');
            });
          }
          $cardEl.addClass('card-opened');
          $cardEl.removeClass('card-opening');
          $cardEl.trigger('card:opened');
          app.emit('cardOpened', $cardEl[0], $pageEl[0]);
        }
        $cardContentEl.css({
          width: `${maxWidth}px`,
          height: `${maxHeight}px`
        }).transform(`translate3d(${app.rtl ? cardLeftOffset + translateX : -cardLeftOffset - translateX}px, 0px, 0) scale(${1 / scaleX}, ${1 / scaleY})`);
        $cardEl.transform(`translate3d(${app.rtl ? -translateX : translateX}px, ${translateY}px, 0) scale(${scaleX}, ${scaleY})`);
        if (cardParams.animate) {
          $cardEl.transitionEnd(() => {
            transitionEnd();
          });
        } else {
          transitionEnd();
        }
        function onResize() {
          $cardEl.removeClass('card-transitioning');
          cardWidth = $cardEl[0].offsetWidth;
          cardHeight = $cardEl[0].offsetHeight;
          pageWidth = $pageEl[0].offsetWidth;
          pageHeight = $pageEl[0].offsetHeight;
          maxWidth = $cardSizeEl[0].offsetWidth || pageWidth;
          maxHeight = $cardSizeEl[0].offsetHeight || pageHeight;
          statusbarHeight = 0;
          if ($navbarEl && !cardParams.hideStatusbarOnOpen && maxHeight === pageHeight) {
            statusbarHeight = parseInt($navbarEl.css('--f7-safe-area-top'), 10);
            if (Number.isNaN(statusbarHeight)) statusbarHeight = 0;
          }
          if (statusbarHeight) {
            maxHeight -= statusbarHeight;
          }
          scaleX = maxWidth / cardWidth;
          scaleY = maxHeight / cardHeight;
          $cardEl.transform('translate3d(0px, 0px, 0) scale(1)');
          offset = $cardEl.offset();
          pageOffset = $pageEl.offset();
          if (statusbarHeight) {
            pageOffset.top += statusbarHeight / 2;
          }
          offset.left -= pageOffset.left;
          offset.top -= pageOffset.top;
          cardLeftOffset = offset.left - (pageWidth - maxWidth) / 2;
          if (app.rtl) cardLeftOffset -= $cardEl[0].scrollLeft;
          cardTopOffset = offset.top - (pageHeight - maxHeight) / 2;
          cardRightOffset = maxWidth - cardWidth - cardLeftOffset;
          cardBottomOffset = maxHeight - cardHeight - cardTopOffset;
          if (app.rtl) {
            [cardLeftOffset, cardRightOffset] = [cardRightOffset, cardLeftOffset];
          }
          translateX = (cardRightOffset - cardLeftOffset) / 2;
          translateY = (cardBottomOffset - cardTopOffset) / 2;
          $cardEl.transform(`translate3d(${app.rtl ? -translateX : translateX}px, ${translateY}px, 0) scale(${scaleX}, ${scaleY})`);
          $cardContentEl.css({
            width: `${maxWidth}px`,
            height: `${maxHeight}px`
          }).transform(`translate3d(${app.rtl ? cardLeftOffset + translateX : -cardLeftOffset - translateX}px, 0px, 0) scale(${1 / scaleX}, ${1 / scaleY})`);
        }
        let cardScrollTop;
        let isTouched;
        let isMoved;
        let touchStartX;
        let touchStartY;
        let touchEndX;
        let touchEndY;
        let isScrolling;
        let progress;
        let isV;
        let isH;
        let $cardScrollableEl;
        function onTouchStart(e) {
          if (!$(e.target).closest($cardEl).length || !e.isTrusted) return;
          if (!$cardEl.hasClass('card-opened')) return;
          $cardScrollableEl = $cardEl.find(cardParams.scrollableEl);
          if ($cardScrollableEl[0] && $cardScrollableEl[0] !== $cardContentEl[0] && !$cardScrollableEl[0].contains(e.target)) {
            cardScrollTop = 0;
          } else {
            cardScrollTop = $cardScrollableEl.scrollTop();
          }
          isTouched = true;
          touchStartX = e.targetTouches[0].pageX;
          touchStartY = e.targetTouches[0].pageY;
          isScrolling = undefined;
          isV = false;
          isH = false;
        }
        function onTouchMove(e) {
          if (!isTouched || !e.isTrusted) return;
          touchEndX = e.targetTouches[0].pageX;
          touchEndY = e.targetTouches[0].pageY;
          if (typeof isScrolling === 'undefined') {
            isScrolling = !!(isScrolling || Math.abs(touchEndY - touchStartY) > Math.abs(touchEndX - touchStartX));
          }
          if (!isH && !isV) {
            if (!isScrolling && e.targetTouches[0].clientX <= 50) {
              isH = true;
            } else {
              isV = true;
            }
          }
          if (!(isH || isV) || isV && cardScrollTop !== 0) {
            isTouched = true;
            isMoved = true;
            return;
          }
          if (!isMoved) {
            $cardEl.removeClass('card-transitioning');
          }
          isMoved = true;
          progress = isV ? Math.max((touchEndY - touchStartY) / 150, 0) : Math.max((touchEndX - touchStartX) / (cardWidth / 2), 0);
          if (progress > 0 && isV || isH) {
            if (isV && device.ios && $cardScrollableEl[0] === $cardContentEl[0]) {
              $cardScrollableEl.css('-webkit-overflow-scrolling', 'auto');
              $cardScrollableEl.scrollTop(0);
            }
            e.preventDefault();
          }
          if (progress > 1) progress **= 0.3;
          if (progress > (isV ? 1.3 : 1.1)) {
            isTouched = false;
            isMoved = false;
            app.card.close($cardEl);
          } else {
            $cardEl.transform(`translate3d(${app.rtl ? -translateX : translateX}px, ${translateY}px, 0) scale(${scaleX * (1 - progress * 0.2)}, ${scaleY * (1 - progress * 0.2)})`);
          }
        }
        function onTouchEnd(e) {
          if (!isTouched || !isMoved || !e.isTrusted) return;
          isTouched = false;
          isMoved = false;
          if (device.ios) {
            $cardScrollableEl.css('-webkit-overflow-scrolling', '');
          }
          if (progress >= 0.8) {
            app.card.close($cardEl);
          } else {
            $cardEl.addClass('card-transitioning').transform(`translate3d(${app.rtl ? -translateX : translateX}px, ${translateY}px, 0) scale(${scaleX}, ${scaleY})`);
          }
        }
        $cardEl[0].detachEventHandlers = function detachEventHandlers() {
          app.off('resize', onResize);
          if (support.touch && cardParams.swipeToClose) {
            app.off('touchstart:passive', onTouchStart);
            app.off('touchmove:active', onTouchMove);
            app.off('touchend:passive', onTouchEnd);
          }
        };
        app.on('resize', onResize);
        if (support.touch && cardParams.swipeToClose) {
          app.on('touchstart:passive', onTouchStart);
          app.on('touchmove:active', onTouchMove);
          app.on('touchend:passive', onTouchEnd);
        }
      },
      close(cardEl, animate) {
        if (cardEl === void 0) {
          cardEl = '.card-expandable.card-opened';
        }
        if (animate === void 0) {
          animate = true;
        }
        const app = this;
        const device = getDevice$1();
        const $cardEl = $(cardEl).eq(0);
        if (!$cardEl || !$cardEl.length) return;
        if (!$cardEl.hasClass('card-opened') || $cardEl.hasClass('card-opening') || $cardEl.hasClass('card-closing')) return;
        const $cardContentEl = $cardEl.children('.card-content');
        const $pageContentEl = $cardEl.parents('.page-content');
        const $pageEl = $cardEl.parents('.page').eq(0);
        if (!$pageEl.length) return;
        const cardParams = Object.assign({
          animate
        }, app.params.card, $cardEl.dataset());
        const $cardScrollableEl = $cardEl.find(cardParams.scrollableEl);
        let $navbarEl;
        let $toolbarEl;
        let $backdropEl;
        if ($cardEl.attr('data-backdrop-el')) {
          $backdropEl = $($cardEl.attr('data-backdrop-el'));
        }
        if (cardParams.backdrop) {
          $backdropEl = $cardEl.parents('.page-content').find('.card-backdrop');
        }
        if (cardParams.hideNavbarOnOpen) {
          $navbarEl = $pageEl.children('.navbar');
          if (!$navbarEl.length) {
            if ($pageEl[0].f7Page) $navbarEl = $pageEl[0].f7Page.$navbarEl;
          }
          if ($navbarEl && $navbarEl.length && !$cardEl[0].f7KeepNavbarOnClose) {
            app.navbar.show($navbarEl, cardParams.animate, true);
          }
        }
        if (cardParams.hideToolbarOnOpen) {
          $toolbarEl = $pageEl.children('.toolbar');
          if (!$toolbarEl.length) {
            $toolbarEl = $pageEl.parents('.view').children('.toolbar');
          }
          if (!$toolbarEl.length) {
            $toolbarEl = $pageEl.parents('.views').children('.toolbar');
          }
          if ($toolbarEl && $toolbarEl.length && !$cardEl[0].f7KeepToolbarOnClose) {
            app.toolbar.show($toolbarEl, cardParams.animate);
          }
        }
        $pageEl.removeClass('page-with-card-opened');
        if (device.ios && $pageContentEl.length) {
          $pageContentEl.css('height', `${$pageContentEl[0].offsetHeight + 1}px`);
          setTimeout(() => {
            $pageContentEl.css('height', '');
          });
        }
        if ($backdropEl && $backdropEl.length) {
          $backdropEl.removeClass('card-backdrop-in').addClass('card-backdrop-out');
        }
        $cardEl.removeClass('card-opened card-transitioning');
        if (cardParams.animate) {
          $cardEl.addClass('card-closing');
        } else {
          $cardEl.addClass('card-no-transition');
        }
        $cardEl.transform('');
        $cardEl.trigger('card:close');
        app.emit('cardClose', $cardEl[0], $pageEl[0]);
        const animateWidth = $cardEl.hasClass('card-expandable-animate-width');
        function transitionEnd() {
          if (!animateWidth) {
            $cardContentEl.css({
              width: '',
              height: ''
            });
          }
          if ($backdropEl && $backdropEl.length) {
            $backdropEl.removeClass('card-backdrop-in card-backdrop-out');
          }
          $cardEl.removeClass('card-closing card-no-transition');
          $cardEl.trigger('card:closed');
          $cardEl.find('.card-expandable-size').remove();
          app.emit('cardClosed', $cardEl[0], $pageEl[0]);
        }
        if (animateWidth) {
          $cardContentEl.css({
            width: '',
            height: ''
          });
        }
        $cardContentEl.transform('').scrollTop(0, animate ? 300 : 0);
        if ($cardScrollableEl.length && $cardScrollableEl[0] !== $cardContentEl[0]) {
          $cardScrollableEl.scrollTop(0, animate ? 300 : 0);
        }
        if (animate) {
          $cardContentEl.transitionEnd(() => {
            transitionEnd();
          });
        } else {
          transitionEnd();
        }
        if ($cardEl[0].detachEventHandlers) {
          $cardEl[0].detachEventHandlers();
          delete $cardEl[0].detachEventHandlers;
        }
      },
      toggle(cardEl, animate) {
        if (cardEl === void 0) {
          cardEl = '.card-expandable';
        }
        const app = this;
        const $cardEl = $(cardEl).eq(0);
        if (!$cardEl.length) return;
        if ($cardEl.hasClass('card-opened')) {
          app.card.close($cardEl, animate);
        } else {
          app.card.open($cardEl, animate);
        }
      }
    };
    var Card = {
      name: 'card',
      params: {
        card: {
          hideNavbarOnOpen: true,
          hideStatusbarOnOpen: true,
          hideToolbarOnOpen: true,
          scrollableEl: '.card-content',
          swipeToClose: true,
          closeByBackdropClick: true,
          backdrop: true
        }
      },
      create() {
        const app = this;
        bindMethods(app, {
          card: CardExpandable
        });
      },
      on: {
        pageBeforeIn(page) {
          const app = this;
          if (app.params.card.hideNavbarOnOpen && page.navbarEl && page.$el.find('.card-opened.card-expandable').length) {
            app.navbar.hide(page.navbarEl, true, app.params.card.hideStatusbarOnOpen, true);
          }
          if (app.params.card.hideToolbarOnOpen && page.$el.find('.card-opened.card-expandable').length) {
            let $toolbarEl = page.$el.children('.toolbar');
            if (!$toolbarEl.length) {
              $toolbarEl = page.$el.parents('.view').children('.toolbar');
            }
            if (!$toolbarEl.length) {
              $toolbarEl = page.$el.parents('.views').children('.toolbar');
            }
            if ($toolbarEl && $toolbarEl.length) {
              app.toolbar.hide($toolbarEl);
            }
          }
        }
      },
      clicks: {
        '.card-close': function closeCard($clickedEl, data) {
          const app = this;
          app.card.close(data.card, data.animate);
        },
        '.card-open': function closeCard($clickedEl, data) {
          const app = this;
          app.card.open(data.card, data.animate);
        },
        '.card-expandable': function toggleExpandableCard($clickedEl, data, e) {
          const app = this;
          if ($clickedEl.hasClass('card-opened') || $clickedEl.hasClass('card-opening') || $clickedEl.hasClass('card-closing')) return;
          if ($(e.target).closest('.card-prevent-open, .card-close').length) return;
          app.card.open($clickedEl);
        },
        '.card-backdrop-in': function onBackdropClick() {
          const app = this;
          let needToClose = false;
          if (app.params.card.closeByBackdropClick) needToClose = true;
          const $openedCardEl = $('.card-opened');
          if (!$openedCardEl.length) return;
          if ($openedCardEl.attr('data-close-by-backdrop-click') === 'true') {
            needToClose = true;
          } else if ($openedCardEl.attr('data-close-by-backdrop-click') === 'false') {
            needToClose = false;
          }
          if (needToClose) app.card.close($openedCardEl);
        }
      }
    };

    var Chip = {
      name: 'chip'
    };

    // Form Data
    const FormData = {
      store(form, data) {
        const app = this;
        const window = getWindow$1();
        let formId = form;
        const $formEl = $(form);
        if ($formEl.length && $formEl.is('form') && $formEl.attr('id')) {
          formId = $formEl.attr('id');
        }
        // Store form data in app.formsData
        app.form.data[`form-${formId}`] = data;

        // Store form data in local storage also
        window.localStorage[`f7form-${formId}`] = JSON.stringify(data);
      },
      get(form) {
        const app = this;
        const window = getWindow$1();
        let formId = form;
        const $formEl = $(form);
        if ($formEl.length && $formEl.is('form') && $formEl.attr('id')) {
          formId = $formEl.attr('id');
        }
        if (window.localStorage[`f7form-${formId}`]) {
          return JSON.parse(window.localStorage[`f7form-${formId}`]);
        }
        if (app.form.data[`form-${formId}`]) {
          return app.form.data[`form-${formId}`];
        }
        return undefined;
      },
      remove(form) {
        const app = this;
        const window = getWindow$1();
        let formId = form;
        const $formEl = $(form);
        if ($formEl.length && $formEl.is('form') && $formEl.attr('id')) {
          formId = $formEl.attr('id');
        }

        // Delete form data from app.formsData
        if (app.form.data[`form-${formId}`]) {
          app.form.data[`form-${formId}`] = '';
          delete app.form.data[`form-${formId}`];
        }

        // Delete form data from local storage also
        if (window.localStorage[`f7form-${formId}`]) {
          window.localStorage[`f7form-${formId}`] = '';
          window.localStorage.removeItem(`f7form-${formId}`);
        }
      }
    };

    // Form Storage
    const FormStorage = {
      init(formEl) {
        const app = this;
        const $formEl = $(formEl);
        const formId = $formEl.attr('id');
        if (!formId) return;
        const initialData = app.form.getFormData(formId);
        if (initialData) {
          app.form.fillFromData($formEl, initialData);
        }
        function store() {
          const data = app.form.convertToData($formEl);
          if (!data) return;
          app.form.storeFormData(formId, data);
          $formEl.trigger('form:storedata', data);
          app.emit('formStoreData', $formEl[0], data);
        }
        $formEl.on('change submit', store);
      },
      destroy(formEl) {
        const $formEl = $(formEl);
        $formEl.off('change submit');
      }
    };

    // Form To/From Data
    function formToData(formEl) {
      const app = this;
      const $formEl = $(formEl).eq(0);
      if ($formEl.length === 0) return undefined;

      // Form data
      const data = {};

      // Skip input types
      const skipTypes = ['submit', 'image', 'button', 'file'];
      const skipNames = [];
      $formEl.find('input, select, textarea').each(inputEl => {
        const $inputEl = $(inputEl);
        if ($inputEl.hasClass('ignore-store-data') || $inputEl.hasClass('no-store-data')) {
          return;
        }
        const name = $inputEl.attr('name');
        const type = $inputEl.attr('type');
        const tag = inputEl.nodeName.toLowerCase();
        if (skipTypes.indexOf(type) >= 0) return;
        if (skipNames.indexOf(name) >= 0 || !name) return;
        if (tag === 'select' && $inputEl.prop('multiple')) {
          skipNames.push(name);
          data[name] = [];
          $formEl.find(`select[name="${name}"] option`).each(el => {
            if (el.selected) data[name].push(el.value);
          });
        } else {
          switch (type) {
            case 'checkbox':
              skipNames.push(name);
              data[name] = [];
              $formEl.find(`input[name="${name}"]`).each(el => {
                if (el.checked) data[name].push(el.value);
              });
              break;
            case 'radio':
              skipNames.push(name);
              $formEl.find(`input[name="${name}"]`).each(el => {
                if (el.checked) data[name] = el.value;
              });
              break;
            default:
              data[name] = $inputEl.val();
              break;
          }
        }
      });
      $formEl.trigger('form:todata', data);
      app.emit('formToData', $formEl[0], data);
      return data;
    }
    function formFromData(formEl, formData) {
      const app = this;
      const $formEl = $(formEl).eq(0);
      if (!$formEl.length) return;
      let data = formData;
      const formId = $formEl.attr('id');
      if (!data && formId) {
        data = app.form.getFormData(formId);
      }
      if (!data) return;

      // Skip input types
      const skipTypes = ['submit', 'image', 'button', 'file'];
      const skipNames = [];
      $formEl.find('input, select, textarea').each(inputEl => {
        const $inputEl = $(inputEl);
        if ($inputEl.hasClass('ignore-store-data') || $inputEl.hasClass('no-store-data')) {
          return;
        }
        const name = $inputEl.attr('name');
        const type = $inputEl.attr('type');
        const tag = inputEl.nodeName.toLowerCase();
        if (typeof data[name] === 'undefined' || data[name] === null) return;
        if (skipTypes.indexOf(type) >= 0) return;
        if (skipNames.indexOf(name) >= 0 || !name) return;
        if (tag === 'select' && $inputEl.prop('multiple')) {
          skipNames.push(name);
          $formEl.find(`select[name="${name}"] option`).each(el => {
            const selectEl = el;
            if (data[name].indexOf(el.value) >= 0) selectEl.selected = true;else selectEl.selected = false;
          });
        } else {
          switch (type) {
            case 'checkbox':
              skipNames.push(name);
              $formEl.find(`input[name="${name}"]`).each(el => {
                const checkboxEl = el;
                if (data[name].indexOf(el.value) >= 0) checkboxEl.checked = true;else checkboxEl.checked = false;
              });
              break;
            case 'radio':
              skipNames.push(name);
              $formEl.find(`input[name="${name}"]`).each(el => {
                const radioEl = el;
                if (data[name] === el.value) radioEl.checked = true;else radioEl.checked = false;
              });
              break;
            default:
              $inputEl.val(data[name]);
              break;
          }
        }
        if (tag === 'select' || tag === 'input' || tag === 'textarea') {
          $inputEl.trigger('change', 'fromdata');
        }
      });
      $formEl.trigger('form:fromdata', data);
      app.emit('formFromData', $formEl[0], data);
    }
    function initAjaxForm() {
      const app = this;
      const window = getWindow$1();
      const document = getDocument$1();
      function onSubmitChange(e, fromData) {
        const $formEl = $(this);
        if (e.type === 'change' && !$formEl.hasClass('form-ajax-submit-onchange')) return;
        if (e.type === 'submit') e.preventDefault();
        if (e.type === 'change' && fromData === 'fromdata') return;
        const method = ($formEl.attr('method') || 'GET').toUpperCase();
        const contentType = $formEl.attr('enctype') || $formEl.prop('enctype');
        let url = $formEl.attr('action');
        if (!url) return;
        let data;
        if (method === 'POST') {
          if (contentType === 'application/x-www-form-urlencoded' || contentType === 'application/json') {
            data = app.form.convertToData($formEl[0]);
            if (contentType === 'application/json') {
              data = JSON.stringify(data);
            }
          } else {
            data = new window.FormData($formEl[0]);
          }
        } else {
          data = serializeObject(app.form.convertToData($formEl[0]));
          if (url.includes('?')) {
            url += `&${data}`;
          } else {
            url += `?${data}`;
          }
        }
        $formEl.trigger('formajax:beforesend', {
          data
        });
        app.emit('formAjaxBeforeSend', $formEl[0], data);
        fetch(url, {
          method,
          headers: {
            'Content-Type': contentType || 'application/x-www-form-urlencoded'
          },
          ...(method === 'POST' || method === 'PUT' ? {
            body: data
          } : {})
        }).then(response => {
          $formEl.trigger('formajax:complete', {
            data,
            response
          });
          app.emit('formAjaxComplete', $formEl[0], data, response);
          $formEl.trigger('formajax:success', {
            data,
            response
          });
          app.emit('formAjaxSuccess', $formEl[0], data, response);
        }).catch(error => {
          $formEl.trigger('formajax:error', {
            data,
            error
          });
          app.emit('formAjaxError', $formEl[0], data, error);
        });
      }
      $(document).on('submit change', 'form.form-ajax-submit, form.form-ajax-submit-onchange', onSubmitChange);
    }
    var Form = {
      name: 'form',
      create() {
        const app = this;
        extend$3(app, {
          form: {
            data: {},
            storeFormData: FormData.store.bind(app),
            getFormData: FormData.get.bind(app),
            removeFormData: FormData.remove.bind(app),
            convertToData: formToData.bind(app),
            fillFromData: formFromData.bind(app),
            storage: {
              init: FormStorage.init.bind(app),
              destroy: FormStorage.destroy.bind(app)
            }
          }
        });
      },
      on: {
        init() {
          const app = this;
          initAjaxForm.call(app);
        },
        tabBeforeRemove(tabEl) {
          const app = this;
          $(tabEl).find('.form-store-data').each(formEl => {
            app.form.storage.destroy(formEl);
          });
        },
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.form-store-data').each(formEl => {
            app.form.storage.init(formEl);
          });
        },
        pageBeforeRemove(page) {
          const app = this;
          page.$el.find('.form-store-data').each(formEl => {
            app.form.storage.destroy(formEl);
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.form-store-data').each(formEl => {
            app.form.storage.init(formEl);
          });
        }
      }
    };

    const Input = {
      ignoreTypes: ['checkbox', 'button', 'submit', 'range', 'radio', 'image'],
      createTextareaResizableShadow() {
        const document = getDocument$1();
        const $shadowEl = $(document.createElement('textarea'));
        $shadowEl.addClass('textarea-resizable-shadow');
        $shadowEl.prop({
          disabled: true,
          readonly: true
        });
        Input.textareaResizableShadow = $shadowEl;
      },
      textareaResizableShadow: undefined,
      resizeTextarea(textareaEl) {
        const app = this;
        const window = getWindow$1();
        const $textareaEl = $(textareaEl);
        if (!Input.textareaResizableShadow) {
          Input.createTextareaResizableShadow();
        }
        const $shadowEl = Input.textareaResizableShadow;
        if (!$textareaEl.length) return;
        if (!$textareaEl.hasClass('resizable')) return;
        if (Input.textareaResizableShadow.parents().length === 0) {
          app.$el.append($shadowEl);
        }
        const styles = window.getComputedStyle($textareaEl[0]);
        'padding-top padding-bottom padding-left padding-right margin-left margin-right margin-top margin-bottom width font-size font-family font-style font-weight line-height font-variant text-transform letter-spacing border box-sizing display'.split(' ').forEach(style => {
          let styleValue = styles[style];
          if ('font-size line-height letter-spacing width'.split(' ').indexOf(style) >= 0) {
            styleValue = styleValue.replace(',', '.');
          }
          $shadowEl.css(style, styleValue);
        });
        const currentHeight = $textareaEl[0].clientHeight;
        $shadowEl.val('');
        const initialHeight = $shadowEl[0].scrollHeight;
        $shadowEl.val($textareaEl.val());
        $shadowEl.css('height', 0);
        const scrollHeight = $shadowEl[0].scrollHeight;
        if (currentHeight !== scrollHeight) {
          if (scrollHeight > initialHeight) {
            $textareaEl.css('height', `${scrollHeight}px`);
          } else if (scrollHeight < currentHeight) {
            $textareaEl.css('height', '');
          }
          if (scrollHeight > initialHeight || scrollHeight < currentHeight) {
            $textareaEl.trigger('textarea:resize', {
              initialHeight,
              currentHeight,
              scrollHeight
            });
            app.emit('textareaResize', {
              initialHeight,
              currentHeight,
              scrollHeight
            });
          }
        }
      },
      validate(inputEl) {
        const $inputEl = $(inputEl);
        if (!$inputEl.length) return true;
        const $itemInputEl = $inputEl.parents('.item-input');
        const $inputWrapEl = $inputEl.parents('.input');
        function unsetReadonly() {
          if ($inputEl[0].f7ValidateReadonly) {
            $inputEl[0].readOnly = false;
          }
        }
        function setReadonly() {
          if ($inputEl[0].f7ValidateReadonly) {
            $inputEl[0].readOnly = true;
          }
        }
        unsetReadonly();
        const validity = $inputEl[0].validity;
        const validationMessage = $inputEl.dataset().errorMessage || $inputEl[0].validationMessage || '';
        if (!validity) {
          setReadonly();
          return true;
        }
        if (!validity.valid) {
          let $errorEl = $inputEl.nextAll('.item-input-error-message, .input-error-message');
          if (validationMessage) {
            if ($errorEl.length === 0) {
              $errorEl = $(`<div class="${$inputWrapEl.length ? 'input-error-message' : 'item-input-error-message'}"></div>`);
              $errorEl.insertAfter($inputEl);
            }
            $errorEl.text(validationMessage);
          }
          if ($errorEl.length > 0) {
            $itemInputEl.addClass('item-input-with-error-message');
            $inputWrapEl.addClass('input-with-error-message');
          }
          $itemInputEl.addClass('item-input-invalid');
          $inputWrapEl.addClass('input-invalid');
          $inputEl.addClass('input-invalid');
          setReadonly();
          return false;
        }
        $itemInputEl.removeClass('item-input-invalid item-input-with-error-message');
        $inputWrapEl.removeClass('input-invalid input-with-error-message');
        $inputEl.removeClass('input-invalid');
        setReadonly();
        return true;
      },
      validateInputs(el) {
        const app = this;
        const validates = $(el).find('input, textarea, select').map(inputEl => app.input.validate(inputEl));
        return validates.indexOf(false) < 0;
      },
      focus(inputEl) {
        const $inputEl = $(inputEl);
        const type = $inputEl.attr('type');
        if (Input.ignoreTypes.indexOf(type) >= 0) return;
        $inputEl.parents('.item-input').addClass('item-input-focused');
        $inputEl.parents('.input').addClass('input-focused');
        $inputEl.addClass('input-focused');
      },
      blur(inputEl) {
        const $inputEl = $(inputEl);
        $inputEl.parents('.item-input').removeClass('item-input-focused');
        $inputEl.parents('.input').removeClass('input-focused');
        $inputEl.removeClass('input-focused');
      },
      checkEmptyState(inputEl) {
        const app = this;
        let $inputEl = $(inputEl);
        if (!$inputEl.is('input, select, textarea, .item-input [contenteditable]')) {
          $inputEl = $inputEl.find('input, select, textarea, .item-input [contenteditable]').eq(0);
        }
        if (!$inputEl.length) return;
        const isContentEditable = $inputEl[0].hasAttribute('contenteditable');
        let value;
        if (isContentEditable) {
          if ($inputEl.find('.text-editor-placeholder').length) value = '';else value = $inputEl.html();
        } else {
          value = $inputEl.val();
        }
        const $itemInputEl = $inputEl.parents('.item-input');
        const $inputWrapEl = $inputEl.parents('.input');
        if (value && typeof value === 'string' && value.trim() !== '' || Array.isArray(value) && value.length > 0) {
          $itemInputEl.addClass('item-input-with-value');
          $inputWrapEl.addClass('input-with-value');
          $inputEl.addClass('input-with-value');
          $inputEl.trigger('input:notempty');
          app.emit('inputNotEmpty', $inputEl[0]);
        } else {
          $itemInputEl.removeClass('item-input-with-value');
          $inputWrapEl.removeClass('input-with-value');
          $inputEl.removeClass('input-with-value');
          $inputEl.trigger('input:empty');
          app.emit('inputEmpty', $inputEl[0]);
        }
      },
      scrollIntoView(inputEl, duration, centered, force) {
        if (duration === void 0) {
          duration = 0;
        }
        const $inputEl = $(inputEl);
        const $scrollableEl = $inputEl.parents('.page-content, .panel, .card-expandable .card-content').eq(0);
        if (!$scrollableEl.length) {
          return false;
        }
        const contentHeight = $scrollableEl[0].offsetHeight;
        const contentScrollTop = $scrollableEl[0].scrollTop;
        const contentPaddingTop = parseInt($scrollableEl.css('padding-top'), 10);
        const contentPaddingBottom = parseInt($scrollableEl.css('padding-bottom'), 10);
        const contentOffsetTop = $scrollableEl.offset().top - contentScrollTop;
        const inputOffsetTop = $inputEl.offset().top - contentOffsetTop;
        const inputHeight = $inputEl[0].offsetHeight;
        const min = inputOffsetTop + contentScrollTop - contentPaddingTop;
        const max = inputOffsetTop + contentScrollTop - contentHeight + contentPaddingBottom + inputHeight;
        const centeredPosition = min + (max - min) / 2;
        if (contentScrollTop > min) {
          $scrollableEl.scrollTop(centered ? centeredPosition : min, duration);
          return true;
        }
        if (contentScrollTop < max) {
          $scrollableEl.scrollTop(centered ? centeredPosition : max, duration);
          return true;
        }
        if (force) {
          $scrollableEl.scrollTop(centered ? centeredPosition : max, duration);
        }
        return false;
      },
      init() {
        const app = this;
        const device = getDevice$1();
        const window = getWindow$1();
        const document = getDocument$1();
        Input.createTextareaResizableShadow();
        function onFocus() {
          const inputEl = this;
          if (app.params.input.scrollIntoViewOnFocus) {
            if (device.android) {
              $(window).once('resize', () => {
                if (document && document.activeElement === inputEl) {
                  app.input.scrollIntoView(inputEl, app.params.input.scrollIntoViewDuration, app.params.input.scrollIntoViewCentered, app.params.input.scrollIntoViewAlways);
                }
              });
            } else {
              app.input.scrollIntoView(inputEl, app.params.input.scrollIntoViewDuration, app.params.input.scrollIntoViewCentered, app.params.input.scrollIntoViewAlways);
            }
          }
          app.input.focus(inputEl);
        }
        function onBlur() {
          const $inputEl = $(this);
          const tag = $inputEl[0].nodeName.toLowerCase();
          app.input.blur($inputEl);
          if ($inputEl.dataset().validate || $inputEl.attr('validate') !== null || $inputEl.attr('data-validate-on-blur') !== null) {
            app.input.validate($inputEl);
          }
          // Resize textarea
          if (tag === 'textarea' && $inputEl.hasClass('resizable')) {
            if (Input.textareaResizableShadow) Input.textareaResizableShadow.remove();
          }
        }
        function onChange() {
          const $inputEl = $(this);
          const type = $inputEl.attr('type');
          const tag = $inputEl[0].nodeName.toLowerCase();
          const isContentEditable = $inputEl[0].hasAttribute('contenteditable');
          if (Input.ignoreTypes.indexOf(type) >= 0) return;

          // Check Empty State
          app.input.checkEmptyState($inputEl);
          if (isContentEditable) return;

          // Check validation
          if ($inputEl.attr('data-validate-on-blur') === null && ($inputEl.dataset().validate || $inputEl.attr('validate') !== null)) {
            app.input.validate($inputEl);
          }

          // Resize textarea
          if (tag === 'textarea' && $inputEl.hasClass('resizable')) {
            app.input.resizeTextarea($inputEl);
          }
        }
        function onInvalid(e) {
          const $inputEl = $(this);
          if ($inputEl.attr('data-validate-on-blur') === null && ($inputEl.dataset().validate || $inputEl.attr('validate') !== null)) {
            e.preventDefault();
            app.input.validate($inputEl);
          }
        }
        function clearInput() {
          const $clicked = $(this);
          const $inputEl = $clicked.siblings('input, textarea').eq(0);
          const previousValue = $inputEl.val();
          $inputEl.val('').trigger('input change').focus().trigger('input:clear', previousValue);
          app.emit('inputClear', previousValue);
        }
        function preventDefault(e) {
          e.preventDefault();
        }
        $(document).on('click', '.input-clear-button', clearInput);
        $(document).on('mousedown', '.input-clear-button', preventDefault);
        $(document).on('change input', 'input, textarea, select, .item-input [contenteditable]', onChange, true);
        $(document).on('focus', 'input, textarea, select, .item-input [contenteditable]', onFocus, true);
        $(document).on('blur', 'input, textarea, select, .item-input [contenteditable]', onBlur, true);
        $(document).on('invalid', 'input, textarea, select', onInvalid, true);
      }
    };
    var Input$1 = {
      name: 'input',
      params: {
        input: {
          scrollIntoViewOnFocus: undefined,
          scrollIntoViewCentered: false,
          scrollIntoViewDuration: 0,
          scrollIntoViewAlways: false
        }
      },
      create() {
        const app = this;
        if (typeof app.params.input.scrollIntoViewOnFocus === 'undefined') {
          app.params.input.scrollIntoViewOnFocus = getDevice$1().android;
        }
        bindMethods(app, {
          input: Input
        });
      },
      on: {
        init() {
          const app = this;
          app.input.init();
        },
        tabMounted(tabEl) {
          const app = this;
          const $tabEl = $(tabEl);
          $tabEl.find('.item-input, .input').each(itemInputEl => {
            const $itemInputEl = $(itemInputEl);
            $itemInputEl.find('input, select, textarea, [contenteditable]').each(inputEl => {
              const $inputEl = $(inputEl);
              if (Input.ignoreTypes.indexOf($inputEl.attr('type')) >= 0) return;
              app.input.checkEmptyState($inputEl);
            });
          });
          $tabEl.find('textarea.resizable').each(textareaEl => {
            app.input.resizeTextarea(textareaEl);
          });
        },
        pageInit(page) {
          const app = this;
          const $pageEl = page.$el;
          $pageEl.find('.item-input, .input').each(itemInputEl => {
            const $itemInputEl = $(itemInputEl);
            $itemInputEl.find('input, select, textarea, [contenteditable]').each(inputEl => {
              const $inputEl = $(inputEl);
              if (Input.ignoreTypes.indexOf($inputEl.attr('type')) >= 0) return;
              app.input.checkEmptyState($inputEl);
            });
          });
          $pageEl.find('textarea.resizable').each(textareaEl => {
            app.input.resizeTextarea(textareaEl);
          });
        },
        'panelBreakpoint panelCollapsedBreakpoint panelResize panelOpen panelSwipeOpen resize viewMasterDetailBreakpoint': function onPanelOpen(instance) {
          const app = this;
          if (instance && instance.$el) {
            instance.$el.find('textarea.resizable').each(textareaEl => {
              app.input.resizeTextarea(textareaEl);
            });
          } else {
            $('textarea.resizable').each(textareaEl => {
              app.input.resizeTextarea(textareaEl);
            });
          }
        }
      }
    };

    var Checkbox = {
      name: 'checkbox'
    };

    var Radio = {
      name: 'radio'
    };

    let Toggle$1 = class Toggle extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const toggle = this;
        const support = getSupport$1();
        const defaults = {};

        // Extend defaults with modules params
        toggle.useModulesParams(defaults);
        toggle.params = extend$3(defaults, params);
        const el = toggle.params.el;
        if (!el) return toggle;
        const $el = $(el);
        if ($el.length === 0) return toggle;
        if ($el[0].f7Toggle) return $el[0].f7Toggle;
        const $inputEl = $el.children('input[type="checkbox"]');
        extend$3(toggle, {
          app,
          $el,
          el: $el[0],
          $inputEl,
          inputEl: $inputEl[0],
          disabled: $el.hasClass('disabled') || $inputEl.hasClass('disabled') || $inputEl.attr('disabled') || $inputEl[0].disabled
        });
        Object.defineProperty(toggle, 'checked', {
          enumerable: true,
          configurable: true,
          set(checked) {
            if (!toggle || typeof toggle.$inputEl === 'undefined') return;
            if (toggle.checked === checked) return;
            $inputEl[0].checked = checked;
            toggle.$inputEl.trigger('change');
          },
          get() {
            return $inputEl[0].checked;
          }
        });
        $el[0].f7Toggle = toggle;
        let isTouched;
        const touchesStart = {};
        let isScrolling;
        let touchesDiff;
        let toggleWidth;
        let touchStartTime;
        let touchStartChecked;
        function handleTouchStart(e) {
          if (isTouched || toggle.disabled) return;
          touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
          touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          touchesDiff = 0;
          isTouched = true;
          isScrolling = undefined;
          touchStartTime = now$1();
          touchStartChecked = toggle.checked;
          toggleWidth = $el[0].offsetWidth;
          nextTick$1(() => {
            if (isTouched) {
              $el.addClass('toggle-active-state');
            }
          });
        }
        function handleTouchMove(e) {
          if (!isTouched || toggle.disabled) return;
          const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
          const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
          const inverter = app.rtl ? -1 : 1;
          if (typeof isScrolling === 'undefined') {
            isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x));
          }
          if (isScrolling) {
            isTouched = false;
            return;
          }
          e.preventDefault();
          touchesDiff = pageX - touchesStart.x;
          let changed;
          if (touchesDiff * inverter < 0 && Math.abs(touchesDiff) > toggleWidth / 3 && touchStartChecked) {
            changed = true;
          }
          if (touchesDiff * inverter > 0 && Math.abs(touchesDiff) > toggleWidth / 3 && !touchStartChecked) {
            changed = true;
          }
          if (changed) {
            touchesStart.x = pageX;
            toggle.checked = !touchStartChecked;
            touchStartChecked = !touchStartChecked;
          }
        }
        function handleTouchEnd() {
          if (!isTouched || toggle.disabled) {
            if (isScrolling) $el.removeClass('toggle-active-state');
            isTouched = false;
            return;
          }
          const inverter = app.rtl ? -1 : 1;
          isTouched = false;
          $el.removeClass('toggle-active-state');
          let changed;
          if (now$1() - touchStartTime < 300) {
            if (touchesDiff * inverter < 0 && touchStartChecked) {
              changed = true;
            }
            if (touchesDiff * inverter > 0 && !touchStartChecked) {
              changed = true;
            }
            if (changed) {
              toggle.checked = !touchStartChecked;
            }
          }
        }
        function handleInputChange() {
          toggle.$el.trigger('toggle:change');
          toggle.emit('local::change toggleChange', toggle);
        }
        toggle.attachEvents = function attachEvents() {
          const passive = support.passiveListener ? {
            passive: true
          } : false;
          $el.on(app.touchEvents.start, handleTouchStart, passive);
          app.on('touchmove', handleTouchMove);
          app.on('touchend:passive', handleTouchEnd);
          toggle.$inputEl.on('change', handleInputChange);
        };
        toggle.detachEvents = function detachEvents() {
          const passive = support.passiveListener ? {
            passive: true
          } : false;
          $el.off(app.touchEvents.start, handleTouchStart, passive);
          app.off('touchmove', handleTouchMove);
          app.off('touchend:passive', handleTouchEnd);
          toggle.$inputEl.off('change', handleInputChange);
        };

        // Install Modules
        toggle.useModules();

        // Init
        toggle.init();
      }
      toggle() {
        const toggle = this;
        toggle.checked = !toggle.checked;
      }
      init() {
        const toggle = this;
        toggle.attachEvents();
      }
      destroy() {
        let toggle = this;
        toggle.$el.trigger('toggle:beforedestroy');
        toggle.emit('local::beforeDestroy toggleBeforeDestroy', toggle);
        delete toggle.$el[0].f7Toggle;
        toggle.detachEvents();
        deleteProps$1(toggle);
        toggle = null;
      }
    };

    var Toggle = {
      name: 'toggle',
      create() {
        const app = this;
        app.toggle = ConstructorMethods({
          defaultSelector: '.toggle',
          constructor: Toggle$1,
          app,
          domProp: 'f7Toggle'
        });
      },
      static: {
        Toggle: Toggle$1
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.toggle-init').each(toggleEl => app.toggle.create({
            el: toggleEl
          }));
        },
        tabBeforeRemove(tabEl) {
          $(tabEl).find('.toggle-init').each(toggleEl => {
            if (toggleEl.f7Toggle) toggleEl.f7Toggle.destroy();
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.toggle-init').each(toggleEl => app.toggle.create({
            el: toggleEl
          }));
        },
        pageBeforeRemove(page) {
          page.$el.find('.toggle-init').each(toggleEl => {
            if (toggleEl.f7Toggle) toggleEl.f7Toggle.destroy();
          });
        }
      },
      vnode: {
        'toggle-init': {
          insert(vnode) {
            const app = this;
            const toggleEl = vnode.elm;
            app.toggle.create({
              el: toggleEl
            });
          },
          destroy(vnode) {
            const toggleEl = vnode.elm;
            if (toggleEl.f7Toggle) toggleEl.f7Toggle.destroy();
          }
        }
      }
    };

    let Range$1 = class Range extends Framework7Class {
      constructor(app, params) {
        super(params, [app]);
        const range = this;
        const support = getSupport$1();
        const defaults = {
          el: null,
          inputEl: null,
          dual: false,
          step: 1,
          label: false,
          min: 0,
          max: 100,
          value: 0,
          draggableBar: true,
          vertical: false,
          verticalReversed: false,
          formatLabel: null,
          scale: false,
          scaleSteps: 5,
          scaleSubSteps: 0,
          formatScaleLabel: null,
          limitKnobPosition: app.theme === 'ios'
        };

        // Extend defaults with modules params
        range.useModulesParams(defaults);
        range.params = extend$3(defaults, params);
        const el = range.params.el;
        if (!el) return range;
        const $el = $(el);
        if ($el.length === 0) return range;
        if ($el[0].f7Range) return $el[0].f7Range;
        const dataset = $el.dataset();
        'step min max value scaleSteps scaleSubSteps'.split(' ').forEach(paramName => {
          if (typeof params[paramName] === 'undefined' && typeof dataset[paramName] !== 'undefined') {
            range.params[paramName] = parseFloat(dataset[paramName]);
          }
        });
        'dual label vertical verticalReversed scale'.split(' ').forEach(paramName => {
          if (typeof params[paramName] === 'undefined' && typeof dataset[paramName] !== 'undefined') {
            range.params[paramName] = dataset[paramName];
          }
        });
        if (!range.params.value) {
          if (typeof dataset.value !== 'undefined') range.params.value = dataset.value;
          if (typeof dataset.valueLeft !== 'undefined' && typeof dataset.valueRight !== 'undefined') {
            range.params.value = [parseFloat(dataset.valueLeft), parseFloat(dataset.valueRight)];
          }
        }
        let $inputEl;
        if (!range.params.dual) {
          if (range.params.inputEl) {
            $inputEl = $(range.params.inputEl);
          } else if ($el.find('input[type="range"]').length) {
            $inputEl = $el.find('input[type="range"]').eq(0);
          }
        }
        const {
          dual,
          step,
          label,
          min,
          max,
          value,
          vertical,
          verticalReversed,
          scale,
          scaleSteps,
          scaleSubSteps,
          limitKnobPosition
        } = range.params;
        extend$3(range, {
          app,
          $el,
          el: $el[0],
          $inputEl,
          inputEl: $inputEl ? $inputEl[0] : undefined,
          dual,
          step,
          label,
          min,
          max,
          value,
          previousValue: value,
          vertical,
          verticalReversed,
          scale,
          scaleSteps,
          scaleSubSteps,
          limitKnobPosition
        });
        if ($inputEl) {
          'step min max'.split(' ').forEach(paramName => {
            if (!params[paramName] && $inputEl.attr(paramName)) {
              range.params[paramName] = parseFloat($inputEl.attr(paramName));
              range[paramName] = parseFloat($inputEl.attr(paramName));
            }
          });
          if (typeof $inputEl.val() !== 'undefined') {
            range.params.value = parseFloat($inputEl.val());
            range.value = parseFloat($inputEl.val());
          }
        }

        // Dual
        if (range.dual) {
          $el.addClass('range-slider-dual');
        }
        if (range.label) {
          $el.addClass('range-slider-label');
        }

        // Vertical
        if (range.vertical) {
          $el.addClass('range-slider-vertical');
          if (range.verticalReversed) {
            $el.addClass('range-slider-vertical-reversed');
          }
        } else {
          $el.addClass('range-slider-horizontal');
        }

        // Check for layout
        const $barEl = $('<div class="range-bar"></div>');
        const $barActiveEl = $('<div class="range-bar-active"></div>');
        $barEl.append($barActiveEl);

        // Create Knobs
        // prettier-ignore
        const knobHTML = `
      <div class="range-knob-wrap">
        <div class="range-knob"></div>
        ${range.label ? '<div class="range-knob-label"></div>' : ''}
      </div>
    `;
        const knobs = [$(knobHTML)];
        if (range.dual) {
          knobs.push($(knobHTML));
        }
        $el.append($barEl);
        knobs.forEach($knobEl => {
          $el.append($knobEl);
        });

        // Labels
        const labels = [];
        if (range.label) {
          labels.push(knobs[0].find('.range-knob-label'));
          if (range.dual) {
            labels.push(knobs[1].find('.range-knob-label'));
          }
        }

        // Scale
        let $scaleEl;
        if (range.scale && range.scaleSteps >= 1) {
          $scaleEl = $(`
        <div class="range-scale">
          ${range.renderScale()}
        </div>
      `);
          $el.append($scaleEl);
        }
        extend$3(range, {
          knobs,
          labels,
          $barEl,
          $barActiveEl,
          $scaleEl
        });
        $el[0].f7Range = range;

        // Touch Events
        let isTouched;
        const touchesStart = {};
        let isScrolling;
        let rangeOffset;
        let rangeOffsetLeft;
        let rangeOffsetTop;
        let $touchedKnobEl;
        let dualValueIndex;
        let valueChangedByTouch;
        let targetTouchIdentifier;
        function onTouchChange() {
          valueChangedByTouch = true;
        }
        function handleTouchStart(e) {
          if (isTouched) return;
          if (!range.params.draggableBar) {
            if ($(e.target).closest('.range-knob').length === 0) {
              return;
            }
          }
          valueChangedByTouch = false;
          touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
          touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          if (e.type === 'touchstart') {
            targetTouchIdentifier = e.targetTouches[0].identifier;
          }
          isTouched = true;
          isScrolling = undefined;
          rangeOffset = $el.offset();
          rangeOffsetLeft = rangeOffset.left;
          rangeOffsetTop = rangeOffset.top;
          let progress;
          if (range.vertical) {
            progress = (touchesStart.y - rangeOffsetTop) / range.rangeHeight;
            if (!range.verticalReversed) progress = 1 - progress;
          } else if (range.app.rtl) {
            progress = (rangeOffsetLeft + range.rangeWidth - touchesStart.x) / range.rangeWidth;
          } else {
            progress = (touchesStart.x - rangeOffsetLeft) / range.rangeWidth;
          }
          let newValue = progress * (range.max - range.min) + range.min;
          if (range.dual) {
            if (Math.abs(range.value[0] - newValue) < Math.abs(range.value[1] - newValue)) {
              dualValueIndex = 0;
              $touchedKnobEl = range.knobs[0];
              newValue = [newValue, range.value[1]];
            } else {
              dualValueIndex = 1;
              $touchedKnobEl = range.knobs[1];
              newValue = [range.value[0], newValue];
            }
          } else {
            $touchedKnobEl = range.knobs[0];
            newValue = progress * (range.max - range.min) + range.min;
          }
          nextTick$1(() => {
            if (isTouched) $touchedKnobEl.addClass('range-knob-active-state');
          }, 70);
          range.on('change', onTouchChange);
          range.setValue(newValue, true);
        }
        function handleTouchMove(e) {
          if (!isTouched) return;
          let pageX;
          let pageY;
          if (e.type === 'touchmove') {
            for (let i = 0; i < e.targetTouches.length; i += 1) {
              if (e.targetTouches[i].identifier === targetTouchIdentifier) {
                pageX = e.targetTouches[i].pageX;
                pageY = e.targetTouches[i].pageY;
              }
            }
          } else {
            pageX = e.pageX;
            pageY = e.pageY;
          }
          if (typeof pageX === 'undefined' && typeof pageY === 'undefined') return;
          if (typeof isScrolling === 'undefined' && !range.vertical) {
            isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x));
          }
          if (isScrolling) {
            isTouched = false;
            return;
          }
          e.preventDefault();
          let progress;
          if (range.vertical) {
            progress = (pageY - rangeOffsetTop) / range.rangeHeight;
            if (!range.verticalReversed) progress = 1 - progress;
          } else if (range.app.rtl) {
            progress = (rangeOffsetLeft + range.rangeWidth - pageX) / range.rangeWidth;
          } else {
            progress = (pageX - rangeOffsetLeft) / range.rangeWidth;
          }
          let newValue = progress * (range.max - range.min) + range.min;
          if (range.dual) {
            let leftValue;
            let rightValue;
            if (dualValueIndex === 0) {
              leftValue = newValue;
              rightValue = range.value[1];
              if (leftValue > rightValue) {
                rightValue = leftValue;
              }
            } else {
              leftValue = range.value[0];
              rightValue = newValue;
              if (rightValue < leftValue) {
                leftValue = rightValue;
              }
            }
            newValue = [leftValue, rightValue];
          }
          range.setValue(newValue, true);
        }
        function handleTouchEnd(e) {
          if (e.type === 'touchend') {
            let touchEnded;
            for (let i = 0; i < e.changedTouches.length; i += 1) {
              if (e.changedTouches[i].identifier === targetTouchIdentifier) touchEnded = true;
            }
            if (!touchEnded) return;
          }
          if (!isTouched) {
            if (isScrolling) $touchedKnobEl.removeClass('range-knob-active-state');
            isTouched = false;
            return;
          }
          range.off('change', onTouchChange);
          isTouched = false;
          $touchedKnobEl.removeClass('range-knob-active-state');
          if (valueChangedByTouch && range.$inputEl && !range.dual) {
            range.$inputEl.trigger('change');
          }
          valueChangedByTouch = false;
          if (typeof range.previousValue !== 'undefined') {
            if (range.dual && (range.previousValue[0] !== range.value[0] || range.previousValue[1] !== range.value[1]) || !range.dual && range.previousValue !== range.value) {
              range.$el.trigger('range:changed', range.value);
              range.emit('local::changed rangeChanged', range, range.value);
            }
          }
        }
        function handleResize() {
          range.calcSize();
          range.layout();
        }
        let parentModals;
        let parentPanel;
        let parentPage;
        range.attachEvents = function attachEvents() {
          const passive = support.passiveListener ? {
            passive: true
          } : false;
          range.$el.on(app.touchEvents.start, handleTouchStart, passive);
          app.on('touchmove', handleTouchMove);
          app.on('touchend:passive', handleTouchEnd);
          app.on('tabShow', handleResize);
          app.on('resize', handleResize);
          parentModals = range.$el.parents('.sheet-modal, .actions-modal, .popup, .popover, .login-screen, .dialog, .toast');
          parentModals.on('modal:open', handleResize);
          parentPanel = range.$el.parents('.panel');
          parentPanel.on('panel:open panel:resize', handleResize);
          parentPage = range.$el.parents('.page').eq(0);
          parentPage.on('page:reinit', handleResize);
        };
        range.detachEvents = function detachEvents() {
          const passive = support.passiveListener ? {
            passive: true
          } : false;
          range.$el.off(app.touchEvents.start, handleTouchStart, passive);
          app.off('touchmove', handleTouchMove);
          app.off('touchend:passive', handleTouchEnd);
          app.off('tabShow', handleResize);
          app.off('resize', handleResize);
          if (parentModals) {
            parentModals.off('modal:open', handleResize);
          }
          if (parentPanel) {
            parentPanel.off('panel:open panel:resize', handleResize);
          }
          if (parentPage) {
            parentPage.off('page:reinit', handleResize);
          }
          parentModals = null;
          parentPanel = null;
          parentPage = null;
        };

        // Install Modules
        range.useModules();

        // Init
        range.init();
        return range;
      }
      calcSize() {
        const range = this;
        if (range.vertical) {
          const height = range.$el.outerHeight();
          if (height === 0) return;
          range.rangeHeight = height;
          range.knobHeight = range.knobs[0].outerHeight();
        } else {
          const width = range.$el.outerWidth();
          if (width === 0) return;
          range.rangeWidth = width;
          range.knobWidth = range.knobs[0].outerWidth();
        }
      }
      layout() {
        const range = this;
        const {
          app,
          knobWidth,
          knobHeight,
          rangeWidth,
          rangeHeight,
          min,
          max,
          knobs,
          $barActiveEl,
          value,
          label,
          labels,
          vertical,
          verticalReversed,
          limitKnobPosition
        } = range;
        const knobSize = vertical ? knobHeight : knobWidth;
        const rangeSize = vertical ? rangeHeight : rangeWidth;
        // eslint-disable-next-line
        const positionProperty = vertical ? verticalReversed ? 'top' : 'bottom' : app.rtl ? 'right' : 'left';
        if (range.dual) {
          const progress = [(value[0] - min) / (max - min), (value[1] - min) / (max - min)];
          $barActiveEl.css({
            [positionProperty]: `${progress[0] * 100}%`,
            [vertical ? 'height' : 'width']: `${(progress[1] - progress[0]) * 100}%`
          });
          knobs.forEach(($knobEl, knobIndex) => {
            let startPos = rangeSize * progress[knobIndex];
            if (limitKnobPosition) {
              const realStartPos = rangeSize * progress[knobIndex] - knobSize / 2;
              if (realStartPos < 0) startPos = knobSize / 2;
              if (realStartPos + knobSize > rangeSize) startPos = rangeSize - knobSize / 2;
            }
            $knobEl.css(positionProperty, `${startPos}px`);
            if (label) labels[knobIndex].text(range.formatLabel(value[knobIndex], labels[knobIndex][0]));
          });
        } else {
          const progress = (value - min) / (max - min);
          $barActiveEl.css(vertical ? 'height' : 'width', `${progress * 100}%`);
          let startPos = rangeSize * progress;
          if (limitKnobPosition) {
            const realStartPos = rangeSize * progress - knobSize / 2;
            if (realStartPos < 0) startPos = knobSize / 2;
            if (realStartPos + knobSize > rangeSize) startPos = rangeSize - knobSize / 2;
          }
          knobs[0].css(positionProperty, `${startPos}px`);
          if (label) labels[0].text(range.formatLabel(value, labels[0][0]));
        }
        if (range.dual && value.indexOf(min) >= 0 || !range.dual && value === min) {
          range.$el.addClass('range-slider-min');
        } else {
          range.$el.removeClass('range-slider-min');
        }
        if (range.dual && value.indexOf(max) >= 0 || !range.dual && value === max) {
          range.$el.addClass('range-slider-max');
        } else {
          range.$el.removeClass('range-slider-max');
        }
      }
      setValue(newValue, byTouchMove) {
        const range = this;
        const {
          step,
          min,
          max
        } = range;
        let valueChanged;
        let oldValue;
        if (range.dual) {
          oldValue = [range.value[0], range.value[1]];
          let newValues = newValue;
          if (!Array.isArray(newValues)) newValues = [newValue, newValue];
          if (newValue[0] > newValue[1]) {
            newValues = [newValues[0], newValues[0]];
          }
          newValues = newValues.map(value => Math.max(Math.min(Math.round(value / step) * step, max), min));
          if (newValues[0] === range.value[0] && newValues[1] === range.value[1]) {
            return range;
          }
          newValues.forEach((value, valueIndex) => {
            range.value[valueIndex] = value;
          });
          valueChanged = oldValue[0] !== newValues[0] || oldValue[1] !== newValues[1];
          range.layout();
        } else {
          oldValue = range.value;
          const value = Math.max(Math.min(Math.round(newValue / step) * step, max), min);
          range.value = value;
          range.layout();
          valueChanged = oldValue !== value;
        }
        if (valueChanged) {
          range.previousValue = oldValue;
        }
        // Events
        if (!valueChanged) return range;
        range.$el.trigger('range:change', range.value);
        if (range.$inputEl && !range.dual) {
          range.$inputEl.val(range.value);
          if (!byTouchMove) {
            range.$inputEl.trigger('input change');
          } else {
            range.$inputEl.trigger('input');
          }
        }
        if (!byTouchMove) {
          range.$el.trigger('range:changed', range.value);
          range.emit('local::changed rangeChanged', range, range.value);
        }
        range.emit('local::change rangeChange', range, range.value);
        return range;
      }
      getValue() {
        return this.value;
      }
      formatLabel(value, labelEl) {
        const range = this;
        if (range.params.formatLabel) return range.params.formatLabel.call(range, value, labelEl);
        return value;
      }
      formatScaleLabel(value) {
        const range = this;
        if (range.params.formatScaleLabel) return range.params.formatScaleLabel.call(range, value);
        return value;
      }
      renderScale() {
        const range = this;
        const {
          app,
          verticalReversed,
          vertical
        } = range;

        // eslint-disable-next-line
        const positionProperty = vertical ? verticalReversed ? 'top' : 'bottom' : app.rtl ? 'right' : 'left';
        let html = '';
        Array.from({
          length: range.scaleSteps + 1
        }).forEach((scaleEl, index) => {
          const scaleStepValue = (range.max - range.min) / range.scaleSteps;
          const scaleValue = range.min + scaleStepValue * index;
          const progress = (scaleValue - range.min) / (range.max - range.min);
          html += `<div class="range-scale-step" style="${positionProperty}: ${progress * 100}%">${range.formatScaleLabel(scaleValue)}</div>`;
          if (range.scaleSubSteps && range.scaleSubSteps > 1 && index < range.scaleSteps) {
            Array.from({
              length: range.scaleSubSteps - 1
            }).forEach((subStepEl, subIndex) => {
              const subStep = scaleStepValue / range.scaleSubSteps;
              const scaleSubValue = scaleValue + subStep * (subIndex + 1);
              const subProgress = (scaleSubValue - range.min) / (range.max - range.min);
              html += `<div class="range-scale-step range-scale-substep" style="${positionProperty}: ${subProgress * 100}%"></div>`;
            });
          }
        });
        return html;
      }
      updateScale() {
        const range = this;
        if (!range.scale || range.scaleSteps < 1) {
          if (range.$scaleEl) range.$scaleEl.remove();
          delete range.$scaleEl;
          return;
        }
        if (!range.$scaleEl) {
          range.$scaleEl = $('<div class="range-scale"></div>');
          range.$el.append(range.$scaleEl);
        }
        range.$scaleEl.html(range.renderScale());
      }
      init() {
        const range = this;
        range.calcSize();
        range.layout();
        range.attachEvents();
        return range;
      }
      destroy() {
        let range = this;
        range.$el.trigger('range:beforedestroy');
        range.emit('local::beforeDestroy rangeBeforeDestroy', range);
        delete range.$el[0].f7Range;
        range.detachEvents();
        deleteProps$1(range);
        range = null;
      }
    };

    var Range = {
      name: 'range',
      create() {
        const app = this;
        app.range = extend$3(ConstructorMethods({
          defaultSelector: '.range-slider',
          constructor: Range$1,
          app,
          domProp: 'f7Range'
        }), {
          getValue(el) {
            if (el === void 0) {
              el = '.range-slider';
            }
            const range = app.range.get(el);
            if (range) return range.getValue();
            return undefined;
          },
          setValue(el, value) {
            if (el === void 0) {
              el = '.range-slider';
            }
            const range = app.range.get(el);
            if (range) return range.setValue(value);
            return undefined;
          }
        });
      },
      static: {
        Range: Range$1
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.range-slider-init').each(rangeEl => new Range$1(app, {
            el: rangeEl
          }));
        },
        tabBeforeRemove(tabEl) {
          $(tabEl).find('.range-slider-init').each(rangeEl => {
            if (rangeEl.f7Range) rangeEl.f7Range.destroy();
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.range-slider-init').each(rangeEl => new Range$1(app, {
            el: rangeEl
          }));
        },
        pageBeforeRemove(page) {
          page.$el.find('.range-slider-init').each(rangeEl => {
            if (rangeEl.f7Range) rangeEl.f7Range.destroy();
          });
        }
      },
      vnode: {
        'range-slider-init': {
          insert(vnode) {
            const rangeEl = vnode.elm;
            const app = this;
            app.range.create({
              el: rangeEl
            });
          },
          destroy(vnode) {
            const rangeEl = vnode.elm;
            if (rangeEl.f7Range) rangeEl.f7Range.destroy();
          }
        }
      }
    };

    let Stepper$1 = class Stepper extends Framework7Class {
      constructor(app, params) {
        super(params, [app]);
        const stepper = this;
        const defaults = {
          el: null,
          inputEl: null,
          valueEl: null,
          value: 0,
          formatValue: null,
          step: 1,
          min: 0,
          max: 100,
          watchInput: true,
          autorepeat: false,
          autorepeatDynamic: false,
          wraps: false,
          manualInputMode: false,
          decimalPoint: 4,
          buttonsEndInputMode: true
        };

        // Extend defaults with modules params
        stepper.useModulesParams(defaults);
        stepper.params = extend$3(defaults, params);
        if (stepper.params.value < stepper.params.min) {
          stepper.params.value = stepper.params.min;
        }
        if (stepper.params.value > stepper.params.max) {
          stepper.params.value = stepper.params.max;
        }
        const el = stepper.params.el;
        if (!el) return stepper;
        const $el = $(el);
        if ($el.length === 0) return stepper;
        if ($el[0].f7Stepper) return $el[0].f7Stepper;
        let $inputEl;
        if (stepper.params.inputEl) {
          $inputEl = $(stepper.params.inputEl);
        } else if ($el.find('.stepper-input-wrap').find('input, textarea').length) {
          $inputEl = $el.find('.stepper-input-wrap').find('input, textarea').eq(0);
        }
        if ($inputEl && $inputEl.length) {
          'step min max'.split(' ').forEach(paramName => {
            if (!params[paramName] && $inputEl.attr(paramName)) {
              stepper.params[paramName] = parseFloat($inputEl.attr(paramName));
            }
          });
          const decimalPoint = parseInt(stepper.params.decimalPoint, 10);
          if (Number.isNaN(decimalPoint)) {
            stepper.params.decimalPoint = 0;
          } else {
            stepper.params.decimalPoint = decimalPoint;
          }
          const inputValue = parseFloat($inputEl.val());
          if (typeof params.value === 'undefined' && !Number.isNaN(inputValue) && (inputValue || inputValue === 0)) {
            stepper.params.value = inputValue;
          }
        }
        let $valueEl;
        if (stepper.params.valueEl) {
          $valueEl = $(stepper.params.valueEl);
        } else if ($el.find('.stepper-value').length) {
          $valueEl = $el.find('.stepper-value').eq(0);
        }
        const $buttonPlusEl = $el.find('.stepper-button-plus');
        const $buttonMinusEl = $el.find('.stepper-button-minus');
        const {
          step,
          min,
          max,
          value,
          decimalPoint
        } = stepper.params;
        extend$3(stepper, {
          app,
          $el,
          el: $el[0],
          $buttonPlusEl,
          buttonPlusEl: $buttonPlusEl[0],
          $buttonMinusEl,
          buttonMinusEl: $buttonMinusEl[0],
          $inputEl,
          inputEl: $inputEl ? $inputEl[0] : undefined,
          $valueEl,
          valueEl: $valueEl ? $valueEl[0] : undefined,
          step,
          min,
          max,
          value,
          decimalPoint,
          typeModeChanged: false
        });
        $el[0].f7Stepper = stepper;

        // Handle Events
        const touchesStart = {};
        let isTouched;
        let isScrolling;
        let preventButtonClick;
        let intervalId;
        let timeoutId;
        let autorepeatAction = null;
        let autorepeatInAction = false;
        let manualInput = false;
        function dynamicRepeat(current, progressions, startsIn, progressionStep, repeatEvery, action) {
          clearTimeout(timeoutId);
          timeoutId = setTimeout(() => {
            if (current === 1) {
              preventButtonClick = true;
              autorepeatInAction = true;
            }
            clearInterval(intervalId);
            action();
            intervalId = setInterval(() => {
              action();
            }, repeatEvery);
            if (current < progressions) {
              dynamicRepeat(current + 1, progressions, startsIn, progressionStep, repeatEvery / 2, action);
            }
          }, current === 1 ? startsIn : progressionStep);
        }
        function onTouchStart(e) {
          if (isTouched) return;
          if (manualInput) {
            return;
          }
          if ($(e.target).closest($buttonPlusEl).length) {
            autorepeatAction = 'increment';
          } else if ($(e.target).closest($buttonMinusEl).length) {
            autorepeatAction = 'decrement';
          }
          if (!autorepeatAction) return;
          touchesStart.x = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
          touchesStart.y = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
          isTouched = true;
          isScrolling = undefined;
          const progressions = stepper.params.autorepeatDynamic ? 4 : 1;
          dynamicRepeat(1, progressions, 500, 1000, 300, () => {
            stepper[autorepeatAction]();
          });
        }
        function onTouchMove(e) {
          if (!isTouched) return;
          if (manualInput) {
            return;
          }
          const pageX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
          const pageY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
          if (typeof isScrolling === 'undefined' && !autorepeatInAction) {
            isScrolling = !!(isScrolling || Math.abs(pageY - touchesStart.y) > Math.abs(pageX - touchesStart.x));
          }
          const distance = ((pageX - touchesStart.x) ** 2 + (pageY - touchesStart.y) ** 2) ** 0.5;
          if (isScrolling || distance > 20) {
            isTouched = false;
            clearTimeout(timeoutId);
            clearInterval(intervalId);
          }
        }
        function onTouchEnd() {
          clearTimeout(timeoutId);
          clearInterval(intervalId);
          autorepeatAction = null;
          autorepeatInAction = false;
          isTouched = false;
        }
        function onMinusClick() {
          if (manualInput) {
            if (stepper.params.buttonsEndInputMode) {
              manualInput = false;
              stepper.endTypeMode(true);
            }
            return;
          }
          if (preventButtonClick) {
            preventButtonClick = false;
            return;
          }
          stepper.decrement(true);
        }
        function onPlusClick() {
          if (manualInput) {
            if (stepper.params.buttonsEndInputMode) {
              manualInput = false;
              stepper.endTypeMode(true);
            }
            return;
          }
          if (preventButtonClick) {
            preventButtonClick = false;
            return;
          }
          stepper.increment(true);
        }
        function onInputClick(e) {
          if (!e.target.readOnly && stepper.params.manualInputMode) {
            manualInput = true;
            if (typeof e.target.selectionStart === 'number') {
              e.target.selectionStart = e.target.value.length;
              e.target.selectionEnd = e.target.value.length;
            }
          }
        }
        function onInputKey(e) {
          if (e.keyCode === 13 || e.which === 13) {
            e.preventDefault();
            manualInput = false;
            stepper.endTypeMode();
          }
        }
        function onInputBlur() {
          manualInput = false;
          stepper.endTypeMode(true);
        }
        function onInput(e) {
          if (manualInput) {
            stepper.typeValue(e.target.value);
            return;
          }
          if (e.detail && e.detail.sentByF7Stepper) return;
          stepper.setValue(e.target.value, true);
        }
        stepper.attachEvents = function attachEvents() {
          $buttonMinusEl.on('click', onMinusClick);
          $buttonPlusEl.on('click', onPlusClick);
          if (stepper.params.watchInput && $inputEl && $inputEl.length) {
            $inputEl.on('input', onInput);
            $inputEl.on('click', onInputClick);
            $inputEl.on('blur', onInputBlur);
            $inputEl.on('keyup', onInputKey);
          }
          if (stepper.params.autorepeat) {
            app.on('touchstart:passive', onTouchStart);
            app.on('touchmove:active', onTouchMove);
            app.on('touchend:passive', onTouchEnd);
          }
        };
        stepper.detachEvents = function detachEvents() {
          $buttonMinusEl.off('click', onMinusClick);
          $buttonPlusEl.off('click', onPlusClick);
          if (stepper.params.watchInput && $inputEl && $inputEl.length) {
            $inputEl.off('input', onInput);
            $inputEl.off('click', onInputClick);
            $inputEl.off('blur', onInputBlur);
            $inputEl.off('keyup', onInputKey);
          }
        };

        // Install Modules
        stepper.useModules();

        // Init
        stepper.init();
        return stepper;
      }
      minus() {
        return this.decrement();
      }
      plus() {
        return this.increment();
      }
      decrement() {
        const stepper = this;
        return stepper.setValue(stepper.value - stepper.step, false, true);
      }
      increment() {
        const stepper = this;
        return stepper.setValue(stepper.value + stepper.step, false, true);
      }
      setValue(newValue, forceUpdate, withWraps) {
        const stepper = this;
        const {
          step,
          min,
          max
        } = stepper;
        const oldValue = stepper.value;
        let value = Math.round(newValue / step) * step;
        if (stepper.params.wraps && withWraps) {
          if (value > max) value = min;
          if (value < min) value = max;
        } else {
          value = Math.max(Math.min(value, max), min);
        }
        if (Number.isNaN(value)) {
          value = oldValue;
        }
        stepper.value = value;
        const valueChanged = oldValue !== value;

        // Events
        if (!valueChanged && !forceUpdate) return stepper;
        stepper.$el.trigger('stepper:change', stepper.value);
        const formattedValue = stepper.formatValue(stepper.value);
        if (stepper.$inputEl && stepper.$inputEl.length) {
          stepper.$inputEl.val(formattedValue);
          stepper.$inputEl.trigger('input change', {
            sentByF7Stepper: true
          });
        }
        if (stepper.$valueEl && stepper.$valueEl.length) {
          stepper.$valueEl.html(formattedValue);
        }
        stepper.emit('local::change stepperChange', stepper, stepper.value);
        return stepper;
      }
      endTypeMode(noBlur) {
        const stepper = this;
        const {
          min,
          max
        } = stepper;
        let value = parseFloat(stepper.value);
        if (Number.isNaN(value)) value = 0;
        value = Math.max(Math.min(value, max), min);
        stepper.value = value;
        if (!stepper.typeModeChanged) {
          if (stepper.$inputEl && stepper.$inputEl.length && !noBlur) {
            stepper.$inputEl.blur();
          }
          return stepper;
        }
        stepper.typeModeChanged = false;
        stepper.$el.trigger('stepper:change', stepper.value);
        const formattedValue = stepper.formatValue(stepper.value);
        if (stepper.$inputEl && stepper.$inputEl.length) {
          stepper.$inputEl.val(formattedValue);
          stepper.$inputEl.trigger('input change', {
            sentByF7Stepper: true
          });
          if (!noBlur) stepper.$inputEl.blur();
        }
        if (stepper.$valueEl && stepper.$valueEl.length) {
          stepper.$valueEl.html(formattedValue);
        }
        stepper.emit('local::change stepperChange', stepper, stepper.value);
        return stepper;
      }
      typeValue(value) {
        const stepper = this;
        stepper.typeModeChanged = true;
        let inputTxt = String(value);
        if (inputTxt.length === 1 && inputTxt === '-') return stepper;
        if (inputTxt.lastIndexOf('.') + 1 === inputTxt.length || inputTxt.lastIndexOf(',') + 1 === inputTxt.length) {
          if (inputTxt.lastIndexOf('.') !== inputTxt.indexOf('.') || inputTxt.lastIndexOf(',') !== inputTxt.indexOf(',')) {
            inputTxt = inputTxt.slice(0, -1);
            stepper.value = inputTxt;
            stepper.$inputEl.val(stepper.value);
            return stepper;
          }
        } else {
          let newValue = parseFloat(inputTxt.replace(',', '.'));
          if (newValue === 0) {
            stepper.value = inputTxt.replace(',', '.');
            stepper.$inputEl.val(stepper.value);
            return stepper;
          }
          if (Number.isNaN(newValue)) {
            stepper.value = 0;
            stepper.$inputEl.val(stepper.value);
            return stepper;
          }
          const powVal = 10 ** stepper.params.decimalPoint;
          newValue = Math.round(newValue * powVal).toFixed(stepper.params.decimalPoint + 1) / powVal;
          stepper.value = parseFloat(String(newValue).replace(',', '.'));
          stepper.$inputEl.val(stepper.value);
          return stepper;
        }
        stepper.value = inputTxt;
        stepper.$inputEl.val(inputTxt);
        return stepper;
      }
      getValue() {
        return this.value;
      }
      formatValue(value) {
        const stepper = this;
        if (!stepper.params.formatValue) return value;
        return stepper.params.formatValue.call(stepper, value);
      }
      init() {
        const stepper = this;
        stepper.attachEvents();
        if (stepper.$valueEl && stepper.$valueEl.length) {
          const formattedValue = stepper.formatValue(stepper.value);
          stepper.$valueEl.html(formattedValue);
        }
        return stepper;
      }
      destroy() {
        let stepper = this;
        stepper.$el.trigger('stepper:beforedestroy');
        stepper.emit('local::beforeDestroy stepperBeforeDestroy', stepper);
        delete stepper.$el[0].f7Stepper;
        stepper.detachEvents();
        deleteProps$1(stepper);
        stepper = null;
      }
    };

    var Stepper = {
      name: 'stepper',
      create() {
        const app = this;
        app.stepper = extend$3(ConstructorMethods({
          defaultSelector: '.stepper',
          constructor: Stepper$1,
          app,
          domProp: 'f7Stepper'
        }), {
          getValue(el) {
            if (el === void 0) {
              el = '.stepper';
            }
            const stepper = app.stepper.get(el);
            if (stepper) return stepper.getValue();
            return undefined;
          },
          setValue(el, value) {
            if (el === void 0) {
              el = '.stepper';
            }
            const stepper = app.stepper.get(el);
            if (stepper) return stepper.setValue(value);
            return undefined;
          }
        });
      },
      static: {
        Stepper: Stepper$1
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.stepper-init').each(stepperEl => {
            const dataset = $(stepperEl).dataset();
            app.stepper.create(extend$3({
              el: stepperEl
            }, dataset || {}));
          });
        },
        tabBeforeRemove(tabEl) {
          $(tabEl).find('.stepper-init').each(stepperEl => {
            if (stepperEl.f7Stepper) stepperEl.f7Stepper.destroy();
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.stepper-init').each(stepperEl => {
            const dataset = $(stepperEl).dataset();
            app.stepper.create(extend$3({
              el: stepperEl
            }, dataset || {}));
          });
        },
        pageBeforeRemove(page) {
          page.$el.find('.stepper-init').each(stepperEl => {
            if (stepperEl.f7Stepper) stepperEl.f7Stepper.destroy();
          });
        }
      },
      vnode: {
        'stepper-init': {
          insert(vnode) {
            const app = this;
            const stepperEl = vnode.elm;
            const dataset = $(stepperEl).dataset();
            app.stepper.create(extend$3({
              el: stepperEl
            }, dataset || {}));
          },
          destroy(vnode) {
            const stepperEl = vnode.elm;
            if (stepperEl.f7Stepper) stepperEl.f7Stepper.destroy();
          }
        }
      }
    };

    /* eslint no-control-regex: "off" */
    const defaultDiacriticsRemovalap = [{
      base: 'A',
      letters: '\u0041\u24B6\uFF21\u00C0\u00C1\u00C2\u1EA6\u1EA4\u1EAA\u1EA8\u00C3\u0100\u0102\u1EB0\u1EAE\u1EB4\u1EB2\u0226\u01E0\u00C4\u01DE\u1EA2\u00C5\u01FA\u01CD\u0200\u0202\u1EA0\u1EAC\u1EB6\u1E00\u0104\u023A\u2C6F'
    }, {
      base: 'AA',
      letters: '\uA732'
    }, {
      base: 'AE',
      letters: '\u00C6\u01FC\u01E2'
    }, {
      base: 'AO',
      letters: '\uA734'
    }, {
      base: 'AU',
      letters: '\uA736'
    }, {
      base: 'AV',
      letters: '\uA738\uA73A'
    }, {
      base: 'AY',
      letters: '\uA73C'
    }, {
      base: 'B',
      letters: '\u0042\u24B7\uFF22\u1E02\u1E04\u1E06\u0243\u0182\u0181'
    }, {
      base: 'C',
      letters: '\u0043\u24B8\uFF23\u0106\u0108\u010A\u010C\u00C7\u1E08\u0187\u023B\uA73E'
    }, {
      base: 'D',
      letters: '\u0044\u24B9\uFF24\u1E0A\u010E\u1E0C\u1E10\u1E12\u1E0E\u0110\u018B\u018A\u0189\uA779'
    }, {
      base: 'DZ',
      letters: '\u01F1\u01C4'
    }, {
      base: 'Dz',
      letters: '\u01F2\u01C5'
    }, {
      base: 'E',
      letters: '\u0045\u24BA\uFF25\u00C8\u00C9\u00CA\u1EC0\u1EBE\u1EC4\u1EC2\u1EBC\u0112\u1E14\u1E16\u0114\u0116\u00CB\u1EBA\u011A\u0204\u0206\u1EB8\u1EC6\u0228\u1E1C\u0118\u1E18\u1E1A\u0190\u018E'
    }, {
      base: 'F',
      letters: '\u0046\u24BB\uFF26\u1E1E\u0191\uA77B'
    }, {
      base: 'G',
      letters: '\u0047\u24BC\uFF27\u01F4\u011C\u1E20\u011E\u0120\u01E6\u0122\u01E4\u0193\uA7A0\uA77D\uA77E'
    }, {
      base: 'H',
      letters: '\u0048\u24BD\uFF28\u0124\u1E22\u1E26\u021E\u1E24\u1E28\u1E2A\u0126\u2C67\u2C75\uA78D'
    }, {
      base: 'I',
      letters: '\u0049\u24BE\uFF29\u00CC\u00CD\u00CE\u0128\u012A\u012C\u0130\u00CF\u1E2E\u1EC8\u01CF\u0208\u020A\u1ECA\u012E\u1E2C\u0197'
    }, {
      base: 'J',
      letters: '\u004A\u24BF\uFF2A\u0134\u0248'
    }, {
      base: 'K',
      letters: '\u004B\u24C0\uFF2B\u1E30\u01E8\u1E32\u0136\u1E34\u0198\u2C69\uA740\uA742\uA744\uA7A2'
    }, {
      base: 'L',
      letters: '\u004C\u24C1\uFF2C\u013F\u0139\u013D\u1E36\u1E38\u013B\u1E3C\u1E3A\u0141\u023D\u2C62\u2C60\uA748\uA746\uA780'
    }, {
      base: 'LJ',
      letters: '\u01C7'
    }, {
      base: 'Lj',
      letters: '\u01C8'
    }, {
      base: 'M',
      letters: '\u004D\u24C2\uFF2D\u1E3E\u1E40\u1E42\u2C6E\u019C'
    }, {
      base: 'N',
      letters: '\u004E\u24C3\uFF2E\u01F8\u0143\u00D1\u1E44\u0147\u1E46\u0145\u1E4A\u1E48\u0220\u019D\uA790\uA7A4'
    }, {
      base: 'NJ',
      letters: '\u01CA'
    }, {
      base: 'Nj',
      letters: '\u01CB'
    }, {
      base: 'O',
      letters: '\u004F\u24C4\uFF2F\u00D2\u00D3\u00D4\u1ED2\u1ED0\u1ED6\u1ED4\u00D5\u1E4C\u022C\u1E4E\u014C\u1E50\u1E52\u014E\u022E\u0230\u00D6\u022A\u1ECE\u0150\u01D1\u020C\u020E\u01A0\u1EDC\u1EDA\u1EE0\u1EDE\u1EE2\u1ECC\u1ED8\u01EA\u01EC\u00D8\u01FE\u0186\u019F\uA74A\uA74C'
    }, {
      base: 'OI',
      letters: '\u01A2'
    }, {
      base: 'OO',
      letters: '\uA74E'
    }, {
      base: 'OU',
      letters: '\u0222'
    }, {
      base: 'OE',
      letters: '\u008C\u0152'
    }, {
      base: 'oe',
      letters: '\u009C\u0153'
    }, {
      base: 'P',
      letters: '\u0050\u24C5\uFF30\u1E54\u1E56\u01A4\u2C63\uA750\uA752\uA754'
    }, {
      base: 'Q',
      letters: '\u0051\u24C6\uFF31\uA756\uA758\u024A'
    }, {
      base: 'R',
      letters: '\u0052\u24C7\uFF32\u0154\u1E58\u0158\u0210\u0212\u1E5A\u1E5C\u0156\u1E5E\u024C\u2C64\uA75A\uA7A6\uA782'
    }, {
      base: 'S',
      letters: '\u0053\u24C8\uFF33\u1E9E\u015A\u1E64\u015C\u1E60\u0160\u1E66\u1E62\u1E68\u0218\u015E\u2C7E\uA7A8\uA784'
    }, {
      base: 'T',
      letters: '\u0054\u24C9\uFF34\u1E6A\u0164\u1E6C\u021A\u0162\u1E70\u1E6E\u0166\u01AC\u01AE\u023E\uA786'
    }, {
      base: 'TZ',
      letters: '\uA728'
    }, {
      base: 'U',
      letters: '\u0055\u24CA\uFF35\u00D9\u00DA\u00DB\u0168\u1E78\u016A\u1E7A\u016C\u00DC\u01DB\u01D7\u01D5\u01D9\u1EE6\u016E\u0170\u01D3\u0214\u0216\u01AF\u1EEA\u1EE8\u1EEE\u1EEC\u1EF0\u1EE4\u1E72\u0172\u1E76\u1E74\u0244'
    }, {
      base: 'V',
      letters: '\u0056\u24CB\uFF36\u1E7C\u1E7E\u01B2\uA75E\u0245'
    }, {
      base: 'VY',
      letters: '\uA760'
    }, {
      base: 'W',
      letters: '\u0057\u24CC\uFF37\u1E80\u1E82\u0174\u1E86\u1E84\u1E88\u2C72'
    }, {
      base: 'X',
      letters: '\u0058\u24CD\uFF38\u1E8A\u1E8C'
    }, {
      base: 'Y',
      letters: '\u0059\u24CE\uFF39\u1EF2\u00DD\u0176\u1EF8\u0232\u1E8E\u0178\u1EF6\u1EF4\u01B3\u024E\u1EFE'
    }, {
      base: 'Z',
      letters: '\u005A\u24CF\uFF3A\u0179\u1E90\u017B\u017D\u1E92\u1E94\u01B5\u0224\u2C7F\u2C6B\uA762'
    }, {
      base: 'a',
      letters: '\u0061\u24D0\uFF41\u1E9A\u00E0\u00E1\u00E2\u1EA7\u1EA5\u1EAB\u1EA9\u00E3\u0101\u0103\u1EB1\u1EAF\u1EB5\u1EB3\u0227\u01E1\u00E4\u01DF\u1EA3\u00E5\u01FB\u01CE\u0201\u0203\u1EA1\u1EAD\u1EB7\u1E01\u0105\u2C65\u0250'
    }, {
      base: 'aa',
      letters: '\uA733'
    }, {
      base: 'ae',
      letters: '\u00E6\u01FD\u01E3'
    }, {
      base: 'ao',
      letters: '\uA735'
    }, {
      base: 'au',
      letters: '\uA737'
    }, {
      base: 'av',
      letters: '\uA739\uA73B'
    }, {
      base: 'ay',
      letters: '\uA73D'
    }, {
      base: 'b',
      letters: '\u0062\u24D1\uFF42\u1E03\u1E05\u1E07\u0180\u0183\u0253'
    }, {
      base: 'c',
      letters: '\u0063\u24D2\uFF43\u0107\u0109\u010B\u010D\u00E7\u1E09\u0188\u023C\uA73F\u2184'
    }, {
      base: 'd',
      letters: '\u0064\u24D3\uFF44\u1E0B\u010F\u1E0D\u1E11\u1E13\u1E0F\u0111\u018C\u0256\u0257\uA77A'
    }, {
      base: 'dz',
      letters: '\u01F3\u01C6'
    }, {
      base: 'e',
      letters: '\u0065\u24D4\uFF45\u00E8\u00E9\u00EA\u1EC1\u1EBF\u1EC5\u1EC3\u1EBD\u0113\u1E15\u1E17\u0115\u0117\u00EB\u1EBB\u011B\u0205\u0207\u1EB9\u1EC7\u0229\u1E1D\u0119\u1E19\u1E1B\u0247\u025B\u01DD'
    }, {
      base: 'f',
      letters: '\u0066\u24D5\uFF46\u1E1F\u0192\uA77C'
    }, {
      base: 'g',
      letters: '\u0067\u24D6\uFF47\u01F5\u011D\u1E21\u011F\u0121\u01E7\u0123\u01E5\u0260\uA7A1\u1D79\uA77F'
    }, {
      base: 'h',
      letters: '\u0068\u24D7\uFF48\u0125\u1E23\u1E27\u021F\u1E25\u1E29\u1E2B\u1E96\u0127\u2C68\u2C76\u0265'
    }, {
      base: 'hv',
      letters: '\u0195'
    }, {
      base: 'i',
      letters: '\u0069\u24D8\uFF49\u00EC\u00ED\u00EE\u0129\u012B\u012D\u00EF\u1E2F\u1EC9\u01D0\u0209\u020B\u1ECB\u012F\u1E2D\u0268\u0131'
    }, {
      base: 'j',
      letters: '\u006A\u24D9\uFF4A\u0135\u01F0\u0249'
    }, {
      base: 'k',
      letters: '\u006B\u24DA\uFF4B\u1E31\u01E9\u1E33\u0137\u1E35\u0199\u2C6A\uA741\uA743\uA745\uA7A3'
    }, {
      base: 'l',
      letters: '\u006C\u24DB\uFF4C\u0140\u013A\u013E\u1E37\u1E39\u013C\u1E3D\u1E3B\u017F\u0142\u019A\u026B\u2C61\uA749\uA781\uA747'
    }, {
      base: 'lj',
      letters: '\u01C9'
    }, {
      base: 'm',
      letters: '\u006D\u24DC\uFF4D\u1E3F\u1E41\u1E43\u0271\u026F'
    }, {
      base: 'n',
      letters: '\u006E\u24DD\uFF4E\u01F9\u0144\u00F1\u1E45\u0148\u1E47\u0146\u1E4B\u1E49\u019E\u0272\u0149\uA791\uA7A5'
    }, {
      base: 'nj',
      letters: '\u01CC'
    }, {
      base: 'o',
      letters: '\u006F\u24DE\uFF4F\u00F2\u00F3\u00F4\u1ED3\u1ED1\u1ED7\u1ED5\u00F5\u1E4D\u022D\u1E4F\u014D\u1E51\u1E53\u014F\u022F\u0231\u00F6\u022B\u1ECF\u0151\u01D2\u020D\u020F\u01A1\u1EDD\u1EDB\u1EE1\u1EDF\u1EE3\u1ECD\u1ED9\u01EB\u01ED\u00F8\u01FF\u0254\uA74B\uA74D\u0275'
    }, {
      base: 'oi',
      letters: '\u01A3'
    }, {
      base: 'ou',
      letters: '\u0223'
    }, {
      base: 'oo',
      letters: '\uA74F'
    }, {
      base: 'p',
      letters: '\u0070\u24DF\uFF50\u1E55\u1E57\u01A5\u1D7D\uA751\uA753\uA755'
    }, {
      base: 'q',
      letters: '\u0071\u24E0\uFF51\u024B\uA757\uA759'
    }, {
      base: 'r',
      letters: '\u0072\u24E1\uFF52\u0155\u1E59\u0159\u0211\u0213\u1E5B\u1E5D\u0157\u1E5F\u024D\u027D\uA75B\uA7A7\uA783'
    }, {
      base: 's',
      letters: '\u0073\u24E2\uFF53\u00DF\u015B\u1E65\u015D\u1E61\u0161\u1E67\u1E63\u1E69\u0219\u015F\u023F\uA7A9\uA785\u1E9B'
    }, {
      base: 't',
      letters: '\u0074\u24E3\uFF54\u1E6B\u1E97\u0165\u1E6D\u021B\u0163\u1E71\u1E6F\u0167\u01AD\u0288\u2C66\uA787'
    }, {
      base: 'tz',
      letters: '\uA729'
    }, {
      base: 'u',
      letters: '\u0075\u24E4\uFF55\u00F9\u00FA\u00FB\u0169\u1E79\u016B\u1E7B\u016D\u00FC\u01DC\u01D8\u01D6\u01DA\u1EE7\u016F\u0171\u01D4\u0215\u0217\u01B0\u1EEB\u1EE9\u1EEF\u1EED\u1EF1\u1EE5\u1E73\u0173\u1E77\u1E75\u0289'
    }, {
      base: 'v',
      letters: '\u0076\u24E5\uFF56\u1E7D\u1E7F\u028B\uA75F\u028C'
    }, {
      base: 'vy',
      letters: '\uA761'
    }, {
      base: 'w',
      letters: '\u0077\u24E6\uFF57\u1E81\u1E83\u0175\u1E87\u1E85\u1E98\u1E89\u2C73'
    }, {
      base: 'x',
      letters: '\u0078\u24E7\uFF58\u1E8B\u1E8D'
    }, {
      base: 'y',
      letters: '\u0079\u24E8\uFF59\u1EF3\u00FD\u0177\u1EF9\u0233\u1E8F\u00FF\u1EF7\u1E99\u1EF5\u01B4\u024F\u1EFF'
    }, {
      base: 'z',
      letters: '\u007A\u24E9\uFF5A\u017A\u1E91\u017C\u017E\u1E93\u1E95\u01B6\u0225\u0240\u2C6C\uA763'
    }];
    const diacriticsMap = {};
    for (let i = 0; i < defaultDiacriticsRemovalap.length; i += 1) {
      const letters = defaultDiacriticsRemovalap[i].letters;
      for (let j = 0; j < letters.length; j += 1) {
        diacriticsMap[letters[j]] = defaultDiacriticsRemovalap[i].base;
      }
    }
    function removeDiacritics(str) {
      return str.replace(/[^\u0000-\u007E]/g, a => diacriticsMap[a] || a);
    }

    let SmartSelect$1 = class SmartSelect extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const ss = this;
        const defaults = extend$3({
          on: {}
        }, app.params.smartSelect);

        // Extend defaults with modules params
        ss.useModulesParams(defaults);
        ss.params = extend$3({}, defaults, params);
        ss.app = app;
        const $el = $(ss.params.el).eq(0);
        if ($el.length === 0) return ss;
        if ($el[0].f7SmartSelect) return $el[0].f7SmartSelect;
        const $selectEl = $el.find('select').eq(0);
        if ($selectEl.length === 0) return ss;
        let $valueEl;
        if (ss.params.setValueText) {
          $valueEl = $(ss.params.valueEl);
          if ($valueEl.length === 0) {
            $valueEl = $el.find('.item-after');
          }
          if ($valueEl.length === 0) {
            $valueEl = $('<div class="item-after"></div>');
            $valueEl.insertAfter($el.find('.item-title'));
          }
        }

        // Url
        let url = params.url;
        if (!url) {
          if ($el.attr('href') && $el.attr('href') !== '#') url = $el.attr('href');else if ($selectEl.attr('name')) url = `${$selectEl.attr('name').toLowerCase()}-select/`;
        }
        if (!url) url = ss.params.url;
        const multiple = $selectEl[0].multiple;
        const inputType = multiple ? 'checkbox' : 'radio';
        const selectId = id$1();
        extend$3(ss, {
          $el,
          el: $el[0],
          $selectEl,
          selectEl: $selectEl[0],
          $valueEl,
          valueEl: $valueEl && $valueEl[0],
          url,
          multiple,
          inputType,
          id: selectId,
          inputName: `${inputType}-${selectId}`,
          selectName: $selectEl.attr('name'),
          maxLength: $selectEl.attr('maxlength') || params.maxLength
        });
        $el[0].f7SmartSelect = ss;

        // Events
        function onClick() {
          ss.open();
        }
        function onChange() {
          const value = ss.$selectEl.val();
          ss.$el.trigger('smartselect:change', value);
          ss.emit('local::change smartSelectChange', ss, value);
          if (ss.vl) {
            ss.vl.clearCache();
          }
          ss.setValueText();
        }
        ss.attachEvents = function attachEvents() {
          $el.on('click', onClick);
          $el.on('change', 'select', onChange);
        };
        ss.detachEvents = function detachEvents() {
          $el.off('click', onClick);
          $el.off('change', 'select', onChange);
        };
        function handleInputChange() {
          let optionEl;
          let text;
          const inputEl = this;
          const value = inputEl.value;
          let optionText = [];
          let displayAs;
          if (inputEl.type === 'checkbox') {
            for (let i = 0; i < ss.selectEl.options.length; i += 1) {
              optionEl = ss.selectEl.options[i];
              if (optionEl.value === value) {
                optionEl.selected = inputEl.checked;
              }
              if (optionEl.selected) {
                displayAs = optionEl.dataset ? optionEl.dataset.displayAs : $(optionEl).data('display-value-as');
                text = displayAs && typeof displayAs !== 'undefined' ? displayAs : optionEl.textContent;
                optionText.push(text.trim());
              }
            }
            if (ss.maxLength) {
              ss.checkMaxLength();
            }
          } else {
            optionEl = ss.$selectEl.find(`option[value="${value}"]`)[0];
            if (!optionEl) {
              optionEl = ss.$selectEl.find('option').filter(optEl => optEl.value === value)[0];
            }
            displayAs = optionEl.dataset ? optionEl.dataset.displayAs : $(optionEl).data('display-as');
            text = displayAs && typeof displayAs !== 'undefined' ? displayAs : optionEl.textContent;
            optionText = [text];
            ss.selectEl.value = value;
          }
          ss.$selectEl.trigger('change');
          if (ss.params.setValueText) {
            ss.formatValueTextContent(optionText);
          }
          if (ss.params.closeOnSelect && ss.inputType === 'radio') {
            ss.close();
          }
        }
        ss.attachInputsEvents = function attachInputsEvents() {
          ss.$containerEl.on('change', 'input[type="checkbox"], input[type="radio"]', handleInputChange);
        };
        ss.detachInputsEvents = function detachInputsEvents() {
          ss.$containerEl.off('change', 'input[type="checkbox"], input[type="radio"]', handleInputChange);
        };

        // Install Modules
        ss.useModules();

        // Init
        ss.init();
        return ss;
      }
      setValue(value) {
        const ss = this;
        let newValue = value;
        let optionText = [];
        let optionEl;
        let displayAs;
        let text;
        if (ss.multiple) {
          if (!Array.isArray(newValue)) newValue = [newValue];
          for (let i = 0; i < ss.selectEl.options.length; i += 1) {
            optionEl = ss.selectEl.options[i];
            if (newValue.indexOf(optionEl.value) >= 0) {
              optionEl.selected = true;
            } else {
              optionEl.selected = false;
            }
            if (optionEl.selected) {
              displayAs = optionEl.dataset ? optionEl.dataset.displayAs : $(optionEl).data('display-value-as');
              text = displayAs && typeof displayAs !== 'undefined' ? displayAs : optionEl.textContent;
              optionText.push(text.trim());
            }
          }
        } else {
          optionEl = ss.$selectEl.find(`option[value="${newValue}"]`)[0];
          if (optionEl) {
            displayAs = optionEl.dataset ? optionEl.dataset.displayAs : $(optionEl).data('display-as');
            text = displayAs && typeof displayAs !== 'undefined' ? displayAs : optionEl.textContent;
            optionText = [text];
          }
          ss.selectEl.value = newValue;
        }
        if (ss.params.setValueText) {
          ss.formatValueTextContent(optionText);
        }
        ss.$selectEl.trigger('change');
        return ss;
      }
      unsetValue() {
        const ss = this;
        if (ss.params.setValueText) {
          ss.formatValueTextContent([]);
        }
        ss.$selectEl.find('option').each(optionEl => {
          optionEl.selected = false;
          optionEl.checked = false;
        });
        ss.$selectEl[0].value = null;
        if (ss.$containerEl) {
          ss.$containerEl.find(`input[name="${ss.inputName}"][type="checkbox"], input[name="${ss.inputName}"][type="radio"]`).prop('checked', false);
        }
        ss.$selectEl.trigger('change');
      }
      getValue() {
        const ss = this;
        return ss.$selectEl.val();
      }
      get view() {
        const {
          params,
          $el
        } = this;
        let view;
        if (params.view) {
          view = params.view;
        }
        if (!view) {
          view = $el.parents('.view').length && $el.parents('.view')[0].f7View;
        }
        if (!view && params.openIn === 'page') {
          throw Error('Smart Select requires initialized View');
        }
        return view;
      }
      checkMaxLength() {
        const ss = this;
        const $containerEl = ss.$containerEl;
        if (ss.selectEl.selectedOptions.length >= ss.maxLength) {
          $containerEl.find('input[type="checkbox"]').each(inputEl => {
            if (!inputEl.checked) {
              $(inputEl).parents('li').addClass('disabled');
            } else {
              $(inputEl).parents('li').removeClass('disabled');
            }
          });
        } else {
          $containerEl.find('.disabled').removeClass('disabled');
        }
      }
      formatValueTextContent(values) {
        const ss = this;
        const valueFormatted = ss.formatValueText(values);
        if (valueFormatted.includes('<') && valueFormatted.includes('>')) {
          ss.$valueEl.html(valueFormatted);
        } else {
          ss.$valueEl.text(valueFormatted);
        }
      }
      formatValueText(values) {
        const ss = this;
        let textValue;
        if (ss.params.formatValueText) {
          textValue = ss.params.formatValueText.call(ss, values, ss);
        } else {
          textValue = values.join(', ');
        }
        return textValue;
      }
      setValueText(value) {
        const ss = this;
        let valueArray = [];
        if (typeof value !== 'undefined') {
          if (Array.isArray(value)) {
            valueArray = value;
          } else {
            valueArray = [value];
          }
        } else {
          ss.$selectEl.find('option').each(optionEl => {
            const $optionEl = $(optionEl);
            if (optionEl.selected) {
              const displayAs = optionEl.dataset ? optionEl.dataset.displayAs : $optionEl.data('display-value-as');
              if (displayAs && typeof displayAs !== 'undefined') {
                valueArray.push(displayAs);
              } else {
                valueArray.push(optionEl.textContent.trim());
              }
            }
          });
        }
        if (ss.params.setValueText) {
          ss.formatValueTextContent(valueArray);
        }
      }
      getItemsData() {
        const ss = this;
        const theme = ss.app.theme;
        const items = [];
        let previousGroupEl;
        ss.$selectEl.find('option').each(optionEl => {
          const $optionEl = $(optionEl);
          const optionData = $optionEl.dataset();
          const optionImage = optionData.optionImage || ss.params.optionImage;
          const optionIcon = optionData.optionIcon || ss.params.optionIcon;
          const optionIconIos = theme === 'ios' && (optionData.optionIconIos || ss.params.optionIconIos);
          const optionIconMd = theme === 'md' && (optionData.optionIconMd || ss.params.optionIconMd);
          const optionInputIconPosition = optionData.inputIconPosition || ss.params.inputIconPosition || '';
          const optionHasMedia = optionImage || optionIcon || optionIconIos || optionIconMd;
          const optionColor = optionData.optionColor;
          let optionClassName = optionData.optionClass || '';
          if ($optionEl[0].disabled) optionClassName += ' disabled';
          const optionGroupEl = $optionEl.parent('optgroup')[0];
          const optionGroupLabel = optionGroupEl && optionGroupEl.label;
          let optionIsLabel = false;
          if (optionGroupEl && optionGroupEl !== previousGroupEl) {
            optionIsLabel = true;
            previousGroupEl = optionGroupEl;
            items.push({
              groupLabel: optionGroupLabel,
              isLabel: optionIsLabel
            });
          }
          items.push({
            value: $optionEl[0].value,
            text: $optionEl[0].textContent.trim(),
            selected: $optionEl[0].selected,
            groupEl: optionGroupEl,
            groupLabel: optionGroupLabel,
            image: optionImage,
            icon: optionIcon,
            iconIos: optionIconIos,
            iconMd: optionIconMd,
            inputIconPosition: optionInputIconPosition,
            color: optionColor,
            className: optionClassName,
            disabled: $optionEl[0].disabled,
            id: ss.id,
            hasMedia: optionHasMedia,
            checkbox: ss.inputType === 'checkbox',
            radio: ss.inputType === 'radio',
            inputName: ss.inputName,
            inputType: ss.inputType
          });
        });
        ss.items = items;
        return items;
      }
      renderSearchbar() {
        const ss = this;
        if (ss.params.renderSearchbar) return ss.params.renderSearchbar.call(ss);
        return $jsx$1("form", {
          class: "searchbar"
        }, $jsx$1("div", {
          class: "searchbar-inner"
        }, $jsx$1("div", {
          class: "searchbar-input-wrap"
        }, $jsx$1("input", {
          type: "search",
          spellcheck: ss.params.searchbarSpellcheck || 'false',
          placeholder: ss.params.searchbarPlaceholder
        }), $jsx$1("i", {
          class: "searchbar-icon"
        }), $jsx$1("span", {
          class: "input-clear-button"
        })), ss.params.searchbarDisableButton && $jsx$1("span", {
          class: "searchbar-disable-button"
        }, ss.params.searchbarDisableText)));
      }
      renderItem(item, index) {
        const ss = this;
        if (ss.params.renderItem) return ss.params.renderItem.call(ss, item, index);
        function getIconContent(iconValue) {
          if (iconValue === void 0) {
            iconValue = '';
          }
          if (iconValue.indexOf(':') >= 0) {
            return iconValue.split(':')[1];
          }
          return '';
        }
        function getIconClass(iconValue) {
          if (iconValue === void 0) {
            iconValue = '';
          }
          if (iconValue.indexOf(':') >= 0) {
            let className = iconValue.split(':')[0];
            if (className === 'f7') className = 'f7-icons';
            if (className === 'material') className = 'material-icons';
            return className;
          }
          return iconValue;
        }
        let itemHtml;
        if (item.isLabel) {
          itemHtml = `<li class="list-group-title">${item.groupLabel}</li>`;
        } else {
          let selected = item.selected;
          let disabled;
          if (ss.params.virtualList) {
            const ssValue = ss.getValue();
            selected = ss.multiple ? ssValue.indexOf(item.value) >= 0 : ssValue === item.value;
            if (ss.multiple) {
              disabled = ss.multiple && !selected && ssValue.length === parseInt(ss.maxLength, 10);
            }
          }
          const {
            icon,
            iconIos,
            iconMd
          } = item;
          const hasIcon = icon || iconIos || iconMd;
          const iconContent = getIconContent(icon || iconIos || iconMd || '');
          const iconClass = getIconClass(icon || iconIos || iconMd || '');
          itemHtml = $jsx$1("li", {
            class: `${item.className || ''}${disabled ? ' disabled' : ''}`
          }, $jsx$1("label", {
            class: `item-${item.inputType} ${item.inputIconPosition ? `item-${item.inputType}-icon-${item.inputIconPosition}` : ''} item-content`
          }, $jsx$1("input", {
            type: item.inputType,
            name: item.inputName,
            value: item.value,
            _checked: selected
          }), $jsx$1("i", {
            class: `icon icon-${item.inputType}`
          }), item.hasMedia && $jsx$1("div", {
            class: "item-media"
          }, hasIcon && $jsx$1("i", {
            class: `icon ${iconClass}`
          }, iconContent), item.image && $jsx$1("img", {
            src: item.image
          })), $jsx$1("div", {
            class: "item-inner"
          }, $jsx$1("div", {
            class: `item-title${item.color ? ` text-color-${item.color}` : ''}`
          }, item.text))));
        }
        return itemHtml;
      }
      renderItems() {
        const ss = this;
        if (ss.params.renderItems) return ss.params.renderItems.call(ss, ss.items);
        const itemsHtml = `
      ${ss.items.map((item, index) => `${ss.renderItem(item, index)}`).join('')}
    `;
        return itemsHtml;
      }
      renderPage() {
        const ss = this;
        if (ss.params.renderPage) return ss.params.renderPage.call(ss, ss.items);
        let pageTitle = ss.params.pageTitle;
        if (typeof pageTitle === 'undefined') {
          const $itemTitleEl = ss.$el.find('.item-title');
          pageTitle = $itemTitleEl.length ? $itemTitleEl.text().trim() : '';
        }
        const cssClass = ss.params.cssClass;
        return $jsx$1("div", {
          class: `page smart-select-page ${cssClass}`,
          "data-name": "smart-select-page",
          "data-select-name": ss.selectName
        }, $jsx$1("div", {
          class: `navbar ${ss.params.navbarColorTheme ? `color-${ss.params.navbarColorTheme}` : ''}`
        }, $jsx$1("div", {
          class: "navbar-bg"
        }), $jsx$1("div", {
          class: `navbar-inner sliding ${ss.params.navbarColorTheme ? `color-${ss.params.navbarColorTheme}` : ''}`
        }, $jsx$1("div", {
          class: "left"
        }, $jsx$1("a", {
          class: "link back"
        }, $jsx$1("i", {
          class: "icon icon-back"
        }), $jsx$1("span", {
          class: "if-not-md"
        }, ss.params.pageBackLinkText))), pageTitle && $jsx$1("div", {
          class: "title"
        }, pageTitle), ss.params.searchbar && $jsx$1("div", {
          class: "subnavbar"
        }, ss.renderSearchbar()))), ss.params.searchbar && $jsx$1("div", {
          class: "searchbar-backdrop"
        }), $jsx$1("div", {
          class: "page-content"
        }, $jsx$1("div", {
          class: `list list-outline-ios list-strong-ios list-dividers-ios smart-select-list-${ss.id} ${ss.params.virtualList ? ' virtual-list' : ''} ${ss.params.formColorTheme ? `color-${ss.params.formColorTheme}` : ''}`
        }, $jsx$1("ul", null, !ss.params.virtualList && ss.renderItems(ss.items)))));
      }
      renderPopup() {
        const ss = this;
        if (ss.params.renderPopup) return ss.params.renderPopup.call(ss, ss.items);
        let pageTitle = ss.params.pageTitle;
        if (typeof pageTitle === 'undefined') {
          const $itemTitleEl = ss.$el.find('.item-title');
          pageTitle = $itemTitleEl.length ? $itemTitleEl.text().trim() : '';
        }
        const cssClass = ss.params.cssClass || '';
        return $jsx$1("div", {
          class: `popup smart-select-popup ${cssClass} ${ss.params.popupTabletFullscreen ? 'popup-tablet-fullscreen' : ''}`,
          "data-select-name": ss.selectName
        }, $jsx$1("div", {
          class: "view"
        }, $jsx$1("div", {
          class: `page smart-select-page ${ss.params.searchbar ? 'page-with-subnavbar' : ''}`,
          "data-name": "smart-select-page"
        }, $jsx$1("div", {
          class: `navbar ${ss.params.navbarColorTheme ? `color-${ss.params.navbarColorTheme}` : ''}`
        }, $jsx$1("div", {
          class: "navbar-bg"
        }), $jsx$1("div", {
          class: "navbar-inner sliding"
        }, pageTitle && $jsx$1("div", {
          class: "title"
        }, pageTitle), $jsx$1("div", {
          class: "right"
        }, $jsx$1("a", {
          class: "link popup-close",
          "data-popup": `.smart-select-popup[data-select-name='${ss.selectName}']`
        }, ss.params.popupCloseLinkText)), ss.params.searchbar && $jsx$1("div", {
          class: "subnavbar"
        }, ss.renderSearchbar()))), ss.params.searchbar && $jsx$1("div", {
          class: "searchbar-backdrop"
        }), $jsx$1("div", {
          class: "page-content"
        }, $jsx$1("div", {
          class: `list list-outline-ios list-strong-ios list-dividers-ios smart-select-list-${ss.id} ${ss.params.virtualList ? ' virtual-list' : ''} ${ss.params.formColorTheme ? `color-${ss.params.formColorTheme}` : ''}`
        }, $jsx$1("ul", null, !ss.params.virtualList && ss.renderItems(ss.items)))))));
      }
      renderSheet() {
        const ss = this;
        if (ss.params.renderSheet) return ss.params.renderSheet.call(ss, ss.items);
        const cssClass = ss.params.cssClass;
        // prettier-ignore
        return $jsx$1("div", {
          class: `sheet-modal smart-select-sheet ${cssClass}`,
          "data-select-name": ss.selectName
        }, $jsx$1("div", {
          class: `toolbar toolbar-top ${ss.params.toolbarColorTheme ? `color-${ss.params.toolbarColorTheme}` : ''}`
        }, $jsx$1("div", {
          class: "toolbar-inner"
        }, $jsx$1("div", {
          class: "left"
        }), $jsx$1("div", {
          class: "right"
        }, $jsx$1("a", {
          class: "link sheet-close"
        }, ss.params.sheetCloseLinkText)))), $jsx$1("div", {
          class: "sheet-modal-inner"
        }, $jsx$1("div", {
          class: "page-content"
        }, $jsx$1("div", {
          class: `list list-strong-ios list-dividers-ios smart-select-list-${ss.id} ${ss.params.virtualList ? ' virtual-list' : ''} ${ss.params.formColorTheme ? `color-${ss.params.formColorTheme}` : ''}`
        }, $jsx$1("ul", null, !ss.params.virtualList && ss.renderItems(ss.items))))));
      }
      renderPopover() {
        const ss = this;
        if (ss.params.renderPopover) return ss.params.renderPopover.call(ss, ss.items);
        const cssClass = ss.params.cssClass;
        // prettier-ignore
        return $jsx$1("div", {
          class: `popover smart-select-popover ${cssClass}`,
          "data-select-name": ss.selectName
        }, $jsx$1("div", {
          class: "popover-inner"
        }, $jsx$1("div", {
          class: `list list-strong-ios list-dividers-ios smart-select-list-${ss.id} ${ss.params.virtualList ? ' virtual-list' : ''} ${ss.params.formColorTheme ? `color-${ss.params.formColorTheme}` : ''}`
        }, $jsx$1("ul", null, !ss.params.virtualList && ss.renderItems(ss.items)))));
      }
      scrollToSelectedItem() {
        const ss = this;
        const {
          params,
          $containerEl
        } = ss;
        if (!ss.opened) return ss;
        if (params.virtualList) {
          let selectedIndex;
          ss.vl.items.forEach((item, index) => {
            if (typeof selectedIndex === 'undefined' && item.selected) {
              selectedIndex = index;
            }
          });
          if (typeof selectedIndex !== 'undefined') {
            ss.vl.scrollToItem(selectedIndex);
          }
        } else {
          const $selectedItemEl = $containerEl.find('input:checked').parents('li');
          if (!$selectedItemEl.length) return ss;
          const $scrollableEl = $containerEl.find('.page-content, .popover-inner');
          if (!$scrollableEl.length) return ss;
          $scrollableEl.scrollTop($selectedItemEl.offset().top - $scrollableEl.offset().top - parseInt($scrollableEl.css('padding-top'), 10));
        }
        return ss;
      }
      onOpen(type, containerEl) {
        const ss = this;
        const app = ss.app;
        const $containerEl = $(containerEl);
        ss.$containerEl = $containerEl;
        ss.openedIn = type;
        ss.opened = true;

        // Init VL
        if (ss.params.virtualList) {
          ss.vl = app.virtualList.create({
            el: $containerEl.find('.virtual-list'),
            items: ss.items,
            renderItem: ss.renderItem.bind(ss),
            height: ss.params.virtualListHeight,
            searchByItem(query, item) {
              if (item.text && removeDiacritics(item.text).toLowerCase().indexOf(query.trim().toLowerCase()) >= 0) return true;
              return false;
            }
          });
        }
        if (ss.params.scrollToSelectedItem) {
          ss.scrollToSelectedItem();
        }

        // Init SB
        if (ss.params.searchbar) {
          let $searchbarEl = $containerEl.find('.searchbar');
          if (type === 'page' && app.theme === 'ios') {
            $searchbarEl = $(app.navbar.getElByPage($containerEl)).find('.searchbar');
          }
          if (ss.params.appendSearchbarNotFound && (type === 'page' || type === 'popup')) {
            let $notFoundEl = null;
            if (typeof ss.params.appendSearchbarNotFound === 'string') {
              $notFoundEl = $(`<div class="block searchbar-not-found">${ss.params.appendSearchbarNotFound}</div>`);
            } else if (typeof ss.params.appendSearchbarNotFound === 'boolean') {
              $notFoundEl = $('<div class="block searchbar-not-found">Nothing found</div>');
            } else {
              $notFoundEl = ss.params.appendSearchbarNotFound;
            }
            if ($notFoundEl) {
              $containerEl.find('.page-content').append($notFoundEl[0]);
            }
          }
          const searchbarParams = extend$3({
            el: $searchbarEl,
            backdropEl: $containerEl.find('.searchbar-backdrop'),
            searchContainer: `.smart-select-list-${ss.id}`,
            searchIn: '.item-title'
          }, typeof ss.params.searchbar === 'object' ? ss.params.searchbar : {});
          ss.searchbar = app.searchbar.create(searchbarParams);
        }

        // Check for max length
        if (ss.maxLength) {
          ss.checkMaxLength();
        }

        // Close on select
        if (ss.params.closeOnSelect) {
          ss.$containerEl.find(`input[type="radio"][name="${ss.inputName}"]:checked`).parents('label').once('click', () => {
            ss.close();
          });
        }

        // Attach input events
        ss.attachInputsEvents();
        ss.$el.trigger('smartselect:open');
        ss.emit('local::open smartSelectOpen', ss);
      }
      onOpened() {
        const ss = this;
        ss.$el.trigger('smartselect:opened');
        ss.emit('local::opened smartSelectOpened', ss);
      }
      onClose() {
        const ss = this;
        if (ss.destroyed) return;

        // Destroy VL
        if (ss.vl && ss.vl.destroy) {
          ss.vl.destroy();
          ss.vl = null;
          delete ss.vl;
        }

        // Destroy SB
        if (ss.searchbar && ss.searchbar.destroy) {
          ss.searchbar.destroy();
          ss.searchbar = null;
          delete ss.searchbar;
        }
        // Detach events
        ss.detachInputsEvents();
        ss.$el.trigger('smartselect:close');
        ss.emit('local::close smartSelectClose', ss);
      }
      onClosed() {
        const ss = this;
        if (ss.destroyed) return;
        ss.opened = false;
        ss.$containerEl = null;
        delete ss.$containerEl;
        ss.$el.trigger('smartselect:closed');
        ss.emit('local::closed smartSelectClosed', ss);
      }
      openPage() {
        const ss = this;
        if (ss.opened) return ss;
        ss.getItemsData();
        const pageHtml = ss.renderPage(ss.items);
        ss.view.router.navigate({
          url: ss.url,
          route: {
            content: pageHtml,
            path: ss.url,
            on: {
              pageBeforeIn(e, page) {
                ss.onOpen('page', page.el);
              },
              pageAfterIn(e, page) {
                ss.onOpened('page', page.el);
              },
              pageBeforeOut(e, page) {
                ss.onClose('page', page.el);
              },
              pageAfterOut(e, page) {
                ss.onClosed('page', page.el);
              }
            }
          }
        });
        return ss;
      }
      openPopup() {
        const ss = this;
        if (ss.opened) return ss;
        ss.getItemsData();
        const popupHtml = ss.renderPopup(ss.items);
        const popupParams = {
          content: popupHtml,
          push: ss.params.popupPush,
          swipeToClose: ss.params.popupSwipeToClose,
          closeByBackdropClick: ss.params.closeByBackdropClick,
          on: {
            popupOpen(popup) {
              ss.onOpen('popup', popup.el);
            },
            popupOpened(popup) {
              ss.onOpened('popup', popup.el);
            },
            popupClose(popup) {
              ss.onClose('popup', popup.el);
            },
            popupClosed(popup) {
              ss.onClosed('popup', popup.el);
            }
          }
        };
        if (ss.params.routableModals && ss.view) {
          ss.view.router.navigate({
            url: ss.url,
            route: {
              path: ss.url,
              popup: popupParams
            }
          });
        } else {
          ss.modal = ss.app.popup.create(popupParams).open();
        }
        return ss;
      }
      openSheet() {
        const ss = this;
        if (ss.opened) return ss;
        ss.getItemsData();
        const sheetHtml = ss.renderSheet(ss.items);
        const sheetParams = {
          content: sheetHtml,
          backdrop: ss.params.sheetBackdrop,
          scrollToEl: ss.$el,
          closeByOutsideClick: true,
          push: ss.params.sheetPush,
          swipeToClose: ss.params.sheetSwipeToClose,
          closeByBackdropClick: ss.params.closeByBackdropClick,
          on: {
            sheetOpen(sheet) {
              ss.onOpen('sheet', sheet.el);
            },
            sheetOpened(sheet) {
              ss.onOpened('sheet', sheet.el);
            },
            sheetClose(sheet) {
              ss.onClose('sheet', sheet.el);
            },
            sheetClosed(sheet) {
              ss.onClosed('sheet', sheet.el);
            }
          }
        };
        if (ss.params.routableModals && ss.view) {
          ss.view.router.navigate({
            url: ss.url,
            route: {
              path: ss.url,
              sheet: sheetParams
            }
          });
        } else {
          ss.modal = ss.app.sheet.create(sheetParams).open();
        }
        return ss;
      }
      openPopover() {
        const ss = this;
        if (ss.opened) return ss;
        ss.getItemsData();
        const popoverHtml = ss.renderPopover(ss.items);
        const popoverParams = {
          content: popoverHtml,
          targetEl: ss.$el,
          closeByBackdropClick: ss.params.closeByBackdropClick,
          on: {
            popoverOpen(popover) {
              ss.onOpen('popover', popover.el);
            },
            popoverOpened(popover) {
              ss.onOpened('popover', popover.el);
            },
            popoverClose(popover) {
              ss.onClose('popover', popover.el);
            },
            popoverClosed(popover) {
              ss.onClosed('popover', popover.el);
            }
          }
        };
        if (ss.params.routableModals && ss.view) {
          ss.view.router.navigate({
            url: ss.url,
            route: {
              path: ss.url,
              popover: popoverParams
            }
          });
        } else {
          ss.modal = ss.app.popover.create(popoverParams).open();
        }
        return ss;
      }
      open(type) {
        const ss = this;
        if (ss.opened) return ss;
        let prevented = false;
        function prevent() {
          prevented = true;
        }
        if (ss.$el) {
          ss.$el.trigger('smartselect:beforeopen', {
            prevent
          });
        }
        ss.emit('local::beforeOpen smartSelectBeforeOpen', ss, prevent);
        if (prevented) return ss;
        const openIn = type || ss.params.openIn;
        ss[`open${openIn.split('').map((el, index) => {
      if (index === 0) return el.toUpperCase();
      return el;
    }).join('')}`]();
        return ss;
      }
      close() {
        const ss = this;
        if (!ss.opened) return ss;
        if (ss.params.routableModals && ss.view || ss.openedIn === 'page') {
          ss.view.router.back();
        } else {
          ss.modal.once('modalClosed', () => {
            nextTick$1(() => {
              if (ss.destroyed) return;
              ss.modal.destroy();
              delete ss.modal;
            });
          });
          ss.modal.close();
        }
        return ss;
      }
      init() {
        const ss = this;
        ss.attachEvents();
        ss.setValueText();
      }
      destroy() {
        const ss = this;
        ss.emit('local::beforeDestroy smartSelectBeforeDestroy', ss);
        ss.$el.trigger('smartselect:beforedestroy');
        ss.detachEvents();
        delete ss.$el[0].f7SmartSelect;
        deleteProps$1(ss);
        ss.destroyed = true;
      }
    };

    var SmartSelect = {
      name: 'smartSelect',
      params: {
        smartSelect: {
          el: undefined,
          valueEl: undefined,
          setValueText: true,
          formatValueText: null,
          openIn: 'page',
          // or 'popup' or 'sheet' or 'popover'
          popupPush: false,
          popupSwipeToClose: undefined,
          // defaults to app
          sheetPush: false,
          sheetSwipeToClose: undefined,
          // defaults to app
          sheetBackdrop: false,
          pageTitle: undefined,
          pageBackLinkText: 'Back',
          popupCloseLinkText: 'Close',
          popupTabletFullscreen: false,
          closeByBackdropClick: true,
          sheetCloseLinkText: 'Done',
          searchbar: false,
          searchbarPlaceholder: 'Search',
          searchbarDisableText: 'Cancel',
          searchbarDisableButton: true,
          searchbarSpellcheck: false,
          closeOnSelect: false,
          virtualList: false,
          virtualListHeight: undefined,
          scrollToSelectedItem: false,
          formColorTheme: undefined,
          navbarColorTheme: undefined,
          routableModals: false,
          url: 'select/',
          inputIconPosition: 'start',
          cssClass: '',
          /*
            Custom render functions
          */
          renderPage: undefined,
          renderPopup: undefined,
          renderSheet: undefined,
          renderPopover: undefined,
          renderItems: undefined,
          renderItem: undefined,
          renderSearchbar: undefined
        }
      },
      static: {
        SmartSelect: SmartSelect$1
      },
      create() {
        const app = this;
        app.smartSelect = extend$3(ConstructorMethods({
          defaultSelector: '.smart-select',
          constructor: SmartSelect$1,
          app,
          domProp: 'f7SmartSelect'
        }), {
          open(smartSelectEl) {
            const ss = app.smartSelect.get(smartSelectEl);
            if (ss && ss.open) return ss.open();
            return undefined;
          },
          close(smartSelectEl) {
            const ss = app.smartSelect.get(smartSelectEl);
            if (ss && ss.close) return ss.close();
            return undefined;
          }
        });
      },
      on: {
        tabMounted(tabEl) {
          const app = this;
          $(tabEl).find('.smart-select-init').each(smartSelectEl => {
            app.smartSelect.create(extend$3({
              el: smartSelectEl
            }, $(smartSelectEl).dataset()));
          });
        },
        tabBeforeRemove(tabEl) {
          $(tabEl).find('.smart-select-init').each(smartSelectEl => {
            if (smartSelectEl.f7SmartSelect && smartSelectEl.f7SmartSelect.destroy) {
              smartSelectEl.f7SmartSelect.destroy();
            }
          });
        },
        pageInit(page) {
          const app = this;
          page.$el.find('.smart-select-init').each(smartSelectEl => {
            app.smartSelect.create(extend$3({
              el: smartSelectEl
            }, $(smartSelectEl).dataset()));
          });
        },
        pageBeforeRemove(page) {
          page.$el.find('.smart-select-init').each(smartSelectEl => {
            if (smartSelectEl.f7SmartSelect && smartSelectEl.f7SmartSelect.destroy) {
              smartSelectEl.f7SmartSelect.destroy();
            }
          });
        }
      },
      clicks: {
        '.smart-select': function open($clickedEl, data) {
          const app = this;
          if (!$clickedEl[0].f7SmartSelect) {
            const ss = app.smartSelect.create(extend$3({
              el: $clickedEl
            }, data));
            ss.open();
          }
        }
      },
      vnode: {
        'smart-select-init': {
          insert(vnode) {
            const app = this;
            const smartSelectEl = vnode.elm;
            app.smartSelect.create(extend$3({
              el: smartSelectEl
            }, $(smartSelectEl).dataset()));
          },
          destroy(vnode) {
            const smartSelectEl = vnode.elm;
            if (smartSelectEl.f7SmartSelect && smartSelectEl.f7SmartSelect.destroy) {
              smartSelectEl.f7SmartSelect.destroy();
            }
          }
        }
      }
    };

    var Grid$1 = {
      name: 'grid'
    };

    let Calendar$1 = class Calendar extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const calendar = this;
        calendar.params = extend$3({}, app.params.calendar, params);
        let $containerEl;
        if (calendar.params.containerEl) {
          $containerEl = $(calendar.params.containerEl);
          if ($containerEl.length === 0) return calendar;
        }
        let $inputEl;
        if (calendar.params.inputEl) {
          $inputEl = $(calendar.params.inputEl);
        }
        const isHorizontal = calendar.params.direction === 'horizontal';
        let inverter = 1;
        if (isHorizontal) {
          inverter = app.rtl ? -1 : 1;
        }
        extend$3(calendar, {
          app,
          $containerEl,
          containerEl: $containerEl && $containerEl[0],
          inline: $containerEl && $containerEl.length > 0,
          $inputEl,
          inputEl: $inputEl && $inputEl[0],
          initialized: false,
          opened: false,
          url: calendar.params.url,
          isHorizontal,
          inverter,
          animating: false,
          allowTouchMove: true,
          hasTimePicker: calendar.params.timePicker && !calendar.params.rangePicker && !calendar.params.multiple
        });
        calendar.dayFormatter = date => {
          const formatter = new Intl.DateTimeFormat(calendar.params.locale, {
            day: 'numeric'
          });
          return formatter.format(date).replace(/日/, '');
        };
        calendar.monthFormatter = date => {
          const formatter = new Intl.DateTimeFormat(calendar.params.locale, {
            month: 'long'
          });
          return formatter.format(date);
        };
        calendar.yearFormatter = date => {
          const formatter = new Intl.DateTimeFormat(calendar.params.locale, {
            year: 'numeric'
          });
          return formatter.format(date);
        };
        calendar.timeSelectorFormatter = date => {
          const formatter = new Intl.DateTimeFormat(calendar.params.locale, calendar.params.timePickerFormat);
          return formatter.format(date);
        };
        const timeFormatCheckDate = calendar.timeSelectorFormatter(new Date()).toLowerCase();
        calendar.is12HoursFormat = timeFormatCheckDate.indexOf('pm') >= 0 || timeFormatCheckDate.indexOf('am') >= 0;

        // Auto names
        let {
          monthNames,
          monthNamesShort,
          dayNames,
          dayNamesShort
        } = calendar.params;
        const {
          monthNamesIntl,
          monthNamesShortIntl,
          dayNamesIntl,
          dayNamesShortIntl
        } = calendar.getIntlNames();
        if (monthNames === 'auto') monthNames = monthNamesIntl;
        if (monthNamesShort === 'auto') monthNamesShort = monthNamesShortIntl;
        if (dayNames === 'auto') dayNames = dayNamesIntl;
        if (dayNamesShort === 'auto') dayNamesShort = dayNamesShortIntl;
        extend$3(calendar, {
          monthNames,
          monthNamesShort,
          dayNames,
          dayNamesShort
        });
        function onInputClick() {
          calendar.open();
        }
        function onInputFocus(e) {
          e.preventDefault();
        }
        function onInputClear() {
          calendar.setValue([]);
          if (calendar.opened) {
            calendar.update();
          }
        }
        function onHtmlClick(e) {
          const $targetEl = $(e.target);
          if (calendar.destroyed || !calendar.params) return;
          if (calendar.isPopover()) return;
          if (!calendar.opened || calendar.closing) return;
          if ($targetEl.closest('[class*="backdrop"]').length) return;
          if (calendar.monthPickerPopover || calendar.yearPickerPopover || calendar.timePickerPopover) return;
          if ($inputEl && $inputEl.length > 0) {
            if ($targetEl[0] !== $inputEl[0] && $targetEl.closest('.sheet-modal, .calendar-modal').length === 0) {
              calendar.close();
            }
          } else if ($(e.target).closest('.sheet-modal, .calendar-modal').length === 0) {
            calendar.close();
          }
        }

        // Events
        extend$3(calendar, {
          attachInputEvents() {
            calendar.$inputEl.on('click', onInputClick);
            calendar.$inputEl.on('input:clear', onInputClear);
            if (calendar.params.inputReadOnly) {
              calendar.$inputEl.on('focus mousedown', onInputFocus);
              if (calendar.$inputEl[0]) {
                calendar.$inputEl[0].f7ValidateReadonly = true;
              }
            }
          },
          detachInputEvents() {
            calendar.$inputEl.off('click', onInputClick);
            calendar.$inputEl.off('input:clear', onInputClear);
            if (calendar.params.inputReadOnly) {
              calendar.$inputEl.off('focus mousedown', onInputFocus);
              if (calendar.$inputEl[0]) {
                delete calendar.$inputEl[0].f7ValidateReadonly;
              }
            }
          },
          attachHtmlEvents() {
            app.on('click', onHtmlClick);
          },
          detachHtmlEvents() {
            app.off('click', onHtmlClick);
          }
        });
        calendar.attachCalendarEvents = function attachCalendarEvents() {
          let allowItemClick = true;
          let isTouched;
          let isMoved;
          let touchStartX;
          let touchStartY;
          let touchCurrentX;
          let touchCurrentY;
          let touchStartTime;
          let touchEndTime;
          let currentTranslate;
          let wrapperWidth;
          let wrapperHeight;
          let percentage;
          let touchesDiff;
          let isScrolling;
          const {
            $el,
            $wrapperEl
          } = calendar;
          function handleTouchStart(e) {
            if (isMoved || isTouched || !e.isTrusted) return;
            isTouched = true;
            touchStartX = e.type === 'touchstart' ? e.targetTouches[0].pageX : e.pageX;
            touchCurrentX = touchStartX;
            touchStartY = e.type === 'touchstart' ? e.targetTouches[0].pageY : e.pageY;
            touchCurrentY = touchStartY;
            touchStartTime = new Date().getTime();
            percentage = 0;
            allowItemClick = true;
            isScrolling = undefined;
            currentTranslate = calendar.monthsTranslate;
          }
          function handleTouchMove(e) {
            if (!isTouched || !e.isTrusted) return;
            const {
              isHorizontal: isH
            } = calendar;
            touchCurrentX = e.type === 'touchmove' ? e.targetTouches[0].pageX : e.pageX;
            touchCurrentY = e.type === 'touchmove' ? e.targetTouches[0].pageY : e.pageY;
            if (typeof isScrolling === 'undefined') {
              isScrolling = !!(isScrolling || Math.abs(touchCurrentY - touchStartY) > Math.abs(touchCurrentX - touchStartX));
            }
            if (isH && isScrolling || !calendar.allowTouchMove) {
              isTouched = false;
              return;
            }
            e.preventDefault();
            if (calendar.animating) {
              isTouched = false;
              return;
            }
            allowItemClick = false;
            if (!isMoved) {
              // First move
              isMoved = true;
              wrapperWidth = $wrapperEl[0].offsetWidth;
              wrapperHeight = $wrapperEl[0].offsetHeight;
              $wrapperEl.transition(0);
            }
            touchesDiff = isH ? touchCurrentX - touchStartX : touchCurrentY - touchStartY;
            percentage = touchesDiff / (isH ? wrapperWidth : wrapperHeight);
            currentTranslate = (calendar.monthsTranslate * calendar.inverter + percentage) * 100;

            // Transform wrapper
            $wrapperEl.transform(`translate3d(${isH ? currentTranslate : 0}%, ${isH ? 0 : currentTranslate}%, 0)`);
          }
          function handleTouchEnd(e) {
            if (!isTouched || !isMoved || !e.isTrusted) {
              isTouched = false;
              isMoved = false;
              return;
            }
            isTouched = false;
            isMoved = false;
            touchEndTime = new Date().getTime();
            if (touchEndTime - touchStartTime < 300) {
              if (Math.abs(touchesDiff) < 10) {
                calendar.resetMonth();
              } else if (touchesDiff >= 10) {
                if (app.rtl) calendar.nextMonth();else calendar.prevMonth();
              } else if (app.rtl) calendar.prevMonth();else calendar.nextMonth();
            } else if (percentage <= -0.5) {
              if (app.rtl) calendar.prevMonth();else calendar.nextMonth();
            } else if (percentage >= 0.5) {
              if (app.rtl) calendar.nextMonth();else calendar.prevMonth();
            } else {
              calendar.resetMonth();
            }

            // Allow click
            setTimeout(() => {
              allowItemClick = true;
            }, 100);
          }
          function handleDayClick(e) {
            if (!allowItemClick) return;
            let $dayEl = $(e.target).parents('.calendar-day');
            if ($dayEl.length === 0 && $(e.target).hasClass('calendar-day')) {
              $dayEl = $(e.target);
            }
            if ($dayEl.length === 0) return;
            if ($dayEl.hasClass('calendar-day-disabled')) return;
            if (!calendar.params.rangePicker) {
              if ($dayEl.hasClass('calendar-day-next')) calendar.nextMonth();
              if ($dayEl.hasClass('calendar-day-prev')) calendar.prevMonth();
            }
            const dateYear = parseInt($dayEl.attr('data-year'), 10);
            const dateMonth = parseInt($dayEl.attr('data-month'), 10);
            const dateDay = parseInt($dayEl.attr('data-day'), 10);
            calendar.emit('local::dayClick calendarDayClick', calendar, $dayEl[0], dateYear, dateMonth, dateDay);
            if (!$dayEl.hasClass('calendar-day-selected') || calendar.params.multiple || calendar.params.rangePicker) {
              const valueToAdd = new Date(dateYear, dateMonth, dateDay, 0, 0, 0);
              if (calendar.hasTimePicker) {
                if (calendar.value && calendar.value[0]) {
                  valueToAdd.setHours(calendar.value[0].getHours(), calendar.value[0].getMinutes());
                } else {
                  valueToAdd.setHours(new Date().getHours(), new Date().getMinutes());
                }
              }
              calendar.addValue(valueToAdd);
            }
            if (calendar.params.closeOnSelect) {
              if (calendar.params.rangePicker && calendar.value.length === 2 || !calendar.params.rangePicker) {
                calendar.close();
              }
            }
          }
          function onNextMonthClick() {
            calendar.nextMonth();
          }
          function onPrevMonthClick() {
            calendar.prevMonth();
          }
          function onNextYearClick() {
            calendar.nextYear();
          }
          function onPrevYearClick() {
            calendar.prevYear();
          }
          function onMonthSelectorClick() {
            calendar.openMonthPicker();
          }
          function onYearSelectorClick() {
            calendar.openYearPicker();
          }
          function onTimeSelectorClick() {
            calendar.openTimePicker();
          }
          const passiveListener = app.touchEvents.start === 'touchstart' && getSupport$1().passiveListener ? {
            passive: true,
            capture: false
          } : false;
          // Selectors clicks
          $el.find('.calendar-prev-month-button').on('click', onPrevMonthClick);
          $el.find('.calendar-next-month-button').on('click', onNextMonthClick);
          $el.find('.calendar-prev-year-button').on('click', onPrevYearClick);
          $el.find('.calendar-next-year-button').on('click', onNextYearClick);
          if (calendar.params.monthPicker) {
            $el.find('.current-month-value').on('click', onMonthSelectorClick);
          }
          if (calendar.params.yearPicker) {
            $el.find('.current-year-value').on('click', onYearSelectorClick);
          }
          if (calendar.hasTimePicker) {
            $el.find('.calendar-time-selector a').on('click', onTimeSelectorClick);
          }
          // Day clicks
          $wrapperEl.on('click', handleDayClick);
          // Touch events
          if (calendar.params.touchMove) {
            $wrapperEl.on(app.touchEvents.start, handleTouchStart, passiveListener);
            app.on('touchmove:active', handleTouchMove);
            app.on('touchend:passive', handleTouchEnd);
          }
          calendar.detachCalendarEvents = function detachCalendarEvents() {
            $el.find('.calendar-prev-month-button').off('click', onPrevMonthClick);
            $el.find('.calendar-next-month-button').off('click', onNextMonthClick);
            $el.find('.calendar-prev-year-button').off('click', onPrevYearClick);
            $el.find('.calendar-next-year-button').off('click', onNextYearClick);
            if (calendar.params.monthPicker) {
              $el.find('.current-month-value').off('click', onMonthSelectorClick);
            }
            if (calendar.params.yearPicker) {
              $el.find('.current-year-value').off('click', onYearSelectorClick);
            }
            if (calendar.hasTimePicker) {
              $el.find('.calendar-time-selector a').off('click', onTimeSelectorClick);
            }
            $wrapperEl.off('click', handleDayClick);
            if (calendar.params.touchMove) {
              $wrapperEl.off(app.touchEvents.start, handleTouchStart, passiveListener);
              app.off('touchmove:active', handleTouchMove);
              app.off('touchend:passive', handleTouchEnd);
            }
          };
        };
        calendar.init();
        return calendar;
      }
      get view() {
        const {
          $inputEl,
          app,
          params
        } = this;
        let view;
        if (params.view) {
          view = params.view;
        } else if ($inputEl) {
          view = $inputEl.parents('.view').length && $inputEl.parents('.view')[0].f7View;
        }
        if (!view) view = app.views.main;
        return view;
      }
      getIntlNames() {
        const calendar = this;
        const locale = calendar.params.locale;
        const monthNamesIntl = [];
        const monthNamesShortIntl = [];
        const dayNamesIntl = [];
        const dayNamesShortIntl = [];
        const formatterMonthNames = new Intl.DateTimeFormat(locale, {
          month: 'long'
        });
        const formatterMonthNamesShort = new Intl.DateTimeFormat(locale, {
          month: 'short'
        });
        const formatterDayNames = new Intl.DateTimeFormat(locale, {
          weekday: 'long'
        });
        const formatterDayNamesShort = new Intl.DateTimeFormat(locale, {
          weekday: 'short'
        });
        let year;
        let yearStarted;
        let yearEnded;
        for (let i = 0; i < 24; i += 1) {
          const date = new Date().setMonth(i, 1);
          const currentYear = calendar.yearFormatter(date);
          if (year && currentYear !== year) {
            if (yearStarted) yearEnded = true;
            yearStarted = true;
            year = currentYear;
          }
          if (!year) {
            year = currentYear;
          }
          if (yearStarted && year === currentYear && !yearEnded) {
            monthNamesIntl.push(formatterMonthNames.format(date));
            monthNamesShortIntl.push(formatterMonthNamesShort.format(date));
          }
        }
        const weekDay = new Date().getDay();
        for (let i = 0; i < 7; i += 1) {
          const date = new Date().getTime() + (i - weekDay) * 24 * 60 * 60 * 1000;
          dayNamesIntl.push(formatterDayNames.format(date));
          dayNamesShortIntl.push(formatterDayNamesShort.format(date));
        }
        return {
          monthNamesIntl,
          monthNamesShortIntl,
          dayNamesIntl,
          dayNamesShortIntl
        };
      }
      normalizeDate(date) {
        const calendar = this;
        const d = new Date(date);
        if (calendar.hasTimePicker) {
          return new Date(d.getFullYear(), d.getMonth(), d.getDate(), d.getHours(), d.getMinutes());
        }
        return new Date(d.getFullYear(), d.getMonth(), d.getDate());
      }
      normalizeValues(values) {
        const calendar = this;
        let newValues = [];
        if (values && Array.isArray(values)) {
          newValues = values.map(val => calendar.normalizeDate(val));
        }
        return newValues;
      }
      initInput() {
        const calendar = this;
        if (!calendar.$inputEl) return;
        if (calendar.params.inputReadOnly) calendar.$inputEl.prop('readOnly', true);
      }
      isPopover() {
        const calendar = this;
        const {
          app,
          modal,
          params
        } = calendar;
        const device = getDevice$1();
        if (params.openIn === 'sheet') return false;
        if (modal && modal.type !== 'popover') return false;
        if (!calendar.inline && calendar.inputEl) {
          if (params.openIn === 'popover') return true;
          if (device.ios) {
            return !!device.ipad;
          }
          if (app.width >= 768) {
            return true;
          }
        }
        return false;
      }
      formatDate(d) {
        const calendar = this;
        const date = new Date(d);
        const year = date.getFullYear();
        const month = date.getMonth();
        const month1 = month + 1;
        const day = date.getDate();
        const weekDay = date.getDay();
        const {
          monthNames,
          monthNamesShort,
          dayNames,
          dayNamesShort
        } = calendar;
        const {
          dateFormat,
          locale
        } = calendar.params;
        function twoDigits(number) {
          return number < 10 ? `0${number}` : number;
        }
        if (typeof dateFormat === 'string') {
          const tokens = {
            yyyy: year,
            yy: String(year).substring(2),
            mm: twoDigits(month1),
            m: month1,
            MM: monthNames[month],
            M: monthNamesShort[month],
            dd: twoDigits(day),
            d: day,
            DD: dayNames[weekDay],
            D: dayNamesShort[weekDay]
          };
          if (calendar.params.timePicker) {
            const hours = date.getHours();
            const minutes = date.getMinutes();
            const seconds = date.getSeconds();
            let hours12 = hours;
            if (hours > 12) hours12 = hours - 12;
            if (hours === 0) hours12 = 12;
            const a = hours >= 12 && hours !== 0 ? 'pm' : 'am';
            Object.assign(tokens, {
              HH: twoDigits(hours),
              H: hours,
              hh: twoDigits(hours12),
              h: hours12,
              ss: twoDigits(seconds),
              s: seconds,
              ':mm': twoDigits(minutes),
              ':m': minutes,
              a,
              A: a.toUpperCase()
            });
          }
          const regexp = new RegExp(Object.keys(tokens).map(t => `(${t})`).join('|'), 'g');
          return dateFormat.replace(regexp, token => {
            if (token in tokens) return tokens[token];
            return token;
          });
        }
        if (typeof dateFormat === 'function') {
          return dateFormat(date);
        }
        // Intl Object
        const formatter = new Intl.DateTimeFormat(locale, dateFormat);
        return formatter.format(date);
      }
      formatValue() {
        const calendar = this;
        const {
          value
        } = calendar;
        if (calendar.params.formatValue) {
          return calendar.params.formatValue.call(calendar, value);
        }
        return value.map(v => calendar.formatDate(v)).join(calendar.params.rangePicker ? ' - ' : ', ');
      }
      addValue(newValue) {
        const calendar = this;
        const {
          multiple,
          rangePicker,
          rangePickerMinDays,
          rangePickerMaxDays
        } = calendar.params;
        if (multiple) {
          if (!calendar.value) calendar.value = [];
          let inValuesIndex;
          for (let i = 0; i < calendar.value.length; i += 1) {
            if (new Date(newValue).getTime() === new Date(calendar.value[i]).getTime()) {
              inValuesIndex = i;
            }
          }
          if (typeof inValuesIndex === 'undefined') {
            calendar.value.push(newValue);
          } else {
            calendar.value.splice(inValuesIndex, 1);
          }
          calendar.updateValue();
        } else if (rangePicker) {
          if (!calendar.value) calendar.value = [];
          if (calendar.value.length === 2 || calendar.value.length === 0) {
            calendar.value = [];
          }
          if (calendar.value.length === 0 || Math.abs(calendar.value[0].getTime() - newValue.getTime()) >= (rangePickerMinDays - 1) * 60 * 60 * 24 * 1000 && (rangePickerMaxDays === 0 || Math.abs(calendar.value[0].getTime() - newValue.getTime()) <= (rangePickerMaxDays - 1) * 60 * 60 * 24 * 1000)) calendar.value.push(newValue);else calendar.value = [];
          calendar.value.sort((a, b) => a - b);
          calendar.updateValue();
        } else {
          calendar.value = [newValue];
          calendar.updateValue();
        }
      }
      setValue(values) {
        const calendar = this;
        const currentValue = calendar.value;
        if (Array.isArray(currentValue) && Array.isArray(values) && currentValue.length === values.length) {
          let equal = true;
          currentValue.forEach((v, index) => {
            if (v !== values[index]) equal = false;
          });
          if (equal) return;
        }
        calendar.value = values;
        calendar.updateValue();
      }
      getValue() {
        const calendar = this;
        return calendar.value;
      }
      updateValue(onlyHeader) {
        const calendar = this;
        const {
          $el,
          $wrapperEl,
          $inputEl,
          value,
          params
        } = calendar;
        let i;
        if ($el && $el.length > 0) {
          $wrapperEl.find('.calendar-day-selected').removeClass('calendar-day-selected calendar-day-selected-range calendar-day-selected-left calendar-day-selected-right');
          let valueDate;
          if (params.rangePicker && value.length === 2) {
            const leftDate = new Date(value[0]).getTime();
            const rightDate = new Date(value[1]).getTime();
            for (i = leftDate; i <= rightDate; i += 24 * 60 * 60 * 1000) {
              valueDate = new Date(i);
              let addClass = 'calendar-day-selected';
              if (leftDate !== rightDate) {
                if (i !== leftDate && i !== rightDate) {
                  addClass += ' calendar-day-selected-range';
                }
                if (i === leftDate) {
                  addClass += ' calendar-day-selected-left';
                }
                if (i === rightDate) {
                  addClass += ' calendar-day-selected-right';
                }
              }
              $wrapperEl.find(`.calendar-day[data-date="${valueDate.getFullYear()}-${valueDate.getMonth()}-${valueDate.getDate()}"]`).addClass(addClass);
            }
            valueDate = new Date(leftDate);
            $wrapperEl.find(`.calendar-day[data-date="${valueDate.getFullYear()}-${valueDate.getMonth()}-${valueDate.getDate()}"]`).removeClass('calendar-day-selected-range').addClass('calendar-day-selected calendar-day-selected-left');
            valueDate = new Date(rightDate);
            $wrapperEl.find(`.calendar-day[data-date="${valueDate.getFullYear()}-${valueDate.getMonth()}-${valueDate.getDate()}"]`).removeClass('calendar-day-selected-range').addClass('calendar-day-selected calendar-day-selected-right');
          } else {
            for (i = 0; i < calendar.value.length; i += 1) {
              valueDate = new Date(value[i]);
              $wrapperEl.find(`.calendar-day[data-date="${valueDate.getFullYear()}-${valueDate.getMonth()}-${valueDate.getDate()}"]`).addClass('calendar-day-selected');
            }
          }
        }
        if (!onlyHeader) {
          calendar.emit('local::change calendarChange', calendar, value);
        }
        if ($el && $el.length > 0 && calendar.hasTimePicker) {
          $el.find('.calendar-time-selector a').text(value && value.length ? calendar.timeSelectorFormatter(value[0]) : calendar.params.timePickerPlaceholder);
        }
        if ($inputEl && $inputEl.length || params.header) {
          const inputValue = calendar.formatValue(value);
          if (params.header && $el && $el.length) {
            $el.find('.calendar-selected-date').text(inputValue);
          }
          if ($inputEl && $inputEl.length && !onlyHeader) {
            $inputEl.val(inputValue);
            $inputEl.trigger('change');
          }
        }
      }
      updateCurrentMonthYear(dir) {
        const calendar = this;
        const {
          $months,
          $el,
          monthNames
        } = calendar;
        let currentLocaleMonth;
        let currentLocaleYear;
        if (typeof dir === 'undefined') {
          calendar.currentMonth = parseInt($months.eq(1).attr('data-month'), 10);
          calendar.currentYear = parseInt($months.eq(1).attr('data-year'), 10);
          currentLocaleMonth = $months.eq(1).attr('data-locale-month');
          currentLocaleYear = $months.eq(1).attr('data-locale-year');
        } else {
          calendar.currentMonth = parseInt($months.eq(dir === 'next' ? $months.length - 1 : 0).attr('data-month'), 10);
          calendar.currentYear = parseInt($months.eq(dir === 'next' ? $months.length - 1 : 0).attr('data-year'), 10);
          currentLocaleMonth = $months.eq(dir === 'next' ? $months.length - 1 : 0).attr('data-locale-month');
          currentLocaleYear = $months.eq(dir === 'next' ? $months.length - 1 : 0).attr('data-locale-year');
        }
        $el.find('.current-month-value').text(monthNames[currentLocaleMonth]);
        $el.find('.current-year-value').text(currentLocaleYear);
      }
      update() {
        const calendar = this;
        const {
          currentYear,
          currentMonth,
          $wrapperEl
        } = calendar;
        const currentDate = new Date(currentYear, currentMonth);
        const prevMonthHtml = calendar.renderMonth(currentDate, 'prev');
        const currentMonthHtml = calendar.renderMonth(currentDate);
        const nextMonthHtml = calendar.renderMonth(currentDate, 'next');
        $wrapperEl.transition(0).html(`${prevMonthHtml}${currentMonthHtml}${nextMonthHtml}`).transform('translate3d(0,0,0)');
        calendar.$months = $wrapperEl.find('.calendar-month');
        calendar.monthsTranslate = 0;
        calendar.setMonthsTranslate();
        calendar.$months.each(monthEl => {
          calendar.emit('local::monthAdd calendarMonthAdd', monthEl);
        });
      }
      onMonthChangeStart(dir) {
        const calendar = this;
        const {
          $months,
          currentYear,
          currentMonth
        } = calendar;
        calendar.updateCurrentMonthYear(dir);
        $months.removeClass('calendar-month-current calendar-month-prev calendar-month-next');
        const currentIndex = dir === 'next' ? $months.length - 1 : 0;
        $months.eq(currentIndex).addClass('calendar-month-current');
        $months.eq(dir === 'next' ? currentIndex - 1 : currentIndex + 1).addClass(dir === 'next' ? 'calendar-month-prev' : 'calendar-month-next');
        calendar.emit('local::monthYearChangeStart calendarMonthYearChangeStart', calendar, currentYear, currentMonth);
      }
      onMonthChangeEnd(dir, rebuildBoth) {
        const calendar = this;
        const {
          currentYear,
          currentMonth,
          $wrapperEl,
          monthsTranslate
        } = calendar;
        calendar.animating = false;
        let nextMonthHtml;
        let prevMonthHtml;
        let currentMonthHtml;
        $wrapperEl.find('.calendar-month:not(.calendar-month-prev):not(.calendar-month-current):not(.calendar-month-next)').remove();
        if (typeof dir === 'undefined') {
          dir = 'next'; // eslint-disable-line
          rebuildBoth = true; // eslint-disable-line
        }

        if (!rebuildBoth) {
          currentMonthHtml = calendar.renderMonth(new Date(currentYear, currentMonth), dir);
        } else {
          $wrapperEl.find('.calendar-month-next, .calendar-month-prev').remove();
          prevMonthHtml = calendar.renderMonth(new Date(currentYear, currentMonth), 'prev');
          nextMonthHtml = calendar.renderMonth(new Date(currentYear, currentMonth), 'next');
        }
        if (dir === 'next' || rebuildBoth) {
          $wrapperEl.append(currentMonthHtml || nextMonthHtml);
        }
        if (dir === 'prev' || rebuildBoth) {
          $wrapperEl.prepend(currentMonthHtml || prevMonthHtml);
        }
        const $months = $wrapperEl.find('.calendar-month');
        calendar.$months = $months;
        calendar.setMonthsTranslate(monthsTranslate);
        calendar.emit('local::monthAdd calendarMonthAdd', calendar, dir === 'next' ? $months.eq($months.length - 1)[0] : $months.eq(0)[0]);
        calendar.emit('local::monthYearChangeEnd calendarMonthYearChangeEnd', calendar, currentYear, currentMonth);
      }
      setMonthsTranslate(translate) {
        const calendar = this;
        const {
          $months,
          isHorizontal: isH,
          inverter
        } = calendar;
        // eslint-disable-next-line
        translate = translate || calendar.monthsTranslate || 0;
        if (typeof calendar.monthsTranslate === 'undefined') {
          calendar.monthsTranslate = translate;
        }
        $months.removeClass('calendar-month-current calendar-month-prev calendar-month-next');
        const prevMonthTranslate = -(translate + 1) * 100 * inverter;
        const currentMonthTranslate = -translate * 100 * inverter;
        const nextMonthTranslate = -(translate - 1) * 100 * inverter;
        $months.eq(0).transform(`translate3d(${isH ? prevMonthTranslate : 0}%, ${isH ? 0 : prevMonthTranslate}%, 0)`).addClass('calendar-month-prev');
        $months.eq(1).transform(`translate3d(${isH ? currentMonthTranslate : 0}%, ${isH ? 0 : currentMonthTranslate}%, 0)`).addClass('calendar-month-current');
        $months.eq(2).transform(`translate3d(${isH ? nextMonthTranslate : 0}%, ${isH ? 0 : nextMonthTranslate}%, 0)`).addClass('calendar-month-next');
      }
      nextMonth(transition) {
        const calendar = this;
        const {
          params,
          $wrapperEl,
          inverter,
          isHorizontal: isH
        } = calendar;
        if (typeof transition === 'undefined' || typeof transition === 'object') {
          transition = ''; // eslint-disable-line
          if (!params.animate) transition = 0; // eslint-disable-line
        }

        const nextMonth = parseInt(calendar.$months.eq(calendar.$months.length - 1).attr('data-month'), 10);
        const nextYear = parseInt(calendar.$months.eq(calendar.$months.length - 1).attr('data-year'), 10);
        const nextDate = new Date(nextYear, nextMonth);
        const nextDateTime = nextDate.getTime();
        const transitionEndCallback = !calendar.animating;
        if (params.maxDate) {
          if (nextDateTime > new Date(params.maxDate).getTime()) {
            calendar.resetMonth();
            return;
          }
        }
        calendar.monthsTranslate -= 1;
        if (nextMonth === calendar.currentMonth) {
          const nextMonthTranslate = -calendar.monthsTranslate * 100 * inverter;
          const nextMonthHtml = $(calendar.renderMonth(nextDateTime, 'next')).transform(`translate3d(${isH ? nextMonthTranslate : 0}%, ${isH ? 0 : nextMonthTranslate}%, 0)`).addClass('calendar-month-next');
          $wrapperEl.append(nextMonthHtml[0]);
          calendar.$months = $wrapperEl.find('.calendar-month');
          calendar.emit('local::monthAdd calendarMonthAdd', calendar.$months.eq(calendar.$months.length - 1)[0]);
        }
        calendar.animating = true;
        calendar.onMonthChangeStart('next');
        const translate = calendar.monthsTranslate * 100 * inverter;
        $wrapperEl.transition(transition).transform(`translate3d(${isH ? translate : 0}%, ${isH ? 0 : translate}%, 0)`);
        if (transitionEndCallback) {
          $wrapperEl.transitionEnd(() => {
            calendar.onMonthChangeEnd('next');
          });
        }
        if (!params.animate) {
          calendar.onMonthChangeEnd('next');
        }
      }
      prevMonth(transition) {
        const calendar = this;
        const {
          params,
          $wrapperEl,
          inverter,
          isHorizontal: isH
        } = calendar;
        if (typeof transition === 'undefined' || typeof transition === 'object') {
          transition = ''; // eslint-disable-line
          if (!params.animate) transition = 0; // eslint-disable-line
        }

        const prevMonth = parseInt(calendar.$months.eq(0).attr('data-month'), 10);
        const prevYear = parseInt(calendar.$months.eq(0).attr('data-year'), 10);
        const prevDate = new Date(prevYear, prevMonth + 1, -1);
        const prevDateTime = prevDate.getTime();
        const transitionEndCallback = !calendar.animating;
        if (params.minDate) {
          let minDate = new Date(params.minDate);
          minDate = new Date(minDate.getFullYear(), minDate.getMonth(), 1);
          if (prevDateTime < minDate.getTime()) {
            calendar.resetMonth();
            return;
          }
        }
        calendar.monthsTranslate += 1;
        if (prevMonth === calendar.currentMonth) {
          const prevMonthTranslate = -calendar.monthsTranslate * 100 * inverter;
          const prevMonthHtml = $(calendar.renderMonth(prevDateTime, 'prev')).transform(`translate3d(${isH ? prevMonthTranslate : 0}%, ${isH ? 0 : prevMonthTranslate}%, 0)`).addClass('calendar-month-prev');
          $wrapperEl.prepend(prevMonthHtml[0]);
          calendar.$months = $wrapperEl.find('.calendar-month');
          calendar.emit('local::monthAdd calendarMonthAdd', calendar.$months.eq(0)[0]);
        }
        calendar.animating = true;
        calendar.onMonthChangeStart('prev');
        const translate = calendar.monthsTranslate * 100 * inverter;
        $wrapperEl.transition(transition).transform(`translate3d(${isH ? translate : 0}%, ${isH ? 0 : translate}%, 0)`);
        if (transitionEndCallback) {
          $wrapperEl.transitionEnd(() => {
            calendar.onMonthChangeEnd('prev');
          });
        }
        if (!params.animate) {
          calendar.onMonthChangeEnd('prev');
        }
      }
      resetMonth(transition) {
        if (transition === void 0) {
          transition = '';
        }
        const calendar = this;
        const {
          $wrapperEl,
          inverter,
          isHorizontal: isH,
          monthsTranslate
        } = calendar;
        const translate = monthsTranslate * 100 * inverter;
        $wrapperEl.transition(transition).transform(`translate3d(${isH ? translate : 0}%, ${isH ? 0 : translate}%, 0)`);
      }
      // eslint-disable-next-line
      setYearMonth(year, month, transition) {
        const calendar = this;
        const {
          params,
          isHorizontal: isH,
          $wrapperEl,
          inverter
        } = calendar;
        // eslint-disable-next-line
        if (typeof year === 'undefined') year = calendar.currentYear;
        // eslint-disable-next-line
        if (typeof month === 'undefined') month = calendar.currentMonth;
        if (typeof transition === 'undefined' || typeof transition === 'object') {
          // eslint-disable-next-line
          transition = '';
          // eslint-disable-next-line
          if (!params.animate) transition = 0;
        }
        let targetDate;
        if (year < calendar.currentYear) {
          targetDate = new Date(year, month + 1, -1).getTime();
        } else {
          targetDate = new Date(year, month).getTime();
        }
        if (params.maxDate && targetDate > new Date(params.maxDate).getTime()) {
          return false;
        }
        if (params.minDate) {
          let minDate = new Date(params.minDate);
          minDate = new Date(minDate.getFullYear(), minDate.getMonth(), 1);
          if (targetDate < minDate.getTime()) {
            return false;
          }
        }
        const currentDate = new Date(calendar.currentYear, calendar.currentMonth).getTime();
        const dir = targetDate > currentDate ? 'next' : 'prev';
        const newMonthHTML = calendar.renderMonth(new Date(year, month));
        calendar.monthsTranslate = calendar.monthsTranslate || 0;
        const prevTranslate = calendar.monthsTranslate;
        let monthTranslate;
        const transitionEndCallback = !calendar.animating && transition !== 0;
        if (targetDate > currentDate) {
          // To next
          calendar.monthsTranslate -= 1;
          if (!calendar.animating) calendar.$months.eq(calendar.$months.length - 1).remove();
          $wrapperEl.append(newMonthHTML);
          calendar.$months = $wrapperEl.find('.calendar-month');
          monthTranslate = -(prevTranslate - 1) * 100 * inverter;
          calendar.$months.eq(calendar.$months.length - 1).transform(`translate3d(${isH ? monthTranslate : 0}%, ${isH ? 0 : monthTranslate}%, 0)`).addClass('calendar-month-next');
        } else {
          // To prev
          calendar.monthsTranslate += 1;
          if (!calendar.animating) calendar.$months.eq(0).remove();
          $wrapperEl.prepend(newMonthHTML);
          calendar.$months = $wrapperEl.find('.calendar-month');
          monthTranslate = -(prevTranslate + 1) * 100 * inverter;
          calendar.$months.eq(0).transform(`translate3d(${isH ? monthTranslate : 0}%, ${isH ? 0 : monthTranslate}%, 0)`).addClass('calendar-month-prev');
        }
        calendar.emit('local::monthAdd calendarMonthAdd', dir === 'next' ? calendar.$months.eq(calendar.$months.length - 1)[0] : calendar.$months.eq(0)[0]);
        calendar.animating = true;
        calendar.onMonthChangeStart(dir);
        const wrapperTranslate = calendar.monthsTranslate * 100 * inverter;
        $wrapperEl.transition(transition).transform(`translate3d(${isH ? wrapperTranslate : 0}%, ${isH ? 0 : wrapperTranslate}%, 0)`);
        if (transitionEndCallback) {
          $wrapperEl.transitionEnd(() => {
            calendar.onMonthChangeEnd(dir, true);
          });
        }
        if (!params.animate || transition === 0) {
          calendar.onMonthChangeEnd(dir, true);
        }
      }
      nextYear() {
        const calendar = this;
        calendar.setYearMonth(calendar.currentYear + 1);
      }
      prevYear() {
        const calendar = this;
        calendar.setYearMonth(calendar.currentYear - 1);
      }
      // eslint-disable-next-line
      dateInRange(dayDate, range) {
        let match = false;
        let i;
        if (!range) return false;
        if (Array.isArray(range)) {
          for (i = 0; i < range.length; i += 1) {
            if (range[i].from || range[i].to) {
              if (range[i].from && range[i].to) {
                if (dayDate <= new Date(range[i].to).getTime() && dayDate >= new Date(range[i].from).getTime()) {
                  match = true;
                }
              } else if (range[i].from) {
                if (dayDate >= new Date(range[i].from).getTime()) {
                  match = true;
                }
              } else if (range[i].to) {
                if (dayDate <= new Date(range[i].to).getTime()) {
                  match = true;
                }
              }
            } else if (range[i].date) {
              if (dayDate === new Date(range[i].date).getTime()) {
                match = true;
              }
            } else if (dayDate === new Date(range[i]).getTime()) {
              match = true;
            }
          }
        } else if (range.from || range.to) {
          if (range.from && range.to) {
            if (dayDate <= new Date(range.to).getTime() && dayDate >= new Date(range.from).getTime()) {
              match = true;
            }
          } else if (range.from) {
            if (dayDate >= new Date(range.from).getTime()) {
              match = true;
            }
          } else if (range.to) {
            if (dayDate <= new Date(range.to).getTime()) {
              match = true;
            }
          }
        } else if (range.date) {
          match = dayDate === new Date(range.date).getTime();
        } else if (typeof range === 'function') {
          match = range(new Date(dayDate));
        }
        return match;
      }
      // eslint-disable-next-line
      daysInMonth(date) {
        const d = new Date(date);
        return new Date(d.getFullYear(), d.getMonth() + 1, 0).getDate();
      }
      renderMonths(date) {
        const calendar = this;
        if (calendar.params.renderMonths) {
          return calendar.params.renderMonths.call(calendar, date);
        }
        return $jsx$1("div", {
          class: "calendar-months-wrapper"
        }, calendar.renderMonth(date, 'prev'), calendar.renderMonth(date), calendar.renderMonth(date, 'next'));
      }
      renderMonth(d, offset) {
        const calendar = this;
        const {
          params,
          value
        } = calendar;
        if (params.renderMonth) {
          return params.renderMonth.call(calendar, d, offset);
        }
        let date = new Date(d);
        let year = date.getFullYear();
        let month = date.getMonth();
        let localeMonth = calendar.monthNames.indexOf(calendar.monthFormatter(date));
        if (localeMonth < 0) localeMonth = month;
        let localeYear = calendar.yearFormatter(date);
        if (offset === 'next') {
          if (month === 11) date = new Date(year + 1, 0);else date = new Date(year, month + 1, 1);
        }
        if (offset === 'prev') {
          if (month === 0) date = new Date(year - 1, 11);else date = new Date(year, month - 1, 1);
        }
        if (offset === 'next' || offset === 'prev') {
          month = date.getMonth();
          year = date.getFullYear();
          localeMonth = calendar.monthNames.indexOf(calendar.monthFormatter(date));
          if (localeMonth < 0) localeMonth = month;
          localeYear = calendar.yearFormatter(date);
        }
        const currentValues = [];
        const today = new Date().setHours(0, 0, 0, 0);
        const minDate = params.minDate ? new Date(params.minDate).getTime() : null;
        const maxDate = params.maxDate ? new Date(params.maxDate).getTime() : null;
        const rows = 6;
        const cols = 7;
        const daysInPrevMonth = calendar.daysInMonth(new Date(date.getFullYear(), date.getMonth()).getTime() - 10 * 24 * 60 * 60 * 1000);
        const daysInMonth = calendar.daysInMonth(date);
        const minDayNumber = params.firstDay === 6 ? 0 : 1;
        let monthHtml = '';
        let dayIndex = 0 + (params.firstDay - 1);
        let disabled;
        let hasEvents;
        let firstDayOfMonthIndex = new Date(date.getFullYear(), date.getMonth()).getDay();
        if (firstDayOfMonthIndex === 0) firstDayOfMonthIndex = 7;
        if (value && value.length) {
          for (let i = 0; i < value.length; i += 1) {
            currentValues.push(new Date(value[i]).setHours(0, 0, 0, 0));
          }
        }
        for (let row = 1; row <= rows; row += 1) {
          let rowHtml = '';
          for (let col = 1; col <= cols; col += 1) {
            dayIndex += 1;
            let dayDate;
            let dayNumber = dayIndex - firstDayOfMonthIndex;
            let addClass = '';
            if (row === 1 && col === 1 && dayNumber > minDayNumber && params.firstDay !== 1) {
              dayIndex -= 7;
              dayNumber = dayIndex - firstDayOfMonthIndex;
            }
            const weekDayIndex = col - 1 + params.firstDay > 6 ? col - 1 - 7 + params.firstDay : col - 1 + params.firstDay;
            if (dayNumber < 0) {
              dayNumber = daysInPrevMonth + dayNumber + 1;
              addClass += ' calendar-day-prev';
              dayDate = new Date(month - 1 < 0 ? year - 1 : year, month - 1 < 0 ? 11 : month - 1, dayNumber).getTime();
            } else {
              dayNumber += 1;
              if (dayNumber > daysInMonth) {
                dayNumber -= daysInMonth;
                addClass += ' calendar-day-next';
                dayDate = new Date(month + 1 > 11 ? year + 1 : year, month + 1 > 11 ? 0 : month + 1, dayNumber).getTime();
              } else {
                dayDate = new Date(year, month, dayNumber).getTime();
              }
            }
            // Today
            if (dayDate === today) addClass += ' calendar-day-today';

            // Selected
            if (params.rangePicker && currentValues.length === 2) {
              if (dayDate >= currentValues[0] && dayDate <= currentValues[1]) {
                addClass += ' calendar-day-selected';
              }
              if (currentValues[0] !== currentValues[1]) {
                if (dayDate > currentValues[0] && dayDate < currentValues[1]) {
                  addClass += ' calendar-day-selected-range';
                }
                if (dayDate === currentValues[0]) {
                  addClass += ' calendar-day-selected-left';
                }
                if (dayDate === currentValues[1]) {
                  addClass += ' calendar-day-selected-right';
                }
              }
            } else if (currentValues.indexOf(dayDate) >= 0) addClass += ' calendar-day-selected';
            // Weekend
            if (params.weekendDays.indexOf(weekDayIndex) >= 0) {
              addClass += ' calendar-day-weekend';
            }
            // Events
            let eventsHtml = '';
            hasEvents = false;
            if (params.events) {
              if (calendar.dateInRange(dayDate, params.events)) {
                hasEvents = true;
              }
            }
            if (hasEvents) {
              addClass += ' calendar-day-has-events';
              // prettier-ignore
              eventsHtml = `
            <span class="calendar-day-events">
              <span class="calendar-day-event"></span>
            </span>
          `;
              if (Array.isArray(params.events)) {
                const eventDots = [];
                params.events.forEach(ev => {
                  const color = ev.color || '';
                  if (eventDots.indexOf(color) < 0 && calendar.dateInRange(dayDate, ev)) {
                    eventDots.push(color);
                  }
                });
                // prettier-ignore
                eventsHtml = `
              <span class="calendar-day-events">
                ${eventDots.map(color => `
                  <span class="calendar-day-event" style="${color ? `background-color: ${color}` : ''}"></span>
                `.trim()).join('')}
              </span>
            `;
              }
            }
            // Custom Ranges
            if (params.rangesClasses) {
              for (let k = 0; k < params.rangesClasses.length; k += 1) {
                if (calendar.dateInRange(dayDate, params.rangesClasses[k].range)) {
                  addClass += ` ${params.rangesClasses[k].cssClass}`;
                }
              }
            }
            // Disabled
            disabled = false;
            if (minDate && dayDate < minDate || maxDate && dayDate > maxDate) {
              disabled = true;
            }
            if (params.disabled) {
              if (calendar.dateInRange(dayDate, params.disabled)) {
                disabled = true;
              }
            }
            if (disabled) {
              addClass += ' calendar-day-disabled';
            }
            dayDate = new Date(dayDate);
            const dayYear = dayDate.getFullYear();
            const dayMonth = dayDate.getMonth();
            const dayNumberDisplay = calendar.dayFormatter(dayDate);
            // prettier-ignore
            rowHtml += `
          <div data-year="${dayYear}" data-month="${dayMonth}" data-day="${dayNumber}" class="calendar-day${addClass}" data-date="${dayYear}-${dayMonth}-${dayNumber}">
            <span class="calendar-day-number">${dayNumberDisplay}${eventsHtml}</span>
          </div>`.trim();
          }
          monthHtml += `<div class="calendar-row">${rowHtml}</div>`;
        }
        monthHtml = `<div class="calendar-month" data-year="${year}" data-month="${month}" data-locale-year="${localeYear}" data-locale-month="${localeMonth}">${monthHtml}</div>`;
        return monthHtml;
      }
      renderWeekHeader() {
        const calendar = this;
        if (calendar.params.renderWeekHeader) {
          return calendar.params.renderWeekHeader.call(calendar);
        }
        const {
          params
        } = calendar;
        let weekDaysHtml = '';
        for (let i = 0; i < 7; i += 1) {
          const dayIndex = i + params.firstDay > 6 ? i - 7 + params.firstDay : i + params.firstDay;
          const dayName = calendar.dayNamesShort[dayIndex];
          weekDaysHtml += `<div class="calendar-week-day">${dayName}</div>`;
        }
        return $jsx$1("div", {
          class: "calendar-week-header"
        }, weekDaysHtml);
      }
      renderMonthSelector() {
        const calendar = this;
        if (calendar.params.renderMonthSelector) {
          return calendar.params.renderMonthSelector.call(calendar);
        }
        return $jsx$1("div", {
          class: "calendar-month-selector"
        }, $jsx$1("a", {
          class: "link icon-only calendar-prev-month-button"
        }, $jsx$1("i", {
          class: "icon icon-prev"
        })), calendar.params.monthPicker ? $jsx$1("a", {
          class: "current-month-value link"
        }) : $jsx$1("span", {
          class: "current-month-value"
        }), $jsx$1("a", {
          class: "link icon-only calendar-next-month-button"
        }, $jsx$1("i", {
          class: "icon icon-next"
        })));
      }
      renderYearSelector() {
        const calendar = this;
        if (calendar.params.renderYearSelector) {
          return calendar.params.renderYearSelector.call(calendar);
        }
        return $jsx$1("div", {
          class: "calendar-year-selector"
        }, $jsx$1("a", {
          class: "link icon-only calendar-prev-year-button"
        }, $jsx$1("i", {
          class: "icon icon-prev"
        })), calendar.params.yearPicker ? $jsx$1("a", {
          class: "current-year-value link"
        }) : $jsx$1("span", {
          class: "current-year-value"
        }), $jsx$1("a", {
          class: "link icon-only calendar-next-year-button"
        }, $jsx$1("i", {
          class: "icon icon-next"
        })));
      }

      // eslint-disable-next-line
      renderTimeSelector() {
        const calendar = this;
        const value = calendar.value && calendar.value[0];
        let timeString;
        if (value) timeString = calendar.timeSelectorFormatter(value);
        return $jsx$1("div", {
          class: "calendar-time-selector"
        }, $jsx$1("span", null, calendar.params.timePickerLabel), $jsx$1("a", {
          class: "link"
        }, timeString || calendar.params.timePickerPlaceholder));
      }
      renderHeader() {
        const calendar = this;
        if (calendar.params.renderHeader) {
          return calendar.params.renderHeader.call(calendar);
        }
        return $jsx$1("div", {
          class: "calendar-header"
        }, $jsx$1("div", {
          class: "calendar-selected-date"
        }, calendar.params.headerPlaceholder));
      }
      renderFooter() {
        const calendar = this;
        const app = calendar.app;
        if (calendar.params.renderFooter) {
          return calendar.params.renderFooter.call(calendar);
        }
        return $jsx$1("div", {
          class: "calendar-footer"
        }, $jsx$1("a", {
          class: `${app.theme === 'md' ? 'button button-round' : 'link'} calendar-close sheet-close popover-close`
        }, calendar.params.toolbarCloseText));
      }
      renderToolbar() {
        const calendar = this;
        if (calendar.params.renderToolbar) {
          return calendar.params.renderToolbar.call(calendar, calendar);
        }
        // prettier-ignore
        return $jsx$1("div", {
          class: "toolbar toolbar-top"
        }, $jsx$1("div", {
          class: "toolbar-inner"
        }, calendar.params.monthSelector ? calendar.renderMonthSelector() : '', calendar.params.yearSelector ? calendar.renderYearSelector() : ''));
      }
      // eslint-disable-next-line
      renderInline() {
        const calendar = this;
        const {
          cssClass,
          toolbar,
          header,
          footer,
          rangePicker,
          weekHeader
        } = calendar.params;
        const {
          value,
          hasTimePicker
        } = calendar;
        const date = value && value.length ? value[0] : new Date().setHours(0, 0, 0);
        return $jsx$1("div", {
          class: `calendar calendar-inline ${rangePicker ? 'calendar-range' : ''} ${cssClass || ''}`
        }, header && calendar.renderHeader(), toolbar && calendar.renderToolbar(), weekHeader && calendar.renderWeekHeader(), $jsx$1("div", {
          class: "calendar-months"
        }, calendar.renderMonths(date)), hasTimePicker && calendar.renderTimeSelector(), footer && calendar.renderFooter());
      }
      renderCustomModal() {
        const calendar = this;
        const {
          cssClass,
          toolbar,
          header,
          footer,
          rangePicker,
          weekHeader
        } = calendar.params;
        const {
          value,
          hasTimePicker
        } = calendar;
        const date = value && value.length ? value[0] : new Date().setHours(0, 0, 0);
        return $jsx$1("div", {
          class: `calendar calendar-modal ${rangePicker ? 'calendar-range' : ''} ${cssClass || ''}`
        }, header && calendar.renderHeader(), toolbar && calendar.renderToolbar(), weekHeader && calendar.renderWeekHeader(), $jsx$1("div", {
          class: "calendar-months"
        }, calendar.renderMonths(date)), hasTimePicker && calendar.renderTimeSelector(), footer && calendar.renderFooter());
      }
      renderSheet() {
        const calendar = this;
        const {
          cssClass,
          toolbar,
          header,
          footer,
          rangePicker,
          weekHeader
        } = calendar.params;
        const {
          value,
          hasTimePicker
        } = calendar;
        const date = value && value.length ? value[0] : new Date().setHours(0, 0, 0);
        return $jsx$1("div", {
          class: `sheet-modal calendar calendar-sheet ${rangePicker ? 'calendar-range' : ''} ${cssClass || ''}`
        }, header && calendar.renderHeader(), toolbar && calendar.renderToolbar(), weekHeader && calendar.renderWeekHeader(), $jsx$1("div", {
          class: "sheet-modal-inner calendar-months"
        }, calendar.renderMonths(date)), hasTimePicker && calendar.renderTimeSelector(), footer && calendar.renderFooter());
      }
      renderPopover() {
        const calendar = this;
        const {
          cssClass,
          toolbar,
          header,
          footer,
          rangePicker,
          weekHeader
        } = calendar.params;
        const {
          value,
          hasTimePicker
        } = calendar;
        const date = value && value.length ? value[0] : new Date().setHours(0, 0, 0);
        return $jsx$1("div", {
          class: "popover calendar-popover"
        }, $jsx$1("div", {
          class: "popover-inner"
        }, $jsx$1("div", {
          class: `calendar ${rangePicker ? 'calendar-range' : ''} ${cssClass || ''}`
        }, header && calendar.renderHeader(), toolbar && calendar.renderToolbar(), weekHeader && calendar.renderWeekHeader(), $jsx$1("div", {
          class: "calendar-months"
        }, calendar.renderMonths(date)), hasTimePicker && calendar.renderTimeSelector(), footer && calendar.renderFooter())));
      }
      render() {
        const calendar = this;
        const {
          params
        } = calendar;
        if (params.render) return params.render.call(calendar);
        if (!calendar.inline) {
          let modalType = params.openIn;
          if (modalType === 'auto') modalType = calendar.isPopover() ? 'popover' : 'sheet';
          if (modalType === 'popover') return calendar.renderPopover();
          if (modalType === 'sheet') return calendar.renderSheet();
          return calendar.renderCustomModal();
        }
        return calendar.renderInline();
      }
      openMonthPicker() {
        const calendar = this;
        const {
          $el,
          app
        } = calendar;
        if (!$el || !$el.length) return;
        $el.append('<div class="popover calendar-popover calendar-month-picker-popover"><div class="popover-inner"><div class="calendar-month-picker"></div></div></div>');
        calendar.monthPickerPopover = app.popover.create({
          el: $el.find('.calendar-month-picker-popover'),
          targetEl: $el.find('.calendar-month-selector'),
          backdrop: true,
          backdropUnique: true,
          on: {
            close() {
              calendar.closeMonthPicker();
            },
            closed() {
              if (calendar.monthPickerPopover.$el) calendar.monthPickerPopover.$el.remove();
              calendar.monthPickerPopover.destroy();
              if (calendar.monthPickerInstance) {
                calendar.monthPickerInstance.close();
                calendar.monthPickerInstance.destroy();
              }
              delete calendar.monthPickerInstance;
              delete calendar.monthPickerPopover;
            }
          }
        });
        calendar.monthPickerPopover.open();
        const localeMonth = parseInt(calendar.$el.find('.calendar-month-current').attr('data-locale-month'), 10);
        const values = [];
        const displayValues = [];
        calendar.monthNames.forEach((m, index) => {
          values.push(index);
          displayValues.push(m);
        });
        calendar.monthPickerInstance = app.picker.create({
          containerEl: calendar.monthPickerPopover.$el.find('.calendar-month-picker'),
          value: [localeMonth],
          toolbar: calendar.params.monthPickerToolbar,
          rotateEffect: false,
          toolbarCloseText: calendar.params.monthPickerCloseText,
          cols: [{
            values,
            displayValues
          }]
        });
      }
      closeMonthPicker() {
        const calendar = this;
        if (calendar.monthPickerPopover && calendar.monthPickerPopover.opened) calendar.monthPickerPopover.close();
        const index = calendar.monthPickerInstance.value[0];
        const localeMonthIndex = parseInt(calendar.$el.find('.calendar-month-current').attr('data-locale-month'), 10);
        const monthIndex = calendar.currentMonth;
        const diff = localeMonthIndex - monthIndex;
        const diffIndex = index - diff;
        calendar.setYearMonth(calendar.currentYear, diffIndex, 0);
      }
      openYearPicker() {
        const calendar = this;
        const {
          $el,
          app
        } = calendar;
        if (!$el || !$el.length) return;
        $el.append('<div class="popover calendar-popover calendar-year-picker-popover"><div class="popover-inner"><div class="calendar-year-picker"></div></div></div>');
        calendar.yearPickerPopover = app.popover.create({
          el: $el.find('.calendar-year-picker-popover'),
          targetEl: $el.find('.calendar-year-selector'),
          backdrop: true,
          backdropUnique: true,
          on: {
            close() {
              calendar.closeYearPicker();
            },
            closed() {
              if (calendar.yearPickerPopover.$el) calendar.yearPickerPopover.$el.remove();
              calendar.yearPickerPopover.destroy();
              if (calendar.yearPickerInstance) {
                calendar.yearPickerInstance.close();
                calendar.yearPickerInstance.destroy();
              }
              delete calendar.yearPickerInstance;
              delete calendar.yearPickerPopover;
            }
          }
        });
        calendar.yearPickerPopover.open();
        const currentYear = calendar.currentYear;
        let yearMin = calendar.params.yearPickerMin || new Date().getFullYear() - 100;
        if (calendar.params.minDate) {
          yearMin = Math.max(yearMin, new Date(calendar.params.minDate).getFullYear());
        }
        let yearMax = calendar.params.yearPickerMax || new Date().getFullYear() + 100;
        if (calendar.params.maxDate) {
          yearMax = Math.min(yearMax, new Date(calendar.params.maxDate).getFullYear());
        }
        const years = [];
        for (let i = yearMin; i <= yearMax; i += 1) {
          years.push(i);
        }
        calendar.yearPickerInstance = app.picker.create({
          containerEl: calendar.yearPickerPopover.$el.find('.calendar-year-picker'),
          value: [currentYear],
          toolbar: calendar.params.yearPickerToolbar,
          rotateEffect: false,
          toolbarCloseText: calendar.params.yearPickerCloseText,
          cols: [{
            values: years
          }]
        });
      }
      closeYearPicker() {
        const calendar = this;
        if (calendar.yearPickerPopover && calendar.yearPickerPopover.opened) calendar.yearPickerPopover.close();
        calendar.setYearMonth(calendar.yearPickerInstance.value[0], undefined, 0);
      }
      openTimePicker() {
        const calendar = this;
        const {
          $el,
          app,
          is12HoursFormat
        } = calendar;
        if (!$el || !$el.length) return;
        $el.append('<div class="popover calendar-popover calendar-time-picker-popover"><div class="popover-inner"><div class="calendar-time-picker"></div></div></div>');
        const hoursArr = [];
        const minutesArr = [];
        const hoursMin = is12HoursFormat ? 1 : 0;
        const hoursMax = is12HoursFormat ? 12 : 23;
        for (let i = hoursMin; i <= hoursMax; i += 1) {
          hoursArr.push(i);
        }
        for (let i = 0; i <= 59; i += 1) {
          minutesArr.push(i);
        }
        let value;
        if (calendar.value && calendar.value.length) {
          value = [calendar.value[0].getHours(), calendar.value[0].getMinutes()];
        } else {
          value = [new Date().getHours(), new Date().getMinutes()];
        }
        if (is12HoursFormat) {
          value.push(value[0] < 12 ? 'AM' : 'PM');
          if (value[0] > 12) value[0] -= 12;
          if (value[0] === 0) value[0] = 12;
        }
        calendar.timePickerPopover = app.popover.create({
          el: $el.find('.calendar-time-picker-popover'),
          targetEl: $el.find('.calendar-time-selector .link'),
          backdrop: true,
          backdropUnique: true,
          on: {
            close() {
              calendar.closeTimePicker();
            },
            closed() {
              if (calendar.timePickerPopover.$el) calendar.timePickerPopover.$el.remove();
              calendar.timePickerPopover.destroy();
              if (calendar.timePickerInstance) {
                calendar.timePickerInstance.close();
                calendar.timePickerInstance.destroy();
              }
              delete calendar.timePickerInstance;
              delete calendar.timePickerPopover;
            }
          }
        });
        calendar.timePickerPopover.open();
        calendar.timePickerInstance = app.picker.create({
          containerEl: calendar.timePickerPopover.$el.find('.calendar-time-picker'),
          value,
          toolbar: calendar.params.timePickerToolbar,
          rotateEffect: false,
          toolbarCloseText: calendar.params.timePickerCloseText,
          cols: [{
            values: hoursArr
          }, {
            divider: true,
            content: ':'
          }, {
            values: minutesArr,
            displayValues: minutesArr.map(m => m < 10 ? `0${m}` : m)
          }, ...(is12HoursFormat ? [{
            values: ['AM', 'PM']
          }] : [])]
        });
      }
      closeTimePicker() {
        const calendar = this;
        const {
          is12HoursFormat
        } = calendar;
        if (calendar.timePickerInstance) {
          const timePickerValue = calendar.timePickerInstance.value;
          let hours = parseInt(timePickerValue[0], 10);
          const minutes = parseInt(timePickerValue[1], 10);
          const period = calendar.timePickerInstance.value[2];
          if (is12HoursFormat) {
            if (period === 'AM' && hours === 12) {
              hours = 0;
            } else if (period === 'PM' && hours !== 12) {
              hours += 12;
            }
          }
          let value = calendar.value && calendar.value.length && calendar.value[0];
          if (!value) {
            value = new Date();
            value.setHours(hours, minutes, 0, 0);
          } else {
            value = new Date(value);
            value.setHours(hours, minutes);
          }
          calendar.setValue([value]);
          if (calendar.timePickerPopover && calendar.timePickerPopover.opened) calendar.timePickerPopover.close();
        }
      }
      onOpen() {
        const calendar = this;
        const {
          initialized,
          $el,
          app,
          $inputEl,
          inline,
          value,
          params
        } = calendar;
        calendar.closing = false;
        calendar.opened = true;
        calendar.opening = true;

        // Init main events
        calendar.attachCalendarEvents();
        const updateValue = !value && params.value;

        // Set value
        if (!initialized) {
          if (value) calendar.setValue(value, 0);else if (params.value) {
            calendar.setValue(calendar.normalizeValues(params.value), 0);
          }
        } else if (value) {
          calendar.setValue(value, 0);
        }

        // Update current month and year
        calendar.updateCurrentMonthYear();

        // Set initial translate
        calendar.monthsTranslate = 0;
        calendar.setMonthsTranslate();

        // Update input value
        if (updateValue) calendar.updateValue();else if (params.header && value) {
          calendar.updateValue(true);
        }

        // Extra focus
        if (!inline && $inputEl && $inputEl.length && app.theme === 'md') {
          $inputEl.trigger('focus');
        }
        calendar.initialized = true;
        calendar.$months.each(monthEl => {
          calendar.emit('local::monthAdd calendarMonthAdd', monthEl);
        });

        // Trigger events
        if ($el) {
          $el.trigger('calendar:open');
        }
        if ($inputEl) {
          $inputEl.trigger('calendar:open');
        }
        calendar.emit('local::open calendarOpen', calendar);
      }
      onOpened() {
        const calendar = this;
        calendar.opening = false;
        if (calendar.$el) {
          calendar.$el.trigger('calendar:opened');
        }
        if (calendar.$inputEl) {
          calendar.$inputEl.trigger('calendar:opened');
        }
        calendar.emit('local::opened calendarOpened', calendar);
      }
      onClose() {
        const calendar = this;
        const app = calendar.app;
        calendar.opening = false;
        calendar.closing = true;
        if (calendar.$inputEl) {
          if (app.theme === 'md') {
            calendar.$inputEl.trigger('blur');
          } else {
            const validate = calendar.$inputEl.attr('validate');
            const required = calendar.$inputEl.attr('required');
            if (validate && required) {
              app.input.validate(calendar.$inputEl);
            }
          }
        }
        if (calendar.detachCalendarEvents) {
          calendar.detachCalendarEvents();
        }
        if (calendar.$el) {
          calendar.$el.trigger('calendar:close');
        }
        if (calendar.$inputEl) {
          calendar.$inputEl.trigger('calendar:close');
        }
        calendar.emit('local::close calendarClose', calendar);
      }
      onClosed() {
        const calendar = this;
        calendar.opened = false;
        calendar.closing = false;
        if (!calendar.inline) {
          nextTick$1(() => {
            if (calendar.modal && calendar.modal.el && calendar.modal.destroy) {
              if (!calendar.params.routableModals) {
                calendar.modal.destroy();
              }
            }
            delete calendar.modal;
          });
        }
        if (calendar.timePickerInstance) {
          if (calendar.timePickerInstance.destroy) calendar.timePickerInstance.destroy();
          delete calendar.timePickerInstance;
        }
        if (calendar.$el) {
          calendar.$el.trigger('calendar:closed');
        }
        if (calendar.$inputEl) {
          calendar.$inputEl.trigger('calendar:closed');
        }
        calendar.emit('local::closed calendarClosed', calendar);
      }
      open() {
        const calendar = this;
        const {
          app,
          opened,
          inline,
          $inputEl,
          params
        } = calendar;
        if (opened) return;
        if (inline) {
          calendar.$el = $(calendar.render());
          calendar.$el[0].f7Calendar = calendar;
          calendar.$wrapperEl = calendar.$el.find('.calendar-months-wrapper');
          calendar.$months = calendar.$wrapperEl.find('.calendar-month');
          calendar.$containerEl.append(calendar.$el);
          calendar.onOpen();
          calendar.onOpened();
          return;
        }
        let modalType = params.openIn;
        if (modalType === 'auto') {
          modalType = calendar.isPopover() ? 'popover' : 'sheet';
        }
        const modalContent = calendar.render();
        const modalParams = {
          targetEl: $inputEl,
          scrollToEl: params.scrollToInput ? $inputEl : undefined,
          content: modalContent,
          backdrop: params.backdrop === true || (modalType === 'popover' || modalType === 'customModal') && app.params.popover.backdrop !== false && params.backdrop !== false,
          closeByBackdropClick: params.closeByBackdropClick,
          on: {
            open() {
              const modal = this;
              calendar.modal = modal;
              calendar.$el = modalType === 'popover' ? modal.$el.find('.calendar') : modal.$el;
              calendar.$wrapperEl = calendar.$el.find('.calendar-months-wrapper');
              calendar.$months = calendar.$wrapperEl.find('.calendar-month');
              calendar.$el[0].f7Calendar = calendar;
              if (modalType === 'customModal') {
                $(calendar.$el).find('.calendar-close').once('click', () => {
                  calendar.close();
                });
              }
              calendar.onOpen();
            },
            opened() {
              calendar.onOpened();
            },
            close() {
              calendar.onClose();
            },
            closed() {
              calendar.onClosed();
            }
          }
        };
        if (modalType === 'sheet') {
          modalParams.push = params.sheetPush;
          modalParams.swipeToClose = params.sheetSwipeToClose;
        }
        if (params.routableModals && calendar.view) {
          calendar.view.router.navigate({
            url: calendar.url,
            route: {
              path: calendar.url,
              [modalType]: modalParams
            }
          });
        } else {
          calendar.modal = app[modalType].create(modalParams);
          calendar.modal.open();
        }
      }
      close() {
        const calendar = this;
        const {
          opened,
          inline
        } = calendar;
        if (!opened) return;
        if (inline) {
          calendar.onClose();
          calendar.onClosed();
          return;
        }
        if (calendar.params.routableModals && calendar.view) {
          calendar.view.router.back();
        } else {
          calendar.modal.close();
        }
      }
      init() {
        const calendar = this;
        calendar.initInput();
        if (calendar.inline) {
          calendar.open();
          calendar.emit('local::init calendarInit', calendar);
          return;
        }
        if (!calendar.initialized && calendar.params.value) {
          calendar.setValue(calendar.normalizeValues(calendar.params.value));
        }

        // Attach input Events
        if (calendar.$inputEl) {
          calendar.attachInputEvents();
        }
        if (calendar.params.closeByOutsideClick) {
          calendar.attachHtmlEvents();
        }
        calendar.emit('local::init calendarInit', calendar);
      }
      destroy() {
        const calendar = this;
        if (calendar.destroyed) return;
        const {
          $el
        } = calendar;
        calendar.emit('local::beforeDestroy calendarBeforeDestroy', calendar);
        if ($el) $el.trigger('calendar:beforedestroy');
        calendar.close();

        // Detach Events
        if (calendar.$inputEl) {
          calendar.detachInputEvents();
        }
        if (calendar.params.closeByOutsideClick) {
          calendar.detachHtmlEvents();
        }
        if (calendar.timePickerInstance) {
          if (calendar.timePickerInstance.destroy) calendar.timePickerInstance.destroy();
          delete calendar.timePickerInstance;
        }
        if ($el && $el.length) delete calendar.$el[0].f7Calendar;
        deleteProps$1(calendar);
        calendar.destroyed = true;
      }
    };

    var Calendar = {
      name: 'calendar',
      static: {
        Calendar: Calendar$1
      },
      create() {
        const app = this;
        app.calendar = ConstructorMethods({
          defaultSelector: '.calendar',
          constructor: Calendar$1,
          app,
          domProp: 'f7Calendar'
        });
        app.calendar.close = function close(el) {
          if (el === void 0) {
            el = '.calendar';
          }
          const $el = $(el);
          if ($el.length === 0) return;
          const calendar = $el[0].f7Calendar;
          if (!calendar || calendar && !calendar.opened) return;
          calendar.close();
        };
      },
      params: {
        calendar: {
          // Calendar settings
          dateFormat: undefined,
          monthNames: 'auto',
          monthNamesShort: 'auto',
          dayNames: 'auto',
          dayNamesShort: 'auto',
          locale: undefined,
          firstDay: 1,
          // First day of the week, Monday
          weekendDays: [0, 6],
          // Sunday and Saturday
          multiple: false,
          rangePicker: false,
          rangePickerMinDays: 1,
          // when calendar is used as rangePicker
          rangePickerMaxDays: 0,
          // when calendar is used as rangePicker, 0 means unlimited
          direction: 'horizontal',
          // or 'vertical'
          minDate: null,
          maxDate: null,
          disabled: null,
          // dates range of disabled days
          events: null,
          // dates range of days with events
          rangesClasses: null,
          // array with custom classes date ranges
          touchMove: true,
          animate: true,
          closeOnSelect: false,
          monthSelector: true,
          monthPicker: true,
          monthPickerToolbar: true,
          monthPickerCloseText: 'Done',
          yearSelector: true,
          yearPicker: true,
          yearPickerToolbar: true,
          yearPickerMin: undefined,
          yearPickerMax: undefined,
          yearPickerCloseText: 'Done',
          timePicker: false,
          timePickerToolbar: true,
          timePickerLabel: 'Time',
          timePickerFormat: {
            hour: 'numeric',
            minute: 'numeric'
          },
          timePickerPlaceholder: 'Select time',
          timePickerCloseText: 'Done',
          weekHeader: true,
          value: null,
          // Common opener settings
          containerEl: null,
          openIn: 'auto',
          // or 'popover' or 'sheet' or 'customModal'
          sheetPush: false,
          sheetSwipeToClose: undefined,
          formatValue: null,
          inputEl: null,
          inputReadOnly: true,
          closeByOutsideClick: true,
          scrollToInput: true,
          header: false,
          headerPlaceholder: 'Select date',
          toolbar: true,
          toolbarCloseText: 'Done',
          footer: false,
          cssClass: null,
          routableModals: false,
          view: null,
          url: 'date/',
          backdrop: null,
          closeByBackdropClick: true,
          // Render functions
          renderWeekHeader: null,
          renderMonths: null,
          renderMonth: null,
          renderMonthSelector: null,
          renderYearSelector: null,
          renderHeader: null,
          renderFooter: null,
          renderToolbar: null,
          renderInline: null,
          renderPopover: null,
          renderSheet: null,
          render: null
        }
      }
    };

    function pickerColumn(colEl, updateItems) {
      const picker = this;
      // const app = picker.app;
      const $colEl = $(colEl);
      const colIndex = $colEl.index();
      const col = picker.cols[colIndex];
      if (col.divider) return;
      col.$el = $colEl;
      col.el = $colEl[0];
      col.$itemsEl = col.$el.find('.picker-items');
      col.items = col.$itemsEl.find('.picker-item');
      let itemHeight;
      let colHeight;
      col.replaceValues = function replaceColValues(values, displayValues) {
        col.detachEvents();
        col.values = values;
        col.displayValues = displayValues;
        col.$itemsEl.html(picker.renderColumn(col, true));
        col.items = col.$itemsEl.find('.picker-item');
        col.calcSize();
        col.setValue(col.values[0], true);
        col.attachEvents();
      };
      col.calcSize = function calcColSize() {
        colHeight = col.$el[0].offsetHeight;
        itemHeight = col.items[0].offsetHeight;
        const hadPadding = col.el.style.getPropertyValue('--f7-picker-scroll-padding');
        col.el.style.setProperty('--f7-picker-scroll-padding', `${(colHeight - itemHeight) / 2}px`);
        if (!hadPadding) {
          col.$itemsEl[0].scrollTop = 0;
        }
      };
      col.setValue = function setColValue(newValue, valueCallbacks) {
        const newActiveIndex = col.$itemsEl.find(`.picker-item[data-picker-value="${newValue}"]`).index();
        if (typeof newActiveIndex === 'undefined' || newActiveIndex === -1) {
          return;
        }
        const newScrollTop = newActiveIndex * itemHeight;
        col.$itemsEl[0].scrollTop = newScrollTop;

        // Update items
        col.updateItems(newActiveIndex, newScrollTop, valueCallbacks);
      };
      col.updateItems = function updateColItems(activeIndex, scrollTop, valueCallbacks) {
        if (typeof scrollTop === 'undefined') {
          // eslint-disable-next-line
          scrollTop = col.$itemsEl[0].scrollTop;
        }

        /* eslint-disable no-param-reassign */
        if (typeof activeIndex === 'undefined') {
          activeIndex = Math.round(scrollTop / itemHeight);
        }
        if (activeIndex < 0) activeIndex = 0;
        if (activeIndex >= col.items.length) activeIndex = col.items.length - 1;
        /* eslint-enable no-param-reassign */
        const previousActiveIndex = col.activeIndex;
        col.activeIndex = activeIndex;
        col.$itemsEl.find('.picker-item-selected').removeClass('picker-item-selected');
        const selectedItem = col.items.eq(activeIndex);
        selectedItem.addClass('picker-item-selected').children().transform('');
        // Set 3D rotate effect
        if (picker.params.rotateEffect) {
          col.items.each(itemEl => {
            const $itemEl = $(itemEl);
            const itemOffset = itemEl.offsetTop - (colHeight - itemHeight) / 2 - scrollTop;
            const percentage = itemOffset / itemHeight;
            const itemsFit = Math.ceil(col.height / itemHeight / 2) + 1;
            let angle = -24 * percentage;
            if (angle > 180) angle = 180;
            if (angle < -180) angle = -180;
            if (Math.abs(percentage) > itemsFit) {
              $itemEl.addClass('picker-item-far');
            } else {
              $itemEl.removeClass('picker-item-far');
            }
            $itemEl.children('span').transform(`translate3d(0, ${-percentage * itemHeight}px, -100px) rotateX(${angle}deg)`);
          });
        }
        if (valueCallbacks || typeof valueCallbacks === 'undefined') {
          // Update values
          col.value = selectedItem.attr('data-picker-value');
          col.displayValue = col.displayValues ? col.displayValues[activeIndex] : col.value;
          // On change callback
          if (previousActiveIndex !== activeIndex) {
            if (col.onChange) {
              col.onChange(picker, col.value, col.displayValue);
            }
            picker.updateValue();
          }
        }
      };
      function handleScroll() {
        col.updateItems();
      }
      function handleClick() {
        const value = $(this).attr('data-picker-value');
        col.setValue(value);
      }
      col.attachEvents = function attachColEvents() {
        col.$itemsEl.on('scroll', handleScroll);
        col.items.on('click', handleClick);
      };
      col.detachEvents = function detachColEvents() {
        col.items.off('click', handleClick);
      };
      col.init = function initCol() {
        col.calcSize();
        if (colIndex === 0) col.$el.addClass('picker-column-first');
        if (colIndex === picker.cols.length - 1) col.$el.addClass('picker-column-last');
        if (picker.params.freeMode) col.$el.addClass('picker-column-free-mode');
        // Update items on init
        if (updateItems) col.updateItems(0);
        col.attachEvents();
      };
      col.destroy = function destroyCol() {
        col.detachEvents();
      };
      col.init();
    }

    let Picker$1 = class Picker extends Framework7Class {
      constructor(app, params) {
        if (params === void 0) {
          params = {};
        }
        super(params, [app]);
        const picker = this;
        const device = getDevice$1();
        const window = getWindow$1();
        picker.params = extend$3({}, app.params.picker, params);
        let $containerEl;
        if (picker.params.containerEl) {
          $containerEl = $(picker.params.containerEl);
          if ($containerEl.length === 0) return picker;
        }
        let $inputEl;
        if (picker.params.inputEl) {
          $inputEl = $(picker.params.inputEl);
        }
        let $scrollToEl = picker.params.scrollToInput ? $inputEl : undefined;
        if (picker.params.scrollToEl) {
          const scrollToEl = $(picker.params.scrollToEl);
          if (scrollToEl.length > 0) {
            $scrollToEl = scrollToEl;
          }
        }
        extend$3(picker, {
          app,
          $containerEl,
          containerEl: $containerEl && $containerEl[0],
          inline: $containerEl && $containerEl.length > 0,
          needsOriginFix: device.ios || window.navigator.userAgent.toLowerCase().indexOf('safari') >= 0 && window.navigator.userAgent.toLowerCase().indexOf('chrome') < 0 && !device.android,
          cols: [],
          $inputEl,
          inputEl: $inputEl && $inputEl[0],
          $scrollToEl,
          initialized: false,
          opened: false,
          url: picker.params.url
        });
        function onResize() {
          picker.resizeCols();
        }
        function onInputClick() {
          picker.open();
        }
        function onInputFocus(e) {
          e.preventDefault();
        }
        let htmlTouchStartTarget = null;
        function onHtmlTouchStart(e) {
          htmlTouchStartTarget = e.target;
        }
        function onHtmlClick(e) {
          if (picker.destroyed || !picker.params) return;
          const $targetEl = $(e.target);
          if (picker.isPopover()) return;
          if (!picker.opened || picker.closing) return;
          if ($targetEl.closest('[class*="backdrop"]').length) return;
          if ($inputEl && $inputEl.length > 0) {
            if (htmlTouchStartTarget === e.target && $targetEl[0] !== $inputEl[0] && $targetEl.closest('.sheet-modal').length === 0) {
              picker.close();
            }
          } else if ($(e.target).closest('.sheet-modal').length === 0) {
            picker.close();
          }
        }

        // Events
        extend$3(picker, {
          attachResizeEvent() {
            app.on('resize', onResize);
          },
          detachResizeEvent() {
            app.off('resize', onResize);
          },
          attachInputEvents() {
            picker.$inputEl.on('click', onInputClick);
            if (picker.params.inputReadOnly) {
              picker.$inputEl.on('focus mousedown', onInputFocus);
              if (picker.$inputEl[0]) {
                picker.$inputEl[0].f7ValidateReadonly = true;
              }
            }
          },
          detachInputEvents() {
            picker.$inputEl.off('click', onInputClick);
            if (picker.params.inputReadOnly) {
              picker.$inputEl.off('focus mousedown', onInputFocus);
              if (picker.$inputEl[0]) {
                delete picker.$inputEl[0].f7ValidateReadonly;
              }
            }
          },
          attachHtmlEvents() {
            app.on('click', onHtmlClick);
            app.on('touchstart', onHtmlTouchStart);
          },
          detachHtmlEvents() {
            app.off('click', onHtmlClick);
            app.off('touchstart', onHtmlTouchStart);
          }
        });
        picker.init();
        return picker;
      }
      get view() {
        const {
          app,
          params,
          $inputEl
        } = this;
        let view;
        if (params.view) {
          view = params.view;
        } else if ($inputEl) {
          view = $inputEl.parents('.view').length && $inputEl.parents('.view')[0].f7View;
        }
        if (!view) view = app.views.main;
        return view;
      }
      initInput() {
        const picker = this;
        if (!picker.$inputEl) return;
        if (picker.params.inputReadOnly) picker.$inputEl.prop('readOnly', true);
      }
      resizeCols() {
        const picker = this;
        if (!picker.opened) return;
        for (let i = 0; i < picker.cols.length; i += 1) {
          if (!picker.cols[i].divider) {
            picker.cols[i].calcSize();
            picker.cols[i].setValue(picker.cols[i].value, false);
          }
        }
      }
      isPopover() {
        const picker = this;
        const {
          app,
          modal,
          params
        } = picker;
        const device = getDevice$1();
        if (params.openIn === 'sheet') return false;
        if (modal && modal.type !== 'popover') return false;
        if (!picker.inline && picker.inputEl) {
          if (params.openIn === 'popover') return true;
          if (device.ios) {
            return !!device.ipad;
          }
          if (app.width >= 768) {
            return true;
          }
        }
        return false;
      }
      formatValue() {
        const picker = this;
        const {
          value,
          displayValue
        } = picker;
        if (picker.params.formatValue) {
          return picker.params.formatValue.call(picker, value, displayValue);
        }
        return value.join(' ');
      }
      setValue(values) {
        const picker = this;
        let valueIndex = 0;
        if (picker.cols.length === 0) {
          picker.value = values;
          picker.updateValue(values);
          return;
        }
        for (let i = 0; i < picker.cols.length; i += 1) {
          if (picker.cols[i] && !picker.cols[i].divider) {
            picker.cols[i].setValue(values[valueIndex]);
            valueIndex += 1;
          }
        }
      }
      getValue() {
        const picker = this;
        return picker.value;
      }
      updateValue(forceValues) {
        const picker = this;
        const newValue = forceValues || [];
        const newDisplayValue = [];
        let column;
        if (picker.cols.length === 0) {
          const noDividerColumns = picker.params.cols.filter(c => !c.divider);
          for (let i = 0; i < noDividerColumns.length; i += 1) {
            column = noDividerColumns[i];
            if (column.displayValues !== undefined && column.values !== undefined && column.values.indexOf(newValue[i]) !== -1) {
              newDisplayValue.push(column.displayValues[column.values.indexOf(newValue[i])]);
            } else {
              newDisplayValue.push(newValue[i]);
            }
          }
        } else {
          for (let i = 0; i < picker.cols.length; i += 1) {
            if (!picker.cols[i].divider) {
              newValue.push(picker.cols[i].value);
              newDisplayValue.push(picker.cols[i].displayValue);
            }
          }
        }
        if (newValue.indexOf(undefined) >= 0) {
          return;
        }
        picker.value = newValue;
        picker.displayValue = newDisplayValue;
        picker.emit('local::change pickerChange', picker, picker.value, picker.displayValue);
        if (picker.inputEl) {
          picker.$inputEl.val(picker.formatValue());
          picker.$inputEl.trigger('change');
        }
      }
      initColumn(colEl, updateItems) {
        const picker = this;
        pickerColumn.call(picker, colEl, updateItems);
      }
      // eslint-disable-next-line
      destroyColumn(colEl) {
        const picker = this;
        const $colEl = $(colEl);
        const index = $colEl.index();
        if (picker.cols[index] && picker.cols[index].destroy) {
          picker.cols[index].destroy();
        }
      }
      renderToolbar() {
        const picker = this;
        if (picker.params.renderToolbar) return picker.params.renderToolbar.call(picker, picker);
        return $jsx$1("div", {
          class: "toolbar toolbar-top"
        }, $jsx$1("div", {
          class: "toolbar-inner"
        }, $jsx$1("div", {
          class: "left"
        }), $jsx$1("div", {
          class: "right"
        }, $jsx$1("a", {
          class: "link sheet-close popover-close"
        }, picker.params.toolbarCloseText))));
      }
      // eslint-disable-next-line
      renderColumn(col, onlyItems) {
        const colClasses = `picker-column ${col.textAlign ? `picker-column-${col.textAlign}` : ''} ${col.cssClass || ''}`;
        let columnHtml;
        let columnItemsHtml;
        if (col.divider) {
          // prettier-ignore
          columnHtml = `
        <div class="${colClasses} picker-column-divider">${col.content}</div>
      `;
        } else {
          // prettier-ignore
          columnItemsHtml = col.values.map((value, index) => `
        <div class="picker-item" data-picker-value="${value}">
          <span>${col.displayValues ? col.displayValues[index] : value}</span>
        </div>
      `).join('');
          // prettier-ignore
          columnHtml = `
        <div class="${colClasses}">
          <div class="picker-items">${columnItemsHtml}</div>
        </div>
      `;
        }
        return onlyItems ? columnItemsHtml.trim() : columnHtml.trim();
      }
      renderInline() {
        const picker = this;
        const {
          rotateEffect,
          cssClass,
          toolbar
        } = picker.params;
        const inlineHtml = $jsx$1("div", {
          class: `picker picker-inline ${rotateEffect ? 'picker-3d' : ''} ${cssClass || ''}`
        }, toolbar && picker.renderToolbar(), $jsx$1("div", {
          class: "picker-columns"
        }, picker.cols.map(col => picker.renderColumn(col)), $jsx$1("div", {
          class: "picker-center-highlight"
        })));
        return inlineHtml;
      }
      renderSheet() {
        const picker = this;
        const {
          rotateEffect,
          cssClass,
          toolbar
        } = picker.params;
        const sheetHtml = $jsx$1("div", {
          class: `sheet-modal picker picker-sheet ${rotateEffect ? 'picker-3d' : ''} ${cssClass || ''}`
        }, toolbar && picker.renderToolbar(), $jsx$1("div", {
          class: "sheet-modal-inner picker-columns"
        }, picker.cols.map(col => picker.renderColumn(col)), $jsx$1("div", {
          class: "picker-center-highlight"
        })));
        return sheetHtml;
      }
      renderPopover() {
        const picker = this;
        const {
          rotateEffect,
          cssClass,
          toolbar
        } = picker.params;
        const popoverHtml = $jsx$1("div", {
          class: "popover picker-popover"
        }, $jsx$1("div", {
          class: "popover-inner"
        }, $jsx$1("div", {
          class: `picker ${rotateEffect ? 'picker-3d' : ''} ${cssClass || ''}`
        }, toolbar && picker.renderToolbar(), $jsx$1("div", {
          class: "picker-columns"
        }, picker.cols.map(col => picker.renderColumn(col)), $jsx$1("div", {
          class: "picker-center-highlight"
        })))));
        return popoverHtml;
      }
      render() {
        const picker = this;
        if (picker.params.render) return picker.params.render.call(picker);
        if (!picker.inline) {
          if (picker.isPopover()) return picker.renderPopover();
          return picker.renderSheet();
        }
        return picker.renderInline();
      }
      onOpen() {
        const picker = this;
        const {
          initialized,
          $el,
          app,
          $inputEl,
          inline,
          value,
          params
        } = picker;
        picker.opened = true;
        picker.closing = false;
        picker.opening = true;

        // Init main events
        picker.attachResizeEvent();

        // Init cols
        $el.find('.picker-column').each(colEl => {
          let updateItems = true;
          if (!initialized && params.value || initialized && value) {
            updateItems = false;
          }
          picker.initColumn(colEl, updateItems);
        });

        // Set value
        if (!initialized) {
          if (value) picker.setValue(value);else if (params.value) {
            picker.setValue(params.value);
          }
        } else if (value) {
          picker.setValue(value);
        }

        // Extra focus
        if (!inline && $inputEl && $inputEl.length && app.theme === 'md') {
          $inputEl.trigger('focus');
        }
        picker.initialized = true;

        // Trigger events
        if ($el) {
          $el.trigger('picker:open');
        }
        if ($inputEl) {
          $inputEl.trigger('picker:open');
        }
        picker.emit('local::open pickerOpen', picker);
      }
      onOpened() {
        const picker = this;
        picker.opening = false;
        if (picker.$el) {
          picker.$el.trigger('picker:opened');
        }
        if (picker.$inputEl) {
          picker.$inputEl.trigger('picker:opened');
        }
        picker.emit('local::opened pickerOpened', picker);
      }
      onClose() {
        const picker = this;
        const app = picker.app;
        picker.opening = false;
        picker.closing = true;

        // Detach events
        picker.detachResizeEvent();
        picker.cols.forEach(col => {
          if (col.destroy) col.destroy();
        });
        if (picker.$inputEl) {
          if (app.theme === 'md') {
            picker.$inputEl.trigger('blur');
          } else {
            const validate = picker.$inputEl.attr('validate');
            const required = picker.$inputEl.attr('required');
            if (validate && required) {
              app.input.validate(picker.$inputEl);
            }
          }
        }
        if (picker.$el) {
          picker.$el.trigger('picker:close');
        }
        if (picker.$inputEl) {
          picker.$inputEl.trigger('picker:close');
        }
        picker.emit('local::close pickerClose', picker);
      }
      onClosed() {
        const picker = this;
        picker.opened = false;
        picker.closing = false;
        if (!picker.inline) {
          nextTick$1(() => {
            if (picker.modal && picker.modal.el && picker.modal.destroy) {
              if (!picker.params.routableModals) {
                picker.modal.destroy();
              }
            }
            delete picker.modal;
          });
        }
        if (picker.$el) {
          picker.$el.trigger('picker:closed');
        }
        if (picker.$inputEl) {
          picker.$inputEl.trigger('picker:closed');
        }
        picker.emit('local::closed pickerClosed', picker);
      }
      open() {
        const picker = this;
        const {
          app,
          opened,
          inline,
          $inputEl,
          $scrollToEl,
          params
        } = picker;
        if (opened) return;
        if (picker.cols.length === 0 && params.cols.length) {
          params.cols.forEach(col => {
            picker.cols.push(col);
          });
        }
        if (inline) {
          picker.$el = $(picker.render());
          picker.$el[0].f7Picker = picker;
          picker.$containerEl.append(picker.$el);
          picker.onOpen();
          picker.onOpened();
          return;
        }
        const isPopover = picker.isPopover();
        const modalType = isPopover ? 'popover' : 'sheet';
        const modalParams = {
          targetEl: $inputEl,
          scrollToEl: $scrollToEl,
          content: picker.render(),
          backdrop: typeof params.backdrop !== 'undefined' ? params.backdrop : isPopover,
          on: {
            open() {
              const modal = this;
              picker.modal = modal;
              picker.$el = isPopover ? modal.$el.find('.picker') : modal.$el;
              picker.$el[0].f7Picker = picker;
              picker.onOpen();
            },
            opened() {
              picker.onOpened();
            },
            close() {
              picker.onClose();
            },
            closed() {
              picker.onClosed();
            }
          }
        };
        if (modalType === 'sheet') {
          modalParams.push = params.sheetPush;
          modalParams.swipeToClose = params.sheetSwipeToClose;
        }
        if (params.routableModals && picker.view) {
          picker.view.router.navigate({
            url: picker.url,
            route: {
              path: picker.url,
              [modalType]: modalParams
            }
          });
        } else {
          picker.modal = app[modalType].create(modalParams);
          picker.modal.open();
        }
      }
      close() {
        const picker = this;
        const {
          opened,
          inline
        } = picker;
        if (!opened) return;
        if (inline) {
          picker.onClose();
          picker.onClosed();
          return;
        }
        if (picker.params.routableModals && picker.view) {
          picker.view.router.back();
        } else {
          picker.modal.close();
        }
      }
      init() {
        const picker = this;
        picker.initInput();
        if (picker.inline) {
          picker.open();
          picker.emit('local::init pickerInit', picker);
          return;
        }
        if (!picker.initialized && picker.params.value) {
          picker.setValue(picker.params.value);
        }

        // Attach input Events
        if (picker.$inputEl) {
          picker.attachInputEvents();
        }
        if (picker.params.closeByOutsideClick) {
          picker.attachHtmlEvents();
        }
        picker.emit('local::init pickerInit', picker);
      }
      destroy() {
        const picker = this;
        if (picker.destroyed) return;
        const {
          $el
        } = picker;
        picker.emit('local::beforeDestroy pickerBeforeDestroy', picker);
        if ($el) $el.trigger('picker:beforedestroy');
        picker.close();

        // Detach Events
        if (picker.$inputEl) {
          picker.detachInputEvents();
        }
        if (picker.params.closeByOutsideClick) {
          picker.detachHtmlEvents();
        }
        if ($el && $el.length) delete picker.$el[0].f7Picker;
        deleteProps$1(picker);
        picker.destroyed = true;
      }
    };

    var Picker = {
      name: 'picker',
      static: {
        Picker: Picker$1
      },
      create() {
        const app = this;
        app.picker = ConstructorMethods({
          defaultSelector: '.picker',
          constructor: Picker$1,
          app,
          domProp: 'f7Picker'
        });
        app.picker.close = function close(el) {
          if (el === void 0) {
            el = '.picker';
          }
          const $el = $(el);
          if ($el.length === 0) return;
          const picker = $el[0].f7Picker;
          if (!picker || picker && !picker.opened) return;
          picker.close();
        };
      },
      params: {
        picker: {
          // Picker settings
          rotateEffect: false,
          freeMode: false,
          cols: [],
          // Common opener settings
          containerEl: null,
          openIn: 'auto',
          // or 'popover' or 'sheet'
          sheetPush: false,
          sheetSwipeToClose: undefined,
          backdrop: undefined,
          // uses Popover or Sheet defaults
          formatValue: null,
          inputEl: null,
          inputReadOnly: true,
          closeByOutsideClick: true,
          scrollToInput: true,
          scrollToEl: undefined,
          toolbar: true,
          toolbarCloseText: 'Done',
          cssClass: null,
          routableModals: false,
          view: null,
          url: 'select/',
          // Render functions
          renderToolbar: null,
          render: null
        }
      }
    };

    const InfiniteScroll = {
      handle(el, e) {
        const app = this;
        const $el = $(el);
        const scrollTop = $el[0].scrollTop;
        const scrollHeight = $el[0].scrollHeight;
        const height = $el[0].offsetHeight;
        let distance = $el[0].getAttribute('data-infinite-distance');
        const virtualListContainer = $el.find('.virtual-list');
        let virtualList;
        const onTop = $el.hasClass('infinite-scroll-top');
        if (!distance) distance = 50;
        if (typeof distance === 'string' && distance.indexOf('%') >= 0) {
          distance = parseInt(distance, 10) / 100 * height;
        }
        if (distance > height) distance = height;
        if (onTop) {
          if (scrollTop < distance) {
            $el.trigger('infinite', e);
            app.emit('infinite', $el[0], e);
          }
        } else if (scrollTop + height >= scrollHeight - distance) {
          if (virtualListContainer.length > 0) {
            virtualList = virtualListContainer.eq(-1)[0].f7VirtualList;
            if (virtualList && !virtualList.reachEnd && !virtualList.params.updatableScroll) {
              return;
            }
          }
          $el.trigger('infinite', e);
          app.emit('infinite', $el[0], e);
        }
      },
      create(el) {
        const $el = $(el);
        const app = this;
        function scrollHandler(e) {
          app.infiniteScroll.handle(this, e);
        }
        $el.each(element => {
          element.f7InfiniteScrollHandler = scrollHandler;
          element.addEventListener('scroll', element.f7InfiniteScrollHandler);
        });
      },
      destroy(el) {
        const $el = $(el);
        $el.each(element => {
          element.removeEventListener('scroll', element.f7InfiniteScrollHandler);
          delete element.f7InfiniteScrollHandler;
        });
      }
    };
    var InfiniteScroll$1 = {
      name: 'infiniteScroll',
      create() {
        const app = this;
        bindMethods(app, {
          infiniteScroll: InfiniteScroll
        });
      },
 