"use strict";
/*
  Copyright 2018 Google LLC

  Use of this source code is governed by an MIT-style
  license that can be found in the LICENSE file or at
  https://opensource.org/licenses/MIT.
*/
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.askRootOfWebApp = void 0;
const assert_1 = __importDefault(require("assert"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const glob_1 = __importDefault(require("glob"));
const inquirer_1 = __importDefault(require("inquirer"));
const common_tags_1 = require("common-tags");
const errors_1 = require("../errors");
const constants_1 = require("../constants");
const ROOT_PROMPT = 'Please enter the path to the root of your web app:';
// The keys used for the questions/answers.
const questionRootDirectory = 'globDirectory';
const questionManualInput = 'manualDirectoryInput';
/**
 * @return {Promise<Array<string>>} The subdirectories of the current
 * working directory, with hidden and ignored ones filtered out.
 */
async function getSubdirectories() {
    return await new Promise((resolve, reject) => {
        (0, glob_1.default)('*/', {
            ignore: constants_1.constants.ignoredDirectories.map((directory) => `${directory}/`),
        }, (error, directories) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(directories);
            }
        });
    });
}
/**
 * @return {Promise<Object>} The answers from inquirer.
 */
async function askQuestion() {
    const subdirectories = await getSubdirectories();
    if (subdirectories.length > 0) {
        const manualEntryChoice = 'Manually enter path';
        return inquirer_1.default.prompt([
            {
                name: questionRootDirectory,
                type: 'list',
                message: (0, common_tags_1.oneLine) `What is the root of your web app (i.e. which directory do
        you deploy)?`,
                choices: subdirectories.concat([
                    new inquirer_1.default.Separator(),
                    manualEntryChoice,
                ]),
            },
            {
                name: questionManualInput,
                when: (answers) => answers.globDirectory === manualEntryChoice,
                message: ROOT_PROMPT,
            },
        ]);
    }
    return inquirer_1.default.prompt([
        {
            name: questionRootDirectory,
            message: ROOT_PROMPT,
            default: '.',
        },
    ]);
}
async function askRootOfWebApp() {
    const { manualDirectoryInput, globDirectory } = await askQuestion();
    try {
        const stat = await fs_extra_1.default.stat(manualDirectoryInput || globDirectory);
        (0, assert_1.default)(stat.isDirectory());
    }
    catch (error) {
        throw new Error(errors_1.errors['glob-directory-invalid']);
    }
    return manualDirectoryInput || globDirectory;
}
exports.askRootOfWebApp = askRootOfWebApp;
